/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.webservice;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Iterator;
import java.util.logging.*;
import java.lang.reflect.Method;
import java.rmi.UnmarshalException;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;


import javax.xml.namespace.QName;
import javax.xml.ws.handler.soap.SOAPHandler;

import javax.xml.ws.handler.Handler;
import javax.xml.ws.handler.HandlerResolver;
import javax.xml.ws.handler.PortInfo;
import javax.xml.ws.Service;

import javax.security.auth.Subject;

import com.sun.enterprise.security.jauth.*;
import com.sun.enterprise.security.SecurityContext;
import com.sun.enterprise.security.wss.WebServiceSecurity;

import com.sun.xml.ws.spi.runtime.Invoker;
import com.sun.xml.ws.spi.runtime.MessageContext;
import com.sun.xml.ws.spi.runtime.SOAPMessageContext;
import com.sun.xml.ws.spi.runtime.SystemHandlerDelegate;
import com.sun.xml.ws.spi.runtime.SystemHandlerDelegateFactory;
import com.sun.xml.ws.handler.SHDSOAPMessageContext;

import com.sun.enterprise.webservice.monitoring.EndpointImpl;
import com.sun.enterprise.webservice.monitoring.JAXWSEndpointImpl;

import com.sun.enterprise.deployment.runtime.common.MessageSecurityBindingDescriptor;
import com.sun.enterprise.deployment.ServiceReferenceDescriptor;
import com.sun.enterprise.deployment.ServiceRefPortInfo;
import com.sun.enterprise.deployment.WebServiceEndpoint;

import com.sun.enterprise.Switch;
import com.sun.ejb.Container;
import com.sun.ejb.Invocation;
import com.sun.enterprise.InvocationManager;

import com.sun.logging.*;
import com.sun.enterprise.util.i18n.StringManager;
/**
 * A SystemHandlerDelegate is used to inject system level functionality into a 
 * message processing runtime. The methods of this interface are invoked by 
 * the client and enpoint message dispatchers of the message processing
 * runtime.
 *
 * @author WS Development Team
 */
public class JAXWSSystemHandlerDelegateFactory 
    extends SystemHandlerDelegateFactory {

    static Logger _logger = LogDomains.getLogger(LogDomains.SECURITY_LOGGER);
   
    static final String CLIENT_AUTH_CONTEXT = 
	"com.sun.enterprise.security.jauth.ClientAuthContext";

    static final String SERVER_AUTH_CONTEXT = 
        "com.sun.enterprise.security.jauth.ServerAuthContext";

    static final Object factory = SystemHandlerDelegateFactory.getFactory();

    // resources...
    static StringManager localStrings = 
        StringManager.getManager(JAXWSSystemHandlerDelegateFactory.class);

    static Map<QName,ClientDelegate> serviceMap = 
        new HashMap<QName,ClientDelegate>();

    private static ClientDelegateHelper helper = new ClientDelegateHelper();

    // used by SOAPBindingImpl to inject a delegate in the binding.
    // we inject the same helper delegate in all SOAPbiningImpl objects,
    // and when invoked, the helper multiplexes to the correct ClientDelegate
    // (with embedded service specific policy).

    public SystemHandlerDelegate create() {
	return helper;
    }

    public boolean isEnabled(MessageContext context) {
	boolean rvalue = (factory != null);
	if (rvalue) {
	    rvalue = (helper.getClientDelegate(context) != null);
	}
	return rvalue;
    }

    private JAXWSSystemHandlerDelegateFactory() {
    }

    public static SystemHandlerDelegate 
	getEjbDelegate(ServerAuthConfig authConfig, 
		       WebServiceEndpoint webServiceEndpoint,
		       JAXWSEndpointImpl endpoint) throws RuntimeException {

	try {
	    
	    // login-config takes precedent over message security binding
	    if (webServiceEndpoint != null && 
		webServiceEndpoint.hasAuthMethod()) {
		authConfig = null;
	    } 

	    if (authConfig != null) {
		return new EjbDelegate(authConfig,endpoint);
	    }

	} catch (Exception e) {
	    _logger.log(Level.SEVERE,"ws.error_server_config",e);
	    if (e instanceof RuntimeException) {
		throw (RuntimeException) e;
	    } else {
		throw new RuntimeException(e);
	    }
	}
	return null;
    }

    public static SystemHandlerDelegate 
	getEjbDelegate(WebServiceEndpoint webServiceEndpoint,
		       JAXWSEndpointImpl endpoint) throws RuntimeException {

	MessageSecurityBindingDescriptor binding = null;
	if (webServiceEndpoint != null) {
	    binding = webServiceEndpoint.getMessageSecurityBinding();
	}

	try {

	    ServerAuthConfig authConfig = ServerAuthConfig.getConfig
		(com.sun.enterprise.security.jauth.AuthConfig.SOAP,
		 binding,WSSCallbackHandler.getInstance());
		
	    return JAXWSSystemHandlerDelegateFactory.getEjbDelegate
		(authConfig,webServiceEndpoint,endpoint);

	} catch (Exception e) {
	     _logger.log(Level.SEVERE,"ws.error_server_config",e);
	    if (e instanceof RuntimeException) {
		throw (RuntimeException) e;
	    } else {
		throw new RuntimeException(e);
	    }
	}
    }

    public static SystemHandlerDelegate 
        getServletDelegate(WebServiceEndpoint webServiceEndpoint) 
	throws Exception {

	MessageSecurityBindingDescriptor binding = null;
	if (webServiceEndpoint != null) {
	    binding = webServiceEndpoint.getMessageSecurityBinding();
	}

	try {
	    
	    ServerAuthConfig authConfig;

	    // login-config takes precedent over message security binding,
	    // although login-config should not be specified in servlet 
	    // WebServiceEndpoint
	    if (webServiceEndpoint != null && 
		webServiceEndpoint.hasAuthMethod()) {

		authConfig = null;

	    } else {

		authConfig = ServerAuthConfig.getConfig
		    (com.sun.enterprise.security.jauth.AuthConfig.SOAP,
		     binding,WSSCallbackHandler.getInstance());

	    }

	    if (authConfig != null) {
		return new ServletDelegate(authConfig);
	    }

	} catch (Exception e) {
	     _logger.log(Level.SEVERE,"ws.error_server_config",e);
	    throw e;
	}
	return null;
    }

    public static void
	configureClientDelegate(Service service,
				ServiceReferenceDescriptor desc) {

	boolean hasConfig = false;

	Map<QName,ClientAuthConfig> portMap = 
	    new HashMap<QName,ClientAuthConfig>();

	MessageSecurityBindingDescriptor binding = null;

	Iterator iter = desc.getPortsInfo().iterator();

	// for each declared port, determine the applicable 
	// message-security-binding

	ClientAuthConfig authConfig;

	try {

	    while (iter.hasNext()) {
 
	        ServiceRefPortInfo portInfo = 
		    (ServiceRefPortInfo) iter.next();

		binding = portInfo.getMessageSecurityBinding();

		// use portInfo and port in CallbackHandler construction

		if (binding != null) {

		    authConfig = ClientAuthConfig.getConfig
			(com.sun.enterprise.security.jauth.AuthConfig.SOAP,
			binding,WSSCallbackHandler.getInstance());

		    if (portInfo.hasWsdlPort()) {
			   portMap.put(portInfo.getWsdlPort(),authConfig);
		    } else {
			throw new RuntimeException
			    ("Service: " + service.getServiceName() + 
			    " portName (i.e. wsdl-Port) not defined");
		    }
		}
	    }
	 
	    // determine the message-security-binding that would apply 
	    // by default to ports without an explicit binding

	    authConfig = ClientAuthConfig.getConfig
		(com.sun.enterprise.security.jauth.AuthConfig.SOAP,
		 null,WSSCallbackHandler.getInstance());

	    if (!portMap.isEmpty() || authConfig != null) {
		portMap.put((QName)null,authConfig);
	    } 

	    serviceMap.put(service.getServiceName(),
			   new ClientDelegate(portMap));
	
	} catch (Exception e) {
	    _logger.log(Level.SEVERE,"ws.error_client_config",e);
	    if (e instanceof RuntimeException) {
		throw (RuntimeException) e;
	    } else {
		throw new RuntimeException(e);
	    }
	} 
    }

    public static void removeClientDelegate(QName serviceName) {
	serviceMap.remove(serviceName);
    }

    static boolean portNamesMatch(QName pN1, QName pN2) {
	boolean rvalue = false;
	if (pN1 == pN2) {
	    rvalue = true;
	} else if (pN1 != null && pN2 != null && pN1.equals(pN2)) {
	    rvalue = true;
	}
	return rvalue;
    }

    static SOAPHandler getClientHandler
        (ServiceReferenceDescriptor descriptor, QName portName) {

	SOAPHandler rvalue = null;

	MessageSecurityBindingDescriptor binding = null;
	 
	ClientAuthConfig authConfig = null;

	Iterator iter = descriptor.getPortsInfo().iterator();

	// if portinfo(s) are declared try to find the one that matches
	// the portName

	while (iter.hasNext()) {
		
	    ServiceRefPortInfo portInfo = (ServiceRefPortInfo) iter.next();
		
	    QName pN = portInfo.getWsdlPort();

	    if (pN != null && portNamesMatch(portName,pN)) {

	        binding = portInfo.getMessageSecurityBinding();

	        break;
	    }
	}
	 
	try {

	    authConfig = ClientAuthConfig.getConfig
		(com.sun.enterprise.security.jauth.AuthConfig.SOAP,
		 binding,WSSCallbackHandler.getInstance());

	} catch (Exception e) { 
	    _logger.log(Level.SEVERE,"ws.error_client_config",e);
	    authConfig = null;
	}

	if (authConfig != null) {
	    rvalue = new ClientHandler(authConfig);
	}
	
	return rvalue;
    }

    public static HandlerResolverImpl 
	getHandlerResolverImpl(Service service,
			       ServiceReferenceDescriptor descriptor) {
	if (factory != null && 
	    (factory instanceof JAXWSSystemHandlerDelegateFactory)) {
	    // If security is configured for some port of the service,
	    // then a ClientDelegate (i.e., a SystemHandlerDelegate) would 
	    // be constructed and "associated" with the service,
	    // or with ports of the service. 
	    configureClientDelegate(service,descriptor);
	    return null;
	} else {
	    return new WSSHandlerResolver(descriptor);
	}
    }
    
    static class ServletDelegate implements SystemHandlerDelegate {

	BaseAuthConfig authConfig;

	ServletDelegate(BaseAuthConfig authConfig) {
	    this.authConfig = authConfig;
	}

	/**
	 * base implementation - designed for servlet endpoint
	 * 
	 * Called by both client and endpoint message dispatchers to activate
	 * injected request message processing. When called by a client side 
	 * message dispatcher, this method must be 
	 * called just before the message (associated with the MessageContext) 
	 * is sent. When called by the message dispatcher at an endpoint, this 
	 * method must be called before MustUnderstand processing on the 
	 * associated message.
	 *
	 * @param context when called for a SOAP binding, the argument
	 * must be an instanceof com.sun.xml.ws.spi.runtime.SOAPMessageContext,
	 * and when called for a SOAP binding at an endpoint, the Invoker (on 
	 * the context) must be available for use by the delegate. 
	 *
	 * @return true if processing by the delegate was such that the caller
	 * should continue with its normal message processing. Returns false 
	 * when the delegate has established, in the MessageContext, 
	 * the response message to be sent. When this method returns false,
	 * the calling message dispatcher must return the response message 
	 * without performing MustUnderstand processing and without invoking 
	 * the endpoint. Only delegates called by endpoint side message 
	 * may return false
	 *
	 * @throws java.lang.Exception when the processing by the delegate 
	 * failed without yielding a response message; in which case, the 
	 * caller shall determine how to process the error.
	 */ 
	public boolean processRequest(MessageContext context) 
	    throws Exception {

		assert context instanceof SOAPMessageContext;

		final SOAPMessageContext soapMC = 
		    (SOAPMessageContext) context;

		boolean status = true;
	
		final ServerAuthContext sAC = 
		    (this.authConfig == null ? null : 
		     ((ServerAuthConfig)this.authConfig).
		     getAuthContext(soapMC));

		try {

		    if (sAC != null) {

			status = false;
			// proceed to process message sescurity
			status= WebServiceSecurity.validateRequest(soapMC,sAC);

			if (status) {

			    soapMC.put( SERVER_AUTH_CONTEXT , sAC);
			    
			    soapMC.setScope
				(SERVER_AUTH_CONTEXT,
				 MessageContext.Scope.HANDLER);

			}
		    }

		} catch (Exception e) {
		    if (e instanceof AuthException) {
		        _logger.log(Level.INFO,"ws.error_validate_request", e);
		    } else {
		       _logger.log(Level.SEVERE,"ws.error_validate_request",e);
		    }

		    throw e;

		} finally {
		    WebServiceSecurity.auditInvocation(soapMC,status);
		}

		if (status) {

		    // only do doAsPriv if SecurityManager in effect.

		    if (System.getSecurityManager() != null && 
			soapMC instanceof SHDSOAPMessageContext) {

			// on this branch, the endpoint invocation and the 
			// processing of the response will be initiated from
			// within the system handler delegate. delegate returns
			// false so that dispatcher will not invoke the 
			// endpoint.

			status = false;

			try {

			    Subject.doAsPrivileged
				(SecurityContext.getCurrent().getSubject(),
				 new PrivilegedExceptionAction() {
				    public Object run() throws Exception {
					Invoker invoker = 
					    ((SHDSOAPMessageContext)soapMC).
					    getInvoker();
					invoker.invoke();
					processResponse(soapMC);
					return null;
				    }
				}, null);

			} catch (PrivilegedActionException pae) {
			    Throwable cause = pae.getCause();
			    if (cause instanceof AuthException){
				_logger.log(Level.SEVERE, 
					    "ws.error_secure_response", cause);
			    }
			    Exception e = null;
			    if (cause instanceof Exception) {
				e = (Exception) cause;
			    } else {
				e = new Exception(cause);
			    }
			    throw e;
			} 
		    }
		}
		return status;
	}
    
	/**
	 * base implementation - generic endpoint
	 * 
	 * Called by both client and endpoint message dispatchers to activate
	 * injected response message processing. When called by the message 
	 * dispatcher at the client, this method must 
	 * be called before MustUnderstand processing on the received message 
	 * (associated with the MessageContext). When called by the message 
	 * dispatcher at an endpoint, this method must be called after the 
	 * endpoint has been invoked, and just before the associated response 
	 * message is sent. In the special case where invocation of the 
	 * endpoint
	 * caused an Exception to be thrown, this method must not be called.
	 *
	 * @param context when called for a SOAP binding the argument
	 * must be an instanceof com.sun.xml.ws.spi.runtime.SOAPMessageContext.
	 *
	 * @throws java.lang.Exception when the processing by the delegate 
	 * failed. In this case, the caller must not send the response message 
	 * but shall 
	 * otherwise determine how to process the error.
	 */
	public void processResponse(MessageContext context) 
	    throws Exception {
        
		assert context instanceof SOAPMessageContext;

		SOAPMessageContext soapMC= (SOAPMessageContext) context;

		ServerAuthContext sAC = 
		    (ServerAuthContext) soapMC.get( SERVER_AUTH_CONTEXT );

		if (sAC == null) {
		    return;
		}

		try {

		    WebServiceSecurity.secureResponse(soapMC,sAC);

		} catch (Exception e) {

		    if (e instanceof AuthException) {
			_logger.log(Level.INFO, "ws.error_secure_response", e);
		    } else {
			_logger.log(Level.SEVERE,"ws.error_secure_response",e);
		    }
		    
		    throw e;
		}
	}

              
	/**
	 * base implementation - generic 
	 *
	 * This method must be called by an endpoint message dispatcher after 
	 * MustUnderstand processing and before endpoint invocation.
	 *
	 * @param context when called for a SOAP binding the argument
	 * must be an instanceof com.sun.xml.ws.spi.runtime.SOAPMessageContext.
	 */
	public void preInvokeEndpointHook(MessageContext context) {
	    // do nothing
	}
    }

    // this delegate specializes methods of ServletDelagate 

    static class EjbDelegate extends ServletDelegate {

	private static WsUtil wsUtil = new WsUtil();
    
	JAXWSEndpointImpl endpoint;

	EjbDelegate(BaseAuthConfig authConfig, 
				 JAXWSEndpointImpl endpoint) {
	    super(authConfig);
	    this.endpoint = endpoint;
	}

	/**
	 * performs ejb authorization check
	 */
	public boolean processRequest(MessageContext context) 
	    throws Exception {

	    assert context instanceof SOAPMessageContext;

	    SOAPMessageContext soapMC = (SOAPMessageContext) context;

	    boolean status = true;

	    ServerAuthContext sAC = 
		(this.authConfig == null ? null : 
		 ((ServerAuthConfig)this.authConfig).getAuthContext(soapMC));
	
	    try {
		
		if (sAC != null) {

		    status = false;
		    // proceed to process message sescurity
		    status = WebServiceSecurity.validateRequest(soapMC,sAC);

		    if (status) {

			soapMC.put( SERVER_AUTH_CONTEXT , sAC);

			soapMC.setScope
			    (SERVER_AUTH_CONTEXT,MessageContext.Scope.HANDLER);

		    }
		}

	    } catch (Exception e) {
		if (e instanceof AuthException) {
		    _logger.log(Level.INFO, "ws.error_validate_request", e);
		} else {
		    _logger.log(Level.SEVERE, "ws.error_validate_request", e);
		}

		throw e;

	    } finally {
		WebServiceSecurity.auditInvocation(soapMC,status);
	    }
	
	    if (status) {

		Switch theSwitch = Switch.getSwitch();
		InvocationManager invManager= theSwitch.getInvocationManager();
		Invocation inv= (Invocation) invManager.getCurrentInvocation();
            
		Method m = BaseAuthConfig.getMethod(soapMC);

		Exception ie = null;
		if (m!=null) {

		    Container container = (Container) inv.container;
		    
		    try {
			inv.method = m;


			if ( !container.authorize(inv) ) {
			    ie = new Exception
				("Client not authorized for invocation of " 
                                 + inv.method);
			} else {
			    // Record the method on which the successful
			    // authorization check was performed. 
			    inv.setWebServiceMethod(inv.method);
			}
		    } catch(Exception e) {
			String errorMsg= "Error unmarshalling method for ejb " 
                            + (endpoint != null ?
			       endpoint.getDescriptor().
			       getEjbComponentImpl().getName() :
			       "unable to determine EjbComponentImp.name");
			ie = new UnmarshalException(errorMsg); 
			ie.initCause(e);
		    } 

		} else {
		    inv.setWebServiceMethod(null);
		}
        
		if( ie != null ) {
		    inv.exception = ie;
		    throw ie;
		} 
	    }

	    return status;
	}

	/**
	 * ensure that authorized method is not changed by jaxrpc handlers
	 */
	public void preInvokeEndpointHook(MessageContext context) {
       
	    assert context instanceof SOAPMessageContext;

	    SOAPMessageContext soapMC = (SOAPMessageContext) context;

	    if (endpoint.getDescriptor().hasHandlerChain()) {
		// some handlers are defined. 
		Switch theSwitch = Switch.getSwitch();
		InvocationManager invManager= theSwitch.getInvocationManager();
		Invocation inv= (Invocation) invManager.getCurrentInvocation();
		Container container = (Container) inv.container;
            
		Method m = BaseAuthConfig.getMethod(soapMC);
		
		if (m!=null) {
		    try {                    
			Method webServiceMethodInPreHandler = 
			    inv.getWebServiceMethod();
                    
			if( webServiceMethodInPreHandler != null ) {
			    // Now that application handlers have run, do 
			    // another method lookup and compare the results 
			    // with the original one. This ensures that the 
			    // application handlers have not changed the
			    // message context in any way that would impact 
			    // which method is invoked.
			    if( !webServiceMethodInPreHandler.equals(m) ) {
				inv.exception = new UnmarshalException
                                    ("Original method " + 
				     webServiceMethodInPreHandler +
				    " does not match post-handler method "+ m);
			    }
			}
		    } catch(Exception e) {
			String errorMsg = "Exception while getting method for "
			    + ((inv != null ) ?
			     ((Container) inv.container).getEjbDescriptor().
			     getName() : "");
			inv.exception = new UnmarshalException(errorMsg);
			inv.exception.initCause(e);
		    }
                
		    if( inv.exception != null ) {
			WsUtil.getDefaultLogger().log
			    (Level.WARNING,"postEjbHandlerError", 
			     inv.exception);

			wsUtil.throwSOAPFaultException
			    (inv.exception.getMessage(),context);
		    }
		}
	    }
	}
    }

    // this delegate is used on the client side, and should eventually be
    // injected by the SOAPBindingImpl.
    //this delegate multiplexes the processing to the correct ClientDelegate.

    static class ClientDelegateHelper implements SystemHandlerDelegate {

	ClientDelegateHelper() {
	}

	static ClientDelegate getClientDelegate(MessageContext context) {

	    ClientDelegate rvalue = null;
	    QName serviceName = 
		(QName)context.get(MessageContext.WSDL_SERVICE);
	    if (serviceName != null) {
		rvalue = serviceMap.get(serviceName);
	    }

	    // the helper should only be included in the binding
	    // when message security is configured for the service;
	    // in which case we should always find a CLientAuthConfig in
	    // the service map (if the helper is invoked)

	    if (rvalue == null) {
		throw new RuntimeException
		    (serviceName == null ? 
		     "no WSDL_SERVICE in MessageContext": 
		     "no ClientDelegate for service" + serviceName);
	    }
	    return rvalue;
	}

	/**
	 * does client side processing
	 */
	public boolean processRequest(MessageContext context) 
	    throws Exception {
		return getClientDelegate(context).processRequest(context);
	}

	/**
	 * does client side processing
	 */
	public void processResponse(MessageContext context) throws 
	    Exception {
		getClientDelegate(context).processResponse(context);
	}    

	// no client side tracing
	public void preInvokeEndpointHook(MessageContext context) {
	    try {
		getClientDelegate(context).preInvokeEndpointHook(context);
	    } catch (Exception e) {
		// extinguish exception but if exception would occur
		// here, it would have alredy occured on processRequest or
		// ProcessResponse (and this methid would not be called).
	    }
	}
    }

    // this delegate is used on the client side, and should eventually be
    // injected by the SOAPBindingImpl. 
    // This delegate is called by the ClientHelper delegate. 

    static class ClientDelegate implements SystemHandlerDelegate {

	private boolean isAppclientContainer;
	Map<QName,ClientAuthConfig> portMap;

	ClientDelegate(Map<QName,ClientAuthConfig> portMap) {

	    this.portMap = portMap;

	    int containerType = Switch.getSwitch().getContainerType();
	    if (containerType == Switch.APPCLIENT_CONTAINER) {
		isAppclientContainer = true;
	    } else {
		isAppclientContainer = false;
	    }
	}

	ClientAuthConfig getAuthConfig(MessageContext context) {

	    ClientAuthConfig rvalue = null;

	    QName portName = (QName)context.get(MessageContext.WSDL_PORT);
	    if (portName != null) {
		if (portMap.containsKey(portName)) {
		    rvalue = (ClientAuthConfig) portMap.get(portName);
		} else if (portMap.containsKey(null)) {
		    rvalue = (ClientAuthConfig) portMap.get((QName)null);
		}
	    } else if (portMap.containsKey((QName)null)) {
		rvalue = (ClientAuthConfig) portMap.get((QName)null);
	    }
	    return rvalue;
	}

	/**
	 * does client side request processing
	 */
	public boolean processRequest(MessageContext context) 
	    throws Exception {

	    assert context instanceof SOAPMessageContext;

	    boolean status = true;
	
	    SOAPMessageContext soapMC= (SOAPMessageContext) context;

	    ClientAuthConfig config = this.getAuthConfig(context);
 
	    final ClientAuthContext cAC = 
		(config == null ? null : config.getAuthContext(soapMC));

	    if (cAC != null) {
	    
		try {

		    status = false;
		    // proceed to process message sescurity
		    WebServiceSecurity.secureRequest(soapMC,cAC,
						     isAppclientContainer);
		    
		    soapMC.put( CLIENT_AUTH_CONTEXT , cAC);

		    soapMC.setScope
			(CLIENT_AUTH_CONTEXT,MessageContext.Scope.HANDLER);
		    
		    status = true;
		    
		} catch (Exception e) {
		    if (e instanceof AuthException) {
			_logger.log(Level.INFO, "ws.error_secure_request", e);
		    } else {
			_logger.log(Level.SEVERE, "ws.error_secure_request",e);
		    }
		    
		    throw e;
		} 
	    }

	    return status;
	}

	/**
	 * does client side response processing
	 */
	public void processResponse(MessageContext context) throws 
	    Exception {

	    assert context instanceof SOAPMessageContext;

	    SOAPMessageContext soapMC = (SOAPMessageContext) context;

	    ClientAuthContext cAC = 
		(ClientAuthContext) soapMC.get( CLIENT_AUTH_CONTEXT );

	    if (cAC == null) {
		return;
	    }


	    try {
		boolean retValue = 
		    WebServiceSecurity.validateResponse(soapMC,cAC);
	    } catch(Exception e) {
		if (e instanceof AuthException) {
		    _logger.log(Level.INFO, "ws.error_validate_response", e);
		} else {
		    _logger.log(Level.SEVERE, "ws.error_validate_response", e);
		}
		
		throw e;
	    }
	}    

	// no client side tracing
	public void preInvokeEndpointHook(MessageContext context) {
	}

    }

    // this following client handler is provided as an alternatve to
    // SHD injection.

    static class ClientHandler implements 
    SOAPHandler<javax.xml.ws.handler.soap.SOAPMessageContext> {

	ClientAuthConfig config;
	private boolean isAppclientContainer;

	ClientHandler(ClientAuthConfig config) {

	    this.config = config;

	    int containerType = Switch.getSwitch().getContainerType();
	    if (containerType == Switch.APPCLIENT_CONTAINER) {
		isAppclientContainer = true;
	    } else {
		isAppclientContainer = false;
	    }
	}

	@PostConstruct
	public void initMe() {
	    // just here for debugging
   	}

	@PreDestroy
	public void destroyMe() {
	    config = null;
	}

	public void close(javax.xml.ws.handler.MessageContext context) {
	}

	public boolean handleFault
	(javax.xml.ws.handler.soap.SOAPMessageContext context) {
	    return true;
	}

	public boolean handleMessage
	(javax.xml.ws.handler.soap.SOAPMessageContext context) {

	    assert context instanceof 
		com.sun.xml.ws.spi.runtime.SOAPMessageContext;

	    boolean rvalue = true;

	    boolean msgOut =
		((Boolean) context.get
		 (javax.xml.ws.handler.
		  MessageContext.MESSAGE_OUTBOUND_PROPERTY)).booleanValue();

	    try {

		if (msgOut) {

		    ClientAuthContext cAC = config.getAuthContext(context);

		    if (cAC != null) {
		
			WebServiceSecurity.secureRequest
			    (context,cAC,isAppclientContainer);

			context.put
			    (JAXWSSystemHandlerDelegateFactory.
			     CLIENT_AUTH_CONTEXT,cAC);

			context.setScope
			    (JAXWSSystemHandlerDelegateFactory.
			     CLIENT_AUTH_CONTEXT,MessageContext.Scope.HANDLER);

		    }
		} else {

		    ClientAuthContext cAC = (ClientAuthContext) context.get
			(JAXWSSystemHandlerDelegateFactory.
			 CLIENT_AUTH_CONTEXT);

		    if (cAC != null) {
			rvalue = WebServiceSecurity.validateResponse
			    (context,cAC);
		    }
		}

	    } catch(Exception e) {
		Level level = (e instanceof AuthException ? 
			       Level.INFO : 
			       Level.SEVERE);
		String msg = (msgOut ? 
			      "wss.container_auth_exception" : 
			      "wss.validate_dispose_failed");
		
		_logger.log(level,msg,e);

		if (e instanceof RuntimeException) {
		    throw (RuntimeException) e;
		} else {
		    throw new RuntimeException(e);
		} 
	    }
	    return rvalue;
	}

	public java.util.Set<QName> getHeaders() {
	    
	    HashSet<QName> headers = new HashSet<QName>();

	    QName qNames[] = config.getMechanisms();
	    for (int i = 0; i < qNames.length; i++) {
		headers.add(qNames[i]);
	    }

	    return headers;
	}
    }

    /**
     * This specialized resolver injects a security handler in every 
     * handler chain it returns.
     */
    static class WSSHandlerResolver extends HandlerResolverImpl {
    
	private Map<PortInfo, List<Handler>> chainMap;

	ServiceReferenceDescriptor d;
    
	public WSSHandlerResolver(ServiceReferenceDescriptor descriptor){
	    chainMap = new HashMap<PortInfo, List<Handler>>();
	    d = descriptor;
	}

	public List<Handler> getHandlerChain(PortInfo info) {
	    List<Handler> chain = null;
	    if (chainMap.containsKey(info)) {
		chain = chainMap.get(info);
	    }
	    else {
		chain = new ArrayList<Handler>();
		this.setHandlerChain(info,chain);
	    }
	    return chain;
	}
    
	public void setHandlerChain(PortInfo info, List<Handler> chain) {

            // If Handlers are already configured for this port, then no need to add
            // the security handler again - just add this list of handlers
            List<Handler> currentList = chainMap.get(info);
            if(currentList != null && !currentList.isEmpty()) {
                currentList.addAll(chain);
                return;
            }

	    Handler secHandler  = null;
	    if (d != null) {
		secHandler = JAXWSSystemHandlerDelegateFactory.getClientHandler
		    (d,info.getPortName());
	    }

	    if (chain == null) {
		chain = new ArrayList<Handler>();
	    }

	    if (secHandler != null) {
		chain.add(secHandler);
	    }

	    chainMap.put(info, chain);
	}
    }
    
}

 

