/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.web;

import com.sun.enterprise.web.stats.PWCRequestStatsImpl;
import java.io.File;
import java.text.MessageFormat;
import java.util.Iterator;
import java.util.List;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.Vector;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.catalina.Container;
import org.apache.catalina.LifecycleException;
import org.apache.catalina.core.StandardHost;

import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.serverbeans.Applications;
import com.sun.enterprise.config.serverbeans.J2eeApplication;
import com.sun.enterprise.config.serverbeans.HttpService;
import com.sun.enterprise.config.serverbeans.Server;
import com.sun.enterprise.config.serverbeans.ApplicationRef;
//import com.sun.enterprise.config.serverbeans.VirtualServerClass;
import com.sun.enterprise.config.serverbeans.WebModule;
import com.sun.enterprise.config.serverbeans.ServerBeansFactory;
import com.sun.enterprise.instance.WebModulesManager;
import com.sun.enterprise.server.ApplicationRegistry;
import com.sun.enterprise.util.StringUtils;
import com.sun.enterprise.util.io.FileUtils;

import com.sun.enterprise.server.ApplicationServer;

import com.sun.enterprise.Switch;
import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.WebBundleDescriptor;
import com.sun.enterprise.deployment.util.DOLLoadingContextFactory;
import com.sun.logging.LogDomains;

/**
 * Standard implementation of a virtual server (aka virtual host) in
 * the iPlanet Application Server.
 */

public class VirtualServer extends StandardHost {

    // ------------------------------------------------------------ Constructor

    /**
     * Default constructor that simply gets a handle to the web container 
     * subsystem's logger.
     */
    public VirtualServer() {
        super();

        setPipeline(new VirtualServerPipeline(this));

        // XXX logger initialization should probably be done in
        //     in LogServiceLifecycle.onInitialization()
        if (_logger == null) {
            _logger = LogDomains.getLogger(LogDomains.WEB_LOGGER);
            _rb = _logger.getResourceBundle();
        }
        _debug = _logger.isLoggable(Level.FINE);
    }

    // ----------------------------------------------------- Instance Variables

    /**
     * The id of this virtual server as specified in the configuration.
     */
    private String _id = null;

    /**
     * The logger to use for logging ALL web container related messages.
     */
    protected static Logger _logger = null;

    /**
     * The resource bundle containing the message strings for _logger.
     */
    protected static ResourceBundle _rb = null;

    /**
     * Indicates whether the logger level is set to any one of 
     * FINE/FINER/FINEST.
     *
     * This flag is used to avoid incurring a perf penalty by making
     * logging calls for debug messages when the logger level is
     * INFO or higher.
     */
    protected boolean _debug = false;

    /**
     * The descriptive information about this implementation.
     */
    private static final String _info =
        "com.sun.enterprise.web.VirtualServer/1.0";

    /**
     * The config bean associated with this VirtualServer
     */
    private com.sun.enterprise.config.serverbeans.VirtualServer vsBean;

    /**
     * The mime mapping associated with this VirtualServer
     */
    private MimeMap mimeMap;

    /*
     * Indicates whether symbolic links from this virtual server's docroot
     * are followed. This setting is inherited by all web modules deployed on
     * this virtual server, unless overridden by a web modules allowLinking
     * property in sun-web.xml.
     */
    private boolean allowLinking = true;
    
    /*
     * default-web.xml location 
     */
    private String defaultWebXmlLocation;


    private String[] cacheControls;


    // Is this virtual server active?
    private boolean isActive;

    
    /**
     * The Stats holder used by this virtual-server.
     */
    private PWCRequestStatsImpl pwcRequestStatsImpl;
    // ------------------------------------------------------------- Properties

    /**
     * Return the virtual server identifier.
     */
    public String getID() {
        return _id;
    }

    /**
     * Set the virtual server identifier string.
     *
     * @param id New identifier for this virtual server
     */
    public void setID(String id) {
        _id = id;
    }

    /**
     * @return true if this virtual server is active, false otherwise
     */
    public boolean isActive() {
        return isActive;
    }

    /**
     * Sets the state of this virtual server.
     *
     * @param isActive true if this virtual server is active, false otherwise
     */
    public void setIsActive(boolean isActive) {
        this.isActive = isActive;
    }

    /**
     * Gets the default-web.xml location of web modules deployed on this
     * virtual server.
     *
     * @return default-web.xml location of web modules deployed on this
     * virtual server
     */
    public String getDefaultWebXmlLocation() {
        return defaultWebXmlLocation;
    }

    /**
     * Sets the default-web.xml location for web modules deployed on this
     * virtual server.
     *
     * @param defaultWebXmlLocation default-web.xml location for web modules
     * deployed on this virtual server
     */
    public void setDefaultWebXmlLocation(String defaultWebXmlLocation) {
        this.defaultWebXmlLocation = defaultWebXmlLocation;
    }

    /**
     * Gets the value of the allowLinking property of this virtual server.
     *
     * @return true if symbolic links from this virtual server's docroot (as
     * well as symbolic links from archives of web modules deployed on this
     * virtual server) are followed, false otherwise
     */
    public boolean getAllowLinking() {
        return allowLinking;
    }

    /**
     * Sets the allowLinking property of this virtual server, which determines
     * whether symblic links from this virtual server's docroot are followed.
     *
     * This property is inherited by all web modules deployed on this virtual
     * server, unless overridden by the allowLinking property in a web module's
     * sun-web.xml.
     *
     * @param allowLinking Value of allowLinking property
     */
    public void setAllowLinking(boolean allowLinking) {
        this.allowLinking = allowLinking;
    }

    /**
     * Gets the config bean associated with this VirtualServer.
     */
    public com.sun.enterprise.config.serverbeans.VirtualServer getBean(){
        return vsBean;
    }

    /**
     * Sets the config bean for this VirtualServer
     */
     public void setBean(com.sun.enterprise.config.serverbeans.VirtualServer vsBean){
        this.vsBean = vsBean;
     }

    /**
     * Gets the mime map associated with this VirtualServer.
     */
    public MimeMap getMimeMap(){
        return mimeMap;
    }

    /**
     * Sets the mime map for this VirtualServer
     */
    public void setMimeMap(MimeMap mimeMap){
        this.mimeMap = mimeMap;
    }

    /**
     * Gets the Cache-Control configuration of this VirtualServer.
     *
     * @return Cache-Control configuration of this VirtualServer, or null if
     * no such configuration exists for this VirtualServer
     */
    public String[] getCacheControls() {
        return cacheControls;
    }

    /**
     * Sets the Cache-Control configuration for this VirtualServer
     *
     * @param cacheControls Cache-Control configuration settings for this
     * VirtualServer
     */
    public void setCacheControls(String[] cacheControls) {
        this.cacheControls = cacheControls;
    }


    // --------------------------------------------------------- Public Methods

    /**
     * Return descriptive information about this Container implementation and
     * the corresponding version number, in the format
     * <code>&lt;description&gt;/&lt;version&gt;</code>.
     */
    public String getInfo() {
        return _info;
    }


    // ------------------------------------------------------ Lifecycle Methods

    /**
     * Gracefully shut down active use of the public methods of this Component.
     *
     * @exception IllegalStateException if this component has not been started
     * @exception LifecycleException if this component detects a fatal error
     *  that needs to be reported
     */
    public synchronized void stop() throws LifecycleException {

        // Remove the descriptor bindings for all the web applications
        // in this virtual server
        Switch sw = Switch.getSwitch();
        Container children[] = findChildren();
        if (children != null) {
            for (int i = 0; i < children.length; i++) {
                sw.removeDescriptorFor(children[i]);
            }
        }
        super.stop();
    }

    // ------------------------------------------------------ Protected Methods

    /**
     * Return the list of enabled web-modules configured for this
     * virtual server.
     *
     * @return     The list of WebModuleConfig objects for all enabled
     *             web-modules hosted under the specified virtual server.
     */
    protected List getWebModules(Server serverBean, String modulesRoot) {

        List modules = new Vector();
        
        //ROB: config changes
        //Applications appsBean = serverBean.getApplications();
        Applications appsBean = null;
        try {
            appsBean = ServerBeansFactory.getApplicationsBean(serverBean.getConfigContext());
        } catch (ConfigException e) {
            String msg = _rb.getString("vs.appsConfigError");
            Object[] params = { getID() };
            msg = MessageFormat.format(msg, params);
            _logger.log(Level.SEVERE, msg, e);
        }

        if (appsBean != null) {
            WebModule[] wmBeans = appsBean.getWebModule();
            if (wmBeans != null && wmBeans.length > 0) {
                for (int i = 0; i < wmBeans.length; i++) {
                    WebModule wm = wmBeans[i];
                    if (isActive(wm)) {                      
                        // skips if the web module is not referenced by 
                        // this server
                        ApplicationRef ref = 
                            serverBean.getApplicationRefByRef(wm.getName());
                        if (ref == null) {
                            continue;
                        }                       

                        String location = wm.getLocation();
                        // If module root is relative then prefix it with the 
                        // location of where all the standalone modules for 
                        // this server instance are deployed
                        File moduleBase = new File(location);
                        if (!moduleBase.isAbsolute()) {
                            location = modulesRoot+File.separator+location;
                            wm.setLocation(location);
                        }
                        WebModuleConfig wmInfo = loadWebModuleConfig(wm);
                        if (wmInfo != null)
                            modules.add(wmInfo);
                    } else {
                        if (_debug) {
                            _logger.finer("Web Module [" + wm.getName() + 
                                          "] is not applicable for virtual " +
                                          " server [" + getID() + "]");
                        }
                    }
                }
            }
        }
        return modules;
    }

    /**
     * Loads and returns the configuration information associated with
     * the web module that the user/configuration has designated to be the 
     * default web module for this virtual server.
     *
     * The default web module for a virtual server is specified via the
     * 'default-web-module' attribute of the 'virtual-server' element in
     * server.xml. This is an optional attribute and if the configuration
     * does not specify another web module (standalone or part of a 
     * j2ee-application) that is configured at a context-root="", then
     * a default web module will be created and loaded. The value for this
     * attribute is either "${standalone-web-module-name}" or 
     * "${j2ee-app-name}:${web-module-uri}".
     *
     * @return null if the default-web-module has not been specified or
     *              if the web module specified either could not be found or
     *              is disabled or does not specify this virtual server (if
     *              it specifies a value for the virtual-servers attribute) or
     *              if there was an error loading its deployment descriptors.
     */
    protected WebModuleConfig getUserDefaultWebModuleConfig(Server serverBean) {

        WebModuleConfig wmInfo = null;
        Applications appsBean = null;
        try{
            appsBean = ServerBeansFactory.getApplicationsBean(serverBean.getConfigContext());
        } catch (ConfigException e) {
            String msg = _rb.getString("vs.appsConfigError");
            Object[] params = { getID() };
            msg = MessageFormat.format(msg, params);
            _logger.log(Level.SEVERE, msg, e);
        }

        String wmID = getDefaultWebModuleID();
        if (wmID != null) {
            // Check if the default-web-module is part of a
            // j2ee-application
            wmInfo = findWebModuleInJ2eeApp(appsBean, wmID);

            // Look up the list of standalone web modules
            if (wmInfo == null) {
                WebModule wm = appsBean.getWebModuleByName(wmID);
                if (wm != null) {
                    if (isActive(wm, false)) {
                        // Create a copy as we need to change the name
                        // and context root to indicate that this web module
                        // is to be loaded as the 'default' web module for
                        // the virtual server
                        // WebModule wmCopy = (WebModule) wm.clone();
                        WebModule wmCopy = new WebModule();
                        wmCopy.setName(wm.getName());
                        wmCopy.setLocation(wm.getLocation());
                        wmCopy.setEnabled(wm.isEnabled());
                        wmCopy.setContextRoot(wm.getContextRoot());
                        //wmCopy.setVirtualServers(_id);
                        wmInfo = loadWebModuleConfig(wmCopy);
                        wmInfo.setVirtualServers(_id);
                        wmCopy.setName(Constants.DEFAULT_WEB_MODULE_PREFIX +
                                       wmCopy.getName());
                        wmCopy.setContextRoot("");
                        wmCopy.setObjectType(wm.getObjectType());
                    } else {
                        Object[] params = { wmID, getID() };
                        _logger.log(Level.SEVERE, "vs.defaultWebModuleDisabled",
                                    params);
                        wm = null;
                    }
                }
            } else {
                WebModule wm = wmInfo.getBean();
                wmInfo.setVirtualServers(_id);
                wm.setName(Constants.DEFAULT_WEB_MODULE_PREFIX + wm.getName());
                wm.setContextRoot("");
            }

            if (wmInfo == null) {
                Object[] params = { wmID, getID() };
                _logger.log(Level.SEVERE, "vs.defaultWebModuleNotFound",
                            params);
            }
        }
        return wmInfo;
    }

    /**
     * If a default web module has not yet been configured and added to this
     * virtual server's list of web modules then return the configuration
     * information needed in order to create a default web module for this
     * virtual server.
     *
     * This method should be invoked only after all the standalone modules
     * and the modules within j2ee-application elements have been added to
     * this virtual server's list of modules (only then will one know whether
     * the user has already configured a default web module or not).
     */
    protected WebModuleConfig createSystemDefaultWebModuleIfNecessary() {

        WebModuleConfig wmInfo = null;
        //
        // Add a default context only if one hasn't already been loaded
        // and then too only if docroot is not null
        //
        String docroot = getAppBase();
        if ((findChild("") == null) && (docroot != null)) {
            wmInfo = new WebModuleConfig();
            WebModule wm = new WebModule();
            wm.setName(Constants.DEFAULT_WEB_MODULE_NAME);
            wm.setContextRoot("");
            wm.setLocation(docroot);
            wmInfo.setBean(wm);
            
            wmInfo.setDescriptor(DOLLoadingContextFactory.getDefaultWebBundleDescriptor());
 
            WebBundleDescriptor wbd = wmInfo.getDescriptor();
            if ( wbd.getApplication() == null ) {
                Application application = new Application();
                application.setVirtual(true);
                application.setName(Constants.DEFAULT_WEB_MODULE_NAME);
                wbd.setApplication(application);
            }
             
        }
        return wmInfo;
    }


    /**
     * Creates and returns an object that contains information about
     * the web module's configuration such as the information specified
     * in server.xml, the deployment descriptor objects etc.
     *
     * @return null if an error occured while reading/parsing the 
     *              deployment descriptors.
     */
    protected WebModuleConfig loadWebModuleConfig(WebModule wm) {

        WebModuleConfig wmInfo = new WebModuleConfig();
        wmInfo.setBean(wm);
        String wmID = wm.getName();
        String location = wm.getLocation();
        try {
            WebModulesManager webModulesManager = new WebModulesManager(
                ApplicationServer.getServerContext().getInstanceEnvironment());
	    Application app = webModulesManager.getDescriptor(wmID, location);	    
            WebBundleDescriptor wbd = (WebBundleDescriptor) app.getStandaloneBundleDescriptor();
            wmInfo.setDescriptor(wbd);
        } catch (ConfigException ce) {
            wmInfo = null;
            String msg = _rb.getString("vs.moduleConfigError");
            Object[] params = { wmID, getID() };
            msg = MessageFormat.format(msg, params);
            _logger.log(Level.SEVERE, msg, ce);
        }
        return wmInfo;
    }

    /**
     * Determines whether the specified web module is "active" under this
     * virtual server.
     */
    private boolean isActive(WebModule wm) {
        return isActive(wm, true);
    }

    /**
     * Determines whether the specified web module is "active" under this
     * virtual server.
     *
     * A web module is active if it meets ALL of the following conditions:
     *   - wm is not null
     *   - the enabled attribute of the web-module element is true
     * If matchVSID is true then the following additional condition must be
     * satisfied.
     *   - the virtual-servers attribute of the web-module element is either
     *     empty/not-specified or if specified then this virtual server's
     *     name/ID is one of the virtual servers specified in the list
     *
     * @param wm        The bean containing the web module configuration
     *                  information as specified in server.xml
     * @param matchVSID This is set to false when testing to see if a
     *                  web module that has been configured to be the
     *                  default-web-module for a VS is active or not. In this
     *                  case the only test is to check if the web module
     *                  is enabled or not. When set to true, the virtual
     *                  server list is examined to ensure that the web
     *                  module has been configured to run on this virtual
     *                  server.
     * @return     <code>true</code> if all the criteria are satisfied and
     *             <code>false</code> otherwise.
     */
    protected boolean isActive(WebModule wm, boolean matchVSID) {

        String vsID = getID();

        boolean active = ((vsID != null) && (vsID.length() > 0));
        active &= (wm != null);

        if (active) {
            // Check if the web module is enabled
            active &= wm.isEnabled();

            //
            // Check if vsID is one of the virtual servers specified
            // in the list of virtual servers that the web module is to
            // be loaded on. If the virtual-servers attribute of the
            // <web-module> element is missing or empty then the implied
            // behaviour is that the web module is active on every virtual
            // server.
            //
            String vsIDs = getVirtualServers(wm.getName());

            //
            // fix for bug# 4913636
            // so that for PE if the vsList is null and the virtual server is
            // admin-vs then return false because we don't want to load user apps
            // on admin-vs
            //
            if (getID().equals(WebContainer.ADMIN_VS) && matchVSID
                                   && ((vsIDs == null) || (vsIDs.length() == 0 ))) {
                return false;
            } 


            if ((vsIDs != null) && matchVSID) {
                List vsList = StringUtils.parseStringList(vsIDs, " ,");
                if (vsList != null)
                    active &= vsList.contains(vsID.trim());
                else
                    active &= true;
            } else
                active &= true;
        }
        return active;
    }

    /**
     * Returns the id of the default web module for this virtual server
     * as specified in the 'default-web-module' attribute of the 
     * 'virtual-server' element.
     *
     * This is an optional attribute.
     */
    protected String getDefaultWebModuleID() {
        String wmID = null;
        
        if (vsBean != null) {
            wmID = vsBean.getDefaultWebModule();
            if (wmID != null && _debug) {
                Object[] params = { wmID, _id };
                _logger.log(Level.FINE, "vs.defaultWebModule", params);
            }
        } else {
            _logger.log(Level.SEVERE, "vs.configError", _id);
        }

        return wmID;
    }

    /**
     * Finds and returns information about a web module embedded within a
     * J2EE application, which is identified by a string of the form
     * <code>a:b</code> or <code>a#b</code>, where <code>a</code> is the name
     * of the J2EE application and <code>b</code> is the name of the embedded
     * web module.
     *
     * @return null if <code>id</code> does not identify a web module embedded
     * within a J2EE application.
     */
    protected WebModuleConfig findWebModuleInJ2eeApp(Applications appsBean,
                                                   String id) {
        WebModuleConfig wmInfo = null;

        int length = id.length();
        // Check for ':' separator
        int separatorIndex = id.indexOf(Constants.NAME_SEPARATOR);
        if (separatorIndex == -1) {
            // Check for '#' separator
            separatorIndex = id.indexOf('#');
        }
        if (separatorIndex != -1) {
            String appID = id.substring(0, separatorIndex);
            String moduleID = id.substring(separatorIndex + 1);
            J2eeApplication j2eeApp = appsBean.getJ2eeApplicationByName(appID);
            if ((j2eeApp != null) && j2eeApp.isEnabled()) {
                String location = j2eeApp.getLocation();
                String moduleDir = FileUtils.makeFriendlyFilename(moduleID);
                ApplicationRegistry registry =
                    ApplicationRegistry.getInstance();
                ClassLoader appLoader =
                    registry.getClassLoaderForApplication(appID);
                if (appLoader != null) {
                    Application appDesc = registry.getApplication(appLoader);
                    if (appDesc != null) {
                        Set wbds = appDesc.getWebBundleDescriptors();
                        WebBundleDescriptor wbd = null;
                        for (Iterator itr = wbds.iterator(); itr.hasNext(); ) {
                            wbd = (WebBundleDescriptor) itr.next();
                            String webUri = wbd.getModuleDescriptor().getArchiveUri();
                            if (moduleID.equals(webUri)) {
                                StringBuffer dir = new StringBuffer(location);
                                dir.append(File.separator);
                                dir.append(moduleDir);
                                WebModule wm = new WebModule();
                                wm.setName(moduleID);
                                wm.setContextRoot(wbd.getContextRoot());
                                wm.setLocation(dir.toString());
                                wm.setEnabled(true);
                                String vsList = getVirtualServers(j2eeApp.getName());
                                wmInfo = new WebModuleConfig();
                                wmInfo.setBean(wm);
                                wmInfo.setDescriptor(wbd);
                                wmInfo.setParentLoader(appLoader);
                                wmInfo.setVirtualServers(vsList);
                                break;
                            }
                        }
                    }
                }
            } else {
                Object[] params = { id, getID() };
                _logger.log(Level.SEVERE, "vs.defaultWebModuleDisabled",
                            params);
            }
        }
        return wmInfo;
    }
    
    /**
     * Virtual servers are maintained in the reference contained 
     * in Server element. First, we need to find the server
     * and then get the virtual server from the correct reference
     *
     * @param appName Name of the app to get vs
     *
     * @return virtual servers as a string (separated by space or comma)
     */
    private String getVirtualServers(String appName) {
        String ret = null;
        try {
            ConfigContext ctx = 
                ApplicationServer.getServerContext().getConfigContext();
            ret = ServerBeansFactory
                    .getVirtualServersByAppName(ctx, appName);
        } catch (ConfigException ce) {
            _logger.log(Level.FINE, "Cannot get virtual server for " + appName, ce);
        }
        
        return ret;
    }
    
    
    /**
     * Delete all aliases.
     */
    public void clearAliases(){
        aliases = new String[0];
    }


    void setIsDisabled(boolean isDisabled) {
        ((VirtualServerPipeline) getPipeline()).setIsDisabled(isDisabled);
    }


    void setIsOff(boolean isOff) {
        ((VirtualServerPipeline) getPipeline()).setIsOff(isOff);
    }


    /**
     * Adds the given redirect instruction to this virtual server.
     *
     * @param from URI prefix to match
     * @param url Redirect URL to return to the client
     * @param urlPrefix New URL prefix to return to the client
     * @param escape true if redirect URL returned to the client is to be
     * escaped, false otherwise
     */
    void addRedirect(String from, String url, String urlPrefix,
                     boolean escape) {

        ((VirtualServerPipeline) getPipeline()).addRedirect(
            from, url, urlPrefix, escape);
    }
    
    
    /**
     * Set the Stat holder.
     */
    public void setPWCRequestStatsImpl(PWCRequestStatsImpl pwcRequestStatsImpl){
        this.pwcRequestStatsImpl = pwcRequestStatsImpl;
    }

    
    /**
     * Get the Stat holder.
     */
    public PWCRequestStatsImpl getPWCRequestStatsImpl(){
        return pwcRequestStatsImpl;
    }
}
