/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.server.logging.diagnostics;

import java.util.ArrayList;
import java.util.ResourceBundle;
import java.util.MissingResourceException;

/**
 * Simple catalog class to locate Diagnostic Information based on
 * message id as the key.
 *
 * @author Hemanth Puttaswamy
 */
public class MessageIdCatalog{
     private static final MessageIdCatalog instance = 
         new MessageIdCatalog( );

     public static MessageIdCatalog getInstance( ) {
         return instance;
     }

     /**
      * Get all the documented DiagnosticCauses for a given message id.
      * The results will be localized based on the current locale of
      * the AppServer's JVM.
      */
     public ArrayList getDiagnosticCausesForMessageId( String messageId ) {
         ResourceBundle rb = 
             ResourceBundleLocator.getResourceBundleForMessageId( messageId );
         String cause = null;
         ArrayList causes = null;
         if( rb != null ) {
             for( int i = 1; i < Constants.MAX_CAUSES_AND_CHECKS; i++ ) { 
                 // The convention used to document diagnostic causes in
                 // resource bundle is
                 // <MsgId>.diag.cause.1= <Cause 1>
                 // <MsgId>.diag.cause.2= <Cause 2> ....
                 try { 
                     cause = rb.getString( messageId + 
                             Constants.CAUSE_PREFIX + i );
                 } catch( MissingResourceException e ) {
                     // We couldn't find any causes listed for the message
                     // id or we have found all. In either case we are
                     // covered here.
                     break;
                 }
                 if( cause == null ) { break; }
                 if( causes == null ) { 
                         causes = new ArrayList( ); 
                 }
                 causes.add( cause );
             }
        }
        return causes;
     }

     /**
      * Get all the documented DiagnosticChecks for a given message id.
      * The results will be localized based on the current locale of
      * the AppServer's JVM.
      */
     public ArrayList getDiagnosticChecksForMessageId( String messageId ) {
         ResourceBundle rb = 
             ResourceBundleLocator.getResourceBundleForMessageId( messageId );
         String check = null;
         ArrayList checks = null;
         if( rb != null ) {
             for( int i = 1; i < Constants.MAX_CAUSES_AND_CHECKS; i++ ) { 
                 // The convention used to document diagnostic checks in
                 // resource bundle is
                 // <MsgId>.diag.check.1= <Check 1>
                 // <MsgId>.diag.check.2= <Check 2> ....
                 try {
                         check = rb.getString( messageId + 
                                 Constants.CHECK_PREFIX + i );
                 } catch( MissingResourceException e ) {
                     // We couldn't find any checks listed for the message
                     // id or we have found all. In either case we are
                     // covered here.
                     break;
                 }
                 if( check == null ) break;
                 if( checks == null ) { 
                     checks = new ArrayList( );
                 }
                 checks.add( check );
             }
         }
         return checks; 
     }

     /**
      * We may collect lot of diagnostic causes and diagnostic checks for
      * some common message id from the field. We may document those 
      * even after the product is shipped. We are planning to generate the
      * HTML's from the resource bundle's diagnostics and update the javadoc
      * or knowledgebase site. This URI should help us to locate the latest
      * and greatest diagnostic info based on the message id.
      */
     public String getDiagnosticURIForMessageId( String messageId ) {
         String moduleId = ResourceBundleLocator.getModuleId( messageId );
         if( moduleId == null ) { return null; }
         return Constants.URI_PREFIX + moduleId + "/" + messageId;
     }

     public Diagnostics getDiagnosticsForMessageId( String messageId ) {
         ArrayList causes = getDiagnosticCausesForMessageId( messageId );
         ArrayList checks = getDiagnosticChecksForMessageId( messageId );
         if( ( causes == null )
           &&( checks == null ) ) {
             return null;
         }
         Diagnostics diagnostics = new Diagnostics( messageId );
         diagnostics.setPossibleCauses( causes );
         diagnostics.setDiagnosticChecks( checks );
         diagnostics.setURI( getDiagnosticURIForMessageId( messageId ) );
         return diagnostics;
     }
}
     
