/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.server;
/**
 * This class overrides the isReferenced method in the ResourcesUtil class.
 * An instance of this class is always returned when ResourcesUtil.getInstance()
 * happens in the DAS
 *
 * @author    Aditya Gore
 * @version
 */

import com.sun.enterprise.config.*;
import com.sun.enterprise.config.serverbeans.*;
import com.sun.enterprise.util.RelativePathResolver;
import com.sun.enterprise.connectors.DeferredResourceConfig;

import java.util.logging.Level;

public class DASResourcesUtil extends ResourcesUtil {
    
    /**
     * Flags if a getConnection call is coming from the connector runtime.
     * Only for such a call, we should not be checking isReferenced
     * The reason for this is that the getConnection call might be happening
     * in the DAS for a jdbc resource which does not have a resource-ref in
     * the DAS. However, we still need to give a connection from that 
     * resource, so we should not check for referencing info.
     */
    public boolean getConnectionFromConnectorRuntime_ = false;
    
    protected DASResourcesUtil() {
        super();
    }
    
    /**
     * Returns true if the given resource is referenced by this server. 
     *
     * @param   resourceName   the name of the resource
     *
     * @return  true if the named resource is used/referred by this server
     *
     * @throws  ConfigException  if an error while parsing domain.xml
     */
    protected boolean isReferenced(String resourceName) throws ConfigException {
        if (_logger.isLoggable(Level.FINE)) {
	    _logger.fine("isReferenced in DASResourcesUtil:: " + resourceName);
	}

	if (isGetConnectionFromConnectorRuntime()) {
            if (_logger.isLoggable(Level.FINE)) {
	        _logger.fine("getConnectionFromConnectorRuntime :: true");
	    }
	    return true;
	}    

	return super.isReferenced( resourceName );
    }

    public boolean isGetConnectionFromConnectorRuntime() {
        return getConnectionFromConnectorRuntime_;
    }

    public void setGetConnectionFromConnectorRuntime(boolean flag) {
        getConnectionFromConnectorRuntime_ = flag;
    }
    
    public String getLocation(String moduleName) {
        if(moduleName == null) {
            return null;
        }

        //Use admin config context, as in DAS, the server config context would
        //not have the latest config context changes until the end of the deploy
        //event
        ConfigContext adminConfigCtx = com.sun.enterprise.admin.server.core.
            AdminService.getAdminService().getAdminContext().getAdminConfigContext();

        try {
            ConnectorModule connectorModule = ServerBeansFactory.getDomainBean(adminConfigCtx).
                            getApplications().getConnectorModuleByName(moduleName);
            
            if (connectorModule == null) {
                return null;
            }
            
            String connectorModuleLocation = connectorModule.getLocation();
            String connectorModuleLocationResolved = RelativePathResolver.
                                resolvePath(connectorModuleLocation);
            return connectorModuleLocationResolved;
        } catch (ConfigException e) {
            _logger.warning(e.getMessage());
            _logger.log(Level.FINE, "Exception while tryign to find location" +
                    "of connector module " + moduleName, new Object[]{e});
        }
        return null;
    }
    
    /**
     * Gets the deployment location for a J2EE application.
     * @param rarName
     * @return
     */
    public String getApplicationDeployLocation(String appName) {
        ConfigContext adminConfigCtx = com.sun.enterprise.admin.server.core.
        AdminService.getAdminService().getAdminContext().getAdminConfigContext();
        
        try {
            J2eeApplication app = ServerBeansFactory.getDomainBean(adminConfigCtx).
            getApplications().getJ2eeApplicationByName(appName);
            return RelativePathResolver.resolvePath(app.getLocation());
        } catch (ConfigException e) {
            e.printStackTrace();
        }
        return null;
    }
 
    /**
     * Gets a PMF resource on the basis of its jndi name
     * 
     * @param jndiName the jndi name of the PMF resource to lookup
     * @param checkReference if true, returns this PMF resource only if it is referenced in
     *                       this server. If false, returns the PMF resource irrespective of
     *                       whether it is referenced or not.
     */
    public PersistenceManagerFactoryResource getPMFResourceByJndiName(
            String jndiName ) {
        reinitialize();

	Resources localRes = getResourcesServerBean();
	if ( localRes == null) {
	    return null;
	}

	if (_logger.isLoggable(Level.FINE)) {
	    _logger.fine("ResourceUtil :: looking up pmf resource, jndiName is :"
	        + jndiName );
	}
	PersistenceManagerFactoryResource pmf = 
	        localRes.getPersistenceManagerFactoryResourceByJndiName( jndiName );
        
	if (_logger.isLoggable(Level.FINE)) {
	    _logger.fine("getPMFResourceByJndiName:: looked up pmf resource : " 
	        + pmf);
	}
	//does the isReferenced method throw NPE for null value? Better be safe
	if (pmf == null) {
	    return null;
	}
     	
        try {
           return isReferenced( jndiName ) ? pmf : null;
        }catch( ConfigException ce ) {
           return null;
        }
    }

    /**
     * Gets a JDBC resource on the basis of its jndi name
     * @param jndiName the jndi name of the JDBC resource to lookup
     * @param checkReference if true, returns this JDBC resource only if it is referenced in
     *                       this server. If false, returns the JDBC resource irrespective of
     *                       whether it is referenced or not.
     */
    public JdbcResource getJdbcResourceByJndiName( String jndiName) { 
        reinitialize();

	Resources localRes = getResourcesServerBean();
	if ( localRes == null ) {
	    return null;
	}

	if (_logger.isLoggable(Level.FINE)) {
	    _logger.fine("ResourcesUtil :: looking up jdbc resource, jndiName is :"
	        + jndiName );
	}

	JdbcResource jdbcRes = 
	        localRes.getJdbcResourceByJndiName( jndiName );
        
	if (_logger.isLoggable(Level.FINE)) {
	    _logger.fine("ResourcesUtil :: looked up jdbc resource:" + jdbcRes.getJndiName() );
	}
	//does the isReferenced method throw NPE for null value? Better be safe
	if (jdbcRes == null) {
	    return null;
	}
	
        try {
            return isReferenced( jndiName ) ? jdbcRes : null;
        }catch( ConfigException ce ) {
            return null;
        }
    }

    /**
     * Checks if this is a PMF resource
     * @param jndiName the jndi name of the PMF resource to lookup
     * @param checkReference if true, returns this PMF resource only if it is referenced in
     *                       this server
     */
    public boolean isPMFResource( String jndiName) {

	if (_logger.isLoggable(Level.FINE)) {
	    _logger.fine("ResourcesUtil :: checking if this is a pmf resource:"
	        + jndiName );
	}
        return (getPMFResourceByJndiName(jndiName) != null); 
    }   


    protected DeferredResourceConfig getDeferredJdbcResourceConfigs(
                  String resourceName) 
    {

        if(resourceName == null) {
            return null;
        }
        reinitialize();
        try {
            //__pm does not have a domain.xml entry and hence will not
            //be referenced
            if(!(resourceName.endsWith("__pm"))){
                if(!isReferenced(resourceName)){
                    return null;
                }
            }
        } catch (ConfigException e) {
            _logger.log(Level.WARNING, e.getMessage());
            _logger.log(Level.FINE, "Error while finding resource references " , e);
        }

	Resources localRes = getResourcesServerBean();
	if ( localRes == null ) {
	    return null;
	}
        
        ConfigBean[] resourcesToload = new ConfigBean[2];
        JdbcResource jdbcResource = localRes.getJdbcResourceByJndiName(resourceName);
        if(jdbcResource == null || !jdbcResource.isEnabled()) {
            String cmpResourceName = 
                        getCorrespondingCmpResourceName(resourceName);
            jdbcResource = localRes.getJdbcResourceByJndiName(cmpResourceName);
            if(jdbcResource == null) {
                return null;
            }
        }
        JdbcConnectionPool jdbcPool = 
                 localRes.getJdbcConnectionPoolByName(jdbcResource.getPoolName());
        if(jdbcPool == null) {
            return null;
        } 
        String rarName = getRAForJdbcConnectionPool(jdbcPool);
        if(rarName != null && belongToSystemRar(rarName)) {
            resourcesToload[0] = jdbcPool;
            resourcesToload[1] = jdbcResource;
            DeferredResourceConfig resourceConfig = 
                            new DeferredResourceConfig(rarName,null,null,
                            null,jdbcPool,jdbcResource,null);
            resourceConfig.setResourcesToLoad(resourcesToload); 
            return resourceConfig;
        } 
        return null;
    }

    protected DeferredResourceConfig getDeferredJdbcPoolConfigs(
                  String poolName) 
    {
        if(poolName == null) {
            return null;
        }
        reinitialize();

        Resources localRes = getResourcesServerBean();
        JdbcConnectionPool jdbcPool = 
                localRes.getJdbcConnectionPoolByName(poolName);
        if(jdbcPool == null) {
            return null;
        }
        String rarName = getRAForJdbcConnectionPool(jdbcPool);
        
        ConfigBean[] resourcesToload = new ConfigBean[1];
        if(rarName != null && belongToSystemRar(rarName)) {
                    resourcesToload[0] = jdbcPool;
                    DeferredResourceConfig resourceConfig = 
                            new DeferredResourceConfig(rarName,null,null,
                            null,jdbcPool,null,null);
                    resourceConfig.setResourcesToLoad(resourcesToload); 
                    return resourceConfig;
        } 
        return null;
    }

    protected Resources getResourcesServerBean() {
	//IF and only IF we are executing in the context of a
	//ConnectorRuntime.getConnection call, we should use the
	//admin config context to get to the Resources MBean
	//so check and return the correct Resources MBean
	
	if (! isGetConnectionFromConnectorRuntime() ) {
	    //return the cached Resources MBean from ResourcesUtil
	    //reinitialize has been called before this, so we are ok
	    return super.getResourcesServerBean(); 
	}

        ConfigContext adminConfigCtx = com.sun.enterprise.admin.server.core.
            AdminService.getAdminService().getAdminContext().getAdminConfigContext();

	Domain localdom = null;
	Resources localres = null;
	try {
            localdom = ServerBeansFactory.getDomainBean( adminConfigCtx );
            localres = localdom.getResources();
        } catch( ConfigException ce ) {
	    _logger.log( Level.WARNING, ce.getMessage() );
	    return null;
	}

	return localres;
    }

    protected DeferredResourceConfig getDeferredConnectorResourceConfigs(
                  String resourceName) 
    {

        if(resourceName == null) {
            return null;
        }
        
        reinitialize();

        try {
            if(!isReferenced(resourceName)){
                return null;
            }
        } catch (ConfigException e) {
            _logger.log(Level.WARNING, e.getMessage());
            _logger.log(Level.FINE, "Error while finding resource references " , e);
        }
        
        Resources localRes = getResourcesServerBean(); 
	if ( localRes == null ) {
	    return null;
	}
        ConfigBean[] resourcesToload = new ConfigBean[2];
        ConnectorResource connectorResource = 
                     localRes.getConnectorResourceByJndiName(resourceName);
        if(connectorResource == null || !connectorResource.isEnabled()) {
            return null;
        }
        String poolName = connectorResource.getPoolName();
        ConnectorConnectionPool ccPool = 
                       localRes.getConnectorConnectionPoolByName(poolName);
        if(ccPool == null) {
            return null;
        }
        String rarName = ccPool.getResourceAdapterName();
        if(rarName != null && belongToSystemRar(rarName)) {
            resourcesToload[0] = ccPool;
            resourcesToload[1] = connectorResource;
            ResourceAdapterConfig[] resourceAdapterConfig = 
                                        new ResourceAdapterConfig[1];
            resourceAdapterConfig[0] = 
                         localRes.getResourceAdapterConfigByResourceAdapterName(
                          rarName);
            DeferredResourceConfig resourceConfig = 
                            new DeferredResourceConfig(rarName,null,ccPool,
                            connectorResource,null,null,
                            resourceAdapterConfig);
            resourceConfig.setResourcesToLoad(resourcesToload); 
            return resourceConfig;
        } 
        return null;
    }

    protected DeferredResourceConfig getDeferredConnectorPoolConfigs(
                  String poolName) 
    {

        if(poolName == null) {
            return null;
        }
        
        reinitialize();
        Resources localRes = getResourcesServerBean();

        ConfigBean[] resourcesToload = new ConfigBean[1];
        ConnectorConnectionPool ccPool = 
                localRes.getConnectorConnectionPoolByName(poolName);
        if(ccPool == null) {
            return null;
        }
        
        String rarName = ccPool.getResourceAdapterName();
        
	//Commenting out the belongToSystemRar call since the method
	//originally was introduced to check if a pool belonged to
	//a system rar while lazily loading already created pools.
	//Now, because of lazy deployment of pools on resource creation,
	//this check is no longer relevant.
	//TODO: revisit later
        if(rarName != null) { //&& belongToSystemRar(rarName)) {
            resourcesToload[0] = ccPool;
            ResourceAdapterConfig[] resourceAdapterConfig = 
                      new ResourceAdapterConfig[1];
            resourceAdapterConfig[0] = 
                   localRes.getResourceAdapterConfigByResourceAdapterName(
                   rarName);
            DeferredResourceConfig resourceConfig = 
                 new DeferredResourceConfig(rarName,null,ccPool,
                 null,null,null,resourceAdapterConfig);
            resourceConfig.setResourcesToLoad(resourcesToload); 
            return resourceConfig;
        } 
        return null;
    }


    protected DeferredResourceConfig getDeferredAdminObjectConfigs(
                  String resourceName) 
    {

        if(resourceName == null) {
            return null;
        }
        reinitialize();
        
        try {
            if(!isReferenced(resourceName)){
                return null;
            }
        } catch (ConfigException e) {
            _logger.log(Level.WARNING, e.getMessage());
            _logger.log(Level.FINE, "Error while finding resource references " , e);
        }

	Resources localRes = getResourcesServerBean();
	if ( localRes == null) {
	    return null;
	}
        
        ConfigBean[] resourcesToload = new ConfigBean[1];
        AdminObjectResource adminObjectResource = 
                      localRes.getAdminObjectResourceByJndiName(resourceName);
        if(adminObjectResource == null || !adminObjectResource.isEnabled()) {
            return null;
        }
        String rarName = adminObjectResource.getResAdapter();
        if(rarName != null && belongToSystemRar(rarName)) {
            resourcesToload[0] = adminObjectResource;
            ResourceAdapterConfig[] resourceAdapterConfig = 
                                        new ResourceAdapterConfig[1];
            resourceAdapterConfig[0] = 
                   localRes.getResourceAdapterConfigByResourceAdapterName(rarName);
            DeferredResourceConfig resourceConfig = 
                         new DeferredResourceConfig(rarName,adminObjectResource,
                         null,null,null,null,resourceAdapterConfig);
            resourceConfig.setResourcesToLoad(resourcesToload); 
            return resourceConfig;
        }
        return null;
    }

    public JdbcConnectionPool[] getJdbcConnectionPools() {
        //the method below is actually a method to indicate that the
        //ConfigContext to be used is the AdminConfigContext and not
        //the normal ConfigContext
        //TODO: Rename the flag
        //This change is needed since 9.0 onwards we create the pool
        //object eagerly at deployment time and hence the pool is not
        //available in the ConfigContext. Note this method is accessed
        //in the isJdbcPool call. There we need to get *all* pools present
        //at that time in the environment
        Resources localRes = null;

        setGetConnectionFromConnectorRuntime( true );
        try {
            localRes = getResourcesServerBean();
        } finally {
            setGetConnectionFromConnectorRuntime( false );
        }

        JdbcConnectionPool[] jdbcConnectionPool = localRes.getJdbcConnectionPool();
        if(jdbcConnectionPool == null) return null;
        if (jdbcConnectionPool.length == 0) {
            return null;
        } else {
            return jdbcConnectionPool;
        }
    }

    public ConnectorConnectionPool[] getConnectorConnectionPools() {
        //the method below is actually a method to indicate that the
        //ConfigContext to be used is the AdminConfigContext and not
        //the normal ConfigContext
        //TODO: Rename the flag
        Resources localRes = null;
        setGetConnectionFromConnectorRuntime( true );
        try { 
            localRes = getResourcesServerBean();
        } finally {
            setGetConnectionFromConnectorRuntime( false );
        }

        ConnectorConnectionPool[] connectorConnectionPools = 
            localRes.getConnectorConnectionPool();
        if(connectorConnectionPools == null) return null;
        if (connectorConnectionPools.length == 0) {
            return null;
        } else {
            return connectorConnectionPools;
        }
    }

}
