/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.security.auth.login;

import java.security.cert.X509Certificate;

/**
 * This class holds the user certificate for the certificate realm and the
 * realm name. This credential is added as a public credential to the
 * JAAS subject.
 */

public class X509CertificateCredential {
    private X509Certificate[] certChain;
    private String realm;
    private String alias;

    /**
     * Construct a credential with the specified X509Certificate certificate
     * chain, realm name and alias.
     * @param the X509Certificate.
     * @param the alias for the certificate
     * @param the realm name. The only value supported for now is "certificate".
     */

    public X509CertificateCredential(X509Certificate[] certChain, 
				    String alias, String realm)
    {
	this.certChain = certChain;
	this.alias = alias;
	this.realm = realm;
    }

    /**
     * Return the alias for the certificate.
     * @return the alias. 
     */
    public String getAlias() {
	return alias;
    }

    /**
     * Return the realm name.
     * @return the realm name. Only value supported for now is "certificate".
     */
    public String getRealm() {
	return realm;
    }

    /**
     * Return the chain of certificates.
     * @return the chain of X509Certificates.
     */
    public X509Certificate[] getX509CertificateChain() {
	return certChain;
    }

    /**
     * Compare two instances of the credential and return true if they are
     * the same and false otherwise.
     * @return true if the instances are equal, false otherwise.
     */
    public boolean equals(Object o) {
	if(o instanceof X509CertificateCredential) {
	    X509CertificateCredential pc = (X509CertificateCredential) o;
	    if(pc.getRealm().equals(realm) && pc.getAlias().equals(alias)) {
		X509Certificate[] certs = pc.getX509CertificateChain();
		for(int i = 0; i < certs.length; i++) {
		    if(!certs[i].equals(certChain[i])) {
			return false;
		    }
		}
		return true;
	    }
	}
	return false;
    }

    /**
     * Return the hashCode computed from the certificate, realm and alias.
     * @return the hash code.
     */
    public int hashCode() {
	return certChain.hashCode() + realm.hashCode() + alias.hashCode();
    }

    /**
     * String representation of the credential.
     */
    public String toString() {
	String s = "Realm=" + realm;
	s = s + " alias=" + alias;
	s = s + " X509Certificate=" + certChain;
	return s;
    }

}
