/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.admin.common.domains.registry;

import java.io.Serializable;
import java.util.HashMap;
import java.io.File;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeMap;
import java.util.NoSuchElementException;


class Registry implements Serializable, Cloneable, DomainRegistryI
{
  TreeMap roots = new TreeMap();
  HashMap entries = new HashMap();

  public boolean isRegistered(String name){
	return roots.containsKey(name);
  }


  public void registerDomain(DomainEntry de) throws DomainRegistryException{
	if (isRegistered(de.getName())) {
	  throw new AlreadyRegisteredException(de.getName());
	}
	if (containsRoot(de.getRoot())){
	  throw new InvalidRootException("The root \""+de.getRoot()+"\" is already registered");
	}
	roots.put(de.getName(), de.getRoot());
	entries.put(de.getName(), de);
  }

 public void unregisterDomain(String domain_name) throws DomainRegistryException{
   if (!isRegistered(domain_name)){
	 throw new UnregisteredDomainException(domain_name);
   }
	delete(domain_name);
  }

  public void unregisterDomain(DomainEntry de) throws DomainRegistryException{
	unregisterDomain(de.getName());
  }
  public void reregisterDomain(DomainEntry de) throws DomainRegistryException {
	if (isRegistered(de.getName())) {
	  if (!roots.get(de.getName()).equals(de.getRoot())){
		throw new InvalidRootException("The given root ("+de.getRoot()+") of domain "+de.getName()+" doesn't match the already registered root for this domain");
	  }
	} else {
	  if (containsRoot(de.getRoot())){
		throw new InvalidRootException("The given root ("+de.getRoot()+") of domain "+de.getName()+" is already registered with a different domain");
	  }
	};

	entries.put(de.getName(), de);
  }

  public Iterator iterator() throws DomainRegistryException{
	return new RegistryIterator(this);
  }

  public boolean containsDomain(DomainEntry de) throws DomainRegistryException{
	return entries.values().contains(de);
  }
  public DomainEntry getDomain(String name) throws DomainRegistryException {
	return (DomainEntry) entries.get(name);
  }
  public int size(){
	return roots.size();
  }

  private boolean containsRoot(File root){
	return roots.containsValue(root);
  }

  private void delete(String name){
	roots.remove(name);
	entries.remove(name);
  }

	

  protected Object clone(){
	try {
	  Registry lhs = (Registry) super.clone();
	  lhs.roots = (TreeMap) this.roots.clone();
	  lhs.entries = (HashMap) this.entries.clone();
	  return lhs;
	}
	catch (CloneNotSupportedException cne){
	  return null;
	}
  }

  
  class RegistryIterator implements Iterator
  {
	Registry registry;
	Iterator iterator;
	  
	RegistryIterator(Registry r){
	  registry = (Registry) r.clone();
	  iterator = registry.roots.keySet().iterator();
	}
	public boolean hasNext(){
	  return iterator.hasNext();
	}
	public Object next() throws NoSuchElementException{
	  return entries.get((String) iterator.next());
	}
	public void remove() throws UnsupportedOperationException{
	  throw new UnsupportedOperationException();
	}
  }

}

