/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
 package com.sun.enterprise.deployment;

import java.util.Set;
import java.util.HashSet;

/**
 * I am an abstract super class of all ejb descriptors.
 *
 * @author Danny Coward
 */

public abstract class EjbAbstractDescriptor extends Descriptor implements NamedDescriptor {
    private String homeClassName;
    private String remoteClassName;
    private String localHomeClassName;
    private String localClassName;

    private Set<String> remoteBusinessClassNames = new HashSet<String>();
    private Set<String> localBusinessClassNames = new HashSet<String>();

    // This is the value of the EJB 2.1 deployment descriptor entry
    // for service endpoint interface.
    private String webServiceEndpointInterfaceName;

    private String jndiName = "";
    private String mappedName = "";
    
       
	/** 
	* Default constructor. 
	*/
    public EjbAbstractDescriptor() {
    }
    
    /**
    * Copy constructor.
    */
    protected EjbAbstractDescriptor(EjbAbstractDescriptor other) {
	super(other);
	this.homeClassName = other.homeClassName;
	this.remoteClassName = other.remoteClassName;
        this.remoteBusinessClassNames = 
            new HashSet<String>(other.remoteBusinessClassNames);
        this.localHomeClassName = other.localHomeClassName;
        this.localClassName = other.localClassName;
        this.localBusinessClassNames = 
            new HashSet<String>(other.localBusinessClassNames);
        this.webServiceEndpointInterfaceName = 
            other.webServiceEndpointInterfaceName;
	this.jndiName = other.jndiName;
    }
    
 
    public abstract String getType();
    public abstract void setType(String type);
    
	/**
	* Returns the classname of the Home interface of this ejb. 
	*/
    public String getHomeClassName() {
	return this.homeClassName;
    }
	
	/** 
	* Sets the classname of the Home interface of this ejb. 
	*/
    public void setHomeClassName(String homeClassName) {
	this.homeClassName = homeClassName;
    }
    
	/**
	* Sets the classname of the Remote interface of this ejb. 
	*/
    public void setRemoteClassName(String remoteClassName) {
	this.remoteClassName = remoteClassName;
    }
    

	/**
	* Returns the classname of the Remote interface of this ejb.
	*/
    public String getRemoteClassName() {
	return this.remoteClassName;
    }	
    
    /**
     * Sets the classname for the local home interface of this ejb
     * 
     * @param localHomeClassName fully qualified class name for the interface
     */
    public void setLocalHomeClassName(String localHomeClassName) {
        this.localHomeClassName = localHomeClassName;
    }
    
    /**
     * @return the fully qualified class name for the local home interface of this ejb
     */
    public String getLocalHomeClassName() {
        return localHomeClassName;
    }
    
    /**
     * Sets the classname for the local interface of this ejb
     * 
     * @param localClassName fully qualified class name for the interface
     */
    public void setLocalClassName(String localClassName) {
        this.localClassName = localClassName;
    }
    
    /**
     * @return the fully qualified class name for the local interface of this ejb
     */
    public String getLocalClassName() {
        return localClassName;
    }

    public void addRemoteBusinessClassName(String className) {
        remoteBusinessClassNames.add(className);
    }

    public void addLocalBusinessClassName(String className) {
        localBusinessClassNames.add(className);
    }

    /**
     * Returns the set of remote business interface names for this ejb.
     * If the bean does not expose a remote business view, return a set
     * of size 0.
     */
    public Set<String> getRemoteBusinessClassNames() {
        return new HashSet<String>( remoteBusinessClassNames );
    }

    /**
     * Returns the set of local business interface names for this ejb.
     * If the bean does not expose a local business view, return a set
     * of size 0.
     */
    public Set<String> getLocalBusinessClassNames() {
        return new HashSet<String>( localBusinessClassNames );
    }

    public void setWebServiceEndpointInterfaceName(String name) {
        this.webServiceEndpointInterfaceName = name;
    }

    public String getWebServiceEndpointInterfaceName() {
        return webServiceEndpointInterfaceName;
    }

	/**
	* Return the JNDI name which will be assigned to the ejb home object ar runtime.
	*/
    public String getJndiName() {
	if (this.jndiName == null) {
	    this.jndiName = "";
	}
	return (jndiName != null && jndiName.length() > 0)?
                jndiName : getMappedName();
    }
   
	/**
	* Sets the JNDI name which will be assigned to the ejb home object ar runtime.
	*/
    public void setJndiName(String jndiName) {
	this.jndiName = jndiName;
	if (this.getName().equals("")) {
	    super.setName(jndiName);
	}
	this.changed();
    }    

    public String getMappedName() {
        return (mappedName != null)? mappedName : "";
    }

    public void setMappedName(String mappedName) {
        this.mappedName = mappedName;
        this.changed();
    }
    
	/**
	* Returns a formatted String of the attributes of this object.
	*/
    public void print(StringBuffer toStringBuffer) {
        super.print(toStringBuffer);
	toStringBuffer.append("\n homeClassName ").append(homeClassName);
	toStringBuffer.append("\n remoteClassName ").append(remoteClassName);
	toStringBuffer.append("\n remoteBusinessIntfs ").append(remoteBusinessClassNames).append("\n");
	toStringBuffer.append("\n localhomeClassName ").append(localHomeClassName);
	toStringBuffer.append("\n localClassName ").append(localClassName);
	toStringBuffer.append("\n localBusinessIntfs ").append(localBusinessClassNames).append("\n");
	toStringBuffer.append("\n jndiName ").append(jndiName).append("\n");
    }
    
    /**
     * @return true if the EJB described has a LocalHome/Local interface
     */
    public boolean isLocalInterfacesSupported() {
        return (getLocalHomeClassName() != null);
    }

    /**
     * @return true if the EJB has 1 or more local business interfaces
     */
    public boolean isLocalBusinessInterfacesSupported() {
        return (localBusinessClassNames.size() > 0);
    }
    
    /**
     * @return true if the EJB has a RemoteHome/Remote interface
     */
    public boolean isRemoteInterfacesSupported() {
        return (getHomeClassName() != null);        
    }

    /**
     * @return true if the EJB has 1 or more remote business interfaces
     */
    public boolean isRemoteBusinessInterfacesSupported() {
        return (remoteBusinessClassNames.size() > 0);
    }


    /**
     * @return true if this is an EJB that implements a web service endpoint.
     */
    public boolean hasWebServiceEndpointInterface() {
        return (getWebServiceEndpointInterfaceName() != null);        
    }

}
    
