/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.deployment.annotation.impl;

import com.sun.enterprise.deployment.annotation.ComponentInfo;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * This class represents the view of a class from annotation.
 *
 * @author Shing Wai Chan
 */
public class ComponentDefinition implements ComponentInfo {
    final private Class clazz;
    final private List<Constructor> constructors = new ArrayList<Constructor>();
    final private List<Class> classes = new ArrayList<Class>();
    final private List<Field> fields = new ArrayList<Field>();
    final private Map<MethodKey, Method> methodMap = new HashMap<MethodKey, Method>();

    public ComponentDefinition(Class clazz) {
        this.clazz = clazz;
        constructClassList();
        initializeConstructors();
        initializeFields();
        initializeMethods();
    }

    public Field[] getFields() {
        return fields.toArray(new Field[fields.size()]);
    }

    public Method[] getMethods() {
        return methodMap.values().toArray(new Method[methodMap.size()]);
    }

    public Constructor[] getConstructors() {
        return constructors.toArray(new Constructor[constructors.size()]);
    }

    private void constructClassList() {
        classes.add(clazz);
        Class parent = clazz;
        while ((parent = parent.getSuperclass()) != null) {
            if (parent.getPackage() == null ||
                    !parent.getPackage().getName().startsWith("java.lang")) {
                classes.add(0, parent);
            } 
        }
    }

    /**
     * In P.148 of "The Java Langugage Specification 2/e", 
     * Constructors, static initializers, and instance initializers are not
     * members and therefore not inherited.
     */
    private void initializeConstructors() {
        for (Class cl : classes) {
            for (Constructor constr : cl.getConstructors()) {
                constructors.add(constr);           
            }
        }
    }

    private void initializeFields() {
        for (Class cl : classes) {
            for (Field f : cl.getDeclaredFields()) {
                fields.add(f);           
            }
        }
    }

    private void initializeMethods() {
        for (Class cl: classes) {
            for (Method method : cl.getDeclaredMethods()) {
                methodMap.put(new MethodKey(method), method);
            }
        }
    }

    private class MethodKey {
        private Method m = null;
        private int hashCode;
        private String className = null;
        private Package classPackage = null;

        private MethodKey(Method m) {
            this.m = m;
            hashCode = m.getName().hashCode();
            // store className and classPackage as getters are native
            className = m.getDeclaringClass().getName();
            classPackage = m.getDeclaringClass().getPackage();
        }

        public int hashCode() { 

            return hashCode;
        }

        /**
         * This equals method is defined in terms of inheritance overriding.
         * We depends on java compiler to rule out irrelvant cases here.
         * @return true for overriding and false otherwise
         */
        public boolean equals(Object o) {
            if (!(o instanceof MethodKey)) {
                return false;
            }

            MethodKey mk2 = (MethodKey)o;
            Method m2 = mk2.m;
            if (m.getName().equals(m2.getName()) && Arrays.equals(
                    m.getParameterTypes(), m2.getParameterTypes())) {
                int modifiers = m.getModifiers();
                int modifiers2 = m2.getModifiers();
                boolean isPackageProtected2 = !Modifier.isPublic(modifiers2) &&
                        !Modifier.isProtected(modifiers2) &&
                        !Modifier.isPrivate(modifiers2);
                boolean isSamePackage = 
                        (classPackage == null && mk2.classPackage == null) ||
                        (classPackage != null && mk2.classPackage != null &&
                            classPackage.getName().equals(
                            mk2.classPackage.getName()));
                if (Modifier.isPrivate(modifiers)) {
                    // need exact match
                    return Modifier.isPrivate(modifiers2) && isSamePackage
                            && className.equals(mk2.className);
                } else { // public, protected, package protected
                    return Modifier.isPublic(modifiers2) ||
                            Modifier.isProtected(modifiers2) ||
                            isPackageProtected2 && isSamePackage;
                }
            }

            return false;
        }
    }
}
