/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.admin.wsmgmt.msg;

import java.util.Collection;
import com.sun.enterprise.admin.wsmgmt.config.spi.Constants;
import com.sun.enterprise.admin.wsmgmt.config.spi.ConfigFactory;
import com.sun.enterprise.admin.wsmgmt.config.spi.ConfigProvider;
import com.sun.enterprise.admin.wsmgmt.config.spi.WebServiceConfig;
import com.sun.enterprise.admin.wsmgmt.pool.spi.Pool;
import com.sun.enterprise.admin.wsmgmt.pool.impl.BoundedPool;
import com.sun.enterprise.admin.wsmgmt.filter.spi.Filter;
import com.sun.enterprise.admin.wsmgmt.filter.spi.FilterRegistry;
import com.sun.appserv.management.ext.wsmgmt.MessageTrace;

import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;
import com.sun.enterprise.util.i18n.StringManager;

/**
 * Keeps track of SOAP messages per endpoint.
 */
class EndpointHandler {

    /**
     * Constructor.
     *
     * @param  endpoint  name of the endpoint
     */
    EndpointHandler(WebServiceConfig wsc, String appId) {

        _applicationId  = appId;
        _endpointId     = wsc.getName();
        String mLevel   = wsc.getMonitoringLevel();

        // SOAP message visualization is only enabled for level HIGH
        if (Constants.HIGH.equals(mLevel)) {
            _pool = new BoundedPool(wsc.getName(), wsc.getMaxHistorySize());
            registerFilter();
        }
    }

    /**
     * Constructor.
     *
     * @param  endpoint  name of the endpoint
     * @param  size  max size of the pool
     */
    EndpointHandler(String endpoint, int size, String appId) {
        _applicationId  = appId;
        _endpointId     = endpoint;
        _pool           = new BoundedPool(endpoint, size);

        registerFilter();
    }

    /**
     * Registers a filter with the filter manager for this endpoint.
     */
    private void registerFilter() {

        // msg filter
        _filter = new MessageFilter(_applicationId, _endpointId, this);
        FilterRegistry fr = FilterRegistry.getInstance();
        String endpoint = getFQEndpointName();

        // registers the filter
        fr.registerFilter(Filter.PROCESS_REQUEST, endpoint, _filter);
        fr.registerFilter(Filter.PROCESS_RESPONSE, endpoint, _filter);
        fr.registerFilter(Filter.POST_PROCESS_RESPONSE, endpoint, _filter);
    }

    /**
     * Sets the number of messages stored in memory for this endpoint. 
     * This method is called to dynamically reconfigure the size.
     *
     * @param  size  number of message stored in memory
     */
    void setMessageHistorySize(int size) {
        if (_pool != null) {
            _pool.resize(size);
            _logger.fine("Set message history size to " + size 
                + " for " + getEndpointName());
        }
    }

    /**
     * Disables monitoring for the endpoint and deregisters the filters.
     */
    void destroy() {
        if (_pool != null) {
            _pool.clear();
            _pool = null;
        }

        if (_filter != null) {
            FilterRegistry fr = FilterRegistry.getInstance();
            String endpoint = getFQEndpointName();

            // unregister filters
            fr.unregisterFilter(Filter.PROCESS_REQUEST, endpoint, _filter);
            fr.unregisterFilter(Filter.PROCESS_RESPONSE, endpoint, _filter);
            fr.unregisterFilter(Filter.POST_PROCESS_RESPONSE, endpoint, _filter);
            _filter = null;
        }
        _logger.finer("Message trace handler destroyed for " 
            + getEndpointName());
    }

    /**
     * Returns all messages for this endpoint.
     *
     * @return  messages associated for this endpoint
     */
    Collection getMessages() {
        return _pool.values();
    }

    /**
     * Adds a message trace to the pool.
     *
     * @param  msgTrace  a message trace object after the invocation
     */
    void addMessage(MessageTrace msgTrace) {
        _pool.put(msgTrace.getMessageID(), msgTrace);
    }

    /**
     * Returns the name of the endpoint. 
     *
     * @return  name of the endpoint
     */
    String getEndpointName() {
        return _endpointId;
    }

    /**
     * Returns the fully qualified name of this endpoint.
     *
     * @return  fully qualified name of this endpoint
     */
    String getFQEndpointName() {
        return _applicationId + DELIM + _endpointId;
    }

    // ---- VARIABLES - PRIVATE ---------------------------------------
    private MessageFilter _filter     = null;
    private Pool _pool                = null;
    private String _endpointId        = null;
    private String _applicationId     = null;
    private static final String DELIM = "#";
    private static final Logger _logger = 
        Logger.getLogger(LogDomains.ADMIN_LOGGER);
    private static final StringManager _stringMgr = 
        StringManager.getManager(EndpointHandler.class);
}
