/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/* EjbContainerChangeHandler.java
 * $Id: EjbContainerChangeHandler.java,v 1.3 2005/12/25 03:43:39 tcfujii Exp $
 * $Revision: 1.3 $
 * $Date: 2005/12/25 03:43:39 $
 * Indentation Information:
 * 0. Please (try to) preserve these settings.
 * 1. Tabs are preferred over spaces.
 * 2. In vi/vim -
 *		:set tabstop=4 :set shiftwidth=4 :set softtabstop=4
 * 3. In S1 Studio -
 *		1. Tools->Options->Editor Settings->Java Editor->Tab Size = 4
 *		2. Tools->Options->Indentation Engines->Java Indentation Engine->Expand Tabs to Spaces = False.
 *		3. Tools->Options->Indentation Engines->Java Indentation Engine->Number of Spaces per Tab = 4.
 */

package com.sun.enterprise.admin.monitor.registry.spi.reconfig;

import java.util.Iterator;
import java.util.Map;
import java.util.logging.Logger;
import com.sun.enterprise.admin.monitor.registry.MonitoringLevel;
import com.sun.enterprise.admin.monitor.registry.MonitoredObjectType;
import com.sun.enterprise.admin.monitor.registry.StatsHolder;
import com.sun.enterprise.admin.monitor.registry.MonitoringLevelListener;

import com.sun.enterprise.admin.monitor.registry.spi.ValueListMap;
import com.sun.enterprise.admin.monitor.registry.spi.MonitoringRegistrationHelper;
import com.sun.enterprise.admin.common.constant.AdminConstants;

/**
 * Provides for dynamic reconfiguration of ejb container and related components like
 * cache, pool, methods. This class decides the actions to take when there are
 * changes to the monitoring level through administrative interfaces.
 * @author  <a href="mailto:Kedar.Mhaswade@sun.com">Kedar Mhaswade</a>
 * @since S1AS8.0
 * @version $Revision: 1.3 $
 */
class EjbContainerChangeHandler implements ChangeHandler {
	
	private final ChangeHandler		successor;
	private final ValueListMap		listeners;
	private static final Logger logger = Logger.getLogger(AdminConstants.kLoggerName);

	EjbContainerChangeHandler(ChangeHandler successor, ValueListMap listeners) {
		this.successor	= successor;
		this.listeners	= listeners;
	}
	
	public void handleChange(MonitoredObjectType t, MonitoringLevel from, MonitoringLevel to) {
		if (isEjbContainerType(t)) {
			handleChange(from, to, t);
		}
		else {
			successor.handleChange(t, from, to);
		}
	}
	
	private boolean isEjbContainerType(MonitoredObjectType t) {
		return (t == MonitoredObjectType.EJB			|| 
				t == MonitoredObjectType.BEAN_CACHE		||
				t == MonitoredObjectType.BEAN_METHOD	||
				t == MonitoredObjectType.BEAN_POOL		||
				t == MonitoredObjectType.ENTITY_BEAN	||
				t == MonitoredObjectType.STATEFUL_BEAN	||
				t == MonitoredObjectType.STATELESS_BEAN	||
				t == MonitoredObjectType.MESSAGE_DRIVEN_BEAN );
	}
	
	private void handleChange(MonitoringLevel from, MonitoringLevel to, MonitoredObjectType t) {
		if (off2Low(from, to) || off2High(from, to)) {
			boolean includeMethods = false;
			notifyListeners(from, to, t);
			// register the ejb method mbeans only if the "to" level is HIGH
			if(to == MonitoringLevel.HIGH)
				includeMethods = true;
			registerMBeans(includeMethods);
		}
		if (low2Off(from, to) || high2Off(from, to)) {
			boolean includeMethods = false;
			// need to include the ejb method mbeans, if the "from" level is HIGH
			if(from == MonitoringLevel.HIGH)
				includeMethods = true;
			unregisterMBeans(includeMethods);
			notifyListeners(from, to, t);
		}
		if (low2High(from, to)) {
			// register the ejb method mbeans
			notifyListeners(from, to, t);
			registerMethodMBeans();
		}
		if (high2Low(from, to)) {
			// unregister the ejb method mbeans
			unregisterMethodMBeans();
			notifyListeners(from, to, t);
		}
		
	}
	
	private void notifyListeners(MonitoringLevel from, MonitoringLevel to, MonitoredObjectType t) {
		logger.finer("DynamicReconfigurator: Now notifying the listeners for ejb stats --- from = " + from.toString() + " to = " + to.toString());
		final Map l = (Map)listeners.get(t); // map of listeners;
		if (l == null)
			return; //do nothing
		final Iterator it = l.keySet().iterator();
		while (it.hasNext()) {
			final MonitoringLevelListener ml = (MonitoringLevelListener)it.next();
			ml.changeLevel(from, to, t);
		}
	}
	
	
	private void registerMBeans(boolean includeMethods) {
		final MonitoringRegistrationHelper registryImpl	= 
			(MonitoringRegistrationHelper) MonitoringRegistrationHelper.getInstance();
		//note that the above refers to the actual implementation rather than interface.

		//registers MBeans pertaining to ejbs, pools, caches, methods
		final Iterator iter = registryImpl.getEjbContainerNodes(includeMethods).iterator();
		while (iter.hasNext()) {
			final StatsHolder c = (StatsHolder) iter.next();
			c.registerMBean();
			logger.finer("DynamicReconfigurator: Now Registering MBean for --- " + c.getName());
		}
	}
	
	private void unregisterMBeans(boolean includeMethods) {
		final MonitoringRegistrationHelper registryImpl	= 
			(MonitoringRegistrationHelper) MonitoringRegistrationHelper.getInstance();
		//note that the above refers to the actual implementation rather than interface.

		final Iterator iter = registryImpl.getEjbContainerNodes(includeMethods).iterator();
		while (iter.hasNext()) {
			final StatsHolder c = (StatsHolder) iter.next();
			c.unregisterMBean();
			logger.finer("DynamicReconfigurator: Now UnRegistering MBean for --- " + c.getName());
		}
	}
	
	
	private void registerMethodMBeans() {
		final MonitoringRegistrationHelper registryImpl	= 
			(MonitoringRegistrationHelper) MonitoringRegistrationHelper.getInstance();

		final Iterator iter = registryImpl.getEjbMethodNodes().iterator();
		while (iter.hasNext()) {
			final StatsHolder c = (StatsHolder) iter.next();
			c.registerMBean();
			logger.finer("DynamicReconfigurator: Now Registering MBean for --- " + c.getName());
		}
	}
	

	private void unregisterMethodMBeans() {
		final MonitoringRegistrationHelper registryImpl	= 
			(MonitoringRegistrationHelper) MonitoringRegistrationHelper.getInstance();

		final Iterator iter = registryImpl.getEjbMethodNodes().iterator();
		while (iter.hasNext()) {
			final StatsHolder c = (StatsHolder) iter.next();
			c.unregisterMBean();
			logger.finer("DynamicReconfigurator: Now UnRegistering MBean for --- " + c.getName());
		}
	}
	
	private boolean off2Low(MonitoringLevel from, MonitoringLevel to) { 
		return ( from == MonitoringLevel.OFF && to == MonitoringLevel.LOW );
	}
	private boolean off2High(MonitoringLevel from, MonitoringLevel to) { 
		return ( from == MonitoringLevel.OFF && to == MonitoringLevel.HIGH );
	}
	private boolean low2Off(MonitoringLevel from, MonitoringLevel to) { 
		return ( from == MonitoringLevel.LOW && to == MonitoringLevel.OFF);
	}
	private boolean high2Off(MonitoringLevel from, MonitoringLevel to) { 
		return ( from == MonitoringLevel.HIGH && to == MonitoringLevel.OFF );
	}
	private boolean low2High(MonitoringLevel from, MonitoringLevel to) { 
		return ( from == MonitoringLevel.LOW && to == MonitoringLevel.HIGH);
	}
	private boolean high2Low(MonitoringLevel from, MonitoringLevel to) {
		return ( from == MonitoringLevel.HIGH && to == MonitoringLevel.LOW );
	}
}
