/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.tools.guiframework.view.descriptors;

import com.iplanet.jato.RequestContext;
import com.iplanet.jato.RequestManager;
import com.iplanet.jato.RequestParticipant;
import com.iplanet.jato.view.ContainerView;
import com.iplanet.jato.view.ContainerViewBase;
import com.iplanet.jato.view.View;
import com.iplanet.jato.view.ViewBean;
import com.iplanet.jato.view.event.DisplayEvent;

import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

import com.sun.enterprise.tools.guiframework.FrameworkDescriptor;
import com.sun.enterprise.tools.guiframework.event.descriptors.EventDescriptor;
import com.sun.enterprise.tools.guiframework.exception.FrameworkError;
import com.sun.enterprise.tools.guiframework.exception.FrameworkException;
import com.sun.enterprise.tools.guiframework.util.LogUtil;
import com.sun.enterprise.tools.guiframework.util.Util;
import com.sun.enterprise.tools.guiframework.view.ChildNullException;
import com.sun.enterprise.tools.guiframework.view.DescriptorContainerView;
import com.sun.enterprise.tools.guiframework.view.DescriptorViewManager;


/**
 *
 */
public class ViewDescriptor implements FrameworkDescriptor {

    /**
     *	Name / Class Name Constructor
     *
     *	@param name	The name (or id) of the descriptor
     */
    public ViewDescriptor(String name) {
	setName(name);
	addChildDescriptors();
    }


    /**
     *	This method sets the instance name for the Described View.
     *
     *	@return The instance name for the Described View.
     */
    public String getName() {
	return _name;
    }


    /**
     *	This method returns the instance name for the Described View.
     *
     *	@param name	The name for the described View.
     */
    public void setName(String name) {
	if (name == null) {
	    throw new IllegalArgumentException("'name' cannot be null!");
	}
	_name = name;
    }


    /**
     *	For future tool support
     */
    public String getDescription() {
	return _description;
    }


    /**
     *	For future tool support
     */
    public void setDescription(String desc) {
	_description = desc;
    }


    /**
     *	This method throws a FrameworkException if it is called.  It is
     *	expected that other descriptors will override this method and provide
     *	the correct way to instantiate the requested View.
     *
     *	<BLOCKQUOTE>
     *	NOTE: This method should not be invoked directly if you want the
     *	ViewManager to manage the instance created.  This method does not
     *	register the newly created instance with the View Manager.
     *	</BLOCKQUOTE>
     *
     *	@param	ctx		The RequestContext
     *	@param	container	The parent container (if any)
     *	@param	name		The name of the View to create
     *
     *	@return A newly created View of getViewClass() type.
     */
    public View getInstance(RequestContext ctx, ContainerView container, String name) {
	throw new FrameworkException("Descriptor type for '"+name+
	    "' is ViewDescriptor, this is not valid.", this, container);
    }


    /**
     *	<P>This method retrieves the corresponding View object for this
     *	ViewDescriptor.  This is useful for obtaining nested Views when you do
     *	not have a reference to the parent container.  Remember you cannot
     *	simply call "getInstance()" on a ViewDescriptor as this will create a
     *	NEW instance.  Instead the top level ViewDescriptor must obtain its
     *	peer via the ViewManager (unless a sub-view registers itself with the
     *	ViewManager).  This method does this for you... it obtains the TopLevel
     *	ViewDescriptor from the given ViewDescriptor, obtains the View peer and
     *	walks the tree to the child that matches the given ViewDescriptor.</P>
     *
     *	<BLOCKQUOTE>
     *	WARNING: Do not invoke this method in a beforeCreate.  Since this
     *	method walks the View tree, it will attempt to create the View that is
     *	the subject of the beforeCreate.  When this happens the beforeCreate
     *	will be fired, which will try to create the View again... which will
     *	fire the before create... etc., etc., etc., etc.  You get the idea,
     *	don't do it.  In general you should always be careful with the tasks
     *	you perform in the beforeCreate events.
     *	</BLOCKQUOTE>
     *
     *	@param	ctx		The RequestContext
     *
     *	@return	View corresponding to this ViewDescriptor
     */
    public View getView(RequestContext ctx) {
	Stack descStack = new Stack();

	// Find the top
	ViewDescriptor topDesc = this;
	while (topDesc.getParent() != null) {
	    descStack.push(topDesc.getName());
	    topDesc = topDesc.getParent();
	}

	// Get the top's peer.  We use the ViewManager in order to get a cached
	// instance.  This should usually work b/c the ViewBean (the top should
	// be a ViewBean) gets created early during the submit cycle.  Although
	// it may be possible this Descriptor is part of a different ViewBean
	// that hasn't been referenced yet.
	View view = ((DescriptorViewManager)ctx.getViewBeanManager()).
	    getView(null, topDesc.getName(), topDesc);

	// Use the ViewDescriptors to get to the correct View child
	while (!descStack.empty()) {
	    view = ((ContainerView)view).getChild((String)descStack.pop());
	}

	// Return the requested View
	return view;
    }


    /**
     *
     */
    public void registerChildren(ContainerViewBase instance) {
	// Do all the children
	Iterator it = getChildDescriptors().iterator();
	ViewDescriptor desc = null;
	View child = null;

	// Trace message...
	if (LogUtil.isLoggable(LogUtil.FINER) && it.hasNext()) {
	    LogUtil.log(LogUtil.FINER, "trace.registerChildren", getName());
	}

	while (it.hasNext()) {
	    desc = (ViewDescriptor)it.next();
	    // NOTE: calling getChild() will often do the child registration,
	    // NOTE: however -- unless the constructor is aware of descriptors
	    // NOTE: -- children declared via descriptors won't get registered
	    // NOTE: unless we do it here
	    try {
		// Trace message...
		if (LogUtil.isLoggable(LogUtil.FINEST) && it.hasNext()) {
		    LogUtil.log(LogUtil.FINEST, "trace.registerChild",
			instance.getName()+"."+desc.getName());
		}

		child = instance.getChild(desc.getName());
		instance.registerChild(desc.getName(), child.getClass());
		if (child instanceof DescriptorContainerView) {
		    // All DescriptorContainerView's are required to register
		    // their own Descriptor children.  Doing so again here
		    // would just waste time.
		    continue;
		} else if (child instanceof ContainerViewBase) {
		    desc.registerChildren((ContainerViewBase)child);
		} else if (desc instanceof FakeContainerDescriptor) {
		    // Fake Containers have children, but aren't necessarily
		    // ContainerView's :(  So we will register their children
		    // in the parent container so acceptRequest() will work
		    desc.registerChildren(instance);
		}
	    } catch (ChildNullException ex) {
		// Ignore b/c some components can't be created (no View
		// corresponding to JSP tag)
	    } catch (Exception ex) {
		throw new FrameworkError(
		    ex, desc, (child == null) ? instance : child);
	    }
	}
    }


    /**
     *	Accessor for the DisplayURL.
     *
     *	@return The DisplayURL for this descriptor.
     */
    public String getDisplayURL() {
	return _displayURL;
    }


    /**
     *	Allows the DisplayURL to be set.
     *
     *	@param url	The DisplayURL to use.
     */
    public void setDisplayURL(String url) {
	_displayURL = url;
    }


    /**
     *	This method returns a EventDescriptor Object for the given type.  If
     *	there is not a registered EventDescriptor Object for the requested
     *	type, then null will be returned.
     *
     *	@param	type	The EventDescriptor.TYPES type
     *
     *	@return EventDescriptor for the requested type.
     */
    public EventDescriptor getEventDescriptor(String type) {
	return (EventDescriptor)_eventDescriptors.get(type);
    }


    /**
     *	This method sets an EventDescriptor.  Only 1 EventDescriptor can be
     *	set for a given type.  If an EventDescriptor has been previously set,
     *	this will replace the previous EventDescriptor.
     *
     *	@param desc	The EventDescriptor
     */
    public void setEventDescriptor(EventDescriptor desc) {
	_eventDescriptors.put(desc.getType(), desc);
    }


    /**
     *	This method allows you to add a child ViewDescriptor.
     *
     *	@param fieldDesc	The ViewDescriptor to add
     */
    public void addChildDescriptor(ViewDescriptor fieldDesc) {
        fieldDesc.setParent(this);
	_childDescriptors.add(fieldDesc);
	_childDescriptorMap = null;
    }


    /**
     *	This method allows you to retrieve the child descriptors as a
     *	List.
     *
     *	@return	List of ViewDescriptors
     */
    public List getChildDescriptors() {
	return _childDescriptors;
    }


    /**
     *	This method allows you to set the complete list of child descriptors
     *	to be used.
     *
     *	@param fields	The List of ViewDescriptors to be set
     */
    public void setChildDescriptors(List fields) {
	_childDescriptors = fields;
	_childDescriptorMap = null;
    }


    /**
     *	This method retrieves the requested child ViewDescriptor.  You may use
     *	.'s in the name to represent a path of child ViewDescriptors to
     *	traverse to obtain the desired ViewDescriptor.  If the ViewDescriptor
     *	is not found, null will be returned.
     *
     *	@param name	The name of the desired child descriptor
     *
     *	@return The requested ViewDescriptor or null if not found.
     */
    public ViewDescriptor getChildDescriptor(String name) {
	if (_childDescriptorMap == null) {
	    // If we haven't created the Map yet, create it.
	    Iterator it = getChildDescriptors().iterator();
	    ViewDescriptor desc = null;
	    Map newMap = new HashMap();
	    while (it.hasNext()) {
		desc = (ViewDescriptor)it.next();
		newMap.put(desc.getName(), desc);
	    }
	    if (_childDescriptorMap == null) {
		_childDescriptorMap = newMap;
	    }
	}
	ViewDescriptor child = (ViewDescriptor )_childDescriptorMap.get(name);
	if (child == null) {
	    // Perhaps this name is a hierarchical .'d name...
	    int pos = name.indexOf('.');
	    if (pos != -1) {
		// Check first name
		child = getChildDescriptor(name.substring(0, pos));
		if (child != null) {
		    // Recurse to take care of the rest of the .'s
		    return child.getChildDescriptor(name.substring(pos+1));
		}
	    }
	}
	return child;
    }


    /**
     *	Returns the Parent ViewDescriptor
     *
     *	@return	The Parent ViewDescriptor
     */
    public ViewDescriptor getParent() {
        return _parent;
    }

    
    /**
     *	Sets the Parent ViewDescriptor
     *
     *	@param	viewDesc    The Parent ViewDescriptor
     */
    public void setParent(ViewDescriptor viewDesc) {
        _parent = viewDesc;
    }
    

    /**
     *	This method allows all the parameters to be set at once by passing in
     *	a Map object.  The Map object should contain parameter-name to
     *	parameter-value pairs.  Parameters may be used to provide additional
     *	instance-specific information.
     *
     *	@param	parameters	The Map of parameters.
     */
    public void setParameters(Map parameters) {
	if (parameters == null) {
	    throw new IllegalArgumentException(
		"The parameter map cannot be null!");
	}
	_parameters = parameters;
    }

    public Set getParameterKeys() {
	return _parameters.keySet();
    }

    /**
     *	<P>This method retrieves the Map of parameters.</P>
     *
     *	@return	The Map of parameters.
     */
    public Map getParameters() {
	Map newMap = new HashMap(_parameters.size());
	Iterator it = _parameters.keySet().iterator();
	Object key;
	Object value;
	// Iterate through the keys, and make the new Map w/ substitutions
	while (it.hasNext()) {
	    key = it.next();
	    value = Util.replaceVariablesWithAttributes(
		_parameters.get(key), this);
	    newMap.put(key, value);
	}
	return newMap;
    }


    /**
     *	This method addes a parameter.
     *
     *	@param name	The Parameter name
     *	@param value	The Parameter value
     */ 
    public void addParameter(String name, Object value) {
	if (name == null) {
	    throw new IllegalArgumentException("'name' cannot be null!");
	}
	_parameters.put(name, value);
    }


    /**
     *	This method retrieves the requested parameter.  If the parameter is not
     *	found, this method will return null.
     *
     *	@param	name	The name of the desired parameter.
     *
     *	@return	The parameter value, or nulll if not found.
     */
    public Object getParameter(String name) {
	return Util.replaceVariablesWithAttributes(_parameters.get(name), this);
    }
    

    /**
     *	The full name of the Model Class.  This used the MODEL_CLASS_NAME
     *	parameter or DEFAULT_MODEL if MODEL_CLASS_NAME is not supplied.
     *
     *	@return The full name of the Model Class.
     */
    public String getModelClassName() {
	String clsName = (String)getParameter(MODEL_CLASS_NAME);
	if (clsName == null) {
	    clsName = DEFAULT_MODEL;
	}
	return clsName;
    }


    /**
     *	This method provides the default Model instance name, which is simply
     *	the name of this ViewDescriptor.  Some ViewDescriptors may override
     *	this method to provide a more unique name if desired.
     *
     *	@return	The default model instance name (this.getName() for this
     *		implementation).
     */
    public String getDefaultModelInstanceName() {
	return getName();
    }


    /**
     *	<P>The model instance name is an arbitrary name to refer to the
     *	instance of the Model.  In most cases the model class name will work
     *	for the instance name, or some other semi-unique value.</P>
     *
     *	<P>This method will use the MODEL_INSTANCE_NAME parameter.  If this is
     *	null, this.getName() will be used which is more unique than the model
     *	class name.</P>
     *
     *	@return The model instance name.
     */
    public String getModelInstanceName() {
	String instName = (String)getParameter(MODEL_INSTANCE_NAME);
	if (instName == null) {
	    instName = getDefaultModelInstanceName();
	}
	return instName;
    }


    /**
     *	Returns true if the model should be retrieved from session.  This
     *	method uses the GET_MODEL_FROM_SESSION parameter to determine if
     *	session should be used.  The default is "false", "true" should be set
     *	as a the parameter value to use session.
     *
     *	@return true if the model should be retrieved from session.
     */
    public boolean shouldGetModelFromSession() {
	return new Boolean(""+getParameter(GET_MODEL_FROM_SESSION)).booleanValue();
    }


    /**
     *	Returns true if the model should be stored in session.  This method
     *	uses the PUT_MODEL_TO_SESSION parameter to determine if session should
     *	be used.  The default is "false", "true" should be set as a the
     *	parameter value to use session.
     *
     *	@return true if the model should be stored in session.
     */
    public boolean shouldPutModelToSession() {
	return new Boolean(""+getParameter(PUT_MODEL_TO_SESSION)).booleanValue();
    }


    /**
     *	This method returns the resource bundle, it must be specified as a
     *	parameter of the ViewDescriptor (or a parameter of one its parents).
     *	If not declared, null will be returned.
     *
     *	@return The resourceBundle
     */
    public String getResourceBundle() {
	String bundle = (String)getParameter(RESOURCE_BUNDLE);

	// Look at parent ViewDescriptors if we need to
	for (ViewDescriptor vd = getParent();
		((bundle == null) && (vd != null)); vd = vd.getParent()) {
	    bundle = (String)vd.getParameter(RESOURCE_BUNDLE);
	}

	// Return the resource bundle or null
	return bundle;
    }


    /**
     *	The XML File name backing the described component (i.e.
     *	propertySheet.xml or table.xml).
     *
     *	@return The XML File Name or null
     */
    public String getXMLFileName() {
	return (String)getParameter(XML_FILE);
    }


    /**
     *	This method attempts to open a stream to the XML file.
     *
     *	@return An InputStream to the XML file.
     */
    public InputStream getXMLFileAsStream() {
	String fileName = getXMLFileName();
	if ((fileName == null) || (fileName.equals(""))) {
	    throw new FrameworkException(
		"Parameter '"+XML_FILE+"' is missing for '"+
		this.getClass().getName()+":"+getName()+"'.");
	}
	try {
            // try loading from the classpath with the call loader. This is cached.
            InputStream inStream = getClass().getClassLoader().getResourceAsStream(fileName);
            if (inStream == null) {
                // try getting the file from the context root. This is not cached.
		inStream = RequestManager.getRequestContext().getServletContext().
		    getResourceAsStream(fileName);
                if (inStream == null) {
                    throw new FrameworkException("InputStream (null) for '"+fileName+"'.");
                }
	    }
	    return inStream;
	} catch (Exception ex) {
	    throw new FrameworkException("Unable to open '"+fileName+"' for '"+
		this.getClass().getName()+":"+getName()+"'.", ex);
	}
    }
    

    /**
     *	The toString() returns getName()
     */
    public String toString() {
	return getName();
    }


    /**
     *	<P>This method gets invoked by the constructor, however it does nothing
     *	by default.  The purpose of this method is to provide a designated
     *	place for programatically adding child descriptors when extending this
     *	class.  To add a child descriptor, invoke the method:</P>
     *
     *	<BLOCKQUOTE>
     *	    addChildDescriptor(ViewDescriptor fieldDesc)
     *	</BLOCKQUOTE>
     *
     *	@see #addChildDescriptor(ViewDescriptor)
     */
    protected void addChildDescriptors() {
    }


    /**
     *	This is the Model class.  If a parameter of this name is supplied, its
     *	value will be used; otherwise the value of DEFAULT_MODEL will be used.
     *	("modelClassName")
     */
    public static final String MODEL_CLASS_NAME =	"modelClassName";

    /**
     *	The default model name.  ("com.iplanet.jato.model.DefaultModel")
     */
    public static final String DEFAULT_MODEL =
	    "com.iplanet.jato.model.DefaultModel";

    /**
     *	This is the ModelManager key the model will be stored as.  If a
     *	parameter of this name is supplied, its value will be used; otherwise
     *	this.getName() will be used as the instance name. ("modelInstanceName")
     */
    public static final String MODEL_INSTANCE_NAME =	"modelInstanceName";

    /**
     *	This parameter should be true or false depending of if you would like
     *	to retrieve the model from session.
     *	"modelFromSession"
     */
    public static final String GET_MODEL_FROM_SESSION =	"modelFromSession";

    /**
     *	This parameter should be true or false depending of if you would like
     *	to store the model in session.
     *	"modelToSession"
     */
    public static final String PUT_MODEL_TO_SESSION =	"modelToSession";


    /**
     *	This is the parameter to specify the XML file backing this component.
     */
    public static final String XML_FILE =		"xmlFile";

    /**
     *	This parameter specifies the resourceBundle to used for I18N.
     *	"resourceBundle"
     */
    public static final String  RESOURCE_BUNDLE 	= "resourceBundle";


    private Map			_eventDescriptors = new HashMap();
    private List		_childDescriptors = new ArrayList();
    private Map			_childDescriptorMap = null;
    private Map			_parameters = new HashMap();
    private String		_name = "";
    private String		_description = "";
    private String		_displayURL = null;
    private ViewDescriptor	_parent = null;
}
