/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */

package com.sun.enterprise.tools.guiframework.event.descriptors;

import com.iplanet.jato.RequestContext;

import com.sun.enterprise.tools.guiframework.FrameworkDescriptor;
import com.sun.enterprise.tools.guiframework.exception.FrameworkException;
import com.sun.enterprise.tools.guiframework.util.PermissionChecker;
import com.sun.enterprise.tools.guiframework.util.Util;
import com.sun.enterprise.tools.guiframework.view.HandlerContext;
import com.sun.enterprise.tools.guiframework.view.descriptors.ViewDescriptor;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 *  This class describes an event handler (method).  Optionally, "mappings"
 *  can be performed before or after a handler.  This class maintains 2 lists
 *  of mappings, one for before and one for after.
 */
public class HandlerDescriptor implements FrameworkDescriptor {

    /**
     *	Constructor.
     *
     *	@param	name	The name
     */
    public HandlerDescriptor(String name) {
	setName(name);
    }


    /**
     *	This method returns the name for this handler definition.
     */
    public String getName() {
	return _name;
    }


    /**
     *	This method sets the handler definitions name (used by the contsrutor).
     */
    protected void setName(String name) {
	if (name == null) {
	    throw new FrameworkException("Handler name cannot be null!");
	}
	_name = name;
    }


    /**
     *	For future tool support
     */
    public String getDescription() {
	return _description;
    }


    /**
     *	For future tool support
     */
    public void setDescription(String desc) {
	_description = desc;
    }


    /**
     *	This method returns the EventDescriptor that contains this handler.
     *
     *	@return The parent descriptor
     */
//NOTE: HandlerDescriptor.getParent() will not work because we are not going to fully instantiate the tree (it may have multiple parents).  See if we really need this anyway.  If we do, consider adding an execution stack instead.
    public FrameworkDescriptor getParent() {
	throw new FrameworkException(
	    "Get parent is not implemented for HandlerDescriptors!");
    }


    /**
     *	This method sets the parent EventDescriptor.
     *
     *	@param	parent	The parent EventDescriptor
     * /
    protected void setParent(EventDescriptor parent) {
	if (parent == null) {
	    throw new RuntimeException("You must provide a the " +
		"EventDescriptor which contains this HandlerDescriptor!");
	}
	_parent = parent;
    }
*/


    /**
     *	<P>This method sets the event handler (method) to be invoked.  The
     *	method should be public; accept 2 parameters: 1 of type
     *	"RequestContext", one of type "HandlerContext"  Example:</P>
     *
     *	<P> <BLOCKQUOTE>
     *		public void beginDisplay(
     *		     	RequestContext ctx, HandlerContext handlerCtx)
     *			throws ModelControlException
     *	    </BLOCKQUOTE></P>
     *
     *	@param cls	    The full class name containing method
     *	@param methodName   The method name of the handler within class
     */
    public void setHandlerMethod(String cls, String methodName) {
	if ((cls == null) || (methodName == null)) {
	    throw new IllegalArgumentException(
		"Class name and method name must be non-null!");
	}

	// Save class / method name info
	_cls = cls;
	_methodName = methodName;
    }

    private void setMethod() {
	if ((_cls == null) || (_methodName == null)) {
	    return;
	}

	// Find the class
	Class clzz = null;
	try {
	    clzz = Class.forName(_cls);
	} catch (ClassNotFoundException ex) {
	    throw new RuntimeException("Class '"+_cls+"' not found!", ex);
	}

	// Find the method on the class
	Method method = null;
	try {
	    method = clzz.getMethod(_methodName, EVENT_ARGS);
	} catch (NoSuchMethodException ex) {
	    throw new RuntimeException("Method '"+_methodName+"' not found!", ex);
	}

	// Add to the list
	_method = method;
    }

    public void setHandlerMethod(Method method) {
	_method = method;
	_cls = method.getDeclaringClass().getName();
	_methodName = method.getName();
    }

    
    public Method getHandlerMethod() {
	if (_method == null) {
	    setMethod();
	}
	return _method;
    }


    /**
     *	This method adds an IODescriptor to the list of input descriptors.
     *	These descriptors define the input parameters to this handler.
     *
     *	@param desc	The input IODescriptor to add
     */
    public void addInputDescriptor(IODescriptor desc) {
	_inputDescriptors.put(desc.getName(), desc);
    }


    /**
     *	This method sets the input IODescriptors for this handler.
     *
     *	@param inputDescriptors	The Map of IODescriptors
     */
    public void setInputDescriptors(Map inputDescriptors) {
	if (inputDescriptors == null) {
	    throw new IllegalArgumentException(
		"inputDescriptors cannot be null!");
	}
	_inputDescriptors = inputDescriptors;
    }


    /**
     *	This method retrieves the Map of input IODescriptors.
     *
     *	@return The Map of IODescriptors
     */
    public Map getInputDescriptors() {
	return _inputDescriptors;
    }


    /**
     *	This method returns the requested IODescriptor, null if not found.
     */
    public IODescriptor getInputDescriptor(String name) {
	return (IODescriptor)_inputDescriptors.get(name);
    }


    /**
     *	This method adds an IODescriptor to the list of output descriptors.
     *	These descriptors define the output parameters to this handler.
     *
     *	@param desc	The IODescriptor to add
     */
    public void addOutputDescriptor(IODescriptor desc) {
	_outputDescriptors.put(desc.getName(), desc);
    }


    /**
     *	This method sets the output IODescriptors for this handler.
     *
     *	@param outputDescriptors    The Map of output IODescriptors
     */
    public void setOutputDescriptors(Map outputDescriptors) {
	if (outputDescriptors == null) {
	    throw new IllegalArgumentException(
		"outputDescriptors cannot be null!");
	}
	_outputDescriptors = outputDescriptors;
    }


    /**
     *	This method retrieves the Map of output IODescriptors.
     *
     *	@return The Map of output IODescriptors
     */
    public Map getOutputDescriptors() {
	return _outputDescriptors;
    }


    /**
     *	This method returns the requested IODescriptor, null if not found.
     */
    public IODescriptor getOutputDescriptor(String name) {
	return (IODescriptor)_outputDescriptors.get(name);
    }


    /**
     *	This method adds a useHandler to the list of child handlers.  Child
     *	handlers are executed PRIOR to this handler executing.
     *
     *	@param desc	The UseHandlerDescriptor to add
     */
    public void addChildHandlerDescriptor(UseHandlerDescriptor desc) {
	_childHandlers.add(desc);
    }


    /**
     *	This method sets the List of child handlers for this handler.
     *
     *	@param childHandlers	The List of child handlers
     */
    public void setChildHandlerDescriptors(List childHandlers) {
	if (childHandlers == null) {
	    throw new IllegalArgumentException(
		"childHandlers cannot be null!");
	}
	_childHandlers = childHandlers;
    }


    /**
     *	This method retrieves the List of child UseHandlerDescriptors.
     *
     *	@return The List of child UseHandlerDescriptors for this handler.
     */
    public List getChildHandlerDescriptors() {
	return _childHandlers;
    }


    /**
     *	Accessor for the ifCheck String
     */
    public String getIfCheck() {
	return _ifCheck;
    }


    /**
     *	Setter for the ifCheck String
     */
    public void setIfCheck(String ifCheck) {
	_ifCheck = ifCheck;
    }


    /**
     *	This method uses a PermissionChecker to evaluate the "ifCheck
     *	String".  The ifCheck String is a boolean expression, see
     *	PermissionChecker for details.  If the ifCheck String is null,
     *	then this method will return true.  $(attribute-name) entries within
     *	the ifCheck String will be evaluated; they should evaluate to the
     *	Strings "true" or "false".  
     *
     *	@param	vd	The ViewDescriptor
     *
     *	@return	True if this Mapping has permission, false if not
     */
    public boolean hasPermission(ViewDescriptor vd) {
	// Get the if="" value
	String ifCheck = getIfCheck();
	if ((ifCheck == null) || (ifCheck.trim().length() == 0)) {
	    // Default is to allow access
	    return true;
	}

	// Delegate to permission checker to determine result
	return new PermissionChecker(ifCheck, vd).hasPermission();
    }


    public static final Class[] EVENT_ARGS = new Class[] {RequestContext.class, HandlerContext.class};

// FIXME: Make sure we don't need this (consider execution stack instead)
//    private EventDescriptor	_parent = null;
    private String		_name			= null;
    private String		_description		= null;
    private String		_cls			= null;
    private String		_methodName		= null;
    private transient Method	_method			= null;
    private String		_ifCheck		= null;
    private Map			_inputDescriptors	= new HashMap(5);
    private Map			_outputDescriptors	= new HashMap(5);
    private List		_childHandlers		= new ArrayList(5);
}
