/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
 
/*
 * $Header: /cvs/glassfish/admin-core/mbeanapi/src/java/com/sun/appserv/management/helper/Attic/Connect.java,v 1.4 2005/12/25 03:50:35 tcfujii Exp $
 * $Revision: 1.4 $
 * $Date: 2005/12/25 03:50:35 $
 */
package com.sun.appserv.management.helper;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import javax.net.ssl.HandshakeCompletedListener;

import com.sun.appserv.management.base.UploadDownloadMgr;

import com.sun.appserv.management.DomainRoot;
import com.sun.appserv.management.client.TLSParams;
import com.sun.appserv.management.client.AppserverConnectionSource;
import com.sun.appserv.management.client.HandshakeCompletedListenerImpl;
import com.sun.appserv.management.client.TrustStoreTrustManager;

/**
	Miscellaneous helper routines for connecting to the Appserver.
	@since AppServer 9.0
 */
public class Connect
{
	private	Connect()	{}
	
	/**
		Connect to an Appserver using TLS (SSL) using the default TrustStore and 
		default TrustStore password.
		<p>
		If the server certificate is unknown, prompting will occur via System.out if
		'promptForNewCertificate' is true; otherwise the connection will be rejected.
		<p>
		If a new server certificate is found, and the user enters "yes", then it
		is added to the default truststore.
		
		@param host	hostname or IP address
		@param port	port to which to connect
		@param user admin user name
		@param userPassword password for specified user
		@param promptForUnknownCertificate	prompts via System.out if the server
			certificate is unrecognized, otherwise rejects the connection
	 */
		public static AppserverConnectionSource
	connectTLS(
		final String host,
		final int	 port,
		final String user,
		final String userPassword,
		final boolean promptForUnknownCertificate )
		throws IOException
	{
		final TLSParams	tlsParams	= createTLSParams( null, promptForUnknownCertificate );
		
		return new AppserverConnectionSource( AppserverConnectionSource.PROTOCOL_RMI,
				host, port, user, userPassword, tlsParams, null);
	}
	
		
		public static AppserverConnectionSource
	connectNoTLS(
		final String host,
		final int	 port,
		final String user,
		final String userPassword)
		throws IOException
	{
		return new AppserverConnectionSource(
				host, port, user, userPassword, null);
	}
	
	
	/**
		The File will use the name
		{@link AppserverConnectionSource#DEFAULT_TRUST_STORE_NAME}
		in the user's home directory.
		
		@return a File for the default truststore.  May not yet exist.
	 */
		public static File
	getDefaultTrustStore()
	{
		final String homeDir = System.getProperty( "user.home" );
		final String sep     = System.getProperty( "file.separator" );
		
		return new File ( homeDir + sep + AppserverConnectionSource.DEFAULT_TRUST_STORE_NAME );
	}
	
	/**
		Get TLSParams for the default truststore, assuming the default password.
		
		@param trustStorePassword a truststore, or null for the default one
		@param promptForNewCertificate
	 */
		public static TLSParams
	createTLSParams(
	    final String trustStorePassword,
	    final boolean promptForNewCertificate)
	{
		return createTLSParams( null, trustStorePassword , promptForNewCertificate);
	}
	

	/**
		Get TLSParams for the specified truststore and password.
		<p>
		A {@link HandshakeCompletedListener} is installed which may be obtained
		by calling {@link TLSParams#getHandshakeCompletedListener}.
		<p>
		If a new server certificate is found, and the user enters "yes" in response
		to prompting, then the certificate is added to the truststore.
		
		@param trustStore a truststore, or null for the default one
		@param trustStorePasswordIn the truststore password, if null the default one will be used
		@param promptForUnknownCertificate	prompts via System.out if the server
			certificate is unrecognized
	 */
		public static TLSParams
	createTLSParams(
		final File		trustStore,
		final String	trustStorePasswordIn,
		final boolean	promptForUnknownCertificate )
	{
		final File		trustStoreFile	= (trustStore == null) ? getDefaultTrustStore() : trustStore;
		final char[]	trustStorePassword	= ((trustStorePasswordIn == null) ?
			AppserverConnectionSource.DEFAULT_TRUST_STORE_PASSWORD : trustStorePasswordIn).toCharArray();
					
		final HandshakeCompletedListener	handshakeCompletedListener	=
			new HandshakeCompletedListenerImpl();
			
		final TrustStoreTrustManager trustMgr =
			new TrustStoreTrustManager( trustStoreFile, trustStorePassword);
			
		trustMgr.setPrompt( promptForUnknownCertificate );

		final TLSParams	tlsParams = new TLSParams( trustMgr, handshakeCompletedListener );

		return( tlsParams );
	}
}







