/*
 * $Header: /home/cvs/jakarta-commons/httpclient/src/test/org/apache/commons/httpclient/TestStreams.java,v 1.11.2.2 2004/08/09 01:22:05 mbecke Exp $
 * $Revision: 1.11.2.2 $
 * $Date: 2004/08/09 01:22:05 $
 * ====================================================================
 *
 *  Copyright 2002-2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.commons.httpclient;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.apache.commons.httpclient.methods.GetMethod;


public class TestStreams extends TestCase {

    public TestStreams(String testName) {
        super(testName);
    }

    public void testChunkedInputStream() throws IOException {
        String correctInput = "10;key=\"value\r\nnewline\"\r\n1234567890123456\r\n5\r\n12345\r\n0\r\nFooter1: abcde\r\nFooter2: fghij\r\n";
        String correctResult = "123456789012345612345";
        HttpMethod method = new SimpleHttpMethod();

        //Test for when buffer is larger than chunk size
        InputStream in = new ChunkedInputStream(new ByteArrayInputStream(HttpConstants.getBytes(correctInput)), method);
        byte[] buffer = new byte[300];
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int len;
        while ((len = in.read(buffer)) > 0) {
            out.write(buffer, 0, len);
        }
        String result = HttpConstants.getContentString(out.toByteArray());
        assertEquals(result, correctResult);
        Header footer = method.getResponseFooter("footer1");
        assertEquals(footer.getValue(), "abcde");
        footer = method.getResponseFooter("footer2");
        assertEquals(footer.getValue(), "fghij");

        // recycle the method so that it can be reused below
        method.recycle();

        //Test for when buffer is smaller than chunk size.
        in = new ChunkedInputStream(new ByteArrayInputStream(HttpConstants.getBytes(correctInput)), method);
        buffer = new byte[7];
        out = new ByteArrayOutputStream();
        while ((len = in.read(buffer)) > 0) {
            out.write(buffer, 0, len);
        }
        result = HttpConstants.getContentString(out.toByteArray());
        assertEquals(result, correctResult);
        footer = method.getResponseFooter("footer1");
        assertEquals(footer.getValue(), "abcde");
        footer = method.getResponseFooter("footer2");
        assertEquals(footer.getValue(), "fghij");
    }

    public void testCorruptChunkedInputStream1() throws IOException {
        //missing \r\n at the end of the first chunk
        String corrupInput = "10;key=\"value\"\r\n123456789012345\r\n5\r\n12345\r\n0\r\nFooter1: abcde\r\nFooter2: fghij\r\n";
        HttpMethod method = new SimpleHttpMethod();

        InputStream in = new ChunkedInputStream(new ByteArrayInputStream(HttpConstants.getBytes(corrupInput)), method);
        byte[] buffer = new byte[300];
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int len;
        try {
            while ((len = in.read(buffer)) > 0) {
                out.write(buffer, 0, len);
            }
            fail("Should have thrown exception");
        } catch(IOException e) {
            /* expected exception */
        }
    }

    public void testEmptyChunkedInputStream() throws IOException {
        String input = "0\r\n";
        HttpMethod method = new SimpleHttpMethod();

        InputStream in = new ChunkedInputStream(new ByteArrayInputStream(HttpConstants.getBytes(input)), method);
        byte[] buffer = new byte[300];
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int len;
        while ((len = in.read(buffer)) > 0) {
            out.write(buffer, 0, len);
        }
        assertEquals(0, out.size());
    }

    public void testContentLengthInputStream() throws IOException {
        String correct = "1234567890123456";
        InputStream in = new ContentLengthInputStream(new ByteArrayInputStream(HttpConstants.getBytes(correct)), 10);
        byte[] buffer = new byte[50];
        int len = in.read(buffer);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        out.write(buffer, 0, len);
        String result = HttpConstants.getContentString(out.toByteArray());
        assertEquals(result, "1234567890");
    }

    public void testContentLengthInputStreamSkip() throws IOException {
        InputStream in = new ContentLengthInputStream(new ByteArrayInputStream(new byte[20]), 10);
        assertEquals(10, in.skip(10));
        assertTrue(in.read() == -1);

        in = new ContentLengthInputStream(new ByteArrayInputStream(new byte[20]), 10);
        in.read();
        assertEquals(9, in.skip(10));
        assertTrue(in.read() == -1);

        in = new ContentLengthInputStream(new ByteArrayInputStream(new byte[20]), 2);
        in.read();
        in.read();
        assertTrue(in.skip(10) <= 0);
        assertTrue(in.read() == -1);
    }

    public void testChunkedConsitance() throws IOException {
        String input = "76126;27823abcd;:q38a-\nkjc\rk%1ad\tkh/asdui\r\njkh+?\\suweb";
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        OutputStream out = new ChunkedOutputStream(buffer);
        out.write(HttpConstants.getBytes(input));
        out.close();
        buffer.close();
        InputStream in = new ChunkedInputStream(new ByteArrayInputStream(buffer.toByteArray()), new GetMethod());

        byte[] d = new byte[10];
        ByteArrayOutputStream result = new ByteArrayOutputStream();
        int len = 0;
        while ((len = in.read(d)) > 0) {
            result.write(d, 0, len);
        }

        String output = HttpConstants.getContentString(result.toByteArray());
        assertEquals(input, output);
    }

    // ------------------------------------------------------- TestCase Methods

    public static Test suite() {
        return new TestSuite(TestStreams.class);
    }

    // ------------------------------------------------------------------- Main
    public static void main(String args[]) {
        String[] testCaseName = { TestStreams.class.getName() };
        junit.textui.TestRunner.main(testCaseName);
    }
}

