/*
 *  file_parser.cc -- BabyTrans ( Babylon Translator front-end for GTK )
 *
 *  Copyright (C) 1999  Frederic Jolliton -- <fjolliton@fnac.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <iostream>
#include <algorithm>
#include <string>
#include <vector>

using std::vector ;
using std::istream ;
using std::string ;
using std::ios ;

#include "file_parser.h"

const char          comment_delimiter = '#' ;

class end_of_identifier
{
    bool                other_than_digit ;
public:
                        end_of_identifier()
                            : other_than_digit( false ) { }
    bool                operator()( char c ) {
        if ( isdigit( c ) )
            return ! other_than_digit ;
        other_than_digit = true ;
        return ! ( ( c == '_' ) || isalpha( c ) ) ;
    }
} ;

class end_of_parameter
{
public:
    bool                operator()( char c ) {
        unsigned char   uc = static_cast< unsigned char >( c ) ;
        return ! ( uc > 32 && uc != ',' && uc != '=' ) ;
    }
} ;

/*
 * Synopsis:
 *
 *   bool read_config_line( istream& is , vector< string >& vs )
 *
 * Description:
 *
 *   Read a single line from a configuration file (via a istream)
 *
 *   The syntax is:
 *   identifier = param1 [, param2]*
 *
 *   A # mark the beggining of a comment, and the rest of the line
 *   is simply ignored.
 *
 *   A blank line (nothing, or blank space) is valid.
 *
 * Return value:
 *
 *   Return true if a line has been read without syntax error,
 *   or if the line is empty, and if stream is not in 'bad' or 'fail'
 *   state.
 *   Otherwise, return false.
 */

#define SPACE_CHARS " \t"

file_parser_status
read_config_line( istream& is , vector<string>& option )
{
    /* Welcome to Return Land */

    string              s ;
    size_t              pos ;

    if ( is.eof() && ! is.bad() ) {
        is.setstate( ios::failbit ) ;
        return file_parser_eof ;
    } else if ( ! is.good() ) {
        is.setstate( ios::failbit ) ;
        return file_parser_io_error ;
    }
    getline( is , s ) ;
    if ( is.bad() )
        return file_parser_io_error ;
    if ( is.fail() )
        return is.eof()
            ? file_parser_eof
            : file_parser_io_error ;

    /* delete comment part */
    pos = s.find( comment_delimiter ) ;
    if ( pos != string::npos )
        s.erase( pos ) ;

    option.clear() ;

    size_t              i , i_end ;
    i = 0 ;
    /* skip blanks */
    i = s.find_first_not_of( SPACE_CHARS , i ) ;
    if ( i == string::npos )
        return file_parser_ok ;
    /* identifier */
    i_end = find_if( s.begin() + i , s.end() , end_of_identifier() )
        - s.begin() ;
    if ( i_end == i )
        return file_parser_syntax_error ;
    option.push_back( s.substr( i , i_end - i ) ) ;
    i = i_end ;
    /* skip blanks */
    i = s.find_first_not_of( SPACE_CHARS , i ) ;
    if ( i == string::npos )
        return file_parser_syntax_error ;
    /* check for '=' symbol */
    if ( s[ i ] != '=' )
        return file_parser_syntax_error ;
    ++ i ;
    for ( ; ; ) {
        /* skip blanks */
        i = s.find_first_not_of( SPACE_CHARS , i ) ;
        if ( i == string::npos )
            return file_parser_syntax_error ;
        /* parameter */
        i_end = find_if( s.begin() + i , s.end() , end_of_parameter() )
            - s.begin() ;
        if ( i == i_end )
            return file_parser_syntax_error ;
        option.push_back( s.substr( i , i_end - i ) ) ;
        i = i_end ;
        /* skip blanks */
        i = s.find_first_not_of( SPACE_CHARS , i ) ;
        if ( i == string::npos )
            return file_parser_ok ;
        /* check for ',' symbol */
        if ( s[ i ] != ',' )
            return file_parser_syntax_error ;
        ++ i ;
    }

    /* NEVER REACHED */
    return file_parser_ok ;
}

#ifdef TEST
template< typename T >
void
print_state( T& s )
{
    cerr << "good(" << s.good()
         << ") eof(" << s.eof()
         << ") fail(" << s.fail()
         << ") bad(" << s.bad() << ")\n" ;
}

int main()
{
    vector<string>      cl ;
    int                 line ;
    bool                ok ;

    line = 0 ;
    while ( cin.good() ) {
        cerr << "BEFORE: " ; print_state( cin ) ;
        ok = read_config_line( cin , cl ) ;
        cerr << "AFTER: " ; print_state( cin ) ;
        if ( ! ok ) {
            cerr << "not ok\n" ;
            break ;
        }
        ++ line ;
        if ( cl.empty() )
            continue ;
        cout << cl[ 0 ] ;
        for ( size_t i = 1 ; i < cl.size() ; ++ i )
            cout << " - " << cl[ i ] ;
        cout << endl ;
    }

    print_state( cin ) ;
}
#endif
