//
// File:        Grammar.java
// Package:     gov.llnl.babel.parsers.sidl
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: Grammar.java,v 1.12 2003/01/24 18:13:14 dahlgren Exp $
// Description: utility class for deciphering productions in the SIDL grammar
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.parsers.sidl;

import gov.llnl.babel.parsers.sidl.ParserConstants;

/**
 * <code>Grammar</code> defines integer constants for the major productions
 * in the SIDL grammar.  These production numbers are used by the parser to
 * identify a particular grammar production in the event of a parse error.
 * This class also provides lookup functions that translate grammar tokens
 * and productions into human-friendly strings.
 */
public class Grammar {
   public static final int SPECIFICATION     =  1;
   public static final int REQUIRE           =  2;
   public static final int IMPORT            =  3;
   public static final int PACKAGE           =  4;
   public static final int DEFINITION        =  5;
   public static final int CLASS             =  6;
   public static final int ENUM              =  7;
   public static final int ENUMERATOR        =  8;
   public static final int INTERFACE         =  9;
   public static final int CLASS_METHOD      = 10;
   public static final int METHOD            = 11;
   public static final int ARGUMENT          = 12;
   public static final int TYPE              = 13;
   public static final int ARRAY             = 14;
   public static final int SCOPED_IDENTIFIER = 15;
   public static final int ARRAY_ORDER       = 16;

   public static final int MIN_PRODUCTION    =  1;
   public static final int MAX_PRODUCTION    =  16;

   /**
    * Static utility method <code>getProduction</code> returns a string
    * representing the specified production integer.
    */
   public static String getProduction(int production) {
      String s = null;
      switch (production) {
      case SPECIFICATION:
         s = "Specification := (Require)* (Import)* (Package)* EOF";
         break;
      case REQUIRE:
         s = "Require := \"require\" ScopedIdentifier \"version\" V1.V2...Vn "
           + "\";\"";
         break;
      case IMPORT:
         s = "Import := \"import\" ScopedIdentifier [ \"version\" V1.V2...Vn"
           + " ] \";\"";
         break;
      case PACKAGE:
         s = "Package := [ \"final\" ] \"package\" ScopedIdentifier "
           + "[ \"version\" V1.V2...Vn ] "
           + "\"{\" (Definition)* \"}\" [ \";\" ]";
         break;
      case DEFINITION:
         s = "Definition := Class | Enum | Interface | Package";
         break;
      case CLASS:
         s = "Class := [ \"abstract\" ] \"class\" Identifier "
           + "[ \"extends\" ScopedIdentifier ] "
           + "[ \"implements-all\" ScopedIdentifier "
           + "( \",\" ScopedIdentifier )* ] "
           + "[ \"implements\" ScopedIdentifier "
           + "( \",\" ScopedIdentifier )* ] "
           + "\"{\" ( ClassMethod )* \"}\" [ \";\" ]";
         break;
      case ENUM:
         s = "Enum := \"enum\" Identifier "
           + "\"{\" Enumerator (\",\" Enumerator)* \"}\" [ \";\" ]";
         break;
      case ENUMERATOR:
         s = "Enumerator := Identifier [\"=\" Integer]";
         break;
      case INTERFACE:
         s = "Interface := \"interface\" Identifier "
           + "[ \"extends\" ScopedIdentifier ( \",\" ScopedIdentifier )* ] "
           + "\"{\" ( Method )* \"}\" [ \";\" ]";
         break;
      case CLASS_METHOD:
         s = "ClassMethod := [ \"abstract\" | \"final\" | \"static\" ] "
           + "Method";
         break;
      case METHOD:
         s = "Method := ( \"void\" | [ \"copy\" ] Type ) Identifier "
           + "[ \"[\" Identifier \"]\" ] "
           + "\"(\" [ Argument ( \",\" Argument )* ] \")\" "
           + "[ \"local\" | \"oneway\" ] "
           + "[ \"throws\" ScopedIdentifier ( \",\" ScopedIdentifier )* ] "
           + "\";\"";
         break;
      case ARGUMENT:
         s = "Argument := [ \"copy\" ] ( \"in\" | \"inout\" | \"out\" ) "
            + "Type Identifier";
         break;
      case TYPE:
         s = "Type : = \"bool\" | \"char\" | \"dcomplex\" | "
           + "\"double\" | \"fcomplex\" | \"float\" | \"int\" | "
           + "\"long\" | \"opaque\" | \"string\" | Array | "
           + "ScopedIdentifier";
         break;
      case ARRAY:
         s = "Array := \"array<\" Type [ \",\" ( Integer "
           + "[ \",\" ArrayOrder ] | ArrayOrder ) ] \">\"";
         break;
      case SCOPED_IDENTIFIER:
         s = "ScopedIdentifier := Identifier ( \".\" Identifier )*";
         break;
      case ARRAY_ORDER:
         s = "ArrayOrder := ( \"column-major\" | \"row-major\" )";
         break;
      default:
         s = "Internal error (unknown production)";
         break;
      }
      return s;
   }

   /**
    * Static utility method <code>getTokenString</code> returns a string
    * token representation for the specified token number.
    */
   public static String getTokenString(int token) {
      switch (token) {
      case ParserConstants.EOF:
         return "\"END-OF-FILE\"";
      case ParserConstants.T_IDENTIFIER:
         return "\"identifier\"";
      case ParserConstants.T_VERSION_STRING:
         return "\"V1.V2...Vn\"";
      case ParserConstants.T_INTEGER:
         return "\"integer\"";
      case ParserConstants.T_DIGIT:
         return "\"[0-9]\"";
      case ParserConstants.T_LETTER:
         return "\"[a-z,A-Z]\"";
      default:
         return ParserConstants.tokenImage[token];
      }
   }

   /**
    * Print the grammar productions to the standard system output.
    */
   public static void main(String[] args) {
      for (int p = MIN_PRODUCTION; p <= MAX_PRODUCTION; p++) {
         System.out.println(getProduction(p));
      }
   }
}
