/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.source;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.ComponentSelector;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.avalon.excalibur.pool.Recyclable;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;


/**
 * Base interface for resolving a source by system identifiers.
 * Instead of using the java.net.URL classes which prevent you
 * to add your own custom protocols in a server environment,
 * you should use this resolver for all URLs.
 *
 * The resolver creates for each source a <code>Source</code>
 * object, which could then be asked for an <code>InputStream</code>
 * etc.
 *
 * When the <code>Source</code> object is no longer needed
 * it must be released using the resolver. This is very similar like
 * looking up components from a <code>ComponentManager</code>
 * and releasing them.
 *
 * It looks for the base URL in the <code>Context</code> object with
 * the "container.rootDir" entry.  If the entry does not exist, it is
 * populated with the system property "user.dir".
 *
 * @author <a href="mailto:cziegeler@apache.org">Carsten Ziegeler</a>
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version $Id: SourceResolverImpl.java,v 1.13 2002/01/25 21:12:56 bloritsch Exp $
 */
public class SourceResolverImpl
extends AbstractLogEnabled
implements Composable,
           Contextualizable,
           Disposable,
           SourceResolver,
           ThreadSafe
{

    /** The component manager */
    protected ComponentManager m_manager;

    /** The special Source factories */
    protected ComponentSelector m_factorySelector;

    /** The context */
    protected Context m_context;

    /**
     * The base URL
     */
    protected URL m_baseURL;

    /**
     * Get the context
     */
    public void contextualize(Context context)
    throws ContextException
    {
        m_context = context;

        try
        {
            m_baseURL = ((File) m_context.get("container.rootDir")).toURL();
        }
        catch (ContextException ce)
        {
            // set the base URL to the current directory
            try
            {
                m_baseURL = new File(System.getProperty("user.dir")).toURL();
                if ( this.getLogger().isDebugEnabled() )
                {
                    this.getLogger().debug("SourceResolver: Using base URL: " + m_baseURL);
                }
            }
            catch (MalformedURLException mue)
            {
                throw new ContextException("Malformed URL for user.dir, and no container.rootDir exists", mue);
            }
        }
        catch (MalformedURLException mue)
        {
            throw new ContextException("Malformed URL for container.rootDir", mue);
        }
    }

    /**
     * Set the current <code>ComponentManager</code> instance used by this
     * <code>Composable</code>.
     */
    public void compose(ComponentManager manager)
    throws ComponentException
    {
        m_manager = manager;
        m_factorySelector = (ComponentSelector)m_manager.lookup(SourceFactory.ROLE + "Selector");
    }

    /**
     * Dispose
     */
    public void dispose()
    {
        if (m_manager != null)
        {
            m_manager.release(m_factorySelector);
            m_factorySelector = null;
        }
    }

    /**
     * Get a <code>Source</code> object.
     */
    public Source resolve(String location)
    throws MalformedURLException, IOException, ComponentException
    {
        return this.resolve(m_baseURL, location, null);
    }

    /**
     * Get a <code>Source</code> object.
     */
    public Source resolve(String location,
                          SourceParameters parameters)
    throws MalformedURLException, IOException, ComponentException
    {
        return this.resolve( m_baseURL, location, parameters );
    }

    /**
     * Get a <code>Source</code> object.
     */
    public Source resolve(URL base, String location)
    throws MalformedURLException, IOException, ComponentException
    {
        return this.resolve(base, location, null);
    }


    /**
     * Get a <code>Source</code> object.
     */
    public Source resolve(URL base,
                          String location,
                          SourceParameters parameters)
    throws MalformedURLException, IOException, ComponentException
    {
        if ( this.getLogger().isDebugEnabled() ) {
            this.getLogger().debug("Resolving '"+location+"' in context '" + base + "'");
        }
        if (location == null) throw new MalformedURLException("Invalid System ID");

        // first step: create systemID
        String systemID;
        if (base == null) base = m_baseURL;

        if (location.length() == 0) {
            systemID = base.toExternalForm();
        } else if (location.indexOf(":") > 1)
        {
            systemID = location;
        }
        else if (location.charAt(0) == '/')
        {
            systemID = new StringBuffer(base.getProtocol())
                           .append(":").append(location).toString();
        // windows: absolute paths can start with drive letter
        }
        else if (location.length() > 1 && location.charAt(1) == ':')
        {
            systemID = new StringBuffer(base.getProtocol())
                           .append(":/").append(location).toString();
        }
        else
        {
            if (base.getProtocol().equals("file") == true)
            {
                File temp = new File(base.toExternalForm().substring("file:".length()), location);
                String path = temp.getAbsolutePath();
                // windows paths starts with drive letter
                if (path.charAt(0) != File.separator.charAt(0))
                {
                    systemID = "file:/" + path;
                }
                else
                {
                    systemID = "file:" + path;
                }
            }
            else
            {
                systemID = new URL(base, location).toExternalForm();
            }
        }
        if ( this.getLogger().isDebugEnabled() )
        {
            this.getLogger().debug("Resolved to systemID '"+systemID+"'");
        }

        Source source = null;
        // search for a SourceFactory implementing the protocol
        final int protocolPos = systemID.indexOf(':');
        if ( protocolPos != -1 )
        {
            final String protocol = systemID.substring(0, protocolPos);
            if ( m_factorySelector.hasComponent(protocol) )
            {
                SourceFactory factory = null;
                try
                {
                    factory = ( SourceFactory )m_factorySelector.select( protocol );
                    source = factory.getSource( systemID, parameters );
                }
                finally
                {
                    m_factorySelector.release( factory );
                }
            }
        }

        if ( null == source )
        {
            // no factory found, so usual url handling stuff...
            try
            {
                if (this.getLogger().isDebugEnabled() == true)
                {
                    getLogger().debug("Making URL from " + systemID);
                }
                source = new URLSource(new URL(systemID), parameters);
            }
            catch (MalformedURLException mue)
            {
                if ( this.getLogger().isDebugEnabled() )
                {
                    getLogger().debug("Making URL - MalformedURLException in getURL:" , mue);
                    getLogger().debug("Making URL a File (assuming that it is full path):" + systemID);
                }
                source = new URLSource((new File(systemID)).toURL(), parameters);
            }
        }
        if (source instanceof LogEnabled)
        {
            ((LogEnabled) source).enableLogging(getLogger());
        }
        try
        {
            if (source instanceof Contextualizable)
            {
                ((Contextualizable) source).contextualize (m_context);
            }
        }
        catch (ContextException ce)
        {
            throw new ComponentException("ContextException occured during source resolving.", ce);
        }

        if (source instanceof Composable)
        {
            ((Composable) source).compose(m_manager);
        }
        return source;
    }

    /**
     * Releases a resolved resource
     */
    public void release( Source source )
    {
        if ( source == null) return;
        if ( source instanceof Recyclable )
        {
            ((Recyclable)source).recycle();
        }
        if ( source instanceof Disposable )
        {
            ((Disposable) source).dispose();
        }
    }

}
