
/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.pool;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.excalibur.collections.Buffer;
import org.apache.avalon.excalibur.collections.VariableSizeBuffer;
import org.apache.avalon.excalibur.concurrent.Mutex;

/**
 * This is an <code>Pool</code> that caches Poolable objects for reuse.
 * Please note that this pool offers no resource limiting whatsoever.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.2 $ $Date: 2001/12/26 16:15:23 $
 * @since 4.1
 */
public final class VariableSizePool
    implements Pool, Disposable, ManagablePool
{
    private boolean             m_disposed = false;
    private final Buffer        m_buffer;
    private final ObjectFactory m_factory;
    private final Mutex         m_mutex = new Mutex();
    private final long          m_key;

    /**
     * Constructor for an unmanaged pool
     */
    public VariableSizePool( ObjectFactory factory, int size )
        throws Exception
    {
        this( factory, size, -1 );
    }

    /**
     * Constructor for a managed pool
     */
    public VariableSizePool( ObjectFactory factory, int size, long key )
        throws Exception
    {
        m_buffer = new VariableSizeBuffer( size );
        m_factory = factory;
        m_key = key;

        for ( int i = 0; i < size; i++ )
        {
            m_buffer.add( m_factory.newInstance() );
        }
    }

    public Poolable get()
    {
        if ( m_disposed )
        {
            throw new IllegalStateException( "Cannot get an object from a disposed pool" );
        }

        return (Poolable) m_buffer.remove();
    }

    public void put( Poolable object )
    {
        if ( m_disposed )
        {
            try
            {
                m_factory.decommission( object );
            }
            catch ( Exception e )
            {
                // We should never get here, but ignore the exception if it happens
            }
        }
        else
        {
            m_buffer.add( object );
        }
    }

    public void dispose()
    {
        m_disposed = true;

        while ( ! m_buffer.isEmpty() )
        {
            try
            {
                m_factory.decommission( m_buffer.remove() );
            }
            catch ( Exception e )
            {
                // We should never get here, but ignore the exception if it happens
            }
        }
    }

    public void shrink( final int byNum, final long key )
        throws IllegalAccessException
    {
        if ( m_key < 0 || m_key != key )
        {
            throw new IllegalAccessException();
        }

        try
        {
            m_mutex.acquire();

            final int num = Math.min( byNum, m_buffer.size() );

            for ( int i = 0; i < num; i++ )
            {
                m_factory.decommission( m_buffer.remove() );
            }
        }
        catch ( Exception e )
        {
            // We should never get here, but ignore the exception if it happens
        }
        finally
        {
            m_mutex.release();
        }
    }

    public void grow( final int byNum, final long key )
        throws IllegalAccessException
    {
        if ( m_key < 0 || m_key != key )
        {
            throw new IllegalAccessException();
        }

        try
        {
            m_mutex.acquire();

            for ( int i = 0; i < byNum; i++ )
            {
                m_buffer.add( m_factory.newInstance() );
            }
        }
        catch ( Exception ie )
        {
            // We should never get here, but ignore the exception if it happens
        }
        finally
        {
            m_mutex.release();
        }
    }

    public int size( final long key )
        throws IllegalAccessException
    {
        if ( m_key < 0 || m_key != key )
        {
            throw new IllegalAccessException();
        }

        return m_buffer.size();
    }
}
