/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.logger;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;

import org.apache.avalon.framework.logger.Logger;

/**
 * Simple Logger which logs all information to an internal StringBuffer.  
 *  When logging is complete call toString() on the logger to obtain the 
 *  logged output.  Useful for testing.
 *
 * @author <a href="mailto:leif@silveregg.co.jp">Leif Mortenson</a>
 * @version CVS $Revision: 1.1 $ $Date: 2002/01/24 02:55:22 $
 * @since 4.0
 */
public class BufferedLogger implements Logger {

    private StringBuffer _sb;

    /*---------------------------------------------------------------
     * Constructors
     *-------------------------------------------------------------*/

    /**
     * Create a new <code>BufferedLogger</code>.
     */
    public BufferedLogger() {
        _sb = new StringBuffer();
    }

    /*---------------------------------------------------------------
     * Logger Methods
     *-------------------------------------------------------------*/

    /**
     * Log a debug message.
     *
     * @param message the message
     */
    public void debug(String message) {
        append("DEBUG", message);
    }

    /**
     * Log a debug message.
     *
     * @param message the message
     * @param throwable the throwable
     */
    public void debug(String message, Throwable throwable) {
        append("DEBUG", message, throwable);
    }

    /**
     * Determine if messages of priority "debug" will be logged.
     *
     * @return true if "debug" messages will be logged
     */
    public boolean isDebugEnabled() {
        return true;
    }

    /**
     * Log a info message.
     *
     * @param message the message
     */
    public void info(String message) {
        append("INFO", message);
    }

    /**
     * Log a info message.
     *
     * @param message the message
     * @param throwable the throwable
     */
    public void info(String message, Throwable throwable) {
        append("INFO", message, throwable);
    }

    /**
     * Determine if messages of priority "info" will be logged.
     *
     * @return true if "info" messages will be logged
     */
    public boolean isInfoEnabled() {
        return true;
    }

    /**
     * Log a warn message.
     *
     * @param message the message
     */
    public void warn(String message) {
        append("WARN", message);
    }

    /**
     * Log a warn message.
     *
     * @param message the message
     * @param throwable the throwable
     */
    public void warn(String message, Throwable throwable) {
        append("WARN", message, throwable);
    }

    /**
     * Determine if messages of priority "warn" will be logged.
     *
     * @return true if "warn" messages will be logged
     */
    public boolean isWarnEnabled() {
        return true;
    }

    /**
     * Log a error message.
     *
     * @param message the message
     */
    public void error(String message) {
        append("ERROR", message);
    }

    /**
     * Log a error message.
     *
     * @param message the message
     * @param throwable the throwable
     */
    public void error(String message, Throwable throwable) {
        append("ERROR", message, throwable);
    }

    /**
     * Determine if messages of priority "error" will be logged.
     *
     * @return true if "error" messages will be logged
     */
    public boolean isErrorEnabled() {
        return true;
    }

    /**
     * Log a fatalError message.
     *
     * @param message the message
     */
    public void fatalError(String message) {
        append("FATAL ERROR", message);
    }

    /**
     * Log a fatalError message.
     *
     * @param message the message
     * @param throwable the throwable
     */
    public void fatalError(String message, Throwable throwable) {
        append("FATAL ERROR", message, throwable);
    }

    /**
     * Determine if messages of priority "fatalError" will be logged.
     *
     * @return true if "fatalError" messages will be logged
     */
    public boolean isFatalErrorEnabled() {
        return true;
    }

    /**
     * Create a new child logger.
     * The name of the child logger is [current-loggers-name].[passed-in-name]
     *
     * @param name the subname of this logger
     * @return the new logger
     */
    public Logger getChildLogger(String name) {
        return this;
    }

    /*---------------------------------------------------------------
     * Public Methods
     *-------------------------------------------------------------*/
    private void append(String message) {

        synchronized (_sb) {
            _sb.append(message);
            _sb.append("\n");
        }
    }

    private void append(String level, String message) {

        synchronized (_sb) {
            _sb.append(level);
            _sb.append(" - ");
            _sb.append(message);
            _sb.append("\n");
        }
    }

    private void append(String level, String message, Throwable throwable) {

        synchronized (_sb) {
            String                tDump = null;
            ByteArrayOutputStream ba    = new ByteArrayOutputStream();
            PrintStream           ps    = new PrintStream(ba);

            try {
                throwable.printStackTrace(ps);

                tDump = ba.toString();
            } finally {
                ps.close();
            }

            _sb.append(level);
            _sb.append(" - ");
            _sb.append(message);
            _sb.append(" : ");
            _sb.append(tDump);
            _sb.append("\n");
        }
    }

    /**
     * Returns the contents of the buffer.
     *
     * @return the buffer contents
     *
     */
    public String toString() {
        return _sb.toString();
    }
}
