/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.i18n;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Iterator;

import org.apache.avalon.excalibur.component.ExcaliburComponentSelector;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.component.ComponentSelector;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.LogKitLogger;
import org.apache.avalon.framework.activity.Initializable;

/**
 * This is the method for getting instances of ResourceBundles.
 *
 * @author <a href="mailto:neeme@one.lv">Neeme Praks</a>
 * @author <a href="mailto:mengelhart@earthtrip.com">Mike Engelhart</a>
 * @version $Id: BundleSelector.java,v 1.10 2002/01/02 19:04:56 neeme Exp $
 */

public class BundleSelector extends ExcaliburComponentSelector {

    /**
     * The role implemented by an <code>BundleSelector</code>.
     */
    public static String ROLE = "org.apache.avalon.excalibur.i18n.BundleSelector";

    /** Component Manager */
    protected ComponentManager manager = null;

    public void compose(ComponentManager manager) {
        this.manager = manager;
    }

    /** Cache for the names of the bundles that were not found */
    protected Map cacheNotFound = new HashMap();
    protected Map cache = new HashMap();

    /** Matchers */
    private Configuration[] matchers = null;

    /** Default bundle type */
    private String defaultType = null;

    /**
     * Configure the component.
     *
     * @param configuration the configuration
     */
    public void configure(Configuration configuration) throws ConfigurationException {
        if (matchers == null) matchers = configuration.getChildren("bundle");
        if (defaultType == null) defaultType = configuration.getAttribute("default");
    }

    /**
     * Select a bundle based on bundle name and locale.
     *
     * @param name              bundle name
     * @param locale            locale
     * @param cacheAtStartup    cache all the keys when constructing?
     * @return                  the bundle
     * @exception ComponentException if a bundle is not found
     */
    public Component select(Object hint) throws ComponentException {
        Component bundle = select((BundleInfo) hint);
        if (bundle == null)
            throw new ComponentException("Unable to locate bundle: " + hint);
        return bundle;
    }

    /**
     * Select a bundle based on bundle name and locale name.
     *
     * @param name              bundle name
     * @param localeName        locale name
     * @return                  the bundle
     * @exception ComponentException if a bundle is not found
     */
    public Component select(String name, String localeName) throws ComponentException {
        return select(new ConfigurableBundleInfo(name, new Locale(localeName, localeName)));
    }

    /**
     * Select a bundle based on bundle name and locale.
     *
     * @param name              bundle name
     * @param locale            locale
     * @param cacheAtStartup    cache all the keys when constructing?
     * @return                  the bundle
     * @exception ComponentException if a bundle is not found
     */
    private Component select(BundleInfo bundleInfo) {
        if (getLogger().isDebugEnabled()) getLogger().debug("_getBundle: " + bundleInfo);
        Bundle bundle = (Bundle) selectCached(bundleInfo);
        if (bundle == null && !isNotFoundBundle(bundleInfo)) {
            if (getLogger().isDebugEnabled()) getLogger().debug("not found in cache, loading: " + bundleInfo);
            synchronized(this) {
                bundle = (Bundle) selectCached(bundleInfo);
                if (bundle == null && !isNotFoundBundle(bundleInfo)) {
                    if (getLogger().isDebugEnabled()) getLogger().debug("synchronized: not found in cache, loading: " + bundleInfo);
                    bundle = loadBundle(bundleInfo);
                    BundleInfo parentBundleInfo = bundleInfo.getParent();
                    while (bundle == null && parentBundleInfo != null) {
                        if (getLogger().isDebugEnabled()) getLogger().debug("synchronized: still not found, trying parent: " + parentBundleInfo);
                        bundle = loadBundle(parentBundleInfo);
                        updateCache(parentBundleInfo, bundle);
                        parentBundleInfo = parentBundleInfo.getParent();
                    }
                    updateCache(bundleInfo, bundle);
                }
            }
        }
        return (Component) bundle;
    }

    private Bundle loadBundle(BundleInfo bundleInfo) {
        ComponentSelector matcherSelector = null;
        ComponentSelector bundleFactorySelector = null;
        BundleFactory bundleFactory = null;
        Bundle bundle = null;
        try {
            matcherSelector = (ComponentSelector) manager.lookup(BundleMatcher.ROLE + "Selector");
            String type = null;
            for (int i = 0; i < matchers.length; i++) {
                BundleMatcher matcher = null;
                try {
                    matcher = (BundleMatcher) matcherSelector.select(matchers[i].getAttribute("matcher"));
                    type = matcher.getType(bundleInfo);
                    matcherSelector.release((Component) matcher);
                } catch (ComponentException e) {
                    getLogger().error("Error while matching bundle!", e);
                } finally {
                    if (matcher != null) matcherSelector.release((Component) matcher);
                }
                if (type != null) break;
            }
            if (type == null) type = defaultType;
            bundleFactorySelector = (ComponentSelector) manager.lookup(BundleFactory.ROLE + "Selector");
            bundleFactory = (BundleFactory) bundleFactorySelector.select(type);
            bundle = bundleFactory.createInstance(bundleInfo);
        } catch (ComponentException e) {
            getLogger().error("Error while matching bundle!", e);
        } catch (ConfigurationException e) {
            getLogger().error("Error while matching bundle!", e);
        } finally {
            if (matcherSelector != null) manager.release(matcherSelector);
            if (bundleFactorySelector != null) {
                if (bundleFactory != null) bundleFactorySelector.release((Component) bundleFactory);
                manager.release(bundleFactorySelector);
            }
        }
        return bundle;
    }

    /**
     * Selects a bundle from the cache.
     *
     * @param fileName          file name of the bundle
     * @return                  the cached bundle; null, if not found
     */
    protected Component selectCached(BundleInfo bundleInfo) {
        return (Component) cache.get(bundleInfo);
        /*
        Component bundle = null;
        try {
            bundle = super.select(bundleInfo);
            if (getLogger().isDebugEnabled()) getLogger().debug("returning from cache: " + bundleInfo);
        }
        catch (ComponentException e) {
            if (getLogger().isDebugEnabled()) getLogger().debug("not found in cache: " + bundleInfo);
        }
        return bundle;
        */
    }

    /**
     * Checks if the bundle is in the &quot;not-found&quot; cache.
     *
     * @param fileName          file name of the bundle
     * @return                  true, if the bundle wasn't found already before;
     *                          otherwise, false.
     */
    protected boolean isNotFoundBundle(BundleInfo bundleInfo) {
        BundleInfo result = (BundleInfo)(cacheNotFound.get(bundleInfo));
        if (result != null) {
            if (getLogger().isDebugEnabled()) getLogger().debug("returning from not_found_cache: " + bundleInfo);
        }
        else {
            if (getLogger().isDebugEnabled()) getLogger().debug("not found in not_found_cache: " + bundleInfo);
        }
        return result != null;
    }

    /**
     * Checks if the bundle is in the &quot;not-found&quot; cache.
     *
     * @param fileName          file name of the bundle
     * @return                  true, if the bundle wasn't found already before;
     *                          otherwise, false.
     */
    protected void updateCache(BundleInfo bundleInfo, Bundle bundle) {
        if (bundle == null) {
            if (getLogger().isDebugEnabled()) getLogger().debug("updating not_found_cache: " + bundleInfo);
            cacheNotFound.put(bundleInfo, bundleInfo);
        }
        else {
            if (getLogger().isDebugEnabled()) getLogger().debug("updating cache: " + bundleInfo);
            //super.addComponentInstance(bundleInfo, (Component) bundle);
            cache.put(bundleInfo, (Component) bundle);
        }
    }

}
