/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.i18n;

import java.util.Map;
import java.util.Locale;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.component.Component;

public abstract class AbstractBundle extends AbstractLogEnabled implements Bundle, Component {

    /** bundle info */
    private BundleInfo bundleInfo;

    /** bundle info mapper */
    private BundleInfoMapper mapper;

    /** Parent of the current bundle */
    private Bundle parent = null;

    /** Last modification time */
    private long lastModified = -1;

    /**
     * Get the bundle info.
     */
    public BundleInfo getBundleInfo() {
        return this.bundleInfo;
    }

    /**
     * Set the bundle info.
     */
    public void setBundleInfo(BundleInfo bundleInfo) {
        if (this.bundleInfo == null) this.bundleInfo = bundleInfo;
    }

    public BundleInfoMapper getMapper() {
        return mapper;
    }

    public void setMapper(BundleInfoMapper mapper) {
        if (this.mapper == null) this.mapper = mapper;
    }

    /**
     * Get the parent bundle of the current bundle.
     *
     * @return          the parent bundle
     */
    public Bundle getParent() {
        return this.parent;
    }

    /**
     * Set the parent bundle of the current bundle.
     *
     * @param parent    the parent bundle
     */
    public void setParent(Bundle parent) {
        if (this.parent == null) this.parent = parent;
    }

    /**
     * Returns the last modification time of this bundle, in milliseconds.
     *
     * @return              last modification time, -1 if N/A
     */
    public long getLastModified() {
        return this.lastModified;
    }

    /**
     * Sets the last modification time of this bundle, in milliseconds.
     *
     * @param lastModified  last modification time
     */
    public void setLastModified(long lastModified) {
        this.lastModified = lastModified;
    }

    /**
     * Get value by key and substitute variables.
     *
     * @param key           key
     * @param variables     map with variable values
     * @return              value with variable values substituted
     * @exception MissingResourceException if value was not found
     */
    public String getString(String key, Map variables) {
        return substitute(getString(key), variables);
    }

    /**
     * Convert the &quot;user view&quot; of the lookup key to the
     * &quot;system view&quot;. Used to hide the implemented storage
     * mechanism and/or XML file schema.
     *
     * The default implementation just returns the key &quot;as-is&quot;.
     *
     * @param key           user key
     * @return              system key
     */
    public String convertKey(String userKey) {
        return userKey;
    }

    /**
     * Substitute the &quot;variables&quot; in the string with the values
     * provided in the map.
     *
     * @param value         value where to search for variables
     * @param dictionary    map with variable values
     * @return              value with variable values substituted
     */
    public String substitute(String value, Map values) {
        if (value == null || values == null) return value;
        if (getLogger().isDebugEnabled()) getLogger().debug("Substituting value: " + value);

        StringBuffer result = new StringBuffer(value.length());
        int startPos = 0;
        int endPos = 0;
        int lastPos = value.length();
        Object varValue = "";
        String varKey = "", oldKey = "";
        while (endPos < lastPos) {
            startPos = endPos;
            endPos = value.indexOf('{', startPos);
            if (endPos == -1)
                endPos = lastPos;
            result.append(value.substring(startPos, endPos));
            if (endPos < lastPos)
                endPos++;
            if (endPos < lastPos) {
                startPos = endPos;
                endPos = value.indexOf('}', startPos);
                if (endPos == -1)
                    endPos = lastPos;
                oldKey = varKey;
                varKey = value.substring(startPos, endPos);
                if (!oldKey.equals(varKey))
                    varValue = values.get(varKey);
                if (varValue != null) {
                    if (getLogger().isDebugEnabled()) getLogger().debug("Substituting var: " + varKey + " --> " + varValue);
                    result.append(varValue);
                }
                else {
                    if (getLogger().isWarnEnabled()) getLogger().warn(bundleInfo + ": var not found: " + varKey);
                    result.append('{').append(varKey).append('}');
                }
                if (endPos < lastPos)
                    endPos++;
            }
        }
        if (getLogger().isDebugEnabled()) getLogger().debug("Returning substituted value: " + result);
        return result.toString();
    }

}
