/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.extension;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import org.apache.avalon.excalibur.io.FileUtil;
import org.apache.avalon.excalibur.util.DeweyDecimal;

/**
 * <p>Interface used to contain "Optional Packages" (formerly known as 
 * "Standard Extensions"). It is assumed that each "Optional Package" is 
 * represented by a single file on the file system. This Repository searches 
 * a path to find the Optional Packages.</p>
 *
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 * @version $Revision: 1.9 $ $Date: 2001/12/11 09:53:34 $
 * @see OptionalPackage
 * @see PackageRepository
 */
public class DefaultPackageRepository
    implements PackageRepository
{
    private static final boolean DEBUG = false;

    /**
     * Map between files and <code>OptionalPackage</code> objects.
     */
    private final HashMap    m_packages = new HashMap();

    /**
     * The set of directories in which to look for Optional Packages
     */
    private File[]           m_path;

    /**
     * Flag set when it is necessary to scan paths to 
     * build "Optional Package" list
     */
    private boolean          m_needToScan;

    /**
     * Construct a package repository with path.
     *
     * @param path The set of directories in which to look for Optional Packages
     */
    public DefaultPackageRepository( final File[] path )
    {
        setPath( path );
    }

    /**
     * Return all the <code>OptionalPackage</code>s that satisfy specified
     * <code>Extension</code>. It is expected that this <code>Extension</code>
     * object will be one retrieved via getLocalExtension() method. If the
     * specified <code>Extension</code> is not local then <code>null</code>
     * is returned.
     *
     * @param extension Description of the optional package
     * @see #isLocal()
     */
    public synchronized OptionalPackage[] getOptionalPackages( final Extension extension )
    {
        if( m_needToScan )
        {
            //Check cache consistency and reload if necessary..
            scanPath();
        }

        final ArrayList results = new ArrayList();
        final ArrayList candidates = (ArrayList)m_packages.get( extension.getExtensionName() );
        if( null != candidates )
        {
            final int size = candidates.size();
            for( int i = 0; i < size; i++ )
            {
                final OptionalPackage optionalPackage = (OptionalPackage)candidates.get( i );
                final Extension[] extensions = optionalPackage.getAvailableExtensions();

                for( int j = 0; j < extensions.length; j++ )
                {
                    if( extensions[ j ].isCompatibleWith( extension ) )
                    {
                        results.add( optionalPackage );
                    }
                }
            }
        }

        //TODO: Sort packages so that most relevent is first
        //ie Sort on spec version first and then on Imp version

        return (OptionalPackage[])results.toArray( new OptionalPackage[ 0 ] );
    }

    /**
     * Set the path for the Repository.
     *
     * @param path the list of directories in which to search
     */
    protected synchronized void setPath( final File[] path )
    {
        if( null == path )
        {
            throw new NullPointerException( "path property is null" );
        }

        for( int i = 0; i < path.length; i++ )
        {
            final File directory = path[ i ];
            
            if( !directory.exists() || !directory.isDirectory() )
            {
                throw new IllegalArgumentException( "path element " + directory + 
                                                    " must exist and must be a directory" );
            }
        }

        m_path = path;
        m_needToScan = true;
    }

    /**
     * Scan the path for this repository and reload all 
     * the "Optional Packages" found in the path.
     *
     */
    protected final synchronized void scanPath()
    {
        clearCache();

        for( int i = 0; i < m_path.length; i++ )
        {
            scanDirectory( m_path[ i ] );
        }
    }

    private synchronized void scanDirectory( final File directory )
    {
        final File[] files = directory.listFiles();
        for( int i = 0; i < files.length; i++ )
        {
            final File file = files[ i ];
            final String name = file.getName();

            if( !name.endsWith( ".jar" ) )
            {
                debug( "Skipping " + file + " as it does not end with '.jar'" );
                continue;
            }

            if( !file.isFile() )
            {
                debug( "Skipping " + file + " as it is not a file." );
                continue;
            }

            if( !file.canRead() )
            {
                debug( "Skipping " + file + " as it is not readable." );
                continue;
            }

            try
            {
                final OptionalPackage optionalPackage = getOptionalPackage( file );
                cacheOptionalPackage( optionalPackage );
            }
            catch( final IOException ioe )
            {
                debug( "Skipping " + file + " as it could not be loaded due to " + ioe );
            }
        }
    }

    /**
     * Clear internal cache of optional packages.
     *
     */
    protected synchronized final void clearCache()
    {
        m_packages.clear();
        m_needToScan = true;
    }

    /**
     * Add OptionalPackage to internal cache of Optional Packages.
     * Note that this method is only protected so that unit tests can sub-class
     * and add entries to PackageRepository by calling this method.
     *
     * @param optionalPackage the OptionalPackage to be added to repository
     */
    protected synchronized final void cacheOptionalPackage( final OptionalPackage optionalPackage )
    {
        m_needToScan = false;
        final Extension extension = optionalPackage.getAvailableExtensions()[ 0 ];
        ArrayList candidates = (ArrayList)m_packages.get( extension.getExtensionName() );
        if( null == candidates )
        {
            candidates = new ArrayList();
            m_packages.put( extension.getExtensionName(), candidates );
        }

        //TODO: Add this in descending order so that don't have to sort in
        //getOptionalPackages ????
        //yes, sort by Spec Version then vendor, Imp Version
        candidates.add( optionalPackage );
    }

    /**
     * Construct an OptionalPackage out of the specified jar archive.
     *
     * @param archive the file object for Jar archive
     * @return the OptionalPackage constructed
     * @exception IOException if an error occurs
     */
    private OptionalPackage getOptionalPackage( final File archive )
        throws IOException
    {
        final File file = archive.getCanonicalFile();
        final JarFile jarFile = new JarFile( file );
        final Manifest manifest = jarFile.getManifest();

        try
        {
            if( null == manifest ) return null;
            final Extension[] available = Extension.getAvailable( manifest );
            final Extension[] required = Extension.getRequired( manifest );

            return new OptionalPackage( file, available, required );
        }
        finally
        {
            jarFile.close();
        }
    }

    protected void debug( final String message )
    {
        if( DEBUG ) System.out.println( message );
        //getLogger().debug( message );
    }
}
