/*
 * Copyright © 2006-2007 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *    Eric Anholt <eric@anholt.net>
 *    Thomas Hellstrom <thomas-at-tungstengraphics-dot-com>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <psb_reg.h>
#include "xf86.h"
#include "psb_driver.h"
#include "i810_reg.h"
#include "i830.h"
#include "i830_bios.h"
#include "X11/Xatom.h"
#include "libmm/mm_defines.h"

typedef struct _PsbLVDSOutputRec
{
    PsbOutputPrivateRec psbOutput;
    CARD32 savePP_ON;
    CARD32 savePP_OFF;
    CARD32 saveLVDS;
    CARD32 savePP_CONTROL;
    CARD32 savePP_CYCLE;
    CARD32 saveBLC_PWM_CTL;
    CARD32 backlight_duty_cycle;
    DisplayModePtr panelFixedMode;
    Bool panelWantsDither;
} PsbLVDSOutputRec, *PsbLVDSOutputPtr;

/**
 * Sets the backlight level.
 *
 * \param level backlight level, from 0 to psbLVDSGetMaxBacklight().
 */
static void
psbLVDSSetBacklight(PsbLVDSOutputPtr pLVDS, int level)
{
    PsbDevicePtr pDevice = pLVDS->psbOutput.pDevice;
    CARD32 blc_pwm_ctl;

    blc_pwm_ctl = PSB_READ32(BLC_PWM_CTL) & ~BACKLIGHT_DUTY_CYCLE_MASK;
    PSB_WRITE32(BLC_PWM_CTL,
	blc_pwm_ctl | (level << BACKLIGHT_DUTY_CYCLE_SHIFT));
}

/**
 * Returns the maximum level of the backlight duty cycle field.
 */
static CARD32
psbLVDSGetMaxBacklight(PsbLVDSOutputPtr pLVDS)
{
    PsbDevicePtr pDevice = pLVDS->psbOutput.pDevice;

    return ((PSB_READ32(BLC_PWM_CTL) & BACKLIGHT_MODULATION_FREQ_MASK) >>
	BACKLIGHT_MODULATION_FREQ_SHIFT) * 2;
}

static void
psbLVDSCheckState(PsbLVDSOutputPtr pLVDS)
{
    PsbDevicePtr pDevice = pLVDS->psbOutput.pDevice;

    PSB_DEBUG(-1, 3, "PanelPower Status = 0x%08x\n",
	(unsigned)PSB_READ32(PP_STATUS));
    PSB_DEBUG(-1, 3, "Pipe B PLL 0x%08x\n", (unsigned)PSB_READ32(DPLL_B));
    PSB_DEBUG(-1, 3, "Pipe B Enabled 0x%08x\n",
	(unsigned)PSB_READ32(PIPEBCONF) & (1 << 31));
}

/**
 * Sets the power state for the panel.
 */
static void
psbLVDSSetPanelPower(PsbLVDSOutputPtr pLVDS, Bool on)
{
    PsbDevicePtr pDevice = pLVDS->psbOutput.pDevice;
    CARD32 pp_status;

    psbLVDSCheckState(pLVDS);
    if (on) {
	PSB_WRITE32(PP_CONTROL, PSB_READ32(PP_CONTROL) | POWER_TARGET_ON);
	do {
	    pp_status = PSB_READ32(PP_STATUS);
	} while ((pp_status & (PP_ON | PP_READY)) == PP_READY);

	psbLVDSSetBacklight(pLVDS, pLVDS->backlight_duty_cycle);
    } else {
	psbLVDSSetBacklight(pLVDS, 0);

	PSB_WRITE32(PP_CONTROL, PSB_READ32(PP_CONTROL) & ~POWER_TARGET_ON);
	do {
	    pp_status = PSB_READ32(PP_STATUS);
	} while ((pp_status & PP_ON) == PP_ON);
    }
}

static void
psbLVDSDPMS(xf86OutputPtr output, int mode)
{
    PsbLVDSOutputPtr pLVDS =
	containerOf(output->driver_private, PsbLVDSOutputRec, psbOutput);

    if (mode == DPMSModeOn)
	psbLVDSSetPanelPower(pLVDS, TRUE);
    else
	psbLVDSSetPanelPower(pLVDS, FALSE);

    /* XXX: We never power down the LVDS pair. */
}

static void
psbLVDSSave(xf86OutputPtr output)
{
    PsbLVDSOutputPtr pLVDS =
	containerOf(output->driver_private, PsbLVDSOutputRec, psbOutput);
    PsbDevicePtr pDevice = pLVDS->psbOutput.pDevice;

    PSB_DEBUG(output->scrn->scrnIndex, 2, "psbLVDSSave\n");

    pLVDS->savePP_ON = PSB_READ32(LVDSPP_ON);
    pLVDS->savePP_OFF = PSB_READ32(LVDSPP_OFF);
    pLVDS->saveLVDS = PSB_READ32(LVDS);
    pLVDS->savePP_CONTROL = PSB_READ32(PP_CONTROL);
    pLVDS->savePP_CYCLE = PSB_READ32(PP_CYCLE);
    pLVDS->saveBLC_PWM_CTL = PSB_READ32(BLC_PWM_CTL);
    pLVDS->backlight_duty_cycle = (pLVDS->saveBLC_PWM_CTL &
	BACKLIGHT_DUTY_CYCLE_MASK);

    /*
     * If the light is off at server startup, just make it full brightness
     */
    if (pLVDS->backlight_duty_cycle == 0)
	pLVDS->backlight_duty_cycle = psbLVDSGetMaxBacklight(pLVDS);
}

static void
psbLVDSRestore(xf86OutputPtr output)
{
    PsbLVDSOutputPtr pLVDS =
	containerOf(output->driver_private, PsbLVDSOutputRec, psbOutput);
    PsbDevicePtr pDevice = pLVDS->psbOutput.pDevice;

    PSB_DEBUG(output->scrn->scrnIndex, 2, "psbLVDSRestore\n");

    PSB_WRITE32(LVDSPP_ON, pLVDS->savePP_ON);
    PSB_WRITE32(LVDSPP_OFF, pLVDS->savePP_OFF);
    PSB_WRITE32(PP_CYCLE, pLVDS->savePP_CYCLE);
    PSB_WRITE32(LVDS, pLVDS->saveLVDS);
    if (pLVDS->savePP_CONTROL & POWER_TARGET_ON)
	psbLVDSSetPanelPower(pLVDS, TRUE);
    else
	psbLVDSSetPanelPower(pLVDS, FALSE);
}

static int
psbLVDSModeValid(xf86OutputPtr output, DisplayModePtr pMode)
{
    PsbLVDSOutputPtr pLVDS =
	containerOf(output->driver_private, PsbLVDSOutputRec, psbOutput);
    DisplayModePtr pFixedMode = pLVDS->panelFixedMode;

    if (pFixedMode) {
	if (pMode->HDisplay > pFixedMode->HDisplay)
	    return MODE_PANEL;
	if (pMode->VDisplay > pFixedMode->VDisplay)
	    return MODE_PANEL;
    }

    return MODE_OK;
}

static Bool
psbLVDSModeFixup(xf86OutputPtr output, DisplayModePtr mode,
    DisplayModePtr adjusted_mode)
{
    ScrnInfoPtr pScrn = output->scrn;
    xf86CrtcConfigPtr xf86_config = XF86_CRTC_CONFIG_PTR(pScrn);
    PsbLVDSOutputPtr pLVDS =
	containerOf(output->driver_private, PsbLVDSOutputRec, psbOutput);
    PsbCrtcPrivatePtr intel_crtc = output->crtc->driver_private;
    DisplayModePtr pFixedMode = pLVDS->panelFixedMode;
    int i;

    psbCheckCrtcs(pLVDS->psbOutput.pDevice);
    for (i = 0; i < xf86_config->num_output; i++) {
	xf86OutputPtr other_output = xf86_config->output[i];

	if (other_output != output && other_output->crtc == output->crtc) {
	    xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		"Can't enable LVDS and another output on the same pipe\n");
	    return FALSE;
	}
    }

    if (intel_crtc->pipe == 0) {
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
	    "Can't support LVDS on pipe A\n");
	return FALSE;
    }

    /* If we have timings from the BIOS for the panel, put them in
     * to the adjusted mode.  The CRTC will be set up for this mode,
     * with the panel scaling set up to source from the H/VDisplay
     * of the original mode.
     */
    if (pFixedMode != NULL) {
	adjusted_mode->HDisplay = pFixedMode->HDisplay;
	adjusted_mode->HSyncStart = pFixedMode->HSyncStart;
	adjusted_mode->HSyncEnd = pFixedMode->HSyncEnd;
	adjusted_mode->HTotal = pFixedMode->HTotal;
	adjusted_mode->VDisplay = pFixedMode->VDisplay;
	adjusted_mode->VSyncStart = pFixedMode->VSyncStart;
	adjusted_mode->VSyncEnd = pFixedMode->VSyncEnd;
	adjusted_mode->VTotal = pFixedMode->VTotal;
	adjusted_mode->Clock = pFixedMode->Clock;
	xf86SetModeCrtc(adjusted_mode, INTERLACE_HALVE_V);
    }

    /* XXX: if we don't have BIOS fixed timings (or we have
     * a preferred mode from DDC, probably), we should use the
     * DDC mode as the fixed timing.
     */

    /* XXX: It would be nice to support lower refresh rates on the
     * panels to reduce power consumption, and perhaps match the
     * user's requested refresh rate.
     */

    return TRUE;
}

static void
psbLVDSModeSet(xf86OutputPtr output, DisplayModePtr mode,
    DisplayModePtr adjusted_mode)
{
    PsbLVDSOutputPtr pLVDS =
	containerOf(output->driver_private, PsbLVDSOutputRec, psbOutput);
    CARD32 pfit_control;
    PsbDevicePtr pDevice = pLVDS->psbOutput.pDevice;

#if 0
    /* The LVDS pin pair needs to be on before the DPLLs are enabled.
     * This is an exception to the general rule that mode_set doesn't turn
     * things on.
     */
    PSB_WRITE32(LVDS, PSB_READ32(LVDS) | LVDS_PORT_EN | LVDS_PIPEB_SELECT);
#endif

    /* Enable automatic panel scaling so that non-native modes fill the
     * screen.  Should be enabled before the pipe is enabled, according to
     * register description and PRM.
     */
    pfit_control = (PFIT_ENABLE |
	VERT_AUTO_SCALE | HORIZ_AUTO_SCALE |
	VERT_INTERP_BILINEAR | HORIZ_INTERP_BILINEAR);

    if (pLVDS->panelWantsDither)
	pfit_control |= PANEL_8TO6_DITHER_ENABLE;

    PSB_WRITE32(PFIT_CONTROL, pfit_control);
}

/**
 * Detect the LVDS connection.
 *
 * This always returns OUTPUT_STATUS_CONNECTED.  This output should only have
 * been set up if the LVDS was actually connected anyway.
 */
static xf86OutputStatus
psbLVDSDetect(xf86OutputPtr output)
{
    PsbOutputPrivatePtr pOutput =
	(PsbOutputPrivatePtr) output->driver_private;

    PSB_DEBUG(output->scrn->scrnIndex, 2, "psbLVDSDetect %d",
	pOutput->pScrn == output->scrn);

    return ((pOutput->pScrn == output->scrn) ?
	XF86OutputStatusConnected : XF86OutputStatusDisconnected);
}

/**
 * Return the list of DDC modes if available, or the BIOS fixed mode otherwise.
 */
static DisplayModePtr
psbLVDSGetModes(xf86OutputPtr output)
{
    PsbLVDSOutputPtr pLVDS =
	containerOf(output->driver_private, PsbLVDSOutputRec, psbOutput);
    xf86MonPtr edid_mon;
    DisplayModePtr modes;

    edid_mon = xf86OutputGetEDID(output, pLVDS->psbOutput.pDDCBus);
    xf86OutputSetEDID(output, edid_mon);

    modes = xf86OutputGetEDIDModes(output);
    if (modes != NULL)
	return modes;

    if (!output->MonInfo) {
	edid_mon = xcalloc(1, sizeof(xf86Monitor));
	if (edid_mon) {
	    /* Set wide sync ranges so we get all modes
	     * handed to valid_mode for checking
	     */
	    edid_mon->det_mon[0].type = DS_RANGES;
	    edid_mon->det_mon[0].section.ranges.min_v = 0;
	    edid_mon->det_mon[0].section.ranges.max_v = 200;
	    edid_mon->det_mon[0].section.ranges.min_h = 0;
	    edid_mon->det_mon[0].section.ranges.max_h = 200;

	    output->MonInfo = edid_mon;
	}
    }

    if (pLVDS->panelFixedMode != NULL)
	return xf86DuplicateMode(pLVDS->panelFixedMode);

    return NULL;
}

static void
psbLVDSDestroy(xf86OutputPtr output)
{
    PsbLVDSOutputPtr pLVDS =
	containerOf(output->driver_private, PsbLVDSOutputRec, psbOutput);

    if (pLVDS && --pLVDS->psbOutput.refCount == 0) {
	psbOutputDestroy(&pLVDS->psbOutput);
	xfree(pLVDS);
    }
    output->driver_private = NULL;
}

#ifdef RANDR_12_INTERFACE
#define BACKLIGHT_NAME	"BACKLIGHT"
static Atom backlight_atom;
#endif /* RANDR_12_INTERFACE */

static void
psbLVDSCreateResources(xf86OutputPtr output)
{
#ifdef RANDR_12_INTERFACE
    ScrnInfoPtr pScrn = output->scrn;
    PsbLVDSOutputPtr pLVDS =
	containerOf(output->driver_private, PsbLVDSOutputRec, psbOutput);
    INT32 range[2];
    int data, err;

    /* Set up the backlight property, which takes effect immediately
     * and accepts values only within the range.
     *
     * XXX: Currently, RandR doesn't verify that properties set are
     * within the range.
     */
    backlight_atom = MakeAtom(BACKLIGHT_NAME, sizeof(BACKLIGHT_NAME) - 1,
	TRUE);

    range[0] = 0;
    range[1] = psbLVDSGetMaxBacklight(pLVDS);
    err = RRConfigureOutputProperty(output->randr_output, backlight_atom,
	FALSE, TRUE, FALSE, 2, range);
    if (err != 0) {
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
	    "RRConfigureOutputProperty error, %d\n", err);
    }
    /* Set the current value of the backlight property */
    data = pLVDS->backlight_duty_cycle;
    err = RRChangeOutputProperty(output->randr_output, backlight_atom,
	XA_INTEGER, 32, PropModeReplace, 1, &data, FALSE, TRUE);
    if (err != 0) {
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
	    "RRChangeOutputProperty error, %d\n", err);
    }
#endif /* RANDR_12_INTERFACE */
}

#ifdef RANDR_12_INTERFACE
static Bool
psbLVDSSetProperty(xf86OutputPtr output, Atom property,
    RRPropertyValuePtr value)
{
    PsbLVDSOutputPtr pLVDS =
	containerOf(output->driver_private, PsbLVDSOutputRec, psbOutput);

    if (property == backlight_atom) {
	INT32 val;

	if (value->type != XA_INTEGER || value->format != 32 ||
	    value->size != 1) {
	    return FALSE;
	}

	val = *(INT32 *) value->data;
	if (val < 0 || val > psbLVDSGetMaxBacklight(pLVDS))
	    return FALSE;

	psbLVDSSetBacklight(pLVDS, val);
	pLVDS->backlight_duty_cycle = val;
	return TRUE;
    }

    return TRUE;
}
#endif /* RANDR_12_INTERFACE */

static const xf86OutputFuncsRec psbLVDSOutputFuncs = {
    .create_resources = psbLVDSCreateResources,
    .dpms = psbLVDSDPMS,
    .save = psbLVDSSave,
    .restore = psbLVDSRestore,
    .mode_valid = psbLVDSModeValid,
    .mode_fixup = psbLVDSModeFixup,
    .prepare = psbOutputPrepare,
    .mode_set = psbLVDSModeSet,
    .commit = psbOutputCommit,
    .detect = psbLVDSDetect,
    .get_modes = psbLVDSGetModes,
#ifdef RANDR_12_INTERFACE
    .set_property = psbLVDSSetProperty,
#endif
    .destroy = psbLVDSDestroy
};

xf86OutputPtr
psbLVDSInit(ScrnInfoPtr pScrn, const char *name)
{
    xf86OutputPtr output;
    PsbLVDSOutputPtr pLVDS;
    DisplayModePtr modes, scan, bios_mode;

    output = xf86OutputCreate(pScrn, &psbLVDSOutputFuncs, name);
    if (!output)
	return NULL;
    pLVDS = xnfcalloc(sizeof(*pLVDS), 1);
    if (!pLVDS) {
	xf86OutputDestroy(output);
	return NULL;
    }
    psbOutputInit(psbDevicePTR(psbPTR(pScrn)), &pLVDS->psbOutput);

    pLVDS->psbOutput.type = PSB_OUTPUT_LVDS;
    pLVDS->psbOutput.refCount = 1;

    output->driver_private = &pLVDS->psbOutput;
    output->subpixel_order = SubPixelHorizontalRGB;
    output->interlaceAllowed = FALSE;
    output->doubleScanAllowed = FALSE;

    /* Set up the LVDS DDC channel.  Most panels won't support it, but it can
     * be useful if available.
     */
    I830I2CInit(pScrn, &pLVDS->psbOutput.pDDCBus, GPIOC, "LVDSDDC_C");

    /* Attempt to get the fixed panel mode from DDC.  Assume that the preferred
     * mode is the right one.
     */
    modes = psbOutputDDCGetModes(output);
    for (scan = modes; scan != NULL; scan = scan->next) {
	if (scan->type & M_T_PREFERRED)
	    break;
    }
    if (scan != NULL) {
	/* Pull our chosen mode out and make it the fixed mode */
	if (modes == scan)
	    modes = modes->next;
	if (scan->prev != NULL)
	    scan->prev = scan->next;
	if (scan->next != NULL)
	    scan->next = scan->prev;
	pLVDS->panelFixedMode = scan;
    }
    /* Delete the mode list */
    while (modes != NULL)
	xf86DeleteMode(&modes, modes);

    /* If we didn't get EDID, try checking if the panel is already turned on.
     * If so, assume that whatever is currently programmed is the correct mode.
     * FIXME: Better method, please.
     */

    if (pLVDS->panelFixedMode == NULL) {
	PsbDevicePtr pDevice = psbDevicePTR(psbPTR(pScrn));
	CARD32 lvds = PSB_READ32(LVDS);
        xf86CrtcRec crtc; /*faked */
	PsbCrtcPrivateRec pCrtc;
	
	memset(&crtc, 0, sizeof(xf86CrtcRec));
	pCrtc.pipe = 1;
	crtc.driver_private = &pCrtc;

	if (lvds & LVDS_PORT_EN) {
	    pLVDS->panelFixedMode = psbCrtcModeGet(pScrn, &crtc);
	    if (pLVDS->panelFixedMode != NULL)
		pLVDS->panelFixedMode->type |= M_T_PREFERRED;
	} else
	    /* Fall through to BIOS mode. */
	  ;
    }
    /* Get the LVDS fixed mode out of the BIOS.  We should support LVDS with
     * the BIOS being unavailable or broken, but lack the configuration options
     * for now.
     */
    bios_mode = i830_bios_get_panel_mode(pScrn, &pLVDS->panelWantsDither);
    if (bios_mode != NULL) {
	if (pLVDS->panelFixedMode != NULL) {
	    if (!xf86ModesEqual(pLVDS->panelFixedMode, bios_mode)) {
		xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
		    "BIOS panel mode data doesn't match probed data, "
		    "continuing with probed.\n");
		xf86DrvMsg(pScrn->scrnIndex, X_INFO, "BIOS mode:\n");
		xf86PrintModeline(pScrn->scrnIndex, bios_mode);
		xf86DrvMsg(pScrn->scrnIndex, X_INFO, "probed mode:\n");
		xf86PrintModeline(pScrn->scrnIndex, pLVDS->panelFixedMode);
		xfree(bios_mode->name);
		xfree(bios_mode);
	    }
	} else {
	    pLVDS->panelFixedMode = bios_mode;
	}
    } else {
	xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
	    "Couldn't detect panel mode.  Disabling panel\n");
	goto disable_exit;
    }

    return output;

  disable_exit:
    xf86DestroyI2CBusRec(pLVDS->psbOutput.pDDCBus, TRUE, TRUE);
    pLVDS->psbOutput.pDDCBus = NULL;
    xf86OutputDestroy(output);
    return NULL;
}
