/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <locale.h>

#include <X11/Xlib.h>

#include <gmodule.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#include <xine.h>

#ifdef HAVE_TAGLIB
#include <taglib/tag_c.h>
#endif

#include "xfmedia-common.h"
#include "xfmedia-icon.h"
#include "mainwin.h"

#if !GTK_CHECK_VERSION(2, 6, 0)
#include "gtk26-stock-icons.h"
#endif
#include "xfmedia-stock-icons.h"

XfceIconTheme *icon_theme = NULL;


GtkWidget *
xfmedia_custom_button_new_from_pixdata(const gchar *text,
        const GdkPixdata *pixdata)
{
    GtkWidget *btn, *img, *hbox, *lbl;
    GdkPixbuf *pix, *tmppix;
    gint iw, ih;
    
    hbox = gtk_hbox_new(FALSE, 4);
    gtk_widget_show(hbox);
    
    gtk_icon_size_lookup(GTK_ICON_SIZE_BUTTON, &iw, &ih);
    pix = gdk_pixbuf_from_pixdata(pixdata, FALSE, NULL);
    ih = gdk_pixbuf_get_height(pix);
    if(iw != ih) {
        tmppix = gdk_pixbuf_scale_simple(pix, iw, iw, GDK_INTERP_BILINEAR);
        g_object_unref(G_OBJECT(pix));
        pix = tmppix;
    }
    img = gtk_image_new_from_pixbuf(pix);
    g_object_unref(G_OBJECT(pix));
    gtk_widget_show(img);
    gtk_box_pack_start(GTK_BOX(hbox), img, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(text);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), hbox);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), btn);
    
    return btn;
}

GtkWidget *
xfmedia_custom_button_new(const gchar *text, const gchar *icon)
{
    GtkWidget *btn, *hbox, *img, *lbl;
    GdkPixbuf *pix;
    gint iw, ih;
    
    g_return_val_if_fail((text && *text) || icon, NULL);
    
    btn = gtk_button_new();
    
    hbox = gtk_hbox_new(FALSE, 4);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 0);
    gtk_widget_show(hbox);
    gtk_container_add(GTK_CONTAINER(btn), hbox);
    
    if(icon) {
        img = gtk_image_new_from_stock(icon, GTK_ICON_SIZE_BUTTON);
        if(!img || gtk_image_get_storage_type(GTK_IMAGE(img)) == GTK_IMAGE_EMPTY) {
            gtk_icon_size_lookup(GTK_ICON_SIZE_BUTTON, &iw, &ih);
            pix = xfce_icon_theme_load(icon_theme, icon, iw);
            if(pix) {
                if(img)
                    gtk_image_set_from_pixbuf(GTK_IMAGE(img), pix);
                else
                    img = gtk_image_new_from_pixbuf(pix);
                g_object_unref(G_OBJECT(pix));
            }
        }
        if(img) {
            gtk_widget_show(img);
            gtk_box_pack_start(GTK_BOX(hbox), img, FALSE, FALSE, 0);
        }
    }
    
    if(text) {
        lbl = gtk_label_new_with_mnemonic(text);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), btn);
    }
    
    return btn;
}

gchar *
xfmedia_filename_to_name(const gchar *filename)
{
    gchar *p, *utf8, *startat;
    GError *err = NULL;
    
    g_return_val_if_fail(filename, NULL);
    
    /* FIXME: i don't like this, but i don't want e.g. "dvd://" to return
     * a blank name */
    if((p=strstr(filename, ":/")) && p-filename < 6 && !strstr(filename, "://"))
        startat = (gchar *)filename;
    else {
        startat = g_strrstr(filename, G_DIR_SEPARATOR_S);
        if(startat)
            startat++;
        else
            startat = (gchar *)filename;
    }
    
    utf8 = g_filename_to_utf8(startat, -1, NULL, NULL, &err);
    if(!utf8) {
        if(err) {
            g_warning("Failed to convert filename to UTF-8 from filesystem encoding, attempting fallback (%d): %s ('%s')",
                    err->code, err->message, startat);
            g_error_free(err);
        }
        utf8 = xfmedia_brutal_utf8_validate(startat);
        
        /* FIXME: needed? */
        if(!utf8)
            utf8 = g_strdup(startat);
    }
    
    /* strip off file extension */
    p = g_strrstr(utf8, ".");
    if(p && strlen(p) < 6)
        *p = 0;
    
    /* convert '_' to space */
    p = strstr(utf8, "_");
    if(p) {
        *p = ' ';
        p++;
        while(*p) {
            if(*p == '_')
                *p = ' ';
            p++;
        }
    }
    
    /* convert '%20' to space */
    p = g_strrstr(utf8, "%20");
    while(p) {
        *p = ' ';
        memmove(p+1, p+3, strlen(p+3)+1);
        p = g_strrstr(utf8, "%20");
    }
    
    return utf8;
}

void
xfmedia_init_icon_theme()
{
    if(icon_theme)
        return;
    
    icon_theme = xfce_icon_theme_get_for_screen(NULL);
    xfce_icon_theme_prepend_search_path(icon_theme, DATADIR "/icons/");
}

gchar *
xfmedia_num_prettify(gint num)
{
    gchar buf[128], *p, *np, *str;
    gint newlen, commas, i;
    const gchar *locale;
    
    locale = setlocale(LC_NUMERIC, NULL);
    if(strcmp(locale, "C") && strcmp(locale, "POSIX"))
        return g_strdup_printf("%'d", num);
    
    g_snprintf(buf, 128, "%d", num);
    
    commas = (strlen(buf)-1) / 3;
    newlen = strlen(buf) + commas;
    str = g_malloc0(newlen+1);
    
    p = (gchar *)buf + strlen(buf) - 1;
    np = str + newlen - 1;
    
    i = 3;
    for(;;) {
        if(i-- == 0) {
            *np = ',';
            np--;
            i = 2;
        }
        *np = *p;
        
        if(p-- == buf || np-- == str)
            break;
    }
    
    return str;
}

GdkWindow *
xfmedia_get_window_decorations(GdkWindow *gwindow)
{
    GdkWindow *wmdec_win = NULL;
    GdkDisplay *gdpy = gdk_drawable_get_display(GDK_DRAWABLE(gwindow));
    Display *dpy = GDK_DISPLAY_XDISPLAY(gdpy);
    Window window = GDK_WINDOW_XWINDOW(gwindow);
    Window root_return, parent_return, *children_return = NULL;
    unsigned int nchildren = 0;
    
    XGrabServer(dpy);
    for(;;) {
        root_return = parent_return = None;
        XQueryTree(dpy, window, &root_return, &parent_return,
                &children_return, &nchildren);
        XFree(children_return);
        if(root_return == None || parent_return == None) {
            XUngrabServer(dpy);
            return NULL;
        }
        else if(root_return == parent_return)
            break;
        window = parent_return;
    };
    XUngrabServer(dpy);
    
    wmdec_win = gdk_window_foreign_new_for_display(gdpy, window);
    
    return wmdec_win;
}

/* ripped off of libgnomesomething, i think. */
GList *
xfmedia_uri_list_extract_uris(const gchar *uri_list)
{
    const gchar *p, *q;
    gchar *retval;
    GList *result = NULL;
    
    g_return_val_if_fail(uri_list != NULL, NULL);
    
    p = uri_list;
    
    /* We don't actually try to validate the URI according to RFC
     * 2396, or even check for allowed characters - we just ignore
     * comments and trim whitespace off the ends.  We also
     * allow LF delimination as well as the specified CRLF.
     */
    while(p) {
        if(*p != '#') {
            while(g_ascii_isspace((int)*p))
                p++;
            
            q = p;
            while(*q && (*q != '\n') && (*q != '\r'))
                q++;
            
            if(q > p) {
                q--;
                while(q > p && g_ascii_isspace((int)*q))
                    q--;
                
                retval = (gchar *)g_malloc(q - p + 2);
                strncpy(retval, p, q - p + 1);
                retval[q - p + 1] = '\0';
                
                result = g_list_prepend (result, retval);
            }
        }
        p = strchr (p, '\n');
        if(p)
            p++;
    }
    
    return g_list_reverse(result);
}

gint
xfmedia_tree_path_to_index(GtkTreePath *path)
{
    gchar *idx_str;
    gint idx;
    
    g_return_val_if_fail(path, -1);
    
    idx_str = gtk_tree_path_to_string(path);
    idx = atoi(idx_str);
    g_free(idx_str);
    
    return (idx >= 0 ? idx : -1);
}

gboolean
xfmedia_list_store_remove_path(GtkListStore *ls, GtkTreePath *path)
{
    GtkTreeIter itr;
    
    if(gtk_tree_model_get_iter(GTK_TREE_MODEL(ls), &itr, path))
        return gtk_list_store_remove(ls, &itr);
    return FALSE;
}

/**
 * xfmedia_brutal_utf8_validate:
 * @string: string to validate
 *
 * Tries its damnedest to convert @string into valid UTF-8.  Any string
 * returned is guaranteed to be UTF-8.  If it returns %NULL, then the string
 * passed in was already valid UTF-8.
 *
 * Returns: Valid UTF-8 string (free with g_free() when done), or %NULL if
 *          @string is already valid UTF-8.
 **/
gchar *
xfmedia_brutal_utf8_validate(const gchar *string)
{
    gchar *utf8 = NULL;
    const gchar *charset = NULL;
    gint read = 0, written = 0;
    GError *err = NULL;
    
    if(!g_utf8_validate(string, -1, NULL)) {
        if(g_get_charset(&charset)) {
            utf8 = g_convert_with_fallback(string, strlen(string), "UTF-8", charset,
                    "?", &read, &written, &err);
        }
        
        if(!utf8 || err) {
            if(err) {
                g_warning("Xfmedia: Unable to convert string to UTF-8 using system charset (%d): %s.  Read %d bytes, wrote %d bytes.",
                        err->code, err->message, read, written);
                g_error_free(err);
                err = NULL;
            }
            if(utf8)  /* will this ever happen? */
                g_free(utf8);
            
            read = written = 0;
            utf8 = g_convert_with_fallback(string, strlen(string), "UTF-8",
                    "ISO-8859-15", "?", &read, &written, &err);
            if(!utf8 || err) {
                gchar *tmp;
                
                if(err) {
                    g_warning("Xfmedia: Unable to convert string to UTF-8 using fallback charset (%d): %s.  Read %d bytes, wrote %d bytes.",
                            err->code, err->message, read, written);
                    g_error_free(err);
                    err = NULL;
                }
                if(utf8)  /* will this ever happen? */
                    g_free(utf8);
                
                tmp = g_strndup(string, read);
                read = written = 0;
                utf8 = g_convert_with_fallback(tmp, strlen(tmp), "UTF-8",
                        "ISO-8859-15", "?", &read, &written, &err);
                g_free(tmp);
                if(!utf8 || err) {
                    if(err) {
                        g_warning("Xfmedia: Unable to convert partial string to UTF-8 using fallback charset (%d): %s.  Read %d bytes, wrote %d bytes.",
                            err->code, err->message, read, written);
                        g_error_free(err);
                    }
                    g_critical("Xfmedia: Unable to convert string to UTF-8 using last-ditch effort.");
                    utf8 = g_strdup(_("(UTF-8 conversion failed)"));
                }
            }
        }
    }
    
    return utf8;
}



void
xfmedia_stock_icons_init()
{
    GtkIconFactory *ifac;
    GtkIconSource *isrc;
    GtkIconSet *iset;
    GdkPixbuf *pix;
    
    ifac = gtk_icon_factory_new();
    gtk_icon_factory_add_default(ifac);
    
    /* volume icon */
    iset = gtk_icon_set_new();
    isrc = gtk_icon_source_new();
    pix = gdk_pixbuf_from_pixdata(&stock_volume, FALSE, NULL);
    gtk_icon_source_set_pixbuf(isrc, pix);
    g_object_unref(G_OBJECT(pix));
    gtk_icon_source_set_size_wildcarded(isrc, TRUE);
    gtk_icon_source_set_state_wildcarded(isrc, TRUE);
    gtk_icon_source_set_direction_wildcarded(isrc, TRUE);
    gtk_icon_source_set_icon_name(isrc, "stock_volume");
    gtk_icon_set_add_source(iset, isrc);
    gtk_icon_factory_add(ifac, XFMEDIA_STOCK_VOLUME, iset);
    gtk_icon_set_unref(iset);
    
#if !GTK_CHECK_VERSION(2, 6, 0)
    
    /* folder icon */
    iset = gtk_icon_set_new();
    isrc = gtk_icon_source_new();
    pix = gdk_pixbuf_from_pixdata(&stock_folder, FALSE, NULL);
    gtk_icon_source_set_pixbuf(isrc, pix);
    g_object_unref(G_OBJECT(pix));
    gtk_icon_source_set_size_wildcarded(isrc, TRUE);
    gtk_icon_source_set_state_wildcarded(isrc, TRUE);
    gtk_icon_source_set_direction_wildcarded(isrc, TRUE);
    gtk_icon_set_add_source(iset, isrc);
    gtk_icon_factory_add(ifac, GTK_STOCK_DIRECTORY, iset);
    gtk_icon_set_unref(iset);
    
    /* media icons */
    
    iset = gtk_icon_set_new();
    isrc = gtk_icon_source_new();
    pix = gdk_pixbuf_from_pixdata(&xfmedia_next, FALSE, NULL);
    gtk_icon_source_set_pixbuf(isrc, pix);
    g_object_unref(G_OBJECT(pix));
    gtk_icon_source_set_size_wildcarded(isrc, TRUE);
    gtk_icon_source_set_state_wildcarded(isrc, TRUE);
    gtk_icon_source_set_direction_wildcarded(isrc, TRUE);
    gtk_icon_set_add_source(iset, isrc);
    gtk_icon_factory_add(ifac, GTK_STOCK_MEDIA_NEXT, iset);
    gtk_icon_set_unref(iset);
    
    iset = gtk_icon_set_new();
    isrc = gtk_icon_source_new();
    pix = gdk_pixbuf_from_pixdata(&xfmedia_pause, FALSE, NULL);
    gtk_icon_source_set_pixbuf(isrc, pix);
    g_object_unref(G_OBJECT(pix));
    gtk_icon_source_set_size_wildcarded(isrc, TRUE);
    gtk_icon_source_set_state_wildcarded(isrc, TRUE);
    gtk_icon_source_set_direction_wildcarded(isrc, TRUE);
    gtk_icon_set_add_source(iset, isrc);
    gtk_icon_factory_add(ifac, GTK_STOCK_MEDIA_PAUSE, iset);
    gtk_icon_set_unref(iset);
    
    iset = gtk_icon_set_new();
    isrc = gtk_icon_source_new();
    pix = gdk_pixbuf_from_pixdata(&xfmedia_play, FALSE, NULL);
    gtk_icon_source_set_pixbuf(isrc, pix);
    g_object_unref(G_OBJECT(pix));
    gtk_icon_source_set_size_wildcarded(isrc, TRUE);
    gtk_icon_source_set_state_wildcarded(isrc, TRUE);
    gtk_icon_source_set_direction_wildcarded(isrc, TRUE);
    gtk_icon_set_add_source(iset, isrc);
    gtk_icon_factory_add(ifac, GTK_STOCK_MEDIA_PLAY, iset);
    gtk_icon_set_unref(iset);
    
    iset = gtk_icon_set_new();
    isrc = gtk_icon_source_new();
    pix = gdk_pixbuf_from_pixdata(&xfmedia_prev, FALSE, NULL);
    gtk_icon_source_set_pixbuf(isrc, pix);
    g_object_unref(G_OBJECT(pix));
    gtk_icon_source_set_size_wildcarded(isrc, TRUE);
    gtk_icon_source_set_state_wildcarded(isrc, TRUE);
    gtk_icon_source_set_direction_wildcarded(isrc, TRUE);
    gtk_icon_set_add_source(iset, isrc);
    gtk_icon_factory_add(ifac, GTK_STOCK_MEDIA_PREVIOUS, iset);
    gtk_icon_set_unref(iset);
    
    iset = gtk_icon_set_new();
    isrc = gtk_icon_source_new();
    pix = gdk_pixbuf_from_pixdata(&xfmedia_stop, FALSE, NULL);
    gtk_icon_source_set_pixbuf(isrc, pix);
    g_object_unref(G_OBJECT(pix));
    gtk_icon_source_set_size_wildcarded(isrc, TRUE);
    gtk_icon_source_set_state_wildcarded(isrc, TRUE);
    gtk_icon_source_set_direction_wildcarded(isrc, TRUE);
    gtk_icon_set_add_source(iset, isrc);
    gtk_icon_factory_add(ifac, GTK_STOCK_MEDIA_STOP, iset);
    gtk_icon_set_unref(iset);
    
    /* about icon */
    iset = gtk_icon_set_new();
    isrc = gtk_icon_source_new();
    pix = gdk_pixbuf_from_pixdata(&about_icon, FALSE, NULL);
    gtk_icon_source_set_pixbuf(isrc, pix);
    g_object_unref(G_OBJECT(pix));
    gtk_icon_source_set_size_wildcarded(isrc, TRUE);
    gtk_icon_source_set_state_wildcarded(isrc, TRUE);
    gtk_icon_source_set_direction_wildcarded(isrc, TRUE);
    gtk_icon_set_add_source(iset, isrc);
    gtk_icon_factory_add(ifac, GTK_STOCK_ABOUT, iset);
    gtk_icon_set_unref(iset);
#endif
}

enum {
    XFMEDIA_SMALLER_ICON_X1_Y23 = 0,
    XFMEDIA_SMALLER_ICON_X23_Y23
};

static GdkPixbuf *
load_smaller_icon(const gchar *stock_id, GtkIconSize size, gint ratio)
{
    GdkPixbuf *old_pix, *pix = NULL;
    GtkWidget *dummy;
    gdouble xratio, yratio;
    
    /* lame */
    dummy = gtk_invisible_new();
    gtk_widget_realize(dummy);
    
    old_pix = gtk_widget_render_icon(dummy, stock_id, size, NULL);
    
    gtk_widget_destroy(dummy);
    
    switch(ratio) {
        case XFMEDIA_SMALLER_ICON_X1_Y23:
            xratio = 1.0;
            yratio = 2.0/3.0;
            break;
        case XFMEDIA_SMALLER_ICON_X23_Y23:
            xratio = yratio = 2.0/3.0;
            break;
        default:
            xratio = yratio = 1.0;
            break;
    }
    
    if(old_pix) {
        gint cur_w = gdk_pixbuf_get_width(old_pix);
        gint cur_h = gdk_pixbuf_get_height(old_pix);
        pix = gdk_pixbuf_scale_simple(old_pix, cur_w*xratio, cur_h*yratio,
                GDK_INTERP_BILINEAR);
        g_object_unref(G_OBJECT(old_pix));
    } else {
        DBG("couldn't render icon for '%s'", stock_id);
    }
    
    return pix;
}

static void
icon_theme_changed_stock_update(GtkIconTheme *icon_theme, gpointer user_data)
{
    GtkWidget *img = user_data;
    GdkPixbuf *pix;
    
    pix = load_smaller_icon(g_object_get_data(G_OBJECT(img), "xfmedia-stock-id"),
            GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(img), "xfmedia-stock-size")),
            GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(img), "xfmedia-stock-ratio")));
    gtk_image_set_from_pixbuf(GTK_IMAGE(img), pix);
    g_object_unref(G_OBJECT(pix));
}

static GtkWidget *
smaller_stock_icon_load_internal(const gchar *stock_id, GtkIconSize size,
        gint ratio)
{
    GdkPixbuf *pix;
    GtkWidget *img;
    
    pix = load_smaller_icon(stock_id, size, ratio);
    
    img = gtk_image_new_from_pixbuf(pix);
    g_object_unref(G_OBJECT(pix));
    g_object_set_data(G_OBJECT(img), "xfmedia-stock-id", (gpointer)stock_id);
    g_object_set_data(G_OBJECT(img), "xfmedia-stock-size",
            GUINT_TO_POINTER(size));
    g_object_set_data(G_OBJECT(img), "xfmedia-stock-ratio",
            GUINT_TO_POINTER(ratio));
    g_signal_connect(G_OBJECT(icon_theme), "changed",
            G_CALLBACK(icon_theme_changed_stock_update), img);
    
    return img;
}

GtkWidget *
xfmedia_squished_stock_icon_load(const gchar *stock_id, GtkIconSize size)
{
    return smaller_stock_icon_load_internal(stock_id, size,
            XFMEDIA_SMALLER_ICON_X1_Y23);
}

GtkWidget *
xfmedia_smaller_stock_icon_load(const gchar *stock_id, GtkIconSize size)
{
    return smaller_stock_icon_load_internal(stock_id, size,
            XFMEDIA_SMALLER_ICON_X23_Y23);
}

gboolean
xfmedia_rectangle_contains_point(GdkRectangle *rect, gint x, gint y)
{
    if(x > rect->x+rect->width
            || x < rect->x
            || y > rect->y+rect->height
            || y < rect->y)
    {
        return FALSE;
    }
    
    return TRUE;
}

void
xfmedia_widget_zero_pointer(GtkWidget **w)
{
    if(w)
        *w = NULL;
}

GtkWidget *
xfmedia_create_framebox(const gchar *title, GtkWidget **frame_bin)
{
#if LIBXFCEGUI4_CHECK_VERSION(4, 3, 4)
    return xfce_create_framebox(title, frame_bin);
#else
    GtkWidget *frame = xfce_framebox_new(title, TRUE);
    *frame_bin = XFCE_FRAMEBOX(frame)->hbox;
    return frame;
#endif
}

gchar *
xfmedia_xine_get_meta_info_helper(gpointer mwin_p, const gchar *uri,
        gint meta_info_id)
{
    XfmediaMainwin *mwin = mwin_p;
    gchar *metadata = NULL;
    
    /* this function isn't really thread-safe, but it should hopefully be ok. */
    
    g_return_val_if_fail(mwin->extra_xine_mx, NULL);
    
    g_mutex_lock(mwin->extra_xine_mx);
    
    if(!mwin->extra_stream)
        g_mutex_unlock(mwin->extra_xine_mx);
    g_return_val_if_fail(mwin->extra_stream, NULL);
    
    if(xine_open(mwin->extra_stream, uri)) {
        metadata = g_strdup(xine_get_meta_info(mwin->extra_stream, meta_info_id));
        xine_close(mwin->extra_stream);
    }
    
    g_mutex_unlock(mwin->extra_xine_mx);
    
    return metadata;
}

struct _XfmediaTagInfo
{
    XfmediaMainwin *mwin;
    gchar *uri;
    
#ifdef HAVE_TAGLIB
    TagLib_File *taglib_file;
    TagLib_Tag *taglib_tag;
    const TagLib_AudioProperties *taglib_ap;
    gboolean is_dirty;
#endif
};

XfmediaTagInfo *
xfmedia_tag_open(gpointer mwin_p, const gchar *uri)
{
    XfmediaTagInfo *tag_info = g_new0(XfmediaTagInfo, 1);
    
#ifdef HAVE_TAGLIB
    tag_info->taglib_file = taglib_file_new(uri);
    if(tag_info->taglib_file) {
        tag_info->taglib_tag = taglib_file_tag(tag_info->taglib_file);
        tag_info->taglib_ap = taglib_file_audioproperties(tag_info->taglib_file);
    }
#endif
    
    tag_info->mwin = mwin_p;
    tag_info->uri = g_strdup(uri);
    
    return tag_info;
}

gchar *
xfmedia_tag_get_info(XfmediaTagInfo *tag_info, gint type)
{
#ifdef HAVE_TAGLIB
    const gchar *info_tmp = NULL;
    gchar *info = NULL;
    
    if(!tag_info->taglib_tag)
        return xfmedia_xine_get_meta_info_helper(tag_info->mwin, tag_info->uri, type);
    
    switch(type) {
        case XINE_META_INFO_TITLE:
            info_tmp = taglib_tag_title(tag_info->taglib_tag);
            break;
        
        case XINE_META_INFO_ARTIST:
            info_tmp =  taglib_tag_artist(tag_info->taglib_tag);
            break;
        
        case XINE_META_INFO_ALBUM:
            info_tmp =  taglib_tag_album(tag_info->taglib_tag);
            break;
        
        case XINE_META_INFO_GENRE:
            info_tmp = taglib_tag_genre(tag_info->taglib_tag);
            break;
        
        case XINE_META_INFO_YEAR:
            {
                guint year = taglib_tag_year(tag_info->taglib_tag);
                if(year == 0)
                    return g_strdup("");
                else
                    return g_strdup_printf("%u", year);
            }
        
        case XINE_META_INFO_TRACK_NUMBER:
            {
                guint track = taglib_tag_track(tag_info->taglib_tag);
                if(track == 0)
                    return g_strdup("");
                else
                    return g_strdup_printf("%u", track);
            }
        
        default:
#endif
            return xfmedia_xine_get_meta_info_helper(tag_info->mwin, tag_info->uri, type);
#ifdef HAVE_TAGLIB
    }
    
    /* if we've fallen down here, we grabbed the info from taglib, but we want
     * to try to validate the UTF-8 first. */
    if(info_tmp) {
        info = xfmedia_brutal_utf8_validate(info_tmp);
        if(!info)
            info = g_strdup(info_tmp);
    }
    
    return info;
#endif
}

void
xfmedia_tag_set_info(XfmediaTagInfo *tag_info, gint type, const gchar *info)
{
#ifdef HAVE_TAGLIB
    if(!tag_info->taglib_tag)
        return;
    
    switch(type) {
        case XINE_META_INFO_TITLE:
            taglib_tag_set_title(tag_info->taglib_tag, info);
            tag_info->is_dirty = TRUE;
            break;
        
        case XINE_META_INFO_ARTIST:
            taglib_tag_set_artist(tag_info->taglib_tag, info);
            tag_info->is_dirty = TRUE;
            break;
        
        case XINE_META_INFO_ALBUM:
            taglib_tag_set_album(tag_info->taglib_tag, info);
            tag_info->is_dirty = TRUE;
            break;
        
        case XINE_META_INFO_COMMENT:
            taglib_tag_set_comment(tag_info->taglib_tag, info);
            tag_info->is_dirty = TRUE;
            break;
        
        case XINE_META_INFO_GENRE:
            taglib_tag_set_genre(tag_info->taglib_tag, info);
            tag_info->is_dirty = TRUE;
            break;
        
        case XINE_META_INFO_YEAR:
            {
                gint year = atoi(info);
                if(year > 0) {
                    taglib_tag_set_year(tag_info->taglib_tag, year);
                    tag_info->is_dirty = TRUE;
                }
            }
            break;
        
        case XINE_META_INFO_TRACK_NUMBER:
            {
                gint track = atoi(info);
                if(track > 0) {
                    taglib_tag_set_track(tag_info->taglib_tag, track);
                    tag_info->is_dirty = TRUE;
                }
            }
            break;
        
        default:
            break;
    }
#endif
}

gint
xfmedia_tag_get_stream_length(XfmediaTagInfo *tag_info)
{
#ifdef HAVE_TAGLIB
    if(tag_info->taglib_ap)
        return taglib_audioproperties_length(tag_info->taglib_ap);
    else
#endif
    {
        XfmediaMainwin *mwin = tag_info->mwin;
        gint pos_stream, pos_time, length = -2;
        
        g_return_val_if_fail(mwin->extra_xine_mx, -1);
    
        g_mutex_lock(mwin->extra_xine_mx);
        
        if(!mwin->extra_stream)
            g_mutex_unlock(mwin->extra_xine_mx);
        g_return_val_if_fail(mwin->extra_stream, -1);
        
        if(xine_open(mwin->extra_stream, tag_info->uri)) {
            if(xine_get_pos_length(mwin->extra_stream, &pos_stream,
                    &pos_time, &length))
            {
                length /= 1000;
            }
            xine_close(mwin->extra_stream);
        }
        
        g_mutex_unlock(mwin->extra_xine_mx);
        
        return length;
    }
}

gchar *
xfmedia_tag_to_string(XfmediaTagInfo *tag_info, const gchar *fmt)
{
    GString *gstr;
    gchar *str = NULL, *c, *filename, *value;
    gboolean have_a_field = FALSE;
    
    gstr = g_string_sized_new(128);
    
    for(c = (gchar *)fmt; *c; c++) {
        if(*c == '%') {
            c++;
            switch(*c) {
                case 'p':
                    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_ARTIST);
                    if(value && *value) {
                        g_string_append(gstr, value);
                        have_a_field = TRUE;
                    }
                    g_free(value);
                    break;
                case 't':
                    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_TITLE);
                    if(value && *value) {
                        g_string_append(gstr, value);
                        have_a_field = TRUE;
                    }
                    g_free(value);
                    break;
                case 'a':
                    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_ALBUM);
                    if(value && *value) {
                        g_string_append(gstr, value);
                        have_a_field = TRUE;
                    }
                    g_free(value);
                    break;
                case 'T':
                    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_TRACK_NUMBER);
                    if(value && *value) {
                        g_string_append(gstr, value);
                        have_a_field = TRUE;
                    }
                    g_free(value);
                    break;
                case 'g':
                    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_GENRE);
                    if(value && *value) {
                        g_string_append(gstr, value);
                        have_a_field = TRUE;
                    }
                    g_free(value);
                    break;
                case 'y':
                    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_YEAR);
                    if(value && *value) {
                        g_string_append(gstr, value);
                        have_a_field = TRUE;
                    }
                    g_free(value);
                    break;
                case 'f':
                    if(tag_info->uri) {
                        filename = xfmedia_filename_to_name(tag_info->uri);
                        g_string_append(gstr, filename);
                        g_free(filename);
                        have_a_field = TRUE;
                    }
                    break;
                case '%':
                    g_string_append(gstr, "%");
                    break;
                default:
                    g_string_append_printf(gstr, "%%%c", *c);
                    break;
            }
        } else
            g_string_append_c(gstr, *c);
    }
    
    if(have_a_field) {
        str = gstr->str;
        g_string_free(gstr, FALSE);
    } else
        g_string_free(gstr, TRUE);
    
    return str;
}

gboolean
xfmedia_tag_is_writable(XfmediaTagInfo *tag_info)
{
#ifdef HAVE_TAGLIB
    /* FIXME: check file permissions */
    if(tag_info->taglib_file)
        return TRUE;
#endif
    
    return FALSE;
}

gboolean
xfmedia_tag_is_dirty(XfmediaTagInfo *tag_info)
{
#ifdef HAVE_TAGLIB
    return tag_info->is_dirty;
#else
    return FALSE;
#endif
}

gboolean
xfmedia_tag_flush(XfmediaTagInfo *tag_info)
{
#ifdef HAVE_TAGLIB
    if(tag_info->is_dirty && tag_info->taglib_file)
        tag_info->is_dirty = !taglib_file_save(tag_info->taglib_file);
    
    return !tag_info->is_dirty;
#else
    return TRUE;
#endif
}

void
xfmedia_tag_close(XfmediaTagInfo *tag_info)
{
    g_return_if_fail(tag_info);
    
#ifdef HAVE_TAGLIB
    if(tag_info->taglib_file)
        taglib_file_free(tag_info->taglib_file);
    
    /* this is so weird */
    taglib_tag_free_strings();
#endif
    
    g_free(tag_info->uri);
    
    g_free(tag_info);
}
