/**
 * @file backend/xcb/grail_gestures.c
 * @brief Wrapper for grail gestures.
 *
 * Copyright 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.	 See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#include "geis_config.h"

#include "geis_attr.h"
#include "geis_class.h"
#include "geis_frame.h"
#include "geis/geis.h"
#include "geis_logging.h"
#include <grail.h>
#include <grail-types.h>
#include "grail_gestures.h"
#include <string.h>


typedef struct _AttrInitializer
{
  GeisString    name;
  GeisAttrType  type;
  void         *value;
} *AttrInitializer;

static GeisFloat zero = 0.0f;

static struct _AttrInitializer s_touch_attrs[] = {
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1, GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1, GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2, GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2, GEIS_ATTR_TYPE_FLOAT, &zero },
  { NULL, 0, &zero }
};

static struct _AttrInitializer s_drag_attrs[] = {
  { GEIS_GESTURE_ATTRIBUTE_DELTA_X,        GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_DELTA_Y,        GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_VELOCITY_X,     GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_VELOCITY_Y,     GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_POSITION_X,     GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_POSITION_Y,     GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1, GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1, GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2, GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2, GEIS_ATTR_TYPE_FLOAT, &zero },
  { NULL, 0, &zero }
};

static struct _AttrInitializer s_pinch_attrs[] = {
  { GEIS_GESTURE_ATTRIBUTE_RADIUS_DELTA,    GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_RADIAL_VELOCITY, GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_RADIUS,          GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { NULL, 0, &zero }
};

static struct _AttrInitializer s_pinch_attrs_v2[] = {
  { GEIS_GESTURE_ATTRIBUTE_RADIUS_DELTA,    GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_RADIAL_VELOCITY, GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_RADIUS,          GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_CENTROID_X,      GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_CENTROID_Y,      GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { NULL, 0, &zero }
};

static struct _AttrInitializer s_rotate_attrs[] = {
  { GEIS_GESTURE_ATTRIBUTE_ANGLE_DELTA,      GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_ANGULAR_VELOCITY, GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_ANGLE,            GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1,   GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1,   GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2,   GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2,   GEIS_ATTR_TYPE_FLOAT, &zero },
  { NULL, 0, &zero }
};

static struct _AttrInitializer s_rotate_attrs_v2[] = {
  { GEIS_GESTURE_ATTRIBUTE_ANGLE_DELTA,      GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_ANGULAR_VELOCITY, GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_ANGLE,            GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_CENTROID_X,       GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_CENTROID_Y,       GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1,   GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1,   GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2,   GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2,   GEIS_ATTR_TYPE_FLOAT, &zero },
  { NULL, 0, &zero }
};

static struct _AttrInitializer s_tap_attrs[] = {
  { GEIS_GESTURE_ATTRIBUTE_TAP_TIME,        GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_POSITION_X,      GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_POSITION_Y,      GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2,  GEIS_ATTR_TYPE_FLOAT, &zero },
  { NULL, 0, &zero }
};
#include <string.h>


/**
 * Maps grail gesture types into geis gesture classes.
 */
typedef struct _GrailClassMap
{
  GeisInteger     grail_type;
  GeisInteger     geis_class_id;
  GeisString      name;
  GeisString      class_name;
  int             grail_version;
  AttrInitializer attr;
  GeisInteger     touch_count;
} GrailClassMap;

static const GrailClassMap s_grail_class_map[] =
{
  { GRAIL_TYPE_DRAG1,   GEIS_GESTURE_PRIMITIVE_DRAG,   GEIS_GESTURE_TYPE_DRAG1,   GEIS_GESTURE_DRAG,   0, s_drag_attrs,      1 },
  { GRAIL_TYPE_DRAG2,   GEIS_GESTURE_PRIMITIVE_DRAG,   GEIS_GESTURE_TYPE_DRAG2,   GEIS_GESTURE_DRAG,   0, s_drag_attrs,      2 },
  { GRAIL_TYPE_DRAG3,   GEIS_GESTURE_PRIMITIVE_DRAG,   GEIS_GESTURE_TYPE_DRAG3,   GEIS_GESTURE_DRAG,   0, s_drag_attrs,      3 },
  { GRAIL_TYPE_EDRAG,   GEIS_GESTURE_PRIMITIVE_DRAG,   GEIS_GESTURE_TYPE_DRAG3,   GEIS_GESTURE_DRAG,   0, s_drag_attrs,      3 },
  { GRAIL_TYPE_DRAG4,   GEIS_GESTURE_PRIMITIVE_DRAG,   GEIS_GESTURE_TYPE_DRAG4,   GEIS_GESTURE_DRAG,   0, s_drag_attrs,      4 },
  { GRAIL_TYPE_MDRAG,   GEIS_GESTURE_PRIMITIVE_DRAG,   GEIS_GESTURE_TYPE_DRAG4,   GEIS_GESTURE_DRAG,   0, s_drag_attrs,      4 },
  { GRAIL_TYPE_DRAG5,   GEIS_GESTURE_PRIMITIVE_DRAG,   GEIS_GESTURE_TYPE_DRAG5,   GEIS_GESTURE_DRAG,   0, s_drag_attrs,      5 },
  { GRAIL_TYPE_PINCH1,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH1,  GEIS_GESTURE_PINCH,  1, s_pinch_attrs,     1 },
  { GRAIL_TYPE_PINCH2,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH2,  GEIS_GESTURE_PINCH,  1, s_pinch_attrs,     2 },
  { GRAIL_TYPE_PINCH3,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH3,  GEIS_GESTURE_PINCH,  1, s_pinch_attrs,     3 },
  { GRAIL_TYPE_EPINCH,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH3,  GEIS_GESTURE_PINCH,  1, s_pinch_attrs,     3 },
  { GRAIL_TYPE_PINCH4,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH4,  GEIS_GESTURE_PINCH,  1, s_pinch_attrs,     4 },
  { GRAIL_TYPE_MPINCH,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH4,  GEIS_GESTURE_PINCH,  1, s_pinch_attrs,     4 },
  { GRAIL_TYPE_PINCH5,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH5,  GEIS_GESTURE_PINCH,  1, s_pinch_attrs,     5 },
  { GRAIL_TYPE_PINCH1,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH1,  GEIS_GESTURE_PINCH,  2, s_pinch_attrs_v2,  1 },
  { GRAIL_TYPE_PINCH2,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH2,  GEIS_GESTURE_PINCH,  2, s_pinch_attrs_v2,  2 },
  { GRAIL_TYPE_PINCH3,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH3,  GEIS_GESTURE_PINCH,  2, s_pinch_attrs_v2,  3 },
  { GRAIL_TYPE_EPINCH,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH3,  GEIS_GESTURE_PINCH,  2, s_pinch_attrs_v2,  3 },
  { GRAIL_TYPE_PINCH4,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH4,  GEIS_GESTURE_PINCH,  2, s_pinch_attrs_v2,  4 },
  { GRAIL_TYPE_MPINCH,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH4,  GEIS_GESTURE_PINCH,  2, s_pinch_attrs_v2,  4 },
  { GRAIL_TYPE_PINCH5,  GEIS_GESTURE_PRIMITIVE_PINCH,  GEIS_GESTURE_TYPE_PINCH5,  GEIS_GESTURE_PINCH,  2, s_pinch_attrs_v2,  5 },
  { GRAIL_TYPE_ROTATE1, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE1, GEIS_GESTURE_PINCH,  1, s_rotate_attrs,    1 },
  { GRAIL_TYPE_ROTATE2, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE2, GEIS_GESTURE_ROTATE, 1, s_rotate_attrs,    2 },
  { GRAIL_TYPE_ROTATE3, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE3, GEIS_GESTURE_ROTATE, 1, s_rotate_attrs,    3 },
  { GRAIL_TYPE_EROTATE, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE3, GEIS_GESTURE_ROTATE, 1, s_rotate_attrs,    3 },
  { GRAIL_TYPE_ROTATE4, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE4, GEIS_GESTURE_ROTATE, 1, s_rotate_attrs,    4 },
  { GRAIL_TYPE_MROTATE, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE4, GEIS_GESTURE_ROTATE, 1, s_rotate_attrs,    4 },
  { GRAIL_TYPE_ROTATE5, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE5, GEIS_GESTURE_ROTATE, 1, s_rotate_attrs,    5 },
  { GRAIL_TYPE_ROTATE1, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE1, GEIS_GESTURE_PINCH,  2, s_rotate_attrs_v2, 1 },
  { GRAIL_TYPE_ROTATE2, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE2, GEIS_GESTURE_ROTATE, 2, s_rotate_attrs_v2, 2 },
  { GRAIL_TYPE_ROTATE3, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE3, GEIS_GESTURE_ROTATE, 2, s_rotate_attrs_v2, 3 },
  { GRAIL_TYPE_EROTATE, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE3, GEIS_GESTURE_ROTATE, 2, s_rotate_attrs_v2, 3 },
  { GRAIL_TYPE_ROTATE4, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE4, GEIS_GESTURE_ROTATE, 2, s_rotate_attrs_v2, 4 },
  { GRAIL_TYPE_MROTATE, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE4, GEIS_GESTURE_ROTATE, 2, s_rotate_attrs_v2, 4 },
  { GRAIL_TYPE_ROTATE5, GEIS_GESTURE_PRIMITIVE_ROTATE, GEIS_GESTURE_TYPE_ROTATE5, GEIS_GESTURE_ROTATE, 2, s_rotate_attrs_v2, 5 },
  { GRAIL_TYPE_TAP1,    GEIS_GESTURE_PRIMITIVE_TAP,    GEIS_GESTURE_TYPE_TAP1,    GEIS_GESTURE_TAP,    0, s_tap_attrs,       1 },
  { GRAIL_TYPE_TAP2,    GEIS_GESTURE_PRIMITIVE_TAP,    GEIS_GESTURE_TYPE_TAP2,    GEIS_GESTURE_TAP,    0, s_tap_attrs,       2 },
  { GRAIL_TYPE_TAP3,    GEIS_GESTURE_PRIMITIVE_TAP,    GEIS_GESTURE_TYPE_TAP3,    GEIS_GESTURE_TAP,    0, s_tap_attrs,       3 },
  { GRAIL_TYPE_TAP4,    GEIS_GESTURE_PRIMITIVE_TAP,    GEIS_GESTURE_TYPE_TAP4,    GEIS_GESTURE_TAP,    0, s_tap_attrs,       4 },
  { GRAIL_TYPE_TAP5,    GEIS_GESTURE_PRIMITIVE_TAP,    GEIS_GESTURE_TYPE_TAP5,    GEIS_GESTURE_TAP,    0, s_tap_attrs,       5 },
  { GRAIL_TYPE_TOUCH1,  GEIS_GESTURE_PRIMITIVE_TOUCH,  GEIS_GESTURE_TYPE_TOUCH1,  GEIS_GESTURE_TOUCH,  0, s_touch_attrs,     1 },
  { GRAIL_TYPE_TOUCH2,  GEIS_GESTURE_PRIMITIVE_TOUCH,  GEIS_GESTURE_TYPE_TOUCH2,  GEIS_GESTURE_TOUCH,  0, s_touch_attrs,     2 },
  { GRAIL_TYPE_TOUCH3,  GEIS_GESTURE_PRIMITIVE_TOUCH,  GEIS_GESTURE_TYPE_TOUCH3,  GEIS_GESTURE_TOUCH,  0, s_touch_attrs,     3 },
  { GRAIL_TYPE_ETOUCH,  GEIS_GESTURE_PRIMITIVE_TOUCH,  GEIS_GESTURE_TYPE_TOUCH3,  GEIS_GESTURE_TOUCH,  0, s_touch_attrs,     3 },
  { GRAIL_TYPE_TOUCH4,  GEIS_GESTURE_PRIMITIVE_TOUCH,  GEIS_GESTURE_TYPE_TOUCH4,  GEIS_GESTURE_TOUCH,  0, s_touch_attrs,     4 },
  { GRAIL_TYPE_MTOUCH,  GEIS_GESTURE_PRIMITIVE_TOUCH,  GEIS_GESTURE_TYPE_TOUCH4,  GEIS_GESTURE_TOUCH,  0, s_touch_attrs,     4 },
  { GRAIL_TYPE_TOUCH5,  GEIS_GESTURE_PRIMITIVE_TOUCH,  GEIS_GESTURE_TYPE_TOUCH5,  GEIS_GESTURE_TOUCH,  0, s_touch_attrs,     5 },
  { -1, -1, NULL, NULL, 0, NULL, 0 } /* sentinel */
};


/*
 * Sets the indicated bit in the bitmask.
 *
 * @todo: convert form uint32_t masks to byte masks
 */
static inline void
_grail_mask_set_bit(uint16_t  mask_len GEIS_UNUSED,
                    uint32_t* mask,
                    int       index)
{
  mask[index >> 5] |= 1 << (index & 0x1f);
}

/*
 * Masks the bits in one bitmask by the bits in another.
 */
static inline void
_grail_mask_and(uint16_t  mask_len, uint32_t* mask_lhs, uint32_t* mask_rhs)
{
  uint16_t i;
  for (i = 0; i < mask_len; ++i)
    mask_lhs[i] &= mask_rhs[i];
}


/*
 * Initializes the XCB gesture bitmap flags.
 *
 * The gestures that are enabled by 'all gesture' depends on the various
 * subscription settings.
 */
void
geis_xcb_backend_gesture_bitmask_init(uint16_t              *mask_len,
                                      uint32_t              *mask)
{
  _grail_mask_clear(*mask_len, mask);

  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_DRAG1);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_PINCH1);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_ROTATE1);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_TAP1);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_TOUCH1);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_DRAG2);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_PINCH2);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_ROTATE2);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_TAP2);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_TOUCH2);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_TAP3);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_TAP4);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_DRAG5);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_PINCH5);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_ROTATE5);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_TAP5);

  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_EDRAG);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_EPINCH);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_EROTATE);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_ETOUCH);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_MDRAG);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_MPINCH);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_MROTATE);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_MTOUCH);

  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_DRAG3);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_PINCH3);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_ROTATE3);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_TOUCH3);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_DRAG4);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_PINCH4);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_ROTATE4);
  _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_TOUCH4);

  geis_debug("initial bitmap is 0x%08x %08x", mask[1], mask[0]);
}


/*
 * Clears all gesture bitmap bits not associated with a named class.
 */
void
geis_xcb_backend_gesture_bitmask_filter_class(GeisString  class_name,
                                              uint16_t   *mask_len,
                                              uint32_t   *mask)
{
  const GrailClassMap *grail_class;
  uint32_t class_mask[*mask_len];

  _grail_mask_clear(*mask_len, class_mask);
  for (grail_class = s_grail_class_map;
       grail_class->geis_class_id >= 0;
       ++grail_class)
  {
    if (0 == strcmp(grail_class->class_name, class_name))
    {
      _grail_mask_set_bit(*mask_len, class_mask, grail_class->grail_type);
    }
  }
  _grail_mask_and(*mask_len, mask, class_mask);
  geis_debug("class=\"%s\" mask is 0x%08x %08x bitmap is now 0x%08x %08x",
             class_name, class_mask[1], class_mask[0], mask[1], mask[0]);
}


/*
 * Masks out all bits that do not cirrespond to the chosen touches.
 */
void
geis_xcb_backend_gesture_bitmask_filter_touches(GeisInteger          touches,
                                                GeisFilterOperation  op,
                                                uint16_t            *mask_len,
                                                uint32_t            *mask)
{
  const GrailClassMap *grail_class;
  uint32_t touch_mask[*mask_len];

  _grail_mask_clear(*mask_len, touch_mask);
  for (grail_class = s_grail_class_map;
       grail_class->geis_class_id >= 0;
       ++grail_class)
  {
    if ((op == GEIS_FILTER_OP_EQ && grail_class->touch_count == touches)
     || (op == GEIS_FILTER_OP_NE && grail_class->touch_count != touches)
     || (op == GEIS_FILTER_OP_GT && grail_class->touch_count > touches)
     || (op == GEIS_FILTER_OP_GE && grail_class->touch_count >= touches)
     || (op == GEIS_FILTER_OP_LT && grail_class->touch_count < touches)
     || (op == GEIS_FILTER_OP_LE && grail_class->touch_count <= touches))
    {
      _grail_mask_set_bit(*mask_len, touch_mask, grail_class->grail_type);
    }
  }
  _grail_mask_and(*mask_len, mask, touch_mask);
  geis_debug("touches=%d op=%d mask is 0x%08x %08x bitmap is now 0x%08x %08x",
             touches, op, touch_mask[1], touch_mask[0], mask[1], mask[0]);
}


void
geis_xcb_backend_gesture_bitmask_filter_system(GeisBoolean is_system,
                                               uint16_t   *mask_len,
                                               uint32_t   *mask)
{
  uint32_t system_mask[*mask_len];

  _grail_mask_clear(*mask_len, system_mask);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_DRAG1);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_PINCH1);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_ROTATE1);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_TAP1);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_TOUCH1);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_DRAG2);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_PINCH2);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_ROTATE2);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_TAP2);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_TOUCH2);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_TAP3);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_TAP4);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_DRAG5);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_PINCH5);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_ROTATE5);
  _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_TAP5);

  if (is_system)
  {
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_EDRAG);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_EPINCH);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_EROTATE);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_ETOUCH);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_MDRAG);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_MPINCH);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_MROTATE);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_MTOUCH);
  }
  else
  {
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_DRAG3);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_PINCH3);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_ROTATE3);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_TOUCH3);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_DRAG4);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_PINCH4);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_ROTATE4);
    _grail_mask_set_bit(*mask_len, system_mask, GRAIL_TYPE_TOUCH4);
  }
  _grail_mask_and(*mask_len, mask, system_mask);
  geis_debug("is_system=%d mask is 0x%08x %08x bitmap is now 0x%08x %08x",
             is_system, system_mask[1], system_mask[0], mask[1], mask[0]);
}


void
geis_xcb_backend_gesture_bitmask_filter_grabs(GeisBoolean  is_grab,
                                              uint16_t    *mask_len,
                                              uint32_t    *mask)
{
  if (is_grab)
  {
    _grail_mask_set_bit(*mask_len, mask, GRAIL_TYPE_SYSFLAG1);
  }
  geis_debug("is_grab=%d bitmap is now 0x%08x %08x", is_grab, mask[1], mask[0]);
}


static GeisStatus
_add_detail_attrs(GeisGestureClass gesture_class, AttrInitializer inits)
{
  GeisStatus status = GEIS_STATUS_UNKNOWN_ERROR;
  GeisInteger i1 = 1;

  GeisAttr attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_TOUCHES,
                                GEIS_ATTR_TYPE_INTEGER,
                                &i1);
  if (!attr)
  {
    geis_error("error allocating touches attr");
    goto error_exit;
  }
  if (GEIS_STATUS_SUCCESS != geis_gesture_class_add_attr(gesture_class, attr))
  {
    geis_error("error inserting touches attr");
    goto error_exit;
  }

  for (AttrInitializer i = inits; i->name; ++i)
  {
    attr = geis_attr_new(i->name, i->type, i->value);
    if (!attr)
    {
      geis_error("error allocating new attr");
      goto error_exit;
    }

    if (GEIS_STATUS_SUCCESS != geis_gesture_class_add_attr(gesture_class, attr))
    {
      geis_error("error inserting new attr");
      goto error_exit;
    }
  }
  status = GEIS_STATUS_SUCCESS;

error_exit:
  return status;
}


GeisStatus
geis_xcb_backend_add_drag_attrs(GeisGestureClass gesture_class)
{
  GeisStatus status = _add_detail_attrs(gesture_class, s_drag_attrs);
  return status;
}


GeisStatus
geis_xcb_backend_add_pinch_attrs(GeisGestureClass gesture_class)
{
  GeisStatus status = _add_detail_attrs(gesture_class, s_pinch_attrs);
  return status;
}


GeisStatus
geis_xcb_backend_add_rotate_attrs(GeisGestureClass gesture_class)
{
  GeisStatus status = _add_detail_attrs(gesture_class, s_rotate_attrs);
  return status;
}


GeisStatus
geis_xcb_backend_add_tap_attrs(GeisGestureClass gesture_class)
{
  GeisStatus status = _add_detail_attrs(gesture_class, s_tap_attrs);
  return status;
}

GeisInteger
geis_xcb_backend_primitive_class(GeisInteger grail_type)
{
  const GrailClassMap *grail_class;
  for (grail_class = s_grail_class_map;
       grail_class->geis_class_id >= 0;
       ++grail_class)
  {
    if (grail_class->grail_type == grail_type)
    {
      return grail_class->geis_class_id;
    }
  }
  return 0;
}


GeisSize
geis_xcb_backend_map_grail_attrs(GeisInteger grail_type,
                                 int         grail_version,
                                 GeisSize    num_properties,
                                 float      *properties,
                                 GeisFrame   frame)
{
  GeisSize num_unmapped_properties = num_properties;
  GeisSize i;

  const GrailClassMap *grail_class;
  for (grail_class = s_grail_class_map;
       grail_class->geis_class_id >= 0;
       ++grail_class)
  {
    if (grail_class->grail_type == grail_type
     && (grail_class->grail_version == 0 || grail_class->grail_version == grail_version))
    {
      GeisAttr attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_GESTURE_NAME,
                                    GEIS_ATTR_TYPE_STRING,
                                    (char *)grail_class->name);
      geis_frame_add_attr(frame, attr);

      attr = geis_attr_new("geis gesture class id",
                           GEIS_ATTR_TYPE_INTEGER,
                           (void *)&grail_class->geis_class_id);
      geis_frame_add_attr(frame, attr);

      for (i = 0; i < num_properties && grail_class->attr[i].name; ++i)
      {
	attr = geis_attr_new(grail_class->attr[i].name,
	                     grail_class->attr[i].type,
	                     &properties[i]);
	geis_frame_add_attr(frame, attr);
	--num_unmapped_properties;
      }
      return num_unmapped_properties;
    }
  }

  return num_unmapped_properties;
}
