/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */
using Dee;
using Zeitgeist;
using Zeitgeist.Timestamp;
using Config;
using Gee;
using GMenu;

namespace Unity.ApplicationsPlace {

  const string ICON_PATH = Config.DATADIR + "/icons/unity-icon-theme/places/svg/";

  public class Daemon : GLib.Object, Unity.Activation
  {
    private Zeitgeist.Log log;
    private Zeitgeist.Index zg_index;
    
    /* The searcher for online material may be null if it fails to load
     * the Xapian index from the Software Center */
    private Unity.Package.Searcher? pkgsearcher;
    public Unity.Package.Searcher appsearcher;

    private Unity.PlaceController control;
    private Unity.PlaceEntryInfo applications;
    
    private Unity.ApplicationsPlace.Runner runner;

    /* For each section we have a set filtering query we use to restrict
     * Xapian queries to that section */
    private Gee.List<string> section_queries;

    private Gee.List<string> image_extensions;
    private HashTable<string,Icon> file_icon_cache;
    
    /* We remember the previous search so we can figure out if we should do
     * incremental filtering of the result models */
    private PlaceSearch? previous_search;
    private PlaceSearch? previous_global_search;
    
    /* To make sure we don't fire of unnecessary queries if the active section
     * is in fact not changed */
    private uint previous_active_section;
    
    private PtrArray zg_templates;
    
    /* Gnome menu structure - also used to check whether apps are installed */
    private uint app_menu_changed_reindex_timeout = 0;
    private GMenu.Tree app_menu = null;

    private bool all_models_synced;
    
    private Regex? uri_regex;

    construct
    {
      var sections_model = new Dee.SharedModel("com.canonical.Unity.ApplicationsPlace.SectionsModel");
      sections_model.set_schema ("s", "s");

      var groups_model = new Dee.SharedModel("com.canonical.Unity.ApplicationsPlace.GroupsModel");
      groups_model.set_schema ("s", "s", "s");

      var global_groups_model = new Dee.SharedModel("com.canonical.Unity.ApplicationsPlace.GlobalGroupsModel");
      global_groups_model.set_schema ("s", "s", "s");

      var results_model = new Dee.SharedModel("com.canonical.Unity.ApplicationsPlace.ResultsModel");
      results_model.set_schema ("s", "s", "u", "s", "s", "s");

      var global_results_model = new Dee.SharedModel("com.canonical.Unity.ApplicationsPlace.GlobalResultsModel");
      global_results_model.set_schema ("s", "s", "u", "s", "s", "s");

      section_queries = new Gee.ArrayList<string> ();
      populate_section_queries();
      populate_zg_templates ();

      applications = new PlaceEntryInfo ("/com/canonical/unity/applicationsplace/applications");
      applications.sections_model = sections_model;
      applications.entry_renderer_info.groups_model = groups_model;
      applications.entry_renderer_info.results_model = results_model;
      applications.global_renderer_info.groups_model = global_groups_model;
      applications.global_renderer_info.results_model = global_results_model;

      applications.icon = @"$(Config.PREFIX)/share/unity/themes/applications.png";

      log = new Zeitgeist.Log();
      zg_index = new Zeitgeist.Index();
      
      pkgsearcher = new Unity.Package.Searcher ();
      if (pkgsearcher == null)
        {
          critical ("Failed to load Software Center index. 'Apps Available for Download' will not be listed");
        }
      
      /* Image file extensions in order of popularity */
      image_extensions = new Gee.ArrayList<string> ();
      image_extensions.add ("png");
      image_extensions.add ("xpm");
      image_extensions.add ("svg");
      image_extensions.add ("tiff");
      image_extensions.add ("ico");
      image_extensions.add ("tif");
      image_extensions.add ("jpg");
      
      previous_search = null;
      previous_global_search = null;
      previous_active_section = Section.LAST_SECTION; /* Must be an invalid section! */
      
      build_app_menu_index ();
      
      file_icon_cache = new HashTable<string,Icon>(str_hash, str_equal);
      
      /* Listen for section changes */
      applications.notify["active-section"].connect (
        (obj, pspec) => {
          if (!all_models_synced)
            return;

          if (previous_active_section == applications.active_section)
            return;

          var search = applications.active_search;
          var section = (Section) applications.active_section;
          update_entry_search.begin (search, section);
          
          previous_search = search;
          previous_active_section = applications.active_section;
        }
      );

      /* Listen for changes to the place entry search */
      applications.notify["active-search"].connect (
        (obj, pspec) => {
          if (!all_models_synced)
            return;

          var search = applications.active_search;
          Section section = (Section) applications.active_section;
          
          if (!Utils.search_has_really_changed (previous_search, search))
            return;
          
          update_entry_search.begin (search, section);
          previous_search = search;
        }
      );

      /* Listen for changes to the global search aka Dash search */
      applications.notify["active-global-search"].connect (
        (obj, pspec) => {
          if (!all_models_synced)
            return;

          var search = applications.active_global_search;
          
          if (!Utils.search_has_really_changed (previous_global_search, search))
            return;
          
          update_global_search.begin(search);
          previous_global_search = search;
        }
      );

      /* Listen for changes in the installed applications */
      AppInfoManager.get_instance().changed.connect (on_appinfo_changed);
      
      /* Now start the RunEntry */
      runner = new Unity.ApplicationsPlace.Runner (this);
      
      try {
        uri_regex = new Regex ("[a-z]+:.+");
      } catch (RegexError e) {
        uri_regex = null;
        critical ("Failed to compile URI regex. URL launching will be disabled");
      }
      

      /* The last thing we do is export the controller. Once that is up,
       * clients will expect the SharedModels to work */
      control = new Unity.PlaceController ("/com/canonical/unity/applicationsplace");
      control.add_entry (applications);
      control.add_entry (runner.place_entry);
      control.activation = this;
      try {
        control.export ();
      } catch (IOError error) {
        critical ("Failed to export DBus service for '%s': %s",
                  control.dbus_path, error.message);
      }

      /* We should not start manipulating any of our models before they are
       * all synchronized. When they are we set all_models_synced = true */
      sections_model.notify["synchronized"].connect (check_models_synced);
      groups_model.notify["synchronized"].connect (check_models_synced);
      global_groups_model.notify["synchronized"].connect (check_models_synced);
      results_model.notify["synchronized"].connect (check_models_synced);
      global_results_model.notify["synchronized"].connect (check_models_synced);
      all_models_synced = false;
    }

    /* The check_models_synced() method acts like a latch - once all models
     * have reported themselves to be synchronized we set
     * all_models_synced = true and tell the searches to re-check their state
     * as they should refuse to run when all_models_synced == false */
    private void check_models_synced (Object obj, ParamSpec pspec)
    {
      if ((applications.sections_model as Dee.SharedModel).synchronized &&
          (applications.entry_renderer_info.groups_model as Dee.SharedModel).synchronized &&
          (applications.entry_renderer_info.results_model as Dee.SharedModel).synchronized &&
          (applications.global_renderer_info.groups_model as Dee.SharedModel).synchronized &&
          (applications.global_renderer_info.results_model as Dee.SharedModel).synchronized) {
        if (all_models_synced == false)
          {
            all_models_synced = true;
            
            populate_sections ();
            populate_groups (applications.entry_renderer_info.groups_model);
            populate_groups (applications.global_renderer_info.groups_model);
            
            /* Emitting notify here will make us recheck if the search results
             * need update. In the negative case this is a noop */
            applications.notify_property ("active-search");
            applications.notify_property ("active-global-search");
          }
      }
    }

    private void populate_sections ()
    {
      var sections = applications.sections_model;

      if (sections.get_n_rows() != 0)
        {
          debug ("Sections model already populated. We probably cloned it off Unity. Rebuilding.");
          sections.clear ();
        }

      sections.append (_("All Applications"), "");
      sections.append (_("Accessories"), "");
      sections.append (_("Universal Access"), "");
      sections.append (_("Developer Tools"), "");
      sections.append (_("Education"), "");
      sections.append (_("Science & Engineering"), "");
      sections.append (_("Games"), "");
      sections.append (_("Graphics"), "");
      sections.append (_("Internet"), "");
      sections.append (_("Multimedia"), "");
      sections.append (_("Office"), "");
      sections.append (_("Themes & Tweaks"), "");
      sections.append (_("System"), "");
    }

    private void populate_groups (Dee.Model groups)
    {
      if (groups.get_n_rows() != 0)
        {
          debug ("The groups model already populated. We probably cloned it off Unity. Rebuilding.");
          groups.clear ();
        }

      groups.append ("UnityShowcaseRenderer",
                     _("Most Frequently Used"),
                     ICON_PATH + "group-mostused.svg");
      groups.append ("UnityDefaultRenderer",
                     _("Installed"),
                     ICON_PATH + "group-installed.svg");
      groups.append ("UnityDefaultRenderer",
                     _("Apps Available for Download"),
                     ICON_PATH + "group-available.svg");
      groups.append ("UnityEmptySearchRenderer",
                     "No search results", // No i18n, should never be rendered
                     "");
      groups.append ("UnityEmptySectionRenderer",
                     "Empty section", // No i18n, should never be rendered
                     "");
      
      /* Always expand the Installed group */
      applications.entry_renderer_info.set_hint ("ExpandedGroups",
                                                 @"$((uint)Group.INSTALLED)");
    }

    private void populate_section_queries ()
    {
      /* XDG category names. Not for translation. */
      /* We need the hack for ALL_APPLICATIONS below because Xapian doesn't
       * like '' or '*' queries */
      section_queries.add ("NOT category:XYZ"); //ALL_APPLICATIONS
      section_queries.add ("(category:Utility AND NOT category:Accessibility)"); //ACCESSORIES
      section_queries.add ("(category:Accessibility AND NOT category:Settings)"); //UNIVERSAL_ACCESS
      section_queries.add ("category:Development"); //DEVELOPER_TOOLS FIXME emacs.desktop should be added
      section_queries.add ("(category:Education AND NOT category:Science)"); // EDUCATION
      section_queries.add ("(category:Science OR category:Engineering)"); // SCIENCE
      section_queries.add ("category:Game"); // GAMES
      section_queries.add ("category:Graphics"); // GRAPHICS
      section_queries.add ("category:Network"); // INTERNET
      section_queries.add ("category:AudioVideo"); // MULTIMEDIA
      section_queries.add ("category:Office"); // OFFICE
      section_queries.add ("category:Settings"); // THEMES
      section_queries.add ("(category:System OR category:Security)"); // SYSTEM
    }
    
    /* Load xdg menu info and build a Xapian index over it.
     * Do throttled re-index if the menu changes */
    private bool build_app_menu_index ()
    {            
      if (app_menu == null)
        {
          debug ("Building initial application menu");
        
          /* We need INCLUDE_NODISPLAY to employ proper de-duping between
           * the Installed and Availabale groups. If a NoDisplay app is installed,
           * eg. Evince, it wont otherwise be in the menu index, only in the
           * S-C index - thus show up in the Available group */
          app_menu = GMenu.Tree.lookup ("unity-place-applications.menu",
                                        GMenu.TreeFlags.INCLUDE_NODISPLAY);
          
          app_menu.add_monitor ((menu) => {
            /* Reschedule the timeout if we already have one. The menu tree triggers
             * many change events during app installation. This way we wait the full
             * delay *after* the last change event has triggered */
            if (app_menu_changed_reindex_timeout != 0)
              Source.remove (app_menu_changed_reindex_timeout);
            
            app_menu_changed_reindex_timeout =
                                  Timeout.add_seconds (5, build_app_menu_index_and_result_models);
          });
        }
      
      debug ("Indexing application menu");
      appsearcher = new Unity.Package.Searcher.for_menu (app_menu);
      app_menu_changed_reindex_timeout = 0;
      
      return false;
    }

    /* Called when our app_menu structure changes - probably because something
     * has been installed or removed. We rebuild the index and update the
     * result models for global and entry. We need to update both because
     * we can't know exactly what Unity may be showing */
    private bool build_app_menu_index_and_result_models ()
    {
      build_app_menu_index ();
      
      debug ("Updating result models");
      previous_search = null;
      previous_global_search = null;
      applications.notify_property ("active-search");
      applications.notify_property ("active-global-search");
      
      return false;
    }

    private void populate_zg_templates ()
    {
      /* Create a template that activation of applications */
      zg_templates = new PtrArray.sized(1);
      var ev = new Zeitgeist.Event.full (ZG_ACCESS_EVENT, ZG_USER_ACTIVITY, "",
                             new Subject.full ("application://*",
                                               "", //NFO_SOFTWARE,
                                               "",
                                               "", "", "", ""));
      zg_templates.add ((ev as GLib.Object).ref());
    }

    private string prepare_zg_search_string (PlaceSearch? search, Section section)
    {
      
      string s;
      if (search != null)
        s = search.get_search_string ();
      else
        s = "";

      s = s.strip ();
      
      if (!s.has_suffix ("*") && s != "")
        s = s + "*";
      
      if (s != "")
        s = @"app:($s)";
      else
        return section_queries.get(section);
      
      if (section == Section.ALL_APPLICATIONS)
        return s;
      else
        return s + @" AND $(section_queries.get(section))";
    }

    private async void update_entry_search (PlaceSearch? search,
                                            Section section)
    {
      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      applications.freeze_notify ();

      var model = applications.entry_renderer_info.results_model;
      model.clear ();
      
      string pkg_search_string = prepare_pkg_search_string (search, section);
      bool has_search = !Utils.search_is_invalid (search);
      
      Timer timer = new Timer ();
      Set<string> installed_uris = new HashSet<string> ();
      Set<string> available_uris = new HashSet<string> ();
      var appresults = appsearcher.search (pkg_search_string, 0,
                                           Unity.Package.SearchType.PREFIX,
                                           has_search ?
                                              Unity.Package.Sort.BY_RELEVANCY :
                                              Unity.Package.Sort.BY_NAME);
      add_pkg_search_result (appresults, installed_uris, available_uris, model,
                             Group.INSTALLED);
      
      timer.stop ();
      debug ("Entry search listed %i Installed apps in %fms for query: %s",
             appresults.num_hits, timer.elapsed ()*1000, pkg_search_string);
      
      /* We force a flush of the shared model's revision queue here because
       * generally the search for Installed apps is orders of magnitude faster
       * than the search for Most Popular and Available apps.
       * So we can update the UI quicker */
      (model as Dee.SharedModel).flush_revision_queue ();
      
      var zg_search_string = prepare_zg_search_string (search, section);      

      try {
        timer.start ();
        var results = yield zg_index.search (zg_search_string,
                                             new Zeitgeist.TimeRange.anytime(),
                                             zg_templates,
                                             Zeitgeist.StorageState.ANY,
                                             20,
                                             Zeitgeist.ResultType.MOST_POPULAR_SUBJECTS,
                                             null);

        append_events_with_group (results, model, Group.MOST_USED, section);
        
        timer.stop ();
        debug ("Entry search found %u/%u Most Used apps in %fms for query '%s'",
               results.size (), results.estimated_matches (),
               timer.elapsed()*1000, zg_search_string);

      } catch (GLib.Error e) {
        warning ("Error performing search '%s': %s",
                 search.get_search_string (), e.message);
      }
      
      /* We force a flush of the shared model's revision queue here because
       * generally the search for Most Used apps has a small result set,
       * while the one for Available apps may be huge. The huge result set
       * would slow the arrival of the Most Used results */
      (model as Dee.SharedModel).flush_revision_queue ();
      
      /* If we don't have a search we display 6 random apps */
      if (has_search && pkgsearcher != null)
        {
          timer.start ();
          var pkgresults = pkgsearcher.search (pkg_search_string, 50,
                                               Unity.Package.SearchType.PREFIX,
                                               Unity.Package.Sort.BY_RELEVANCY);
          add_pkg_search_result (pkgresults, installed_uris, available_uris,
                                 model, Group.AVAILABLE);
          timer.stop ();
          debug ("Entry search listed %i Available apps in %fms for query: %s",
                 pkgresults.num_hits, timer.elapsed ()*1000, pkg_search_string);
        }
      else if (pkgsearcher != null)
        {
          timer.start ();
          string? category = (section == Section.ALL_APPLICATIONS ?
                                           null : section_queries.get(section));
          var random_pkgresults = pkgsearcher.get_random_apps (category, 12);
          add_pkg_search_result (random_pkgresults, installed_uris, available_uris,
                                 model, Group.AVAILABLE, 6);
          timer.stop ();
          debug ("Entry search listed %i random Available apps in %fms",
                 random_pkgresults.num_hits, timer.elapsed ()*1000);
        }
      
      if (has_search)
        check_empty_search (search, model);
      else
        check_empty_section (section, model);
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        applications.thaw_notify ();
        return false;
      });
      
      if (search != null)
        search.finished ();
    }
    
    private async void update_global_search (PlaceSearch? search)
    {
      var model = applications.global_renderer_info.results_model;
    
      model.clear ();
      
      if (Utils.search_is_invalid (search))
      {        
        return;
      }
      
      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remember to thaw the notifys again! */
      applications.freeze_notify ();      
      
      var search_string = prepare_pkg_search_string (search,
                                                     Section.ALL_APPLICATIONS);
      Set<string> installed_uris = new HashSet<string> ();
      Set<string> available_uris = new HashSet<string> ();
      var timer = new Timer ();
      var appresults = appsearcher.search (search_string, 0,
                                           Unity.Package.SearchType.PREFIX,
                                           Unity.Package.Sort.BY_RELEVANCY);      
      add_pkg_search_result (appresults, installed_uris, available_uris, model,
                             Group.INSTALLED);
      
      timer.stop ();
      debug ("Global search listed %i Installed apps in %fms for query: %s",
             appresults.num_hits, timer.elapsed ()*1000, search_string);

      // Dowloadable Apps search disabled from global search
      // See https://bugs.launchpad.net/unity-place-applications/+bug/733669
      /*
      if (pkgsearcher != null)
        {      
          // We force a flush of the shared model's revision queue here because
          // generally the search for Installed apps is orders of magnitude faster
          // than the search for Available apps. So we can update the UI quicker         

          (model as Dee.SharedModel).flush_revision_queue ();
          
          timer.start ();
          var pkgresults = pkgsearcher.search (search_string, 20,
                                               Unity.Package.SearchType.PREFIX,
                                               Unity.Package.Sort.BY_RELEVANCY);
          add_pkg_search_result (pkgresults, installed_uris, available_uris,
                                 model, Group.AVAILABLE);
          timer.stop ();
          debug ("Global search listed %i Available apps in %fms for query: %s",
                 pkgresults.num_hits, timer.elapsed ()*1000, search_string);
        }
      */
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        applications.thaw_notify ();
        return false;
      });
      
      if (search != null)
        search.finished ();
    }
    
    private string prepare_pkg_search_string (PlaceSearch? search, Section section)
    {       
      if (Utils.search_is_invalid (search))
        {
          if (section == Section.ALL_APPLICATIONS)
            return "type:Application";
          else
            return @"type:Application AND $(section_queries.get(section))";
        }
      else
        {
          var s = search.get_search_string ();

          s = s.strip ();
          
          /* The Xapian query parser seems to handle hyphens in a special way,
           * namely that it forces the joined tokens into a phrase query
           * no matter if it appears as the last word in a query and we have
           * the PARTIAL flag set on the query parser. This makes 'd-f' not
           * match 'd-feet' etc. */
          s = s.delimit ("-", ' ');

          if (section == Section.ALL_APPLICATIONS)
            return @"type:Application AND $s";
          else
            return @"type:Application AND $(section_queries.get(section)) AND $s";
        }
    }
    
    public Icon find_pkg_icon (Unity.Package.PackageInfo pkginfo)
    {
      string desktop_id = Path.get_basename (pkginfo.desktop_file);
      bool installed = AppInfoManager.get_instance().lookup (desktop_id) != null;
      
      /* If the app is already installed we should be able to pull the
       * icon from the theme */
      if (installed)
        return new ThemedIcon (pkginfo.icon);
      
      /* App is not installed - we need to find the right icon in the bowels
       * of the software center */
      if (pkginfo.icon.has_prefix ("/"))
        {
          return new FileIcon (File.new_for_path (pkginfo.icon));
        }
      else
        {
          Icon icon = file_icon_cache.lookup (pkginfo.icon);
          
          if (icon != null)
            return icon;
          
          /* If the icon name contains a . it probably already have a
           * type postfix - so test icon name directly */
          string path;
          if ("." in pkginfo.icon)
            {
              path  = @"$(Config.DATADIR)/app-install/icons/$(pkginfo.icon)";
              if (FileUtils.test (path, FileTest.EXISTS))
                {
                  icon = new FileIcon (File.new_for_path (path));
                  file_icon_cache.insert (pkginfo.icon, icon);
                  return icon;
                }
            }
          
          /* Now try appending all the image extensions we know */
          foreach (var ext in image_extensions)
          {
            path = @"$(Config.DATADIR)/app-install/icons/$(pkginfo.icon).$(ext)";
            if (FileUtils.test (path, FileTest.EXISTS))
              {
                /* Got it! Cache the icon path and return the icon */
                icon = new FileIcon (File.new_for_path (path));
                file_icon_cache.insert (pkginfo.icon, icon);
                return icon;
              }
          }
        }
      
      /* Cache the fact that we couldn't find this icon */
      var icon = new ThemedIcon ("applications-other");
      file_icon_cache.insert (pkginfo.icon, icon);
      
      return icon;
    }

    private void on_appinfo_changed (string id, AppInfo? appinfo)
    {
      debug ("Application changed: %s", id);
      //update_entry_results_model.begin ();
    }
    
    private void add_pkg_search_result (Unity.Package.SearchResult results,
                                        Set<string> installed_uris,
                                        Set<string> available_uris,
                                        Dee.Model model,
                                        Group group,
                                        uint max_add=0)
    {
      var appmanager = AppInfoManager.get_instance();
      uint n_added = 0;
    
      foreach (var pkginfo in results.results)
      {
      	if (pkginfo.desktop_file == null)
          continue;
                
        string desktop_id = Path.get_basename (pkginfo.desktop_file);
        AppInfo? app = appmanager.lookup (desktop_id);
        
        /* De-dupe by 'application://foo.desktop' URI. Also note that we need
         * to de-dupe before we chuck out NoDisplay app infos, otherwise they'd
         * show up from alternate sources */
        string uri = @"application://$(desktop_id)";
        if (uri in installed_uris || uri in available_uris)
          continue;
        
        /* Extract basic metadata and register de-dupe keys */
        string display_name;
        string comment;
        switch (group)
        {
          case Group.INSTALLED:
            installed_uris.add (uri);
            display_name = app.get_display_name ();
            comment = app.get_description ();
            break;
          case Group.AVAILABLE:
            available_uris.add (uri);
            display_name = pkginfo.application_name;
            comment = "";
            break;
          default:
            warning (@"Illegal group for package search $(group)");
            continue;
        } 
        
        /* We can only chuck out NoDisplay and OnlyShowIn app infos after
         * we have registered a de-dupe key for them - which is done in the
         * switch block above) */
        if (app != null && !app.should_show ())
          continue;
                
        if (group == Group.AVAILABLE)
          {
            /* If we have an available item, which is not a dupe, but is
             * installed anyway, we weed it out here, because it's probably
             * left out from the Installed section because of some rule in the
             * .menu file */
            if (app != null)
              continue;            
            
            /* Apps that are not installed, ie. in the Available group
             * use the 'unity-install://pkgname/Full App Name' URI scheme,
             * but only use that after we've de-duped the results.
             * But only change the URI *after* we've de-duped the results! */
            uri = @"unity-install://$(pkginfo.package_name)/$(pkginfo.application_name)";
            available_uris.add (uri);            
          }
        
        Icon icon = find_pkg_icon (pkginfo);
        
        model.append (uri, icon.to_string (),
                      group,"application/x-desktop",
                      display_name != null ? display_name : "",
                      comment != null ? comment : "");
        
        /* Stop if we added the number of items requested */
        n_added++;
        if (max_add > 0 && n_added >= max_add)
          return;
      }
    }

    /**
     * Override of the default activation handler. The apps place daemon
     * can handle activation of installable apps using the Software Center
     */
    public async uint32 activate (string uri)
    {
      string[] args;
      string exec_or_dir = null;
      if (uri.has_prefix ("unity-install://"))
        {
          unowned string pkg = uri.offset (16); // strip off "unity-install://" prefix
          debug ("Installing: %s", pkg);
          args = new string[2];
          args[0] = "software-center";
          args[1] = pkg;
        }
      else if (uri.has_prefix ("unity-runner://"))
        {
          string orig;
          orig = uri.offset (15);
          if (uri_regex != null && uri_regex.match (orig)) {
            try {
              AppInfo.launch_default_for_uri (orig, null);
            } catch (GLib.Error error) {
              warning ("Failed to launch URI %s", orig);
              return ActivationStatus.NOT_ACTIVATED;
            }
            return ActivationStatus.ACTIVATED_HIDE_DASH;
          } else {
            exec_or_dir = Utils.subst_tilde (orig);
            args = exec_or_dir.split (" ", 0);
            for (int i = 0; i < args.length; i++)
              args[i] = Utils.subst_tilde (args[i]);
          }
          this.runner.add_history (orig);
        }
      else
        {
          debug ("Declined activation of URI '%s': Expected URI scheme unity-install:// or unity-runner://", uri);
          return ActivationStatus.NOT_ACTIVATED;
        }

      if ((exec_or_dir != null) && FileUtils.test (exec_or_dir, FileTest.IS_DIR))
      {
        try {
            AppInfo.launch_default_for_uri ("file://" + exec_or_dir, null);
        } catch (GLib.Error err) {
            warning ("Failed to open current folder '%s' in file manager: %s",
                     exec_or_dir, err.message);
          return ActivationStatus.NOT_ACTIVATED;
        }
      }
      else
      {
          try {
            unowned string home_dir = GLib.Environment.get_home_dir ();
            Process.spawn_async (home_dir, args, null, SpawnFlags.SEARCH_PATH, null, null);
          } catch (SpawnError e) {
            warning ("Failed to spawn software-center or direct URI activation '%s': %s",
                     uri, e.message);
            return ActivationStatus.NOT_ACTIVATED;
          }
      }

      return ActivationStatus.ACTIVATED_HIDE_DASH;
      
    }

    /* Appends the subject URIs from a set of Zeitgeist.Events to our Dee.Model
     * assuming that these events are already sorted */
    public void append_events_with_group (Zeitgeist.ResultSet events,
                                          Dee.Model results,
                                          uint group_id,
                                          int section_filter = -1)
    {
      foreach (var ev in events)
        {
          string app_uri;
          if (ev.num_subjects () > 0)
            app_uri = ev.get_subject (0).get_uri ();
          else
            {
              warning ("Unexpected event without subject");
              continue;
            }
        
          /* Assert that we indeed have a known application as actor */
          AppInfo? app = Utils.get_app_info_for_actor (app_uri);
          
          if (app == null)
            continue;
          
          if (!app.should_show ())
            continue;          
          
          results.append (app_uri, app.get_icon().to_string(), group_id,
                          "application/x-desktop", app.get_display_name (),
                          app.get_description ());
        }
    }
    
    public void check_empty_search (PlaceSearch? search,
                                    Dee.Model results_model)
    {
      if (results_model.get_n_rows () > 0)
        return;
      
      if (Utils.search_is_invalid (search))
        return;
      
      results_model.append ("", "", Group.EMPTY_SEARCH, "",
                            _("Your search did not match any applications"),
                            "");      
      
      // FIXME: Use prefered browser
      // FIXME: URL escape search string
      results_model.append (@"http://google.com/#q=$(search.get_search_string())",
                            "", Group.EMPTY_SEARCH, "",
                             _("Search the web"), "");
    }
    
    public void check_empty_section (Section section,
                                     Dee.Model results_model)
    {
      if (results_model.get_n_rows () > 0)
        return;
            
      string msg;
      
      switch (section)
      {
        case Section.ALL_APPLICATIONS:
          msg = _("There are no applications installed on this computer");
          break;
        case Section.ACCESSORIES:
          msg = _("There are no accessories installed on this computer");
          break;
        case Section.UNIVERSAL_ACCESS:
          msg = _("There are no universal access applications installed on this computer");
          break;
        case Section.DEVELOPER_TOOLS:
          msg = _("There are no developer tools installed on this computer");
          break;
        case Section.EDUCATION:
          msg = _("There are no educational applications installed on this computer");
          break;
        case Section.SCIENCE:
          msg = _("There are no scientific or engineering applications installed on this computer");
          break;
        case Section.GAMES:
          msg = _("There are no games installed on this computer");
          break;
        case Section.GRAPHICS:
          msg = _("There are no graphics applications installed on this computer");
          break;
        case Section.INTERNET:
          msg = _("There are no internet applications installed on this computer");
          break;
        case Section.MULTIMEDIA:
          msg = _("There are no multimedia applications installed on this computer");
          break;
        case Section.OFFICE:
          msg = _("There are no office applications installed on this computer");
          break;
        case Section.THEMES:
          msg = _("There are no theming or tweaking applications installed on this computer");
          break;
        case Section.SYSTEM:
          msg = _("There are no system applications installed on this computer");
          break;
        default:
          msg = _("There are no applications installed on this computer");
          warning ("Unknown section: %u", section);
          break;
      }
      
      results_model.append ("", "", Group.EMPTY_SECTION, "", msg, "");
    }
  
  } /* END: class Daemon */

} /* namespace */
