# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Control Panel."""

from twisted.internet import defer

from ubuntuone.controlpanel.gui import (
    LOADING,
    PLEASE_WAIT,
)
from ubuntuone.controlpanel.gui.qt import filesyncstatus as gui
from ubuntuone.controlpanel.gui.qt.tests import BaseTestCase

backend = gui.backend  # pylint: disable=C0103


class FileSyncStatusTestCase(BaseTestCase):
    """Test the qt control panel."""

    innerclass_ui = gui.filesyncstatus_ui
    innerclass_name = "Ui_Form"
    class_ui = gui.FileSyncStatus

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FileSyncStatusTestCase, self).setUp()
        self.ui.load()

    def assert_status_correct(self, status_bd, status_ui, action, msg_bd=None,
                              callback=None, tooltip=None):
        """The shown status is correct.

        * The ui's label shows 'status_ui'.
        * If action is not None, the ui's button shows that 'action' as label
        and when clicking it, 'callback' gets executed.
        * If action is None, the ui's button should be hidden.
        * If a tooltip is given, then it exists with correct text.

        """
        if msg_bd is None:
            msg_bd = None
            expected_text = status_ui
        else:
            expected_text = status_ui + ' (' + msg_bd + ')'

        status = {backend.STATUS_KEY: status_bd, backend.MSG_KEY: msg_bd}
        self.ui.process_info(status)

        actual_text = unicode(self.ui.ui.sync_status_label.text())
        self.assertEqual(expected_text, actual_text)

        actual_icon = self.ui.ui.sync_status_icon.pixmap()
        pixmap_name = gui.icon_name_from_status(status_bd)
        expected_icon = gui.pixmap_from_name(pixmap_name)
        self.assertEqualPixmaps(expected_icon, actual_icon)

        self.assertTrue(self.ui.ui.sync_status_button.isEnabled())
        self.assertEqual(unicode(self.ui.ui.sync_status_button.text()),
            action)

        is_default = self.ui.ui.sync_status_button.isDefault()
        expected_default = (action == gui.FILE_SYNC_CONNECT)
        self.assertEqual(expected_default, is_default)

        self.ui.ui.sync_status_button.click()
        self.assertFalse(self.ui.ui.sync_status_button.isEnabled())
        self.assert_backend_called(callback)

        is_default = self.ui.ui.sync_status_button.isDefault()
        self.assertFalse(is_default)

        if tooltip is not None:
            self.assertEqual(self.ui.ui.sync_status_button.toolTip(), tooltip)
        else:
            self.assertEqual(self.ui.ui.sync_status_button.toolTip(), '')

    def test_backend(self):
        """Backend is created."""
        self.assertIsInstance(self.ui.backend,
                              backend.ControlBackend)

    def test_process_info_changed(self):
        """Backend's file sync status changed callback is connected."""
        self.assertEqual(self.ui.backend.status_changed_handler,
                         self.ui.process_info)

    def test_file_sync_status_is_requested_on_load(self):
        """The file sync status is requested to the backend."""
        self.assert_backend_called('file_sync_status')

    def test_process_info_disabled(self):
        """File sync status disabled update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_DISABLED,
                                   status_ui=gui.FILE_SYNC_DISABLED,
                                   action=gui.FILE_SYNC_ENABLE,
                                   callback='enable_files',
                                   tooltip=gui.FILE_SYNC_ENABLE_TOOLTIP)

    def test_process_info_disconnected(self):
        """File sync status disconnected update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_DISCONNECTED,
                                   status_ui=gui.FILE_SYNC_DISCONNECTED,
                                   action=gui.FILE_SYNC_CONNECT,
                                   callback='connect_files',
                                   tooltip=gui.FILE_SYNC_CONNECT_TOOLTIP)

    def test_process_info_error(self):
        """File sync status error update the label."""
        msg = gui.WARNING_MARKUP % gui.FILE_SYNC_ERROR
        self.assert_status_correct(status_bd=backend.FILE_SYNC_ERROR,
                                   status_ui=msg,
                                   msg_bd='what an error!',
                                   action=gui.FILE_SYNC_RESTART,
                                   callback='restart_files',
                                   tooltip=gui.FILE_SYNC_RESTART_TOOLTIP)

    def test_process_info_idle(self):
        """File sync status idle update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_IDLE,
                                   status_ui=gui.FILE_SYNC_IDLE,
                                   action=gui.FILE_SYNC_DISCONNECT,
                                   callback='disconnect_files',
                                   tooltip=gui.FILE_SYNC_DISCONNECT_TOOLTIP)

    def test_process_info_starting(self):
        """File sync status starting update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_STARTING,
                                   status_ui=gui.FILE_SYNC_STARTING,
                                   action=gui.FILE_SYNC_STOP,
                                   callback='stop_files',
                                   tooltip=gui.FILE_SYNC_STOP_TOOLTIP)

    def test_process_info_stopped(self):
        """File sync status stopped update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_STOPPED,
                                   status_ui=gui.FILE_SYNC_STOPPED,
                                   action=gui.FILE_SYNC_START,
                                   callback='start_files',
                                   tooltip=gui.FILE_SYNC_START_TOOLTIP)

    def test_process_info_syncing(self):
        """File sync status syncing update the label."""
        self.assert_status_correct(status_bd=backend.FILE_SYNC_SYNCING,
                                   status_ui=gui.FILE_SYNC_SYNCING,
                                   action=gui.FILE_SYNC_DISCONNECT,
                                   callback='disconnect_files',
                                   tooltip=gui.FILE_SYNC_DISCONNECT_TOOLTIP)

    def test_process_info_unknown(self):
        """File sync status unknown update the label."""
        msg = gui.WARNING_MARKUP % gui.FILE_SYNC_ERROR
        self.assert_status_correct(status_bd=backend.FILE_SYNC_UNKNOWN,
                                   status_ui=msg,
                                   msg_bd='yadda oops',
                                   action=gui.FILE_SYNC_RESTART,
                                   callback='restart_files',
                                   tooltip=gui.FILE_SYNC_RESTART_TOOLTIP)

    def test_on_sync_status_button_clicked(self):
        """Check the labels and icon when the button is pressed."""
        # Ensure the _backend_method is not None to execute the first
        # part of the If.
        self.ui._backend_method = lambda: 'Not None'

        # Simulate the click event
        self.ui.on_sync_status_button_clicked()

        self.assertFalse(self.ui.ui.sync_status_button.isEnabled())
        self.assertFalse(self.ui.ui.sync_status_button.isDefault())
        actual_text = unicode(self.ui.ui.sync_status_label.text())
        self.assertEqual(actual_text, LOADING)
        self.assertEqual(unicode(self.ui.ui.sync_status_button.text()),
            PLEASE_WAIT)
        actual_icon = self.ui.ui.sync_status_icon.pixmap()
        expected_icon = gui.pixmap_from_name('sync_status_loading')
        self.assertEqualPixmaps(expected_icon, actual_icon)
