# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Control Panel."""

import logging
import operator
import os

from twisted.internet import defer
from ubuntuone.devtools.handlers import MementoHandler

from ubuntuone.controlpanel.gui.tests import (
    FAKE_VOLUMES_INFO,
    FAKE_VOLUMES_MINIMAL_INFO,
    FAKE_VOLUMES_NO_FREE_SPACE_INFO,
    MUSIC_FOLDER, ROOT, USER_HOME,
)
from ubuntuone.controlpanel.gui.qt import folders as gui
from ubuntuone.controlpanel.gui.qt.tests import (
    FakedDialog,
)
from ubuntuone.controlpanel.gui.qt.tests.test_ubuntuonebin import (
    UbuntuOneBinTestCase,
)


# Access to a protected member
# Instance of 'ControlBackend' has no '_called' member
# pylint: disable=W0212, E1103


def _build_name(name):
    """Helper to build the name expected when showing folder info."""
    if name:
        name = gui.FOLDER_SHARED_BY % {'other_user_display_name': name}
    else:
        name = gui.FOLDER_OWNED_BY
    return name


class FoldersPanelTestCase(UbuntuOneBinTestCase):
    """Test the qt cloud folders tab."""

    innerclass_ui = gui.folders_ui
    innerclass_name = "Ui_Form"
    class_ui = gui.FoldersPanel

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FoldersPanelTestCase, self).setUp()
        self.ui.backend.next_result = FAKE_VOLUMES_MINIMAL_INFO

        self.memento = MementoHandler()
        self.memento.setLevel(logging.DEBUG)
        gui.logger.addHandler(self.memento)

        old_home = os.environ['HOME']
        os.environ['HOME'] = USER_HOME
        self.addCleanup(lambda: os.environ.__setitem__('HOME', old_home))


class FoldersPanelVolumesInfoTestCase(FoldersPanelTestCase):
    """Test the qt cloud folders tab."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FoldersPanelVolumesInfoTestCase, self).setUp()
        yield self.ui.load()

    def assert_folder_group_header_correct(self, item, name):
        """Check that the folder group header is correct."""
        self.assertEqual(item.text(gui.FOLDER_NAME_COL), name)
        self.assertEqual(item.text(gui.SUBSCRIPTION_COL), gui.SYNC_LOCALLY)
        self.assertEqual(item.text(gui.EXPLORE_COL), '')

    @defer.inlineCallbacks
    def test_is_processing_while_asking_info(self):
        """The ui is processing while the contents are loaded."""
        def check(with_storage_info):
            """The ui must be is_processing."""
            self.assertTrue(self.ui.is_processing, 'ui must be processing')
            return FAKE_VOLUMES_MINIMAL_INFO

        self.patch(self.ui.backend, 'volumes_info', check)

        yield self.ui.load()  # trigger the info request

    def test_is_not_processing_after_info_ready(self):
        """The ui is not processing when contents are load."""
        self.ui.process_info(FAKE_VOLUMES_INFO)

        self.assertFalse(self.ui.is_processing)

    def test_info_is_requested_on_load(self):
        """The volumes info is requested to the backend."""
        self.assert_backend_called('volumes_info', with_storage_info=False)

    def test_process_info(self):
        """The volumes info is processed when ready."""
        self.ui.process_info(FAKE_VOLUMES_INFO)
        folders = self.ui.ui.folders

        root = folders.invisibleRootItem()
        self.assertEqual(len(FAKE_VOLUMES_INFO), root.childCount())

        treeiter = gui.QtGui.QTreeWidgetItemIterator(folders)
        for name, _, volumes in FAKE_VOLUMES_INFO:
            name = _build_name(name)

            # get first child, matching "My Folders"
            item = treeiter.value()
            self.assertFalse(item.is_empty)

            self.assert_folder_group_header_correct(item, name)

            # check children
            self.assertEqual(len(volumes), item.childCount())
            sorted_vols = sorted(volumes, key=operator.itemgetter('path'))
            for volume in sorted_vols:
                treeiter += 1
                item = treeiter.value()  # get child folder
                self.assertTrue(item is not None)

                name = volume['path'].replace(USER_HOME + os.path.sep, '')
                expected_path = volume['path']
                if volume['type'] == self.ui.backend.SHARE_TYPE:
                    name = volume['name']
                    expected_path = volume['realpath']
                label = item.text(gui.FOLDER_NAME_COL)
                self.assertEqual(label, name)

                if volume['type'] == self.ui.backend.ROOT_TYPE:
                    # no check box but the ALWAYS_SUBSCRIBED legend
                    self.assertEqual(item.text(gui.SUBSCRIPTION_COL),
                                     gui.ALWAYS_SUBSCRIBED)
                else:
                    subscribed = item.checkState(gui.SUBSCRIPTION_COL) == \
                                 gui.CHECKED
                    self.assertEqual(subscribed, bool(volume['subscribed']))

                icon_name = item.icon_obj.icon_name
                if volume['type'] != self.ui.backend.SHARE_TYPE:
                    self.assertEqual(icon_name, gui.FOLDER_ICON_NAME)
                else:
                    self.assertEqual(icon_name, gui.SHARE_ICON_NAME)

                self.assertEqual(item.volume_id, volume['volume_id'])
                self.assertEqual(item.volume_path, expected_path)

                # tooltips are correct
                self.assertEqual(item.toolTip(gui.FOLDER_NAME_COL), name)
                self.assertEqual(item.toolTip(gui.EXPLORE_COL), gui.EXPLORE)

                # explore button is in place
                model_index = folders.indexFromItem(item, gui.EXPLORE_COL)
                button = folders.indexWidget(model_index)
                self.assertEqual(button.isFlat(), True)
                self.assertEqual(button.icon_obj.icon_name,
                                 gui.FOLDER_ICON_NAME)
                self.assertEqual(button.iconSize().width(), 12)
                self.assertEqual(button.iconSize().height(), 12)
                self.assertEqual(button.isEnabled(),
                                 bool(volume['subscribed']))

            treeiter += 1
            item = treeiter.value()

    def test_process_info_clears_the_list(self):
        """The old volumes info is cleared before updated."""
        self.ui.process_info(FAKE_VOLUMES_INFO)
        self.ui.process_info(FAKE_VOLUMES_INFO)

        root = self.ui.ui.folders.invisibleRootItem()
        self.assertEqual(len(FAKE_VOLUMES_INFO), root.childCount())

    def test_process_info_with_no_volumes(self):
        """When there are no volumes, a notification is shown."""
        self.ui.process_info([])

        root = self.ui.ui.folders.invisibleRootItem()
        self.assertEqual(root.childCount(), 0)

    def test_process_info_highlights_little_free_space(self):
        """The free space is red if is zero (or close to 0)."""
        self.ui.process_info(FAKE_VOLUMES_NO_FREE_SPACE_INFO)

        child_index = 0
        root = self.ui.ui.folders.invisibleRootItem()
        for name, _, _ in FAKE_VOLUMES_NO_FREE_SPACE_INFO:
            name = _build_name(name)

            item = root.child(child_index)
            self.assertFalse(item.is_empty)
            self.assert_folder_group_header_correct(item, name)

            child_index += 1
            item = root.child(child_index)

    def test_process_info_handles_no_quota_info(self):
        """The lack of free space is handled."""
        info = [
            (u'', self.ui.backend.FREE_BYTES_NOT_AVAILABLE, [ROOT]),
            (u'No free space available',
             self.ui.backend.FREE_BYTES_NOT_AVAILABLE,
             [{u'volume_id': u'0', u'name': u'full', u'display_name': u'test',
               u'path': u'full-share', u'subscribed': u'',
               u'type': self.ui.backend.SHARE_TYPE}]),
        ]
        self.ui.process_info(info)

        child_index = 0
        root = self.ui.ui.folders.invisibleRootItem()
        for name, _, _ in info:
            name = _build_name(name)

            item = root.child(child_index)
            self.assertFalse(item.is_empty)
            self.assert_folder_group_header_correct(item, name)

            child_index += 1
            item = root.child(child_index)

    def test_clicking_on_row_opens_folder(self):
        """The folder activated is opened."""
        self.patch(gui.os.path, 'exists', lambda *a: True)
        self.patch(gui, 'uri_hook', self._set_called)

        path = ROOT['path']
        item = gui.QtGui.QTreeWidgetItem()
        item.volume_path = path
        self.ui.on_folders_itemActivated(item)

        self.assertEqual(self._called,
            ((unicode(gui.QtCore.QUrl.fromLocalFile(path).toString()),), {}))
        self.assertTrue(gui.QtCore.QUrl(self._called[0][0]).isValid())

    def test_clicking_on_row_handles_path_none(self):
        """None paths are properly handled."""
        self.patch(gui, 'uri_hook', self._set_called)

        item = gui.QtGui.QTreeWidgetItem()
        item.volume_path = None
        self.ui.on_folders_itemActivated(item)

        self.assertTrue(self.memento.check_warning(str(item),
                                                   'volume_path', 'is None'))
        self.assertEqual(self._called, False)

    def test_clicking_on_row_handles_path_non_existent(self):
        """Not-existent paths are properly handled."""
        self.patch(gui.os.path, 'exists', lambda *a: False)
        self.patch(gui, 'uri_hook', self._set_called)

        path = 'not-in-disk'
        item = gui.QtGui.QTreeWidgetItem()
        item.volume_path = path
        self.ui.on_folders_itemActivated(item)

        self.assertTrue(self.memento.check_warning(path, 'does not exist'))
        self.assertEqual(self._called, False)

    def test_process_info_with_music_folder(self):
        """The volumes info is processed when ready."""
        self.ui.process_info(FAKE_VOLUMES_MINIMAL_INFO)
        folders = self.ui.ui.folders

        treeiter = gui.QtGui.QTreeWidgetItemIterator(folders)

        # walk to 'My folders' children
        treeiter += 1

        # grab next row since first one is root
        treeiter += 1
        item = treeiter.value()

        volume = MUSIC_FOLDER

        label = item.text(gui.FOLDER_NAME_COL)
        self.assertEqual(label, gui.MUSIC_DISPLAY_NAME)

        subscribed = item.checkState(gui.SUBSCRIPTION_COL) == gui.CHECKED
        self.assertEqual(subscribed, bool(volume['subscribed']))

        icon_name = item.icon_obj.icon_name
        self.assertEqual(icon_name, gui.MUSIC_ICON_NAME)

        self.assertEqual(item.volume_id, volume['volume_id'])
        self.assertEqual(item.volume_path, volume['path'])

    def test_share_publish_button(self):
        """When clicking the share/publish button, the proper url is opened."""
        self.assert_uri_hook_called(self.ui.ui.share_publish_button,
                                    gui.MANAGE_FILES_LINK)


class FoldersPanelAddFolderTestCase(FoldersPanelTestCase):
    """The test suite for the folder creation from a local dir."""

    timeout = 3

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FoldersPanelAddFolderTestCase, self).setUp()

        # simulate volumes info properly processed
        self.ui.process_info([])
        # reset backend state
        self.ui.backend._called.clear()

    def test_not_is_processing(self):
        """Before clicking the add folder button, the UI is not processing."""
        self.assertFalse(self.ui.is_processing, 'ui must not be processing')

    @defer.inlineCallbacks
    def test_reload_volumes_info_on_success(self):
        """Once the folder is created, the volumes info is reloaded."""
        d = defer.Deferred()
        self.patch(self.ui, 'load', lambda *a, **kw: d.callback((a, kw)))

        self.ui.ui.add_folder_button.folderCreated.emit(USER_HOME)

        result = yield d  # when d is fired, self.ui.load() was called
        self.assertEqual(result, ((), {}))


class FoldersPanelSubscriptionTestCase(FoldersPanelTestCase):
    """The test suite for the folder subscription."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FoldersPanelSubscriptionTestCase, self).setUp()
        self.patch(gui.os.path, 'exists', lambda path: True)
        FakedDialog.response = gui.YES

        self.ui.process_info(FAKE_VOLUMES_MINIMAL_INFO)
        # the music folder
        self.item = self.ui.ui.folders.topLevelItem(0).child(1)

    @defer.inlineCallbacks
    def test_on_folders_item_changed(self):
        """Clicking on 'subscribed' updates the folder subscription."""
        self.patch(self.ui, 'load', self._set_called)
        volume = MUSIC_FOLDER
        fid = volume['volume_id']
        subscribed = not bool(volume['subscribed'])
        check_state = gui.CHECKED if subscribed else gui.UNCHECKED

        self.ui.is_processing = True
        self.item.setCheckState(gui.SUBSCRIPTION_COL, check_state)
        self.ui.is_processing = False

        yield self.ui.on_folders_itemChanged(self.item)

        # backend was called
        self.assert_backend_called('change_volume_settings',
                                   fid, {'subscribed': subscribed})

        value = self.item.checkState(gui.SUBSCRIPTION_COL) == gui.CHECKED
        self.assertEqual(value, bool(subscribed))

        # folder list was reloaded
        self.assertEqual(self._called, ((), {}))

    @defer.inlineCallbacks
    def test_on_folders_item_changed_is_processing(self):
        """Clicking on 'subscribed' sets is_processing flag until done."""
        def check(volume_id, settings):
            """The ui must be is_processing when a change was requested."""
            self.assertTrue(self.ui.is_processing, 'ui must be processing')

        self.patch(self.ui.backend, 'change_volume_settings', check)

        yield self.ui.on_folders_itemChanged(self.item)

        # the ui is no loner processing
        self.assertFalse(self.ui.is_processing, 'ui must not be processing')

    @defer.inlineCallbacks
    def test_on_folders_item_changed_volume_id_is_none(self):
        """If item.volume_id is None, log warning and return."""
        self.patch(self.ui.backend, 'change_volume_settings', self._set_called)

        self.item.volume_id = None
        yield self.ui.on_folders_itemChanged(self.item)

        self.assertTrue(self.memento.check_warning(str(self.item),
                                                   'volume_id', 'is None'))
        self.assertEqual(self._called, False)

    @defer.inlineCallbacks
    def test_on_folders_item_changed_does_nothing_if_processing(self):
        """If UI is already processing, do nothing and return."""
        self.patch(self.ui.backend, 'change_volume_settings', self._set_called)

        self.ui.is_processing = True

        yield self.ui.on_folders_itemChanged(self.item)

        self.assertEqual(self._called, False)

    @defer.inlineCallbacks
    def test_confirm_dialog_if_path_exists(self):
        """The confirmation dialog is correct."""
        self.patch(gui.os.path, 'exists', lambda path: True)

        # make sure the item is subscribed
        self.ui.is_processing = True
        self.item.setCheckState(gui.SUBSCRIPTION_COL, gui.CHECKED)
        self.ui.is_processing = False

        yield self.ui.on_folders_itemChanged(self.item)

        volume_path = self.item.volume_path
        msg = gui.FOLDERS_CONFIRM_MERGE % {'folder_path': volume_path}
        buttons = gui.YES | gui.NO | gui.CANCEL
        self.assertEqual(FakedDialog.args,
                         (self.ui, '', msg, buttons, gui.YES))
        self.assertEqual(FakedDialog.kwargs, {})

    @defer.inlineCallbacks
    def test_confirm_dialog_if_path_does_not_exist(self):
        """The confirmation dialog is not created if not needed."""
        self.patch(gui.os.path, 'exists', lambda path: False)

        # make sure the item is unsubscribed
        self.ui.is_processing = True
        self.item.setCheckState(gui.SUBSCRIPTION_COL, gui.UNCHECKED)
        self.ui.is_processing = False

        yield self.ui.on_folders_itemChanged(self.item)

        self.assertEqual(FakedDialog.args, None)
        self.assertEqual(FakedDialog.kwargs, None)

    @defer.inlineCallbacks
    def test_subscribe_does_not_call_backend_if_dialog_closed(self):
        """Backend is not called if users closes the confirmation dialog."""
        FakedDialog.response = gui.CANCEL

        # make sure the item is subscribed
        self.ui.is_processing = True
        self.item.setCheckState(gui.SUBSCRIPTION_COL, gui.CHECKED)
        self.ui.is_processing = False

        yield self.ui.on_folders_itemChanged(self.item)

        self.assertFalse(FakedDialog.args is None, 'warning was called')
        self.assertNotIn('change_volume_settings', self.ui.backend._called)
        self.assertFalse(self.ui.is_processing)

        subscribed = self.item.checkState(gui.SUBSCRIPTION_COL) == gui.CHECKED
        self.assertEqual(subscribed, False)

    @defer.inlineCallbacks
    def test_subscribe_does_not_call_backend_if_answer_is_no(self):
        """Backend is not called if users clicks on 'No'."""
        FakedDialog.response = gui.NO

        # make sure the item is subscribed
        self.ui.is_processing = True
        self.item.setCheckState(gui.SUBSCRIPTION_COL, gui.CHECKED)
        self.ui.is_processing = False

        yield self.ui.on_folders_itemChanged(self.item)

        self.assertFalse(FakedDialog.args is None, 'warning was called')
        self.assertNotIn('change_volume_settings', self.ui.backend._called)
        self.assertFalse(self.ui.is_processing)

        subscribed = self.item.checkState(gui.SUBSCRIPTION_COL) == gui.CHECKED
        self.assertEqual(subscribed, False)

    @defer.inlineCallbacks
    def test_no_confirmation_if_unsubscribing(self):
        """The confirmation dialog is not shown if unsubscribing."""
        # make sure the item is unsubscribed
        self.ui.is_processing = True
        self.item.setCheckState(gui.SUBSCRIPTION_COL, gui.UNCHECKED)
        self.ui.is_processing = False

        # the confirm dialog was not called so far
        assert FakedDialog.args is None
        assert FakedDialog.kwargs is None

        yield self.ui.on_folders_itemChanged(self.item)

        self.assertTrue(FakedDialog.args is None, 'dialog was not run')
        self.assertTrue(FakedDialog.kwargs is None, 'dialog was hid')
