# -*- coding: utf-8 -*-

# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the devices tab."""

from twisted.internet import defer

from ubuntuone.controlpanel.gui.qt import devices as gui
from ubuntuone.controlpanel.gui.qt.tests import (
    SAMPLE_DEVICES_INFO,
)
from ubuntuone.controlpanel.gui.qt.tests.test_ubuntuonebin import (
    UbuntuOneBinTestCase,
)


class DevicesPanelTestCase(UbuntuOneBinTestCase):
    """Test the qt control panel."""

    innerclass_ui = gui.devices_ui
    innerclass_name = "Ui_Form"
    class_ui = gui.DevicesPanel

    @defer.inlineCallbacks
    def setUp(self):
        yield super(DevicesPanelTestCase, self).setUp()
        self.ui.backend.next_result = SAMPLE_DEVICES_INFO

    def test_is_processing_while_asking_info(self):
        """The ui is processing while the contents are loaded."""
        def check():
            """The ui must be is_processing."""
            self.assertTrue(self.ui.is_processing, 'ui must be processing')
            return SAMPLE_DEVICES_INFO

        self.patch(self.ui.backend, 'device_names_info', check)

        return self.ui.load()  # trigger the info request

    def test_is_not_processing_after_info_ready(self):
        """The ui is not processing when contents are load."""
        self.ui.process_info(SAMPLE_DEVICES_INFO)

        self.assertFalse(self.ui.is_processing)

    @defer.inlineCallbacks
    def test_info_is_requested_on_load(self):
        """The info is requested to the backend."""
        yield self.ui.load()
        self.assert_backend_called('device_names_info')

    def test_no_devices_at_startup(self):
        """The UI is reset at startup."""
        self.assertEqual(self.ui.ui.list_devices.count(), 0)
        self.assertEqual(self.ui.ui.local_device_box.count(), 0)

    def test_process_info(self):
        """The widget is updated with the info."""
        self.ui.process_info(SAMPLE_DEVICES_INFO)

        local, remote = SAMPLE_DEVICES_INFO[0], SAMPLE_DEVICES_INFO[1:]

        self.assertEqual(self.ui.ui.local_device_box.count(), 1)
        local_device = self.ui.ui.local_device_box.itemAt(0).widget()
        self.assertEqual(local_device.ui.device_name_label.text(),
                         local['name'])
        self.assertEqual(local_device.id, local['device_id'])

        self.assertEqual(self.ui.ui.list_devices.count(),
                         len(remote))
        for i, remote_device in enumerate(remote):
            device = self.ui.ui.list_devices.item(i)
            self.assertEqual(device.text(), remote_device['name'])

    def test_process_info_twice(self):
        """The widget is updated with the info."""
        self.test_process_info()
        self.test_process_info()

    def test_manage_devices_button(self):
        """Clicking the manage devices button opens the proper url."""
        self.assert_uri_hook_called(self.ui.ui.manage_devices_button,
                                    gui.EDIT_DEVICES_LINK)

    def test_remove_device_widget_after_removal(self):
        """When a device widget was deleted, remove it from the UI."""
        self.ui.process_info(SAMPLE_DEVICES_INFO)

        local_device = self.ui.ui.local_device_box.itemAt(0).widget()
        local_device.removed.emit()

        self.assertTrue(self.ui.ui.local_device_box.itemAt(0) is None)

    def test_local_device_removed_signal(self):
        """When the local device is removed, emit localDeviceRemoved signal."""
        self.ui.localDeviceRemoved.connect(self._set_called)
        self.ui.process_info(SAMPLE_DEVICES_INFO)

        local_device = self.ui.ui.local_device_box.itemAt(0).widget()
        local_device.removed.emit()

        self.assertEqual(self._called, ((), {}))
