# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The test suite for the control panel user interface."""

from __future__ import division

import operator

from ubuntuone.controlpanel.gui.gtk import gui
from ubuntuone.controlpanel.gui.gtk.tests import (
    FakedConfirmDialog,
    FAKE_REPLICATIONS_INFO,
)
from ubuntuone.controlpanel.gui.tests import (
    FAKE_ACCOUNT_INFO,
    FAKE_DEVICE_INFO,
    FAKE_DEVICES_INFO,
    FAKE_FOLDERS_INFO,
    FAKE_VOLUMES_INFO,
    FAKE_VOLUMES_MINIMAL_INFO,
    FAKE_VOLUMES_NO_FREE_SPACE_INFO,
    MUSIC_FOLDER, ROOT, USER_HOME,
)
from ubuntuone.controlpanel.gui.gtk.tests.test_gui_basic import (
    ControlPanelMixinTestCase,
)
from ubuntuone.controlpanel.gui.gtk.tests.test_package_manager import (
    SUCCESS, FAILURE)
from ubuntuone.controlpanel.gui import humanize


# Attribute 'yyy' defined outside __init__, access to a protected member
# pylint: disable=W0201, W0212

# Unused variable 'skip'
#pylint: disable=W0612


class DashboardTestCase(ControlPanelMixinTestCase):
    """The test suite for the dashboard panel."""

    klass = gui.DashboardPanel
    ui_filename = 'dashboard.ui'

    def assert_account_info_correct(self, info):
        """Check that the displayed account info matches 'info'."""
        self.assertEqual(self.ui.name_label.get_label(),
                         FAKE_ACCOUNT_INFO['name'])
        self.assertEqual(self.ui.type_label.get_label(),
                         FAKE_ACCOUNT_INFO['type'])
        self.assertEqual(self.ui.email_label.get_label(),
                         FAKE_ACCOUNT_INFO['email'])

    def test_is_an_ubuntuone_bin(self):
        """Inherits from UbuntuOneBin."""
        self.assertIsInstance(self.ui, gui.UbuntuOneBin)

    def test_inner_widget_is_packed(self):
        """The 'itself' vbox is packed into the widget."""
        self.assertIn(self.ui.itself, self.ui.get_children())

    def test_is_visible(self):
        """Is visible."""
        self.assertTrue(self.ui.get_visible())

    def test_account_info_is_not_visible(self):
        """Account info is not visible."""
        self.assertFalse(self.ui.account.get_visible())

    def test_backend_signals(self):
        """The proper signals are connected to the backend."""
        self.assertEqual(self.ui.backend._signals['AccountInfoReady'],
                         [self.ui.on_account_info_ready])
        self.assertEqual(self.ui.backend._signals['AccountInfoError'],
                         [self.ui.on_account_info_error])

    def test_is_processing_at_startup(self):
        """The ui is processing when info is being loaded."""
        self.assertTrue(self.ui.is_processing)

    def test_is_not_processing_on_info_ready(self):
        """The ui is not processing when info is ready."""
        self.ui.on_account_info_ready(FAKE_ACCOUNT_INFO)

        self.assertFalse(self.ui.is_processing)

    def test_on_account_info_ready(self):
        """The account info is processed when ready."""
        self.ui.on_account_info_ready(FAKE_ACCOUNT_INFO)

        self.assert_account_info_correct(FAKE_ACCOUNT_INFO)
        self.assertTrue(self.ui.account.get_visible())

    def test_is_not_processing_on_info_error(self):
        """The ui is not processing when info is ready."""
        self.ui.on_account_info_error()

        self.assertFalse(self.ui.is_processing)

    def test_on_account_info_error(self):
        """The account info couldn't be retrieved."""
        self.ui.on_account_info_error()

        self.assertFalse(self.ui.account.get_visible())
        self.assert_warning_correct(self.ui.message, self.ui.VALUE_ERROR)


class VolumesTestCase(ControlPanelMixinTestCase):
    """The test suite for the volumes panel."""

    klass = gui.VolumesPanel
    ui_filename = 'volumes.ui'

    def setUp(self):
        super(VolumesTestCase, self).setUp()
        self.ui.load()

    def test_is_an_ubuntuone_bin(self):
        """Inherits from UbuntuOneBin."""
        self.assertIsInstance(self.ui, gui.UbuntuOneBin)

    def test_inner_widget_is_packed(self):
        """The 'itself' vbox is packed into the widget."""
        self.assertIn(self.ui.itself, self.ui.get_children())

    def test_is_visible(self):
        """Is visible."""
        self.assertTrue(self.ui.get_visible())

    def test_backend_signals(self):
        """The proper signals are connected to the backend."""
        self.assertEqual(self.ui.backend._signals['VolumesInfoReady'],
                         [self.ui.on_volumes_info_ready])
        self.assertEqual(self.ui.backend._signals['VolumesInfoError'],
                         [self.ui.on_volumes_info_error])
        self.assertEqual(self.ui.backend._signals['VolumeSettingsChanged'],
                         [self.ui.on_volume_settings_changed])
        self.assertEqual(self.ui.backend._signals['VolumeSettingsChangeError'],
                         [self.ui.on_volume_settings_change_error])

    def test_volumes_info_is_requested_on_load(self):
        """The volumes info is requested to the backend."""
        # clean backend calls
        self.ui.backend._called.pop('volumes_info', None)
        self.ui.load()

        self.assert_backend_called('volumes_info')

    def test_is_processing_after_load(self):
        """The ui is processing when contents are load."""
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)
        self.ui.load()

        self.assertTrue(self.ui.is_processing)

    def test_is_not_processing_after_volumes_info_ready(self):
        """The ui is processing when contents are load."""
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)

        self.assertFalse(self.ui.is_processing)

    def test_message_after_non_empty_volumes_info_ready(self):
        """The volumes label is a LabelLoading."""
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)

        self.assertFalse(self.ui.message.active)

    def test_message_after_empty_volumes_info_ready(self):
        """When there are no volumes, a notification is shown."""
        self.ui.on_volumes_info_ready([])

        self.assertFalse(self.ui.message.active)
        self.assertEqual(self.ui.message.get_text(), gui.NO_FOLDERS)

    def test_on_volumes_info_ready(self):
        """The volumes info is processed when ready."""
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)

        self.assertEqual(len(FAKE_VOLUMES_INFO) + 1,  # count the empty row
                         len(self.ui.volumes_store))
        treeiter = self.ui.volumes_store.get_iter_root()
        for name, free_bytes, volumes in FAKE_VOLUMES_INFO:
            name = "%s's" % name if name else gui.MY_FOLDERS
            free_bytes = humanize(int(free_bytes))
            header = (name, self.ui.FREE_SPACE % {'free_space': free_bytes})

            # check parent row
            row = self.ui.volumes_store.get(treeiter,
                                            *xrange(self.ui.MAX_COLS))

            self.assertEqual(row[0], self.ui.ROW_HEADER % header)
            self.assertTrue(row[1], 'parent will always be subscribed')
            self.assertEqual(row[2], gui.CONTACT_ICON_NAME)
            self.assertFalse(row[3], 'no toggle should be shown on parent!')
            self.assertFalse(row[4], 'toggle should be non sensitive.')
            self.assertEqual(row[5], gui.gtk.ICON_SIZE_LARGE_TOOLBAR)
            self.assertEqual(row[6], None)
            self.assertEqual(row[7], None)

            # check children
            self.assertEqual(len(volumes),
                             self.ui.volumes_store.iter_n_children(treeiter))
            childiter = self.ui.volumes_store.iter_children(treeiter)

            sorted_vols = sorted(volumes, key=operator.itemgetter('path'))
            for volume in sorted_vols:
                row = self.ui.volumes_store.get(childiter,
                                                *xrange(self.ui.MAX_COLS))

                sensitive = True
                path = volume['path'].replace(USER_HOME + '/', '')
                if volume['type'] == 'ROOT':
                    sensitive = False
                    path = self.ui.ROOT % (path, gui.ORANGE,
                                           gui.ALWAYS_SUBSCRIBED)
                elif volume['type'] == 'SHARE':
                    path = volume['name']

                self.assertEqual(row[0], path)
                self.assertEqual(row[1], bool(volume['subscribed']))
                if volume['type'] != 'SHARE':
                    self.assertEqual(row[2], gui.FOLDER_ICON_NAME)
                else:
                    self.assertEqual(row[2], gui.SHARE_ICON_NAME)
                self.assertTrue(row[3], 'toggle should be shown on child!')
                self.assertEqual(row[4], sensitive)
                self.assertEqual(row[5], gui.gtk.ICON_SIZE_MENU)
                self.assertEqual(row[6], volume['volume_id'])
                self.assertEqual(row[7], volume['path'])

                childiter = self.ui.volumes_store.iter_next(childiter)

            treeiter = self.ui.volumes_store.iter_next(treeiter)

            if treeiter is not None:
                # skip the empty row
                row = self.ui.volumes_store.get(treeiter,
                                                *xrange(self.ui.MAX_COLS))
                self.assertEqual(row, self.ui._empty_row)

                # grab next non-empty row
                treeiter = self.ui.volumes_store.iter_next(treeiter)

    def test_on_volumes_info_ready_clears_the_list(self):
        """The old volumes info is cleared before updated."""
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)

        self.assertEqual(len(FAKE_VOLUMES_INFO) + 1,
                         len(self.ui.volumes_store))

    def test_on_volumes_info_ready_with_no_volumes(self):
        """When there are no volumes, a notification is shown."""
        self.ui.on_volumes_info_ready([])

        self.assertEqual(len(self.ui.volumes_store), 0)

    def test_on_volumes_info_ready_highlights_little_free_space(self):
        """The free space is red if is zero (or close to 0)."""
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_NO_FREE_SPACE_INFO)

        treeiter = self.ui.volumes_store.get_iter_root()
        for name, free_bytes, volumes in FAKE_VOLUMES_NO_FREE_SPACE_INFO:
            name = "%s's" % name if name else gui.MY_FOLDERS
            free_bytes = humanize(int(free_bytes))
            free_bytes = self.ui.NO_FREE_SPACE % {'free_space': free_bytes}

            # check parent row
            row = self.ui.volumes_store.get(treeiter,
                                            *xrange(self.ui.MAX_COLS))

            self.assertEqual(row[0], self.ui.ROW_HEADER % (name, free_bytes))

            treeiter = self.ui.volumes_store.iter_next(treeiter)

            if treeiter is not None:
                # skip the empty row
                row = self.ui.volumes_store.get(treeiter,
                                                *xrange(self.ui.MAX_COLS))
                self.assertEqual(row, self.ui._empty_row)

                # grab next non-empty row
                treeiter = self.ui.volumes_store.iter_next(treeiter)

    def test_on_volumes_info_ready_handles_no_quota_info(self):
        """The lack of free space is handled."""
        info = [
            (u'', gui.backend.ControlBackend.FREE_BYTES_NOT_AVAILABLE, [ROOT]),
            (u'No free space available',
             gui.backend.ControlBackend.FREE_BYTES_NOT_AVAILABLE,
             [{u'volume_id': u'0', u'name': u'full', u'display_name': u'test',
               u'path': u'full-share', u'subscribed': u'',
               u'type': gui.backend.ControlBackend.SHARE_TYPE}]),
        ]
        self.ui.on_volumes_info_ready(info)

        treeiter = self.ui.volumes_store.get_iter_root()
        for name, free_bytes, volumes in info:
            name = "%s's" % name if name else gui.MY_FOLDERS

            # check parent row
            row = self.ui.volumes_store.get(treeiter,
                                            *xrange(self.ui.MAX_COLS))

            self.assertEqual(row[0], self.ui.ROW_HEADER % (name, ''))

            treeiter = self.ui.volumes_store.iter_next(treeiter)

            if treeiter is not None:
                # skip the empty row
                row = self.ui.volumes_store.get(treeiter,
                                                *xrange(self.ui.MAX_COLS))
                self.assertEqual(row, self.ui._empty_row)

                # grab next non-empty row
                treeiter = self.ui.volumes_store.iter_next(treeiter)

    def test_on_volumes_info_error(self):
        """The volumes info couldn't be retrieved."""
        self.ui.on_volumes_info_error()
        self.assert_warning_correct(warning=self.ui.message,
                                    text=gui.VALUE_ERROR)
        self.assertFalse(self.ui.message.active)

    def test_on_volumes_info_error_after_success(self):
        """The volumes info couldn't be retrieved after a prior success."""
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)

        self.ui.on_volumes_info_error()

        self.test_on_volumes_info_error()
        self.test_on_volumes_info_ready_with_no_volumes()

    def test_clicking_on_row_opens_folder(self):
        """The folder activated is opened."""
        self.patch(gui.os.path, 'exists', lambda *a: True)
        self.patch(gui, 'uri_hook', self._set_called)
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)

        self.ui.volumes_view.row_activated('0:0',
                                           self.ui.volumes_view.get_column(0))

        self.assertEqual(self._called,
                         ((None, gui.FILE_URI_PREFIX + ROOT['path']), {}))

    def test_clicking_on_row_handles_path_none(self):
        """None paths are properly handled."""
        self.patch(gui, 'uri_hook', self._set_called)
        self.patch(self.ui.volumes_store, 'get_value', lambda *a: None)
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)

        self.ui.volumes_view.row_activated('0:0',
                                           self.ui.volumes_view.get_column(0))

        self.assertTrue(self.memento.check_warning('tree_path (0, 0)',
                                                   'volume_path', 'is None'))
        self.assertEqual(self._called, False)

    def test_clicking_on_row_handles_path_non_existent(self):
        """Not-existent paths are properly handled."""
        self.patch(gui.os.path, 'exists', lambda *a: False)
        self.patch(gui, 'uri_hook', self._set_called)
        path = 'not-in-disk'
        self.patch(self.ui.volumes_store, 'get_value', lambda *a: path)
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)

        self.ui.volumes_view.row_activated('0:0',
                                           self.ui.volumes_view.get_column(0))

        self.assertTrue(self.memento.check_warning(path, 'does not exist'))
        self.assertEqual(self._called, False)

    def test_on_volumes_info_ready_with_music_folder(self):
        """The volumes info is processed when ready."""
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_MINIMAL_INFO)

        treeiter = self.ui.volumes_store.get_iter_root()
        row = self.ui.volumes_store.get(treeiter, *xrange(self.ui.MAX_COLS))

        # walk 'Mine' folders children
        treeiter = self.ui.volumes_store.iter_children(treeiter)

        # grab next row since first one is root
        treeiter = self.ui.volumes_store.iter_next(treeiter)
        row = self.ui.volumes_store.get(treeiter, *xrange(self.ui.MAX_COLS))

        volume = MUSIC_FOLDER
        self.assertEqual(row[0], gui.MUSIC_DISPLAY_NAME)
        self.assertEqual(row[1], bool(volume['subscribed']))
        self.assertEqual(row[2], gui.MUSIC_ICON_NAME)
        self.assertTrue(row[3], 'toggle should be shown on child!')
        self.assertTrue(row[4], 'toggle should be sensitive')
        self.assertEqual(row[5], gui.gtk.ICON_SIZE_MENU)
        self.assertEqual(row[6], volume['volume_id'])
        self.assertEqual(row[7], volume['path'])


class VolumesSubscriptionTestCase(VolumesTestCase):
    """The test suite for the volumes panel."""

    kwargs = {'main_window': object()}
    tree_path = '0:3'  # this is the /home/tester/foo folder, not subscribed

    def setUp(self):
        super(VolumesSubscriptionTestCase, self).setUp()
        self.patch(gui.os.path, 'exists', lambda path: True)
        self.ui.confirm_dialog.response_code = gui.gtk.RESPONSE_YES
        self.ui.on_volumes_info_ready(FAKE_VOLUMES_INFO)

    def test_on_subscribed_toggled(self):
        """Clicking on 'subscribed' updates the folder subscription."""
        real_rows = len(FAKE_VOLUMES_INFO)
        data = zip(range(real_rows)[::2], FAKE_VOLUMES_INFO)  # skip emtpy rows
        for parent, (_, _, volumes) in data:

            sorted_vols = sorted(volumes, key=operator.itemgetter('path'))
            for child, volume in enumerate(sorted_vols):
                if volume['type'] == 'ROOT':
                    continue  # not editable

                path = '%s:%s' % (parent, child)
                self.ui.on_subscribed_toggled(widget=None, path=path)

                fid = volume['volume_id']
                subscribed = gui.bool_str(not bool(volume['subscribed']))
                # backend was called
                self.assert_backend_called('change_volume_settings',
                                           fid, {'subscribed': subscribed})
                # store was updated
                it = self.ui.volumes_store.get_iter(path)
                value = self.ui.volumes_store.get_value(it, 1)
                self.assertEqual(value, bool(subscribed))

                # the ui is processing
                self.assertTrue(self.ui.is_processing, 'ui must be processing')

                # simulate success for setting change
                self.ui.on_volume_settings_changed(volume_id=fid)

    def test_on_volume_setting_changed(self):
        """The setting for a volume was successfully changed."""
        self.ui.on_subscribed_toggled(None, self.tree_path)

        self.ui.on_volume_settings_changed(volume_id=None)  # id not used

        # the ui is no longer processing
        self.assertFalse(self.ui.is_processing, 'ui must not be processing')

    def test_on_volume_setting_change_error(self):
        """The setting for a volume was not successfully changed."""
        self.ui.on_subscribed_toggled(None, self.tree_path)

        self.patch(self.ui, 'load', self._set_called)
        self.ui.on_volume_settings_change_error(volume_id=None,
                                                error_dict=None)  # id not used
        # reload folders list to sanitize the info in volumes_store
        self.assertTrue(self._called, ((), {}))

    def test_confirm_dialog(self):
        """The confirmation dialog is correct."""
        dialog = self.ui.confirm_dialog

        self.assertEqual(dialog._args, ())
        flags = gui.gtk.DIALOG_MODAL | gui.gtk.DIALOG_DESTROY_WITH_PARENT
        kwargs = dict(parent=self.kwargs['main_window'],
                      flags=flags, type=gui.gtk.MESSAGE_WARNING,
                      buttons=gui.gtk.BUTTONS_YES_NO)
        self.assertEqual(dialog._kwargs, kwargs)

    def test_subscribe_shows_confirmation_dialog(self):
        """Clicking on subscribe displays a confirmation dialog."""
        self.ui.on_subscribed_toggled(None, self.tree_path)

        path = FAKE_FOLDERS_INFO[-1]['path']
        self.assertTrue(self.ui.confirm_dialog.was_run, 'dialog was run')
        self.assertEqual(self.ui.confirm_dialog.markup,
                         gui.FOLDERS_CONFIRM_MERGE % {'folder_path': path})
        self.assertFalse(self.ui.confirm_dialog.is_visible, 'dialog was hid')

    def test_subscribe_does_not_call_backend_if_dialog_closed(self):
        """Backend is not called if users closes the confirmation dialog."""
        self.ui.confirm_dialog.response_code = gui.gtk.RESPONSE_DELETE_EVENT
        self.ui.on_subscribed_toggled(None, self.tree_path)

        self.assertNotIn('change_volume_settings', self.ui.backend._called)
        self.assertFalse(self.ui.is_processing)

    def test_subscribe_does_not_call_backend_if_answer_is_no(self):
        """Backend is not called if users clicks on 'No'."""
        self.ui.confirm_dialog.response_code = gui.gtk.RESPONSE_NO
        self.ui.on_subscribed_toggled(None, self.tree_path)

        self.assertNotIn('change_volume_settings', self.ui.backend._called)
        self.assertFalse(self.ui.is_processing)

    def test_no_confirmation_if_no_local_folder(self):
        """The confirmation dialog is not shown if local folder not present."""
        self.patch(gui.os.path, 'exists', lambda path: False)
        self.ui.on_subscribed_toggled(None, self.tree_path)

        self.assertFalse(self.ui.confirm_dialog.was_run, 'dialog was not run')
        self.assertFalse(self.ui.confirm_dialog.is_visible, 'dialog was hid')

    def test_no_confirmation_if_unsubscribing(self):
        """The confirmation dialog is not shown if unsubscribing."""
        self.ui.on_subscribed_toggled(None, self.tree_path)

        treeiter = self.ui.volumes_store.get_iter(self.tree_path)
        assert self.ui.volumes_store.get_value(treeiter, 1)

        # reset flags
        self.ui.confirm_dialog.was_run = False
        self.ui.confirm_dialog.is_visible = False

        self.ui.on_subscribed_toggled(None, self.tree_path)

        self.assertFalse(self.ui.confirm_dialog.was_run, 'dialog was not run')
        self.assertFalse(self.ui.confirm_dialog.is_visible, 'dialog was hid')


class DeviceTestCase(ControlPanelMixinTestCase):
    """The test suite for the device widget."""

    klass = gui.Device
    ui_filename = 'device.ui'

    def assert_device_equal(self, device, expected):
        """Assert that the device has the values from expected."""
        self.assertEqual(device.id,
                         expected['device_id'])
        value = expected['device_name'].replace(gui.DEVICE_REMOVABLE_PREFIX,
                                                '')
        self.assertEqual(device.device_name.get_text(), value)
        self.assertEqual(device.device_type.get_icon_name()[0],
                         expected['device_type'].lower())
        self.assertEqual(device.is_local,
                         bool(expected['is_local']))
        self.assertEqual(device.configurable,
                         bool(expected['configurable']))
        self.assertEqual(device.show_all_notifications.get_active(),
                         bool(expected['show_all_notifications']))
        self.assertEqual(device.limit_bandwidth.get_active(),
                         bool(expected['limit_bandwidth']))

        config_enabled = self.ui.config_settings.get_sensitive()
        self.assertEqual(device.configurable, config_enabled)

        limit_enabled = self.ui.throttling_limits.get_sensitive()
        self.assertEqual(device.limit_bandwidth.get_active(), limit_enabled)

        value = int(expected['max_upload_speed']) // gui.KILOBYTES
        self.assertEqual(device.max_upload_speed.get_value_as_int(), value)
        value = int(expected['max_download_speed']) // gui.KILOBYTES
        self.assertEqual(device.max_download_speed.get_value_as_int(), value)

    def assert_device_settings_changed(self):
        """Changing throttling settings updates the backend properly."""
        expected = self.ui.__dict__
        self.assert_backend_called('change_device_settings',
                                   self.ui.id, expected)
        self.assertEqual(self.ui.warning_label.get_text(), '')

        limit_enabled = self.ui.throttling_limits.get_sensitive()
        self.assertEqual(self.ui.limit_bandwidth.get_active(), limit_enabled)

    def modify_settings(self):
        """Modify settings so values actually change."""
        new_val = not self.ui.show_all_notifications.get_active()
        self.ui.show_all_notifications.set_active(new_val)

        new_val = not self.ui.limit_bandwidth.get_active()
        self.ui.limit_bandwidth.set_active(new_val)

        new_val = self.ui.max_upload_speed.get_value_as_int() + 1
        self.ui.max_upload_speed.set_value(new_val)

        new_val = self.ui.max_download_speed.get_value_as_int() + 1
        self.ui.max_download_speed.set_value(new_val)

    def test_is_a_container(self):
        """Inherits from a container class."""
        self.assertIsInstance(self.ui, gui.gtk.Bin)

    def test_inner_widget_is_packed(self):
        """The 'itself' vbox is packed into the widget."""
        self.assertIn(self.ui.itself, self.ui.get_children())

    def test_is_visible(self):
        """Is visible."""
        self.assertTrue(self.ui.get_visible())

    def test_is_sensitive(self):
        """Is sensitive."""
        self.assertTrue(self.ui.get_sensitive())

    def test_warning_label_is_cleared(self):
        """The warning label is cleared."""
        self.assertEqual(self.ui.warning_label.get_text(), '')

    def test_default_values(self):
        """Default values are correct."""
        self.assertEqual(self.ui.id, None)
        self.assertEqual(self.ui.device_name.get_text(), '')
        self.assertEqual(self.ui.device_type.get_icon_name()[0],
                         gui.DEVICE_TYPE_COMPUTER.lower())
        self.assertEqual(self.ui.is_local, False)
        self.assertEqual(self.ui.configurable, False)
        self.assertEqual(self.ui.show_all_notifications.get_active(), True)
        self.assertEqual(self.ui.limit_bandwidth.get_active(), False)
        self.assertEqual(self.ui.max_upload_speed.get_value_as_int(), 0)
        self.assertEqual(self.ui.max_download_speed.get_value_as_int(), 0)

    def test_init_does_not_call_backend(self):
        """When updating, the backend is not called."""
        self.assertEqual(self.ui.backend._called, {})

    def test_update_device_name(self):
        """A device can be updated from a dict."""
        value = 'The death star'
        self.ui.update(device_name=gui.DEVICE_REMOVABLE_PREFIX + value)
        self.assertEqual(value, self.ui.device_name.get_text())

    def test_update_unicode_device_name(self):
        """A device can be updated from a dict."""
        value = u'Ñoño Ñandú'
        self.ui.update(device_name=gui.DEVICE_REMOVABLE_PREFIX + value)
        self.assertEqual(value, self.ui.device_name.get_text())

    def test_update_device_type_computer(self):
        """A device can be updated from a dict."""
        dtype = gui.DEVICE_TYPE_COMPUTER
        self.ui.update(device_type=dtype)
        self.assertEqual((dtype.lower(), gui.gtk.ICON_SIZE_LARGE_TOOLBAR),
                         self.ui.device_type.get_icon_name())

    def test_update_device_type_phone(self):
        """A device can be updated from a dict."""
        dtype = gui.DEVICE_TYPE_PHONE
        self.ui.update(device_type=dtype)
        self.assertEqual((dtype.lower(), gui.gtk.ICON_SIZE_LARGE_TOOLBAR),
                         self.ui.device_type.get_icon_name())

    def test_update_is_not_local(self):
        """A device can be updated from a dict."""
        self.ui.update(is_local='')
        self.assertFalse(self.ui.is_local)

    def test_update_is_local(self):
        """A device can be updated from a dict."""
        self.ui.update(is_local='True')
        self.assertTrue(self.ui.is_local)

    def test_update_non_configurable(self):
        """A device can be updated from a dict."""
        self.ui.update(configurable='')
        self.assertFalse(self.ui.configurable)
        self.assertFalse(self.ui.config_settings.get_visible())

    def test_update_configurable(self):
        """A device can be updated from a dict."""
        self.ui.update(configurable='True')
        self.assertTrue(self.ui.configurable)
        self.assertTrue(self.ui.config_settings.get_visible())

    def test_update_show_all_notifications(self):
        """A device can be updated from a dict."""
        self.ui.update(show_all_notifications='')
        self.assertFalse(self.ui.show_all_notifications.get_active())

        self.ui.update(show_all_notifications='True')
        self.assertTrue(self.ui.show_all_notifications.get_active())

    def test_update_limit_bandwidth(self):
        """A device can be updated from a dict."""
        self.ui.update(limit_bandwidth='')
        self.assertFalse(self.ui.limit_bandwidth.get_active())
        self.assertFalse(self.ui.throttling_limits.get_sensitive())

        self.ui.update(limit_bandwidth='True')
        self.assertTrue(self.ui.limit_bandwidth.get_active())
        self.assertTrue(self.ui.throttling_limits.get_sensitive())

    def test_update_upload_speed(self):
        """A device can be updated from a dict."""
        value = '12345'
        self.ui.update(max_upload_speed=value)
        self.assertEqual(int(value) // gui.KILOBYTES,
                         self.ui.max_upload_speed.get_value_as_int())

    def test_update_download_speed(self):
        """A device can be updated from a dict."""
        value = '987654'
        self.ui.update(max_download_speed=value)
        self.assertEqual(int(value) // gui.KILOBYTES,
                         self.ui.max_download_speed.get_value_as_int())

    def test_update_does_not_call_backend(self):
        """When updating, the backend is not called."""
        self.ui.update(**FAKE_DEVICE_INFO)
        self.assertEqual(self.ui.backend._called, {})
        self.assert_device_equal(self.ui, FAKE_DEVICE_INFO)

    def test_on_show_all_notifications_toggled(self):
        """When toggling show_all_notifications, backend is updated."""
        value = not self.ui.show_all_notifications.get_active()
        self.ui.show_all_notifications.set_active(value)
        self.assert_device_settings_changed()

    def test_on_limit_bandwidth_toggled(self):
        """When toggling limit_bandwidth, backend is updated."""
        value = not self.ui.limit_bandwidth.get_active()
        self.ui.limit_bandwidth.set_active(value)
        self.assert_device_settings_changed()

    def test_on_max_upload_speed_value_changed(self):
        """When setting max_upload_speed, backend is updated."""
        self.ui.max_upload_speed.set_value(25)
        self.assert_device_settings_changed()

    def test_on_max_download_speed_value_changed(self):
        """When setting max_download_speed, backend is updated."""
        self.ui.max_download_speed.set_value(52)
        self.assert_device_settings_changed()

    def test_backend_signals(self):
        """The proper signals are connected to the backend."""
        self.assertEqual(self.ui.backend._signals['DeviceSettingsChanged'],
                         [self.ui.on_device_settings_changed])
        self.assertEqual(self.ui.backend._signals['DeviceSettingsChangeError'],
                         [self.ui.on_device_settings_change_error])
        self.assertEqual(self.ui.backend._signals['DeviceRemoved'],
                         [self.ui.on_device_removed])
        self.assertEqual(self.ui.backend._signals['DeviceRemovalError'],
                         [self.ui.on_device_removal_error])

    def test_on_device_settings_changed(self):
        """When settings were changed for this device, enable it."""
        self.modify_settings()
        self.ui.on_device_settings_changed(device_id=self.ui.id)

        self.assertTrue(self.ui.get_sensitive())
        self.assertEqual(self.ui.warning_label.get_text(), '')
        self.assertEqual(self.ui.__dict__, self.ui._last_settings)

    def test_on_device_settings_change_after_error(self):
        """Change success after error."""
        self.modify_settings()
        self.ui.on_device_settings_change_error(device_id=self.ui.id)  # error

        self.test_on_device_settings_changed()

    def test_on_device_settings_changed_different_id(self):
        """When settings were changed for other device, nothing changes."""
        self.modify_settings()
        self.ui.on_device_settings_changed(device_id='yadda')

        self.assertEqual(self.ui.warning_label.get_text(), '')

    def test_on_device_settings_change_error(self):
        """When settings were not changed for this device, notify the user.

        Also, confirm that old values were restored.

        """
        self.ui.update(**FAKE_DEVICE_INFO)  # use known values

        self.modify_settings()

        self.ui.on_device_settings_change_error(device_id=self.ui.id)  # error

        self.assertTrue(self.ui.get_sensitive())
        self.assert_warning_correct(self.ui.warning_label,
                                    gui.DEVICE_CHANGE_ERROR)
        self.assert_device_equal(self.ui, FAKE_DEVICE_INFO)  # restored info

    def test_on_device_settings_change_error_after_success(self):
        """Change error after success."""
        self.modify_settings()
        self.ui.on_device_settings_changed(device_id=self.ui.id)

        self.test_on_device_settings_change_error()

    def test_on_device_settings_change_error_different_id(self):
        """When settings were not changed for other device, do nothing."""
        self.modify_settings()
        self.ui.on_device_settings_change_error(device_id='yudo')
        self.assertEqual(self.ui.warning_label.get_text(), '')

    def test_remove(self):
        """Clicking on remove calls the backend properly."""
        self.ui.is_local = False
        self.ui.remove.clicked()

        self.assert_backend_called('remove_device', self.ui.id)
        self.assertFalse(self.ui.get_sensitive(),
                         'Must be disabled while removing.')

    def test_on_device_removed(self):
        """On this device removed, hide and destroy."""
        self.ui.remove.clicked()
        self.ui.on_device_removed(device_id=self.ui.id)

        self.assertFalse(self.ui.get_visible(),
                         'Must not be visible after removed.')

    def test_on_device_removed_other_id(self):
        """On other device removed, do nothing."""
        self.ui.remove.clicked()
        self.ui.on_device_removed(device_id='foo')

        self.assertTrue(self.ui.get_visible(),
                        'Must be visible after other device was removed.')

    def test_on_device_removal_error(self):
        """On this device removal error, re-enable and show error."""
        self.ui.remove.clicked()
        self.ui.on_device_removal_error(device_id=self.ui.id)

        self.assertTrue(self.ui.get_sensitive(),
                        'Must be enabled after removal error.')
        self.assert_warning_correct(self.ui.warning_label,
                                    gui.DEVICE_REMOVAL_ERROR)

    def test_on_device_removal_error_other_id(self):
        """On other device removal error, do nothing."""
        self.ui.remove.clicked()
        self.ui.on_device_removal_error(device_id='foo')

        self.assertFalse(self.ui.get_sensitive(),
                         'Must be disabled after other device removal error.')


class RemoveDeviceTestCase(DeviceTestCase):
    """The test suite for the device widget when prompting for removal."""

    confirm_dialog = FakedConfirmDialog()
    kwargs = {'confirm_remove_dialog': confirm_dialog}

    def test_remove(self):
        """Clicking on remove calls the backend properly."""
        self.confirm_dialog.response_code = gui.gtk.RESPONSE_YES
        super(RemoveDeviceTestCase, self).test_remove()

    def test_on_device_removal_error_other_id(self):
        """On other device removal error, do nothing."""
        self.confirm_dialog.response_code = gui.gtk.RESPONSE_YES
        parent_test = super(RemoveDeviceTestCase, self)
        parent_test.test_on_device_removal_error_other_id()

    def test_remove_shows_confirmation_dialog(self):
        """Clicking on remove displays a confirmation dialog."""
        self.ui.remove.clicked()

        self.assertTrue(self.confirm_dialog.was_run, 'dialog was run')
        self.assertFalse(self.confirm_dialog.is_visible, 'dialog was hid')

    def test_remove_does_not_call_backend_if_dialog_closed(self):
        """Backend is not called if users closes the confirmation dialog."""
        self.confirm_dialog.response_code = gui.gtk.RESPONSE_DELETE_EVENT
        self.ui.remove.clicked()

        self.assertNotIn('remove_device', self.ui.backend._called)
        self.assertTrue(self.ui.is_sensitive())

    def test_remove_does_not_call_backend_if_answer_is_no(self):
        """Backend is not called if users clicks on 'No'."""
        self.confirm_dialog.response_code = gui.gtk.RESPONSE_NO
        self.ui.remove.clicked()

        self.assertNotIn('remove_device', self.ui.backend._called)
        self.assertTrue(self.ui.is_sensitive())


class DevicesTestCase(ControlPanelMixinTestCase):
    """The test suite for the devices panel."""

    klass = gui.DevicesPanel
    ui_filename = 'devices.ui'
    kwargs = {'main_window': object()}

    def setUp(self):
        super(DevicesTestCase, self).setUp()
        self.ui.load()

    def test_is_an_ubuntuone_bin(self):
        """Inherits from UbuntuOneBin."""
        self.assertIsInstance(self.ui, gui.UbuntuOneBin)

    def test_inner_widget_is_packed(self):
        """The 'itself' vbox is packed into the widget."""
        self.assertIn(self.ui.itself, self.ui.get_children())

    def test_is_visible(self):
        """Is visible."""
        self.assertTrue(self.ui.get_visible())

    def test_confirm_remove_dialog(self):
        """The confirmation dialog is correct."""
        dialog = self.ui.confirm_remove_dialog

        self.assertEqual(dialog._args, ())
        flags = gui.gtk.DIALOG_MODAL | gui.gtk.DIALOG_DESTROY_WITH_PARENT
        kwargs = dict(parent=self.kwargs['main_window'],
                      flags=flags, type=gui.gtk.MESSAGE_WARNING,
                      buttons=gui.gtk.BUTTONS_YES_NO,
                      message_format=gui.DEVICE_CONFIRM_REMOVE)
        self.assertEqual(dialog._kwargs, kwargs)

    def test_backend_signals(self):
        """The proper signals are connected to the backend."""
        self.assertEqual(self.ui.backend._signals['DevicesInfoReady'],
                         [self.ui.on_devices_info_ready])
        self.assertEqual(self.ui.backend._signals['DevicesInfoError'],
                         [self.ui.on_devices_info_error])
        self.assertEqual(self.ui.backend._signals['DeviceRemoved'],
                         [self.ui.on_device_removed])

    def test_devices_info_is_requested_on_load(self):
        """The devices info is requested to the backend."""
        # clean backend calls
        self.ui.backend._called.pop('devices_info', None)
        self.ui.load()

        self.assert_backend_called('devices_info')

    def test_is_processing_after_load(self):
        """The ui is processing when contents are load."""
        self.ui.on_devices_info_ready(FAKE_DEVICES_INFO)
        self.ui.load()

        self.assertTrue(self.ui.is_processing)

    def test_is_not_processing_after_non_empty_devices_info_ready(self):
        """The ui is no longer processing after a non empty device list."""
        self.ui.on_devices_info_ready(FAKE_DEVICES_INFO)

        self.assertFalse(self.ui.is_processing)

    def test_is_not_processing_after_empty_devices_info_ready(self):
        """The ui is no longer processing after a empty device list."""
        self.ui.on_devices_info_ready([])

        self.assertFalse(self.ui.is_processing)

    def test_show_message_after_empty_devices_info_ready(self):
        """When there are no devices, a notification is shown."""
        self.ui.on_devices_info_ready([])

        self.assertEqual(self.ui.message.get_text(), gui.NO_DEVICES)

    def test_on_devices_info_ready(self):
        """The devices info is processed when ready."""
        self.ui.on_devices_info_ready(FAKE_DEVICES_INFO)

        children = self.ui.devices.get_children()
        self.assertEqual(len(children), len(FAKE_DEVICES_INFO))

        for child, device in zip(children, FAKE_DEVICES_INFO):
            self.assertIsInstance(child, gui.Device)

            self.assertEqual(device['device_id'], child.id)
            value = device['device_name'].replace(gui.DEVICE_REMOVABLE_PREFIX,
                                                  '')
            self.assertEqual(value, child.device_name.get_text())
            self.assertEqual(device['device_type'].lower(),
                             child.device_type.get_icon_name()[0])
            self.assertEqual(bool(device['is_local']),
                             child.is_local)
            self.assertEqual(bool(device['configurable']),
                             child.configurable)

            if bool(device['configurable']):
                self.assertEqual(bool(device['show_all_notifications']),
                                 child.show_all_notifications.get_active())
                self.assertEqual(bool(device['limit_bandwidth']),
                                 child.limit_bandwidth.get_active())
                value = int(device['max_upload_speed']) // gui.KILOBYTES
                self.assertEqual(value,
                                 child.max_upload_speed.get_value_as_int())
                value = int(device['max_download_speed']) // gui.KILOBYTES
                self.assertEqual(value,
                                 child.max_download_speed.get_value_as_int())

            self.assertIs(child.confirm_dialog, self.ui.confirm_remove_dialog)

    def test_on_devices_info_ready_have_devices_cached(self):
        """The devices are cached for further removal."""
        self.ui.on_devices_info_ready(FAKE_DEVICES_INFO)

        for child in self.ui.devices.get_children():
            self.assertTrue(self.ui._devices[child.id] is child)

    def test_on_devices_info_ready_clears_the_list(self):
        """The old devices info is cleared before updated."""
        self.ui.on_devices_info_ready(FAKE_DEVICES_INFO)
        self.ui.on_devices_info_ready(FAKE_DEVICES_INFO)

        devices = self.ui.devices.get_children()
        self.assertEqual(len(devices), len(FAKE_DEVICES_INFO))

    def test_on_devices_info_ready_with_no_devices(self):
        """When there are no devices, a notification is shown."""
        self.ui.on_devices_info_ready([])
        self.assertEqual(len(self.ui.devices.get_children()), 0)

    def test_is_not_processing_after_on_devices_info_error(self):
        """The ui is no longer processing on devices info error."""
        self.ui.on_devices_info_error()

        self.assertFalse(self.ui.is_processing)

    def test_on_devices_info_error(self):
        """The devices info couldn't be retrieved."""
        self.ui.on_devices_info_error()

        self.assert_warning_correct(warning=self.ui.message,
                                    text=gui.VALUE_ERROR)

    def test_on_devices_info_error_after_success(self):
        """The devices info couldn't be retrieved after a prior success."""
        self.ui.on_devices_info_ready(FAKE_DEVICES_INFO)

        self.ui.on_devices_info_error()

        self.test_on_devices_info_error()
        self.test_on_devices_info_ready_with_no_devices()

    def test_on_device_removed(self):
        """When a child device was removed, remove and destroy."""
        self.ui.on_devices_info_ready(FAKE_DEVICES_INFO)
        did = FAKE_DEVICES_INFO[0]['device_id']
        device = self.ui._devices[did]
        self.ui.on_device_removed(device_id=did)

        self.assertTrue(device not in self.ui.devices.get_children())
        self.assertTrue(did not in self.ui._devices)

    def test_on_local_device_removed(self):
        """Removing the local device emits local-device-removed."""
        self.ui.connect('local-device-removed', self._set_called)

        self.ui.on_devices_info_ready(FAKE_DEVICES_INFO)
        local_device = FAKE_DEVICES_INFO[-1]
        assert bool(local_device['is_local'])
        local_device_id = local_device['device_id']
        assert self.ui._devices[local_device_id].is_local

        self.ui.on_device_removed(device_id=local_device_id)

        self.assertEqual(self._called, ((self.ui,), {}))

    def test_on_device_removed_for_no_child_device(self):
        """On other device removed, do nothing."""
        self.ui.on_devices_info_ready(FAKE_DEVICES_INFO)
        old_devices = self.ui.devices.get_children()

        self.ui.on_device_removed(device_id='foo')

        new_devices = self.ui.devices.get_children()
        self.assertEqual(new_devices, old_devices)


class InstallPackageTestCase(ControlPanelMixinTestCase):
    """The test suite for the install widget."""

    klass = gui.InstallPackage
    ui_filename = 'install.ui'
    kwargs = {'package_name': 'a test package'}

    def test_is_an_box(self):
        """Inherits from gtk.VBox."""
        self.assertIsInstance(self.ui, gui.gtk.VBox)

    def test_inner_widget_is_packed(self):
        """The 'itself' vbox is packed into the widget."""
        self.assertIn(self.ui.itself, self.ui.get_children())

    def test_is_visible(self):
        """Is visible."""
        self.assertTrue(self.ui.get_visible())

    def test_package_name(self):
        """The package_name is stored."""
        self.assertEqual(self.ui.package_name, self.kwargs['package_name'])

    def test_children(self):
        """The children is correct."""
        children = self.ui.itself.get_children()
        self.assertEqual(len(children), 2)
        self.assertEqual(self.ui.install_label, children[0])
        self.assertIn(self.ui.install_button, children[1].get_children())

    def test_install_label(self):
        """The install label is correct."""
        msg = gui.INSTALL_PACKAGE % self.kwargs
        self.assertEqual(self.ui.install_label.get_label(), msg)

    @gui.package_manager.inline_callbacks
    def test_install_button_clicked_shows_progress(self):
        """The install button is correct."""
        yield self.ui.install_button.clicked()

        children = self.ui.itself.get_children()
        self.assertEqual(len(children), 2)
        self.assertEqual(self.ui.progress_bar, children[1])
        self.assertTrue(self.ui.progress_bar.get_visible())
        self.assertIsInstance(self.ui.progress_bar,
                              gui.package_manager.PackageManagerProgressBar)

    def test_install_button_clicked_install_label(self):
        """The install label is correct."""
        yield self.ui.install_button.clicked()

        children = self.ui.itself.get_children()
        self.assertEqual(len(children), 2)
        self.assertEqual(self.ui.install_label, children[0])
        msg = gui.INSTALLING % self.kwargs
        self.assertEqual(self.ui.install_label.get_label(), msg)

    @gui.package_manager.inline_callbacks
    def test_install_button_clicked_install_returns_string_success(self):
        """The install immediatly returning a string is handled."""
        self.patch(self.ui.package_manager, 'install', lambda name: SUCCESS)
        yield self.ui.install_button.clicked()

        msg = gui.SUCCESS_INSTALL % self.kwargs
        self.assertEqual(self.ui.install_label.get_label(), msg)

    @gui.package_manager.inline_callbacks
    def test_install_button_clicked_transaction(self):
        """The install button transaction is correct."""
        yield self.ui.install_button.clicked()

        transaction = self.ui.transaction
        self.assertTrue(transaction.packages, [self.ui.package_name])
        self.assertIn(self.ui.on_install_finished,
                      transaction._signals['finished'])
        self.assertTrue(transaction.was_run)

    @gui.package_manager.inline_callbacks
    def test_install_button_clicked_fails(self):
        """The install button transaction is correct."""

        def fail(*args):
            """Simulate an error."""
            raise Exception(args)

        self.patch(self.ui.package_manager, 'install', fail)
        yield self.ui.install_button.clicked()

        msg = gui.FAILED_INSTALL % self.kwargs
        self.assert_warning_correct(self.ui.install_label, msg)

    @gui.package_manager.inline_callbacks
    def test_on_install_finished_success(self):
        """The install finished."""
        self.ui.connect('finished', self._set_called)
        yield self.ui.install_button.clicked()
        self.ui.on_install_finished(object(), SUCCESS)

        self.assertFalse(self.ui.progress_bar.get_sensitive())
        msg = gui.SUCCESS_INSTALL % self.kwargs
        self.assertEqual(self.ui.install_label.get_label(), msg)
        self.assertEqual(self._called, ((self.ui,), {}))

    @gui.package_manager.inline_callbacks
    def test_on_install_finished_failed(self):
        """The install finished."""
        yield self.ui.install_button.clicked()
        self.ui.on_install_finished(object(), FAILURE)

        self.assertFalse(self.ui.progress_bar.get_sensitive())
        msg = gui.FAILED_INSTALL % self.kwargs
        self.assert_warning_correct(self.ui.install_label, msg)


class ServiceTestCase(ControlPanelMixinTestCase):
    """The test suite for a service."""

    klass = gui.Service
    service_id = 'dc_test'
    name = u'Qué lindo test!'
    kwargs = {'service_id': service_id, 'name': name,
              'container': None, 'check_button': None}

    def test_is_an_box(self):
        """Inherits from gtk.VBox."""
        self.assertIsInstance(self.ui, gui.gtk.VBox)

    def test_is_visible(self):
        """Is visible."""
        self.assertTrue(self.ui.get_visible())

    def test_warning_label_is_cleared(self):
        """The warning label is cleared."""
        self.assertEqual(self.ui.warning_label.get_text(), '')

    def test_warning_label_packed(self):
        """The warning label is packed as child."""
        self.assertIn(self.ui.warning_label, self.ui.get_children())

    def test_check_button_packed(self):
        """A check button is packed as child."""
        self.assertIn(self.ui.button, self.ui.get_children())

    def test_label(self):
        """The label is set."""
        self.assertEqual(self.name, self.ui.button.get_label())

    def test_service_id(self):
        """The service id is set."""
        self.assertEqual(self.service_id, self.ui.id)


class FileSyncServiceTestCase(ServiceTestCase):
    """The test suite for the file sync service."""

    klass = gui.FileSyncService
    service_id = 'file-sync'
    name = gui.FILE_SYNC_SERVICE_NAME
    kwargs = {'container': None, 'check_button': None, 'action_button': None}

    def test_backend_account_signals(self):
        """The proper signals are connected to the backend."""
        self.assertEqual(self.ui.backend._signals['FileSyncStatusChanged'],
                         [self.ui.on_file_sync_status_changed])
        self.assertEqual(self.ui.backend._signals['FilesEnabled'],
                         [self.ui.on_files_enabled])
        self.assertEqual(self.ui.backend._signals['FilesDisabled'],
                         [self.ui.on_files_disabled])

    def test_file_sync_status_is_requested(self):
        """The file sync status is requested to the backend."""
        self.assert_backend_called('file_sync_status')

    def test_is_disabled(self):
        """Until file sync status is given, the widget is disabled."""
        self.assertFalse(self.ui.get_sensitive())

    def test_is_enabled_on_file_sync_status_changed(self):
        """When the file sync status is given, the widget is enabled."""
        self.ui.on_file_sync_status_changed('something')
        self.assertTrue(self.ui.get_sensitive())

    def test_active(self):
        """Is active when file status is anything but 'file-sync-disabled'."""
        self.ui.on_file_sync_status_changed('something not disabled')
        self.assertTrue(self.ui.button.get_active())

    def test_not_active(self):
        """Is not active when status is exactly but 'file-sync-disabled'."""
        self.ui.on_file_sync_status_changed(gui.backend.FILE_SYNC_DISABLED)
        self.assertFalse(self.ui.button.get_active())

    def test_on_button_toggled(self):
        """When toggling the button, the file sync service is updated."""
        self.ui.on_file_sync_status_changed('something not disabled')
        assert self.ui.button.get_active()

        self.ui.button.set_active(not self.ui.button.get_active())
        self.assert_backend_called('disable_files')

        self.ui.button.set_active(not self.ui.button.get_active())
        self.assert_backend_called('enable_files')

    def test_on_file_sync_enabled(self):
        """When file sync is enabled, the button is active."""
        self.ui.on_files_disabled()
        assert not self.ui.button.get_active()

        self.ui.on_files_enabled()
        self.assertTrue(self.ui.button.get_active())

    def test_on_file_sync_disabled(self):
        """When file sync is disabled, the button is not active."""
        self.ui.on_files_enabled()
        assert self.ui.button.get_active()

        self.ui.on_files_disabled()
        self.assertFalse(self.ui.button.get_active())

FileSyncServiceTestCase.skip = 'LP: #729349'


class DesktopcouchServiceTestCase(ServiceTestCase):
    """The test suite for a desktopcouch service."""

    klass = gui.DesktopcouchService
    enabled = True

    def setUp(self):
        self.kwargs['enabled'] = self.enabled
        super(DesktopcouchServiceTestCase, self).setUp()

    def modify_settings(self):
        """Modify settings so values actually change."""
        self.ui.button.set_active(not self.ui.button.get_active())

    def test_backend_account_signals(self):
        """The proper signals are connected to the backend."""
        self.assertEqual(
            self.ui.backend._signals['ReplicationSettingsChanged'],
            [self.ui.on_replication_settings_changed])
        self.assertEqual(
            self.ui.backend._signals['ReplicationSettingsChangeError'],
            [self.ui.on_replication_settings_change_error])

    def test_active(self):
        """Is active if enabled."""
        self.assertEqual(self.enabled, self.ui.button.get_active())

    def test_on_button_toggled(self):
        """When toggling the button, the DC exclude list is updated."""
        self.ui.button.set_active(not self.ui.button.get_active())

        args = (self.service_id,
                {'enabled': gui.bool_str(self.ui.button.get_active())})
        self.assert_backend_called('change_replication_settings', *args)

    def test_dependency(self):
        """The dependency box is None."""
        self.assertTrue(self.ui.dependency is None)

    def test_button_sensitiveness(self):
        """The check button is sensitive."""
        self.assertTrue(self.ui.button.get_sensitive())

    def test_on_replication_settings_changed(self):
        """When settings were changed for this replication, enable it."""
        new_val = not self.ui.button.get_active()
        self.ui.button.set_active(new_val)

        self.ui.on_replication_settings_changed(replication_id=self.ui.id)

        self.assertEqual(self.ui.warning_label.get_text(), '')
        self.assertEqual(new_val, self.ui.button.get_active())

    def test_on_replication_settings_changed_after_error(self):
        """Change success after error."""
        self.ui.button.set_active(not self.ui.button.get_active())
        self.ui.on_replication_settings_change_error(replication_id=self.ui.id)

        self.test_on_replication_settings_changed()

    def test_on_replication_settings_changed_different_id(self):
        """When settings were changed for other rep, nothing changes."""
        self.ui.button.set_active(not self.ui.button.get_active())
        self.ui.on_replication_settings_changed(replication_id='yadda')

        self.assertEqual(self.ui.warning_label.get_text(), '')

    def test_on_replication_settings_changed_different_id_after_error(self):
        """When settings were changed for other + error, nothing changes."""
        self.ui.on_replication_settings_change_error(replication_id=self.ui.id)
        self.ui.on_replication_settings_changed(replication_id='yadda')

        self.assert_warning_correct(self.ui.warning_label,
                                    gui.SETTINGS_CHANGE_ERROR)

    def test_on_replication_settings_change_error(self):
        """When settings were not changed, notify the user.

        Also, confirm that old value was restored.

        """
        old_val = self.ui.button.get_active()
        self.ui.button.set_active(not old_val)
        self.ui.on_replication_settings_change_error(replication_id=self.ui.id)

        self.assert_warning_correct(self.ui.warning_label,
                                    gui.SETTINGS_CHANGE_ERROR)
        self.assertEqual(old_val, self.ui.button.get_active())

    def test_on_replication_settings_change_error_after_success(self):
        """Change error after success."""
        self.ui.button.set_active(not self.ui.button.get_active())
        self.ui.on_replication_settings_changed(replication_id=self.ui.id)

        self.test_on_replication_settings_change_error()

    def test_on_replication_settings_change_error_different_id(self):
        """When settings were not changed for other replication, do nothing."""
        self.ui.button.set_active(not self.ui.button.get_active())
        self.ui.on_replication_settings_change_error(replication_id='yudo')

        self.assertEqual(self.ui.warning_label.get_text(), '')

DesktopcouchServiceTestCase.skip = 'LP: #729349'


class DesktopcouchServiceDisabledAtStartupTestCase(ServiceTestCase):
    """The test suite for a desktopcouch service when enabled=False."""

    enabled = False


class DesktopcouchServiceWithDependencyTestCase(DesktopcouchServiceTestCase):
    """The test suite for a desktopcouch service when it needs a dependency."""

    def setUp(self):
        self.kwargs['dependency'] = 'a package'
        super(DesktopcouchServiceWithDependencyTestCase, self).setUp()

    def test_dependency(self):
        """The dependency bos is not hidden."""
        self.assertIsInstance(self.ui.dependency, gui.InstallPackage)
        self.assertEqual(self.ui.dependency.package_name,
                         self.kwargs['dependency'])

    def test_dependency_is_packed(self):
        """The dependency is packed in the ui."""
        self.assertIn(self.ui.dependency, self.ui.get_children())

    def test_button_sensitiveness(self):
        """The check button is not sensitive until depedency installed."""
        self.assertFalse(self.ui.button.get_sensitive())

    def test_button_is_enabled_on_dependency_installed(self):
        """The check button is sensitive when depedency is installed."""
        self.ui.dependency.emit('finished')

        self.assertTrue(self.ui.button.get_sensitive())

    def test_install_widget_is_removed_on_dependency_installed(self):
        """The install button is removed when depedency is installed."""
        self.ui.dependency.emit('finished')

        self.assertTrue(self.ui.dependency is None)
        self.assertEqual(sorted(self.ui.get_children()),
                         sorted([self.ui.button, self.ui.warning_label]))

DesktopcouchServiceWithDependencyTestCase.skip = 'LP: #729349'


class ServicesTestCase(ControlPanelMixinTestCase):
    """The test suite for the services panel."""

    klass = gui.ServicesPanel
    ui_filename = 'services.ui'

    def test_is_an_ubuntuone_bin(self):
        """Inherits from UbuntuOneBin."""
        self.assertIsInstance(self.ui, gui.UbuntuOneBin)

    def test_inner_widget_is_packed(self):
        """The 'itself' vbox is packed into the widget."""
        self.assertIn(self.ui.itself, self.ui.get_children())

    def test_is_visible(self):
        """Is visible."""
        self.assertTrue(self.ui.get_visible())

    def test_package_manager(self):
        """Has a package manager."""
        self.assertIsInstance(self.ui.package_manager,
                              gui.package_manager.PackageManager)

    def test_install_box(self):
        """The install box is None."""
        self.assertTrue(self.ui.install_box is None)

    def test_backend_signals(self):
        """The proper signals are connected to the backend."""
        self.assertEqual(self.ui.backend._signals['ReplicationsInfoReady'],
                         [self.ui.on_replications_info_ready])
        self.assertEqual(self.ui.backend._signals['ReplicationsInfoError'],
                         [self.ui.on_replications_info_error])


class ServicesFilesTestCase(ServicesTestCase):
    """The test suite for the services panel (files section)."""

    def test_files_is_visible(self):
        """Files section is visible."""
        self.assertTrue(self.ui.files.get_visible())

    def test_files_is_a_file_sync_service(self):
        """Files contains a FilesService."""
        child, = self.ui.files.get_children()
        self.assertIsInstance(child, gui.FileSyncService)

    test_files_is_a_file_sync_service.skip = 'LP: #729349'


class ServicesWithoutDesktopcouchTestCase(ServicesTestCase):
    """The test suite for the services panel when DC is not installed."""

    def setUp(self):
        super(ServicesWithoutDesktopcouchTestCase, self).setUp()
        self.patch(self.ui.package_manager, 'is_installed', lambda *a: False)
        self.ui.load()

    def test_message(self):
        """Global load message is stopped and cleared."""
        self.assertFalse(self.ui.message.active)
        self.assertEqual(self.ui.message.get_text(), '')

    def test_has_desktopcouch(self):
        """Has desktopcouch installed?"""
        self.assertFalse(self.ui.has_desktopcouch)

    def test_install_box_is_hidden(self):
        """The install box is not hidden."""
        self.assertTrue(self.ui.install_box.get_visible())

    def test_replications_is_hidden(self):
        """The replications section is disabled."""
        self.assertFalse(self.ui.replications.get_visible())

    def test_install_box(self):
        """The install box is enabled."""
        self.assertTrue(self.ui.install_box.get_visible())
        self.assertIn(self.ui.install_box, self.ui.itself.get_children())
        self.assertEqual(self.ui.install_box.package_name,
                         gui.DESKTOPCOUCH_PKG)

    def test_install_box_finished_connected(self):
        """The install box 'finished' signal is connected."""
        self.patch(self.ui, 'load_replications', self._set_called)
        self.ui.load()  # ensure signal connection uses the new method

        self.ui.install_box.emit('finished')

        self.assertEqual(self._called, ((self.ui.install_box,), {}))

    def test_load_replications(self):
        """The load_replications starts the spinner and calls the backend."""
        self.ui.load_replications()

        self.assertTrue(self.ui.message.active)
        self.assert_backend_called('replications_info')


class ServicesWithDesktopcouchTestCase(ServicesTestCase):
    """The test suite for the services panel."""

    def setUp(self):
        super(ServicesWithDesktopcouchTestCase, self).setUp()
        self.ui.package_manager.is_installed = lambda name: True
        self.ui.on_replications_info_ready(info=FAKE_REPLICATIONS_INFO)

    def test_message(self):
        """Global load message is stopped and proper test is shown."""
        self.assertFalse(self.ui.message.active)
        self.assertEqual(self.ui.message.get_text(), '')

    def test_has_desktopcouch(self):
        """Has desktopcouch installed?"""
        self.assertTrue(self.ui.has_desktopcouch)

    def test_replications(self):
        """Has proper child for each desktopcouch replication available."""
        self.assertTrue(self.ui.replications.get_visible())

        children = self.ui.replications.get_children()
        self.assertEqual(len(children), len(FAKE_REPLICATIONS_INFO))
        for expected, child in zip(FAKE_REPLICATIONS_INFO, children):
            self.assertIsInstance(child, gui.DesktopcouchService)
            self.assertEqual(expected['replication_id'], child.id)
            self.assertEqual(expected['name'], child.button.get_label())
            self.assertEqual(bool(expected['enabled']),
                             child.button.get_active())
            self.assertTrue(child.dependency is None)  # all deps are installed

    def test_replications_with_dependencies_not_installed(self):
        """Has proper child for each desktopcouch replication available."""
        self.ui.package_manager.is_installed = \
            lambda name: True if name == gui.DESKTOPCOUCH_PKG else False
        self.ui.on_replications_info_ready(info=FAKE_REPLICATIONS_INFO)

        children = self.ui.replications.get_children()
        self.assertEqual(len(children), len(FAKE_REPLICATIONS_INFO))
        for expected, child in zip(FAKE_REPLICATIONS_INFO, children):
            if expected['dependency']:
                self.assertTrue(child.dependency is not None)
                self.assertEqual(expected['dependency'],
                                 child.dependency.package_name)
            else:
                self.assertTrue(child.dependency is None)

    def test_replications_after_getting_info_twice(self):
        """Has proper child after getting backend info twice."""
        self.ui.on_replications_info_ready(info=FAKE_REPLICATIONS_INFO)
        self.test_replications()

    def test_service_name_in_mapping(self):
        """If available, use a user-friendly, translatable service name."""
        for sid, name in self.ui.service_names.iteritems():
            info = [{'replication_id': sid, 'name': 'Bar',
                     'enabled': 'True', 'dependency': ''}]
            self.ui.on_replications_info_ready(info=info)

            child, = self.ui.replications.get_children()
            self.assertEqual(child.button.get_label(), name)

    def test_service_name_not_in_mapping(self):
        """If available, use a user-friendly, translatable service name."""
        sid = 'not-in-mapping'
        assert sid not in self.ui.service_names

        info = [{'replication_id': sid, 'name': 'Bar',
                 'enabled': 'True', 'dependency': ''}]
        self.ui.on_replications_info_ready(info=info)

        child, = self.ui.replications.get_children()
        self.assertEqual(child.button.get_label(), info[0]['name'])

ServicesWithDesktopcouchTestCase.skip = 'LP: #729349'


class ServicesWithDesktopcouchErrorTestCase(ServicesTestCase):
    """The test suite for the services panel."""

    def setUp(self):
        super(ServicesWithDesktopcouchErrorTestCase, self).setUp()
        self.ui.package_manager._installed[gui.DESKTOPCOUCH_PKG] = True

    def test_no_pairing_record(self):
        """The pairing record is not in place."""
        error_dict = {'error_type': 'NoPairingRecord'}
        self.ui.on_replications_info_error(error_dict)

        self.assertEqual(self.ui.replications.get_children(), [])
        self.assertFalse(self.ui.message.active)
        self.assert_warning_correct(self.ui.message, gui.NO_PAIRING_RECORD)

    def test_other_error(self):
        """There was an error other than no pairing record."""
        error_dict = {'error_type': 'OtherError'}
        self.ui.on_replications_info_error(error_dict)

        self.assertEqual(self.ui.replications.get_children(), [])
        self.assertFalse(self.ui.message.active)
        self.assert_warning_correct(self.ui.message, gui.VALUE_ERROR)

    def test_empty_dict(self):
        """Handle empty dicts errors."""
        self.ui.on_replications_info_error(error_dict={})

        self.assertEqual(self.ui.replications.get_children(), [])
        self.assertFalse(self.ui.message.active)
        self.assert_warning_correct(self.ui.message, gui.VALUE_ERROR)

    def test_error_dict_none(self):
        """HGandle empty dicts errors."""
        self.ui.on_replications_info_error(error_dict=None)

        self.assertEqual(self.ui.replications.get_children(), [])
        self.assertFalse(self.ui.message.active)
        self.assert_warning_correct(self.ui.message, gui.VALUE_ERROR)

ServicesWithDesktopcouchErrorTestCase.skip = 'LP: #729349'


class FileSyncStatusTestCase(ControlPanelMixinTestCase):
    """Test case for a file sync status widget."""

    klass = gui.FileSyncStatus

    def assert_status_correct(self, status, action=None,
                              callback=None, tooltip=None):
        """The shown status is correct.

        * The ui's label shows 'status'.
        * If action is not None, the ui's button shows that 'action' as label
        and when clicking it, 'self._set_called' gets executed.
        * If action is None, the ui's button should be hidden.
        * If a tooltip is required, then it exists with correct text.

        """
        self.assertTrue(self.ui.label.get_visible())
        self.assertFalse(self.ui.label.active)
        msg = '%r does not end with %r' % (self.ui.label.get_label(), status)
        self.assertTrue(self.ui.label.get_label().endswith(status), msg)

        self.assertTrue(self.ui.button.is_sensitive())
        self.assertFalse(self.ui.button.get_visited())

        if action is not None:
            self.assertTrue(self.ui.button.get_visible())
            self.assertTrue(self.ui.button.get_sensitive())
            self.assertEqual(self.ui.button.get_label(), action)

            self.ui.button.clicked()
            self.assertFalse(self.ui.button.get_visited())
            self.assertFalse(self.ui.button.get_sensitive())
            self.assertEqual(self._called, ((self.ui.button,), {}))
        else:
            self.assertFalse(self.ui.button.get_visible())

        if tooltip is not None:
            self.assertTrue(self.ui.button.get_has_tooltip())
            self.assertEqual(self.ui.button.get_tooltip_text(), tooltip)
        else:
            self.assertFalse(self.ui.button.get_has_tooltip())

    def test_is_a_box(self):
        """Inherits from gtk.Box."""
        self.assertIsInstance(self.ui, gui.gtk.Box)

    def test_startup_visibility(self):
        """The widget is visible at startup."""
        self.assertTrue(self.ui.get_visible(),
                        'must be visible at startup.')

    def test_label(self):
        """The label is packed."""
        self.assertIsInstance(self.ui.label, gui.LabelLoading)
        self.assertTrue(self.ui.label.active)
        self.assertIn(self.ui.label, self.ui.get_children())

    def test_button(self):
        """The button is packed."""
        self.assertIn(self.ui.button, self.ui.get_children())

    def test_backend_file_sync_signals(self):
        """The proper signals are connected to the backend."""
        matches = (
            # status
            ('FileSyncStatusDisabled', [self.ui.on_file_sync_status_disabled]),
            ('FileSyncStatusStarting', [self.ui.on_file_sync_status_starting]),
            ('FileSyncStatusStopped', [self.ui.on_file_sync_status_stopped]),
            ('FileSyncStatusDisconnected',
             [self.ui.on_file_sync_status_disconnected]),
            ('FileSyncStatusSyncing', [self.ui.on_file_sync_status_syncing]),
            ('FileSyncStatusIdle', [self.ui.on_file_sync_status_idle]),
            ('FileSyncStatusError', [self.ui.on_file_sync_status_error]),
            ('FilesStartError', [self.ui.on_files_start_error]),
            ('FilesDisabled', [self.ui.on_file_sync_status_disabled]),
            ('FilesEnabled', [self.ui.on_file_sync_status_starting]),
        )
        for sig, handlers in matches:
            self.assertEqual(self.ui.backend._signals[sig], handlers)

    def test_file_sync_status_is_requested_on_load(self):
        """The file sync status is requested to the backend."""
        self.ui.load()
        self.assert_backend_called('file_sync_status')

    def test_on_file_sync_status_disabled(self):
        """The file sync is disabled.

        * The correct connection status is displayed.
        * The button has a tooltip with correct text.

        """
        self.patch(self.ui, 'on_enable_clicked', self._set_called)
        self.ui.on_file_sync_status_disabled('msg')

        self.assert_status_correct(gui.FILE_SYNC_DISABLED,
                                   action=gui.FILE_SYNC_ENABLE,
                                   tooltip=gui.FILE_SYNC_ENABLE_TOOLTIP)

    def test_on_file_sync_status_starting(self):
        """The file sync status is starting.

        * The correct connection status is displayed.
        * The button has a tooltip with correct text.

        """
        self.patch(self.ui, 'on_stop_clicked', self._set_called)
        self.ui.on_file_sync_status_starting('msg')

        self.assert_status_correct(gui.FILE_SYNC_STARTING,
                                   action=gui.FILE_SYNC_STOP,
                                   tooltip=gui.FILE_SYNC_STOP_TOOLTIP)

    def test_on_file_sync_status_stopped(self):
        """The file sync is stopped.

        * The correct connection status is displayed.
        * The button has a tooltip with correct text.

        """
        self.patch(self.ui, 'on_start_clicked', self._set_called)
        self.ui.on_file_sync_status_stopped('msg')

        self.assert_status_correct(gui.FILE_SYNC_STOPPED,
                                   action=gui.FILE_SYNC_START,
                                   tooltip=gui.FILE_SYNC_START_TOOLTIP)

    def test_on_file_sync_status_disconnected(self):
        """The file sync status is disconnected.

        * The correct connection status is displayed.
        * The button has a tooltip with correct text.

        """
        self.patch(self.ui, 'on_connect_clicked', self._set_called)
        self.ui.on_file_sync_status_disconnected('msg')

        self.assert_status_correct(gui.FILE_SYNC_DISCONNECTED,
                                   action=gui.FILE_SYNC_CONNECT,
                                   tooltip=gui.FILE_SYNC_CONNECT_TOOLTIP)

    def test_on_file_sync_status_syncing(self):
        """The file sync status is syncing.

        * The correct connection status is displayed.
        * The button has a tooltip with correct text.

        """
        self.patch(self.ui, 'on_disconnect_clicked', self._set_called)
        self.ui.on_file_sync_status_syncing('msg')

        self.assert_status_correct(gui.FILE_SYNC_SYNCING,
                                   action=gui.FILE_SYNC_DISCONNECT,
                                   tooltip=gui.FILE_SYNC_DISCONNECT_TOOLTIP)

    def test_on_file_sync_status_idle(self):
        """The file sync status is idle.

        * The correct connection status is displayed.
        * The button has a tooltip with correct text.

        """
        self.patch(self.ui, 'on_disconnect_clicked', self._set_called)
        self.ui.on_file_sync_status_idle('msg')

        self.assert_status_correct(gui.FILE_SYNC_IDLE,
                                   action=gui.FILE_SYNC_DISCONNECT,
                                   tooltip=gui.FILE_SYNC_DISCONNECT_TOOLTIP)

    def test_on_file_sync_status_error_with_error_msg(self):
        """The file sync status couldn't be retrieved."""
        self.patch(self.ui, 'on_restart_clicked', self._set_called)
        msg = 'error message'
        self.ui.on_file_sync_status_error({'error_msg': msg})

        msg = gui.WARNING_MARKUP % (gui.FILE_SYNC_ERROR + ' (' + msg + ')')
        self.assert_status_correct(msg,
                                   action=gui.FILE_SYNC_RESTART,
                                   tooltip=gui.FILE_SYNC_RESTART_TOOLTIP)

    def test_on_file_sync_status_error_without_error_msg(self):
        """The file sync status couldn't be retrieved."""
        self.patch(self.ui, 'on_restart_clicked', self._set_called)
        self.ui.on_file_sync_status_error()

        msg = gui.WARNING_MARKUP % gui.FILE_SYNC_ERROR
        self.assert_status_correct(msg,
                                   action=gui.FILE_SYNC_RESTART,
                                   tooltip=gui.FILE_SYNC_RESTART_TOOLTIP)

    def test_on_files_start_error(self):
        """The files service could not be started."""
        self.ui.backend._called.clear()
        self.ui.on_files_start_error({'error_msg': 'error msg'})

        self.assert_backend_called('file_sync_status')

    def test_on_connect_clicked(self):
        """User requested connection."""
        self.ui.on_connect_clicked(self.ui.button)

        self.assert_backend_called('connect_files')

    def test_on_disconnect_clicked(self):
        """User requested disconnection."""
        self.ui.on_disconnect_clicked(self.ui.button)

        self.assert_backend_called('disconnect_files')

    def test_on_enable_clicked(self):
        """User requested enable the service."""
        self.ui.on_enable_clicked(self.ui.button)

        self.assert_backend_called('enable_files')

    def test_on_restart_clicked(self):
        """User requested restart the service."""
        self.ui.on_restart_clicked(self.ui.button)

        self.assert_backend_called('restart_files')

    def test_on_start_clicked(self):
        """User requested start the service."""
        self.ui.on_start_clicked(self.ui.button)

        self.assert_backend_called('start_files')

    def test_on_stop_clicked(self):
        """User requested stop the service."""
        self.ui.on_stop_clicked(self.ui.button)

        self.assert_backend_called('stop_files')


class ManagementPanelTestCase(ControlPanelMixinTestCase):
    """The test suite for the management panel."""

    klass = gui.ManagementPanel
    ui_filename = 'management.ui'

    def assert_account_info_correct(self, info, progressbar_fraction=None):
        """Check that the displayed account info matches 'info'."""
        used = int(info['quota_used'])
        total = int(info['quota_total'])
        percentage = round((used / total) * 100, 2)
        expected = {'used': humanize(used),
                    'total': humanize(total),
                    'percentage': percentage}
        msg = gui.QUOTA_LABEL % expected
        self.assertEqual(self.ui.quota_label.get_text(), msg)

        if percentage >= gui.QUOTA_THRESHOLD * 100:
            self.assert_warning_correct(self.ui.quota_label, msg)

        if progressbar_fraction is None:
            progressbar_fraction = min(percentage / 100, 1)
        self.assertEqual(self.ui.quota_progressbar.get_fraction(),
                         progressbar_fraction)
        self.assertTrue(self.ui.quota_progressbar.get_sensitive())

    def test_is_a_vbox(self):
        """Inherits from gtk.VBox."""
        self.assertIsInstance(self.ui, gui.gtk.VBox)

    def test_startup_visibility(self):
        """The widget is visible at startup."""
        self.assertTrue(self.ui.get_visible(),
                        'must be visible at startup.')

    def test_inner_widget_is_packed(self):
        """The 'itself' vbox is packed into the widget."""
        self.assertIn(self.ui.itself, self.ui.get_children())

    def test_tabs_are_not_shown(self):
        """Tabs are not shown."""
        self.assertFalse(self.ui.notebook.get_show_tabs())

    def test_default_page_is_dashboard(self):
        """The default page is Dashboard."""
        self.assertEqual(self.ui.notebook.get_current_page(),
                         self.ui.DASHBOARD_PAGE)
        self.assertTrue(self.ui.dashboard_button.get_active())

    def test_buttons_set_notebook_pages(self):
        """The notebook pages are set when clicking buttons."""
        msg = 'Page num should be %i when %s was clicked (got %i instead).'
        for tab in self.ui.tabs:
            button = '%s_button' % tab
            getattr(self.ui, button).clicked()
            expected = getattr(self.ui, ('%s_page' % tab).upper())
            actual = self.ui.notebook.get_current_page()
            self.assertEqual(actual, expected,
                             msg % (expected, button, actual))

    def test_buttons_indicates_current_page(self):
        """Only one button is activated at a time."""
        msg = 'Only button %s should be active (%s was active as well).'
        for tab in self.ui.tabs:
            button = '%s_button' % tab
            getattr(self.ui, button).clicked()
            for other in self.ui.tabs:
                if other is tab:
                    continue
                active = getattr(self.ui, '%s_button' % other).get_active()
                self.assertFalse(active, msg % (button, other))

    def test_backend_account_signals(self):
        """The proper signals are connected to the backend."""
        self.assertEqual(self.ui.backend._signals['AccountInfoReady'],
                         [self.ui.on_account_info_ready])
        self.assertEqual(self.ui.backend._signals['AccountInfoError'],
                         [self.ui.on_account_info_error])

    def test_backend_unauthorized_signal(self):
        """The proper signals are connected to the backend."""
        self.assertEqual(self.ui.backend._signals['UnauthorizedError'],
                         [self.ui.on_unauthorized_error])

    def test_no_backend_calls_before_load(self):
        """No calls are made to the backend before load() is called."""
        self.assertEqual(self.ui.backend._called, {})

    def test_account_info_is_requested_on_load(self):
        """The account info is requested to the backend."""
        self.ui.load()
        self.assert_backend_called('account_info')

    def test_file_sync_status_info_is_requested_on_load(self):
        """The file sync status info is requested to the backend."""
        self.patch(self.ui.status_label, 'load', self._set_called)
        self.ui.load()
        self.assertEqual(self._called, ((), {}))

    def test_replications_info_is_requested_on_load(self):
        """The replications info is requested to the backend."""
        self.patch(self.ui.services, 'load', self._set_called)
        self.ui.load()
        self.assertEqual(self._called, ((), {}))

    def test_dashboard_panel_is_packed(self):
        """The dashboard panel is packed."""
        self.assertIsInstance(self.ui.dashboard, gui.DashboardPanel)
        actual = self.ui.notebook.get_nth_page(self.ui.DASHBOARD_PAGE)
        self.assertTrue(self.ui.dashboard is actual)

    def test_volumes_panel_is_packed(self):
        """The volumes panel is packed."""
        self.assertIsInstance(self.ui.volumes, gui.VolumesPanel)
        actual = self.ui.notebook.get_nth_page(self.ui.VOLUMES_PAGE)
        self.assertTrue(self.ui.volumes is actual)

    def test_devices_panel_is_packed(self):
        """The devices panel is packed."""
        self.assertIsInstance(self.ui.devices, gui.DevicesPanel)
        actual = self.ui.notebook.get_nth_page(self.ui.DEVICES_PAGE)
        self.assertTrue(self.ui.devices is actual)

    def test_services_panel_is_packed(self):
        """The services panel is packed."""
        self.assertIsInstance(self.ui.services, gui.ServicesPanel)
        actual = self.ui.notebook.get_nth_page(self.ui.SERVICES_PAGE)
        self.assertTrue(self.ui.services is actual)

    def test_entering_volumes_tab_loads_content(self):
        """The volumes info is loaded when entering the Volumes tab."""
        self.patch(self.ui.volumes, 'load', self._set_called)
        # clean backend calls
        self.ui.volumes_button.clicked()

        self.assertEqual(self._called, ((), {}))

    def test_entering_devices_tab_loads_content(self):
        """The devices info is loaded when entering the Devices tab."""
        self.patch(self.ui.devices, 'load', self._set_called)
        # clean backend calls
        self.ui.devices_button.clicked()

        self.assertEqual(self._called, ((), {}))

    def test_entering_services_tab_does_not_load_content(self):
        """The services info is not loaded when entering the Services tab."""
        self.patch(self.ui.services, 'load', self._set_called)
        # clean backend calls
        self.ui.services_button.clicked()

        self.assertEqual(self._called, False)

    def test_quota_placeholder_is_loading(self):
        """Placeholders for quota label is a Loading widget."""
        self.assertIsInstance(self.ui.quota_label, gui.LabelLoading)
        self.assertIn(self.ui.quota_label, self.ui.quota_box.get_children())

    def test_on_account_info_ready(self):
        """The account info is processed when ready."""
        self.ui.on_account_info_ready(FAKE_ACCOUNT_INFO)
        self.assert_account_info_correct(FAKE_ACCOUNT_INFO)
        self.assertFalse(self.ui.quota_label.active)

    def test_on_account_info_error(self):
        """The account info couldn't be retrieved."""
        self.ui.on_account_info_error()
        self.assertEqual(self.ui.quota_label.get_text(), '')
        self.assertEqual(self.ui.quota_progressbar.get_fraction(), 0)
        self.assertFalse(self.ui.quota_progressbar.get_sensitive())
        self.assertFalse(self.ui.quota_label.active)

    def test_on_account_info_error_after_success(self):
        """The account info couldn't be retrieved."""
        self.test_on_account_info_ready()
        self.test_on_account_info_error()

    def test_on_account_info_ready_quota_unused(self):
        """The used quota is correct when unused."""
        info = FAKE_ACCOUNT_INFO.copy()
        info['quota_used'] = '0'
        self.ui.on_account_info_ready(FAKE_ACCOUNT_INFO)
        self.assert_account_info_correct(FAKE_ACCOUNT_INFO)

    def test_on_account_info_ready_quota_little_used(self):
        """The used quota shows a minimun when little is used."""
        info = FAKE_ACCOUNT_INFO.copy()
        info['quota_used'] = '10'
        self.ui.on_account_info_ready(info)

        self.assert_account_info_correct(info, progressbar_fraction=0.05)

    def test_on_account_info_ready_quota_used_at_threshold(self):
        """Show red notification when quota usage is same as threshold."""
        info = FAKE_ACCOUNT_INFO.copy()

        info['quota_total'] = '12345678'
        info['quota_used'] = str(int(int(info['quota_total']) *
                                     gui.QUOTA_THRESHOLD))
        self.ui.on_account_info_ready(info)

        self.assert_account_info_correct(info)

    def test_on_account_info_ready_quota_over_threshold(self):
        """Show red notification when quota usage is higher than threshold."""
        info = FAKE_ACCOUNT_INFO.copy()
        info['quota_total'] = '12345678'
        info['quota_used'] = info['quota_total'] + '0'
        self.ui.on_account_info_ready(info)

        self.assert_account_info_correct(info)

    def test_file_sync_status(self):
        """The file sync status is shown correctly."""
        self.assertIsInstance(self.ui.status_label, gui.FileSyncStatus)
        self.assertIn(self.ui.status_label, self.ui.status_box.get_children())

    def test_backend_file_sync_signals(self):
        """The proper signals are connected to the backend."""
        self.assertEqual(self.ui.backend._signals['FilesEnabled'],
                         [self.ui.enable_volumes])
        self.assertEqual(self.ui.backend._signals['FilesDisabled'],
                         [self.ui.disable_volumes])

    def test_enable_volumes(self):
        """The volumes tab is properly enabled."""
        self.ui.enable_volumes()
        self.assertTrue(self.ui.volumes_button.get_sensitive())

    def test_disable_volumes(self):
        """The volumes tab is properly disabled."""
        self.ui.disable_volumes()
        self.assertFalse(self.ui.volumes_button.get_sensitive())

    def test_local_device_removed_is_emitted(self):
        """Signal local-device-removed is sent when DevicesPanel emits it."""
        self.ui.connect('local-device-removed', self._set_called)

        self.ui.devices.emit('local-device-removed')

        self.assertEqual(self._called, ((self.ui,), {}))

    def test_button_names(self):
        """The dashboard_button widget has the proper name."""
        for tab in (u'dashboard', u'services'):
            actual = getattr(self.ui, '%s_button' % tab).get_name()
            expected = getattr(self.ui, '%s_BUTTON_NAME' % tab.upper())
            self.assertEqual(actual, expected)

    def test_devices_button_tooltip(self):
        """The devices button widget has the proper tooltip."""
        for tab in self.ui.tabs:
            has_tooltip = getattr(self.ui, '%s_button' % tab).get_has_tooltip()
            self.assertTrue(has_tooltip,
                            '%s_button should have a tooltip set' % tab)

            actual = getattr(self.ui, '%s_button' % tab).get_tooltip_text()
            expected = getattr(gui, '%s_BUTTON_TOOLTIP' % tab.upper())
            self.assertEqual(actual, expected)

    def test_on_unauthorized_error(self):
        """On invalid credentials, proper signal is sent."""
        self.ui.connect('unauthorized', self._set_called)
        self.ui.on_unauthorized_error()
        self.assertEqual(self._called, ((self.ui,), {}))
