# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The test suite for the GTK UI for the control panel for Ubuntu One."""

import logging

from collections import defaultdict

from twisted.internet import defer
from ubuntuone.devtools.handlers import MementoHandler

from ubuntuone.controlpanel.gui.gtk import gui
from ubuntuone.controlpanel.gui.gtk.tests.test_package_manager import (
    FakedTransaction)
# Unused imports, they are here to maintain old module API
# pylint: disable=W0611
from ubuntuone.controlpanel.gui.tests import (FakedObject,
    FAKE_ACCOUNT_INFO,
    FAKE_DEVICE_INFO,
    FAKE_DEVICES_INFO,
    FAKE_FOLDERS_INFO,
    FAKE_SHARES_INFO,
    FAKE_VOLUMES_INFO,
    FAKE_VOLUMES_NO_FREE_SPACE_INFO,
    MUSIC_FOLDER,
    ROOT,
    USER_HOME,
)
# pylint: enable=W0611
from ubuntuone.controlpanel.tests import TestCase


# Attribute 'yyy' defined outside __init__, access to a protected member
# pylint: disable=W0201, W0212


FAKE_REPLICATIONS_INFO = [
    {'replication_id': 'foo', 'name': 'Bar',
     'enabled': 'True', 'dependency': ''},
    {'replication_id': 'yadda', 'name': 'Foo',
     'enabled': '', 'dependency': 'a very weird one'},
    {'replication_id': 'yoda', 'name': 'Figthers',
     'enabled': 'True', 'dependency': 'other dep'},
]


class FakedNMState(FakedObject):
    """Fake a NetworkManagerState."""

    exposed_methods = ['find_online_state']


class FakedDBusBackend(FakedObject):
    """Fake a DBus Backend."""

    bus_name = None
    object_path = None
    iface = None

    def __init__(self, obj, dbus_interface, *args, **kwargs):
        if dbus_interface != self.iface:
            raise TypeError()
        self._signals = defaultdict(list)
        super(FakedDBusBackend, self).__init__(*args, **kwargs)

    def connect_to_signal(self, signal, handler):
        """Bind 'handler' to be callback'd when 'signal' is fired."""
        self._signals[signal].append(handler)


class FakedCredentialsBackend(FakedObject):
    """Fake a credentials backend."""

    exposed_methods = ['find_credentials', 'clear_credentials',
                       'login', 'register']
    next_result = defer.succeed(None)


class FakedControlPanelBackend(FakedDBusBackend):
    """Fake a Control Panel Backend, act as a dbus.Interface."""

    bus_name = gui.DBUS_BUS_NAME
    object_path = gui.DBUS_PREFERENCES_PATH
    iface = gui.DBUS_PREFERENCES_IFACE
    exposed_methods = [
        'account_info',  # account
        'devices_info', 'change_device_settings', 'remove_device',  # devices
        'volumes_info', 'change_volume_settings',  # volumes
        'replications_info', 'change_replication_settings',  # replications
        'file_sync_status', 'enable_files', 'disable_files',  # files
        'connect_files', 'disconnect_files',
        'restart_files', 'start_files', 'stop_files', 'shutdown',
    ]


class FakedGUIBackend(FakedDBusBackend):
    """Fake a Control Panel GUI Service, act as a dbus.Interface."""

    bus_name = gui.DBUS_BUS_NAME_GUI
    object_path = gui.DBUS_PATH_GUI
    iface = gui.DBUS_IFACE_GUI
    exposed_methods = ['draw_attention', 'switch_to']


class FakedSessionBus(object):
    """Fake a session bus."""

    def get_object(self, bus_name, object_path, introspect=True,
                   follow_name_owner_changes=False, **kwargs):
        """Return a faked proxy for the given remote object."""
        return None


class FakedInterface(object):
    """Fake a dbus interface."""

    def __new__(cls, obj, dbus_interface, *args, **kwargs):
        if dbus_interface == gui.DBUS_PREFERENCES_IFACE:
            return FakedControlPanelBackend(obj, dbus_interface,
                                            *args, **kwargs)
        if dbus_interface == gui.DBUS_IFACE_GUI:
            return FakedGUIBackend(
                obj, dbus_interface, *args, **kwargs)


class FakedPackageManager(object):
    """Faked a package manager."""

    def __init__(self):
        self._installed = {}
        self.is_installed = lambda package_name: \
            self._installed.setdefault(package_name, False)

    @gui.package_manager.inline_callbacks
    def install(self, package_name):
        """Install 'package_name' if is not installed in this system."""
        yield
        self._installed[package_name] = True
        gui.package_manager.return_value(FakedTransaction([package_name]))


class FakedConfirmDialog(object):
    """Fake a confirmation dialog."""

    def __init__(self, *args, **kwargs):
        self._args = args
        self._kwargs = kwargs
        self.was_run = False
        self.is_visible = False
        self.markup = kwargs.get('message_format', None)
        self.show = lambda: setattr(self, 'is_visible', True)
        self.hide = lambda: setattr(self, 'is_visible', False)
        self.response_code = None

    def run(self):
        """Set flag and return 'self.response_code'."""
        self.was_run = True
        return self.response_code

    def set_markup(self, msg):
        """Set the markup."""
        self.markup = msg


class BaseTestCase(TestCase):
    """Basics for testing."""

    # self.klass is not callable
    # pylint: disable=E1102
    klass = None
    kwargs = {}
    backend_is_dbus = True

    def setUp(self):
        super(BaseTestCase, self).setUp()
        self.patch(gui, 'CredentialsManagementTool', FakedCredentialsBackend)
        self.patch(gui.os.path, 'expanduser',
                   lambda path: path.replace('~', USER_HOME))
        self.patch(gui.gtk, 'main', lambda: None)
        self.patch(gui.gtk, 'MessageDialog', FakedConfirmDialog)
        self.patch(gui.dbus, 'SessionBus', FakedSessionBus)
        self.patch(gui.dbus, 'Interface', FakedInterface)
        self.patch(gui.networkstate, 'NetworkManagerState', FakedNMState)
        self.patch(gui.package_manager, 'PackageManager', FakedPackageManager)

        if self.klass is not None:
            self.ui = self.klass(**self.kwargs)

        self.memento = MementoHandler()
        self.memento.setLevel(logging.DEBUG)
        gui.logger.addHandler(self.memento)

    def tearDown(self):
        try:
            self.ui.hide()
            del self.ui
            self.ui = None
        except AttributeError:
            pass
        super(BaseTestCase, self).tearDown()

    def assert_image_equal(self, image, filename):
        """Check that expected and actual represent the same image."""
        pb = gui.gtk.gdk.pixbuf_new_from_file(gui.get_data_file(filename))
        self.assertEqual(image.get_pixbuf().get_pixels(), pb.get_pixels())

    def assert_backend_called(self, method_name, *args, **kwargs):
        """Check that the control panel backend 'method_name' was called."""
        if self.backend_is_dbus:
            kwargs = {'reply_handler': gui.NO_OP,
                      'error_handler': gui.error_handler}
        self.assertIn(method_name, self.ui.backend._called)
        self.assertEqual(self.ui.backend._called[method_name], (args, kwargs))

    def assert_warning_correct(self, warning, text):
        """Check that 'warning' is visible, showing 'text'."""
        self.assertTrue(warning.get_visible(), 'Must be visible.')
        self.assertEqual(warning.get_label(), gui.WARNING_MARKUP % text)

    def assert_function_decorated(self, decorator, func):
        """Check that 'func' is decorated with 'decorator'."""
        expected = decorator(lambda: None)
        self.assertEqual(expected.func_code, func.im_func.func_code)
