# -*- coding: utf-8 -*-
#
# tests.eventlog.test_zg_listener - test logging ZG events
#
# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Test the event logging from SyncDaemon into Zeitgeist."""

import logging
import os
import shutil
import sys
import unittest
import uuid

from os.path import basename
from twisted.internet import defer
from zeitgeist.datamodel import Interpretation, Manifestation

# python-zeitgeist on lucid has no mimetypes module, and it's usually
# disabled, but fake it for running tests on lucid
try:
    import zeitgeist.mimetypes
    assert(zeitgeist.mimetypes is not None) # make pyflakes happy
except ImportError:

    class FakeMimetypes(object):
        """A fake zg.mimetypes module."""

        def get_interpretation_for_mimetype(self, mimetype):
            """This fake always returns AUDIO."""
            return Interpretation.AUDIO

    sys.modules["zeitgeist.mimetypes"] = FakeMimetypes()

from contrib.testing.testcase import (
        FakeMain, FakeMainTestCase, BaseTwistedTestCase)
from ubuntuone.devtools.handlers import MementoHandler
from ubuntuone.platform.linux import get_udf_path
from ubuntuone.storageprotocol import client, delta
from ubuntuone.storageprotocol.request import ROOT
from ubuntuone.storageprotocol.sharersp import NotifyShareHolder
from ubuntuone.syncdaemon.action_queue import (
        RequestQueue, Upload, MakeFile, MakeDir)
from ubuntuone.eventlog.zg_listener import (
        zglog, ZeitgeistListener, ACTOR_UBUNTUONE,
        EVENT_INTERPRETATION_U1_FOLDER_SHARED,
        EVENT_INTERPRETATION_U1_FOLDER_UNSHARED,
        EVENT_INTERPRETATION_U1_SHARE_ACCEPTED,
        EVENT_INTERPRETATION_U1_SHARE_UNACCEPTED,
        EVENT_INTERPRETATION_U1_CONFLICT_RENAME,
        EVENT_INTERPRETATION_U1_UDF_CREATED,
        EVENT_INTERPRETATION_U1_UDF_DELETED,
        EVENT_INTERPRETATION_U1_UDF_SUBSCRIBED,
        EVENT_INTERPRETATION_U1_UDF_UNSUBSCRIBED,
        MANIFESTATION_U1_CONTACT_DATA_OBJECT, DIRECTORY_MIMETYPE,
        INTERPRETATION_U1_CONTACT, URI_PROTOCOL_U1,
        STORAGE_DELETED, STORAGE_NETWORK, STORAGE_LOCAL)
from ubuntuone.syncdaemon.sync import Sync
from ubuntuone.syncdaemon.volume_manager import Share, Shared, UDF
from tests.syncdaemon.test_action_queue import (
    ConnectedBaseTestCase,
    FakeSemaphore,
)

VOLUME = uuid.UUID('12345678-1234-1234-1234-123456789abc')


class MockLogger(object):
    """A mock logger that stores whatever is logged into it."""

    def __init__(self):
        """Initialize this instance."""
        self.events = []
        self.deferreds = []

    def log(self, event):
        """Log the event."""
        self.events.append(event)
        if self.deferreds:
            self.callback_next_deferred(event)

    def callback_next_deferred(self, event):
        """Pop the next deferred and callback it."""
        d = self.deferreds.pop()
        if d:
            d.callback(event)

def listen_for(event_q, event, callback, count=1, collect=False):
    """Setup a EQ listener for the specified event."""
    class Listener(object):
        """A basic listener to handle the pushed event."""

        def __init__(self):
            self.hits = 0
            self.events = []

        def _handle_event(self, *args, **kwargs):
            self.hits += 1
            if collect:
                self.events.append((args, kwargs))
            if self.hits == count:
                event_q.unsubscribe(self)
                if collect:
                    callback(self.events)
                elif kwargs:
                    callback((args, kwargs))
                else:
                    callback(args)

    listener = Listener()
    setattr(listener, 'handle_'+event, listener._handle_event)
    event_q.subscribe(listener)
    return listener


class ZeitgeistListenerTestCase(FakeMainTestCase):
    """Tests for ZeitgeistListener."""

    def setUp(self):
        """Initialize this instance."""
        super(ZeitgeistListenerTestCase, self).setUp()
        self.patch(zglog, "ZeitgeistLogger", MockLogger)
        self.listener = ZeitgeistListener(self.fs, self.vm)
        self.event_q.subscribe(self.listener)
        self.patch(self.event_q, "ignored_base_exception", RuntimeError)

    def _listen_for(self, *args, **kwargs):
        return listen_for(self.main.event_q, *args, **kwargs)


class ZeitgeistSharesTestCase(ZeitgeistListenerTestCase):
    """Tests for all Share-related zeitgeist events."""

    @defer.inlineCallbacks
    def test_share_created_with_username_is_logged(self):
        """A ShareCreated event is logged."""
        fake_username = "fake user"
        path = os.path.join(self.vm.root.path, 'shared_path')
        sample_node_id = "node id"
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, sample_node_id)

        def fake_create_share(node_id, user, name, access_level, marker, path):
            """Fake the creation of the share on the server."""
            self.assertIn(marker, self.vm.marker_share_map)
            share_id = self.fs.get_by_mdid(marker).share_id
            self.main.event_q.push('AQ_CREATE_SHARE_OK',
                                   share_id=share_id,
                                   marker=marker)

        d = defer.Deferred()
        self._listen_for('AQ_CREATE_SHARE_OK', d.callback, 1, collect=True)
        self.patch(self.main.action_q, "create_share", fake_create_share)
        self.vm.create_share(path, fake_username, 'shared_name', 'View')

        yield d

        self.assert_folder_shared_is_logged(path, fake_username)

    def test_share_created_with_email_is_logged(self):
        """A ShareCreated event is logged."""
        fake_username = "fakeuser@somewhere.com"
        path = os.path.join(self.vm.root.path, 'shared_path')
        sample_node_id = "node id"
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, sample_node_id)

        def fake_create_share(node_id, user, name, access_level, marker, path):
            """Fake the creation of the share on the server."""
            self.assertIn(marker, self.vm.marker_share_map)
            self.main.event_q.push('AQ_SHARE_INVITATION_SENT',
                                   marker=marker)

        self.patch(self.main.action_q, "create_share", fake_create_share)
        self.vm.create_share(path, fake_username, 'shared_name', 'View')

        self.assert_folder_shared_is_logged(path, fake_username)

    def assert_folder_shared_is_logged(self, path, fake_username):
        """Assert that the FolderShared event was logged."""

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]
        foldername = basename(path)

        self.assertEqual(event.interpretation,
                          EVENT_INTERPRETATION_U1_FOLDER_SHARED)
        self.assertEqual(event.manifestation,
                          Manifestation.USER_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        folder = event.subjects[0]
        self.assertTrue(folder.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(folder.manifestation,
                          Manifestation.REMOTE_DATA_OBJECT)
        self.assertTrue(folder.origin.endswith(path))
        self.assertEqual(folder.text, foldername)
        self.assertEqual(folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(folder.storage, STORAGE_NETWORK)

        other_user = event.subjects[1]
        self.assertEqual(other_user.uri, "mailto:" + fake_username)
        self.assertEqual(other_user.interpretation, INTERPRETATION_U1_CONTACT)
        self.assertEqual(other_user.manifestation,
                          MANIFESTATION_U1_CONTACT_DATA_OBJECT)
        self.assertEqual(other_user.text, fake_username)


    @defer.inlineCallbacks
    def test_share_deleted_is_logged(self):
        """Test VolumeManager.delete_share."""
        sample_share_id = "share id"
        sample_node_id = "node id"
        fake_username = "fake user"
        folder_name = "shared_path"
        path = os.path.join(self.vm.root.path, folder_name)
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, sample_node_id)
        share = Shared(path=path, volume_id=sample_share_id,
                       node_id=sample_node_id, other_username=fake_username)
        yield self.vm.add_shared(share)

        def fake_delete_share(share_id):
            """Fake delete_share."""
            self.assertEqual(share_id, share.volume_id)
            self.main.event_q.push('AQ_DELETE_SHARE_OK', share_id=share_id)

        self.patch(self.main.action_q, 'delete_share', fake_delete_share)
        d = defer.Deferred()
        self._listen_for('VM_SHARE_DELETED', d.callback, 1, collect=True)
        self.vm.delete_share(share.volume_id)
        yield d

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          EVENT_INTERPRETATION_U1_FOLDER_UNSHARED)
        self.assertEqual(event.manifestation,
                          Manifestation.USER_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        folder = event.subjects[0]
        self.assertTrue(folder.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(folder.manifestation,
                          Manifestation.REMOTE_DATA_OBJECT)
        self.assertTrue(folder.origin.endswith(path))
        self.assertEqual(folder.text, folder_name)
        self.assertEqual(folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(folder.storage, STORAGE_NETWORK)

        other_user = event.subjects[1]
        self.assertEqual(other_user.uri, "mailto:" + fake_username)
        self.assertEqual(other_user.interpretation, INTERPRETATION_U1_CONTACT)
        self.assertEqual(other_user.manifestation,
                          MANIFESTATION_U1_CONTACT_DATA_OBJECT)
        self.assertEqual(other_user.text, fake_username)

    @defer.inlineCallbacks
    def test_share_accepted_is_logged(self):
        """Test that an accepted share event is logged."""
        # initialize the the root
        self.vm._got_root('root_uuid')
        fake_username = "fake user"
        folder_name = "shared_path"
        path = os.path.join(self.vm.root.path, folder_name)
        self.main.fs.create(path, "")
        share_path = os.path.join(self.shares_dir, folder_name)
        share = Share(path=share_path, volume_id='volume_id', node_id="node_id",
                      other_username=fake_username)
        yield self.vm.add_share(share)

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          EVENT_INTERPRETATION_U1_SHARE_ACCEPTED)
        self.assertEqual(event.manifestation,
                          Manifestation.USER_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        folder = event.subjects[0]
        self.assertTrue(folder.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(folder.manifestation,
                          Manifestation.REMOTE_DATA_OBJECT)
        self.assertTrue(folder.origin.endswith(share_path))
        self.assertEqual(folder.text, folder_name)
        self.assertEqual(folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(folder.storage, STORAGE_NETWORK)

        other_user = event.subjects[1]
        self.assertEqual(other_user.uri, "mailto:" + fake_username)
        self.assertEqual(other_user.interpretation, INTERPRETATION_U1_CONTACT)
        self.assertEqual(other_user.manifestation,
                          MANIFESTATION_U1_CONTACT_DATA_OBJECT)
        self.assertEqual(other_user.text, fake_username)


    @defer.inlineCallbacks
    def test_share_unaccepted_is_logged(self):
        """Test that an unaccepted share event is logged."""
        fake_username = "fake user"
        folder_name = u"share"
        d = defer.Deferred()

        share_path = os.path.join(self.main.shares_dir, folder_name)
        holder = NotifyShareHolder.from_params(uuid.uuid4(),
                                                     uuid.uuid4(),
                                                     u'fake_share',
                                                     fake_username,
                                                     u'visible_name', 'Read')

        share = Share.from_notify_holder(holder, share_path)
        yield self.main.vm.add_share(share)
        self._listen_for('VM_VOLUME_DELETED', d.callback, 1, collect=True)
        self.main.event_q.push('SV_SHARE_DELETED', share_id=holder.share_id)
        yield d

        self.assertEqual(len(self.listener.zg.events), 2)
        event = self.listener.zg.events[1]

        self.assertEqual(event.interpretation,
                          EVENT_INTERPRETATION_U1_SHARE_UNACCEPTED)
        self.assertEqual(event.manifestation,
                          Manifestation.USER_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        folder = event.subjects[0]
        self.assertTrue(folder.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(folder.manifestation,
                          Manifestation.REMOTE_DATA_OBJECT)
        self.assertTrue(folder.origin.endswith(share_path))
        self.assertEqual(folder.text, folder_name)
        self.assertEqual(folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(folder.storage, STORAGE_NETWORK)

        other_user = event.subjects[1]
        self.assertEqual(other_user.uri, "mailto:" + fake_username)
        self.assertEqual(other_user.interpretation, INTERPRETATION_U1_CONTACT)
        self.assertEqual(other_user.manifestation,
                          MANIFESTATION_U1_CONTACT_DATA_OBJECT)
        self.assertEqual(other_user.text, fake_username)


class ZeitgeistUDFsTestCase(ZeitgeistListenerTestCase):
    """Tests for all UDFs-related zeitgeist events."""

    def setUp(self):
        """Initialize this test instance."""
        super(ZeitgeistUDFsTestCase, self).setUp()
        self.home_dir = self.mktemp('ubuntuonehacker')
        self._old_home = os.environ['HOME']
        os.environ['HOME'] = self.home_dir

    def tearDown(self):
        """Finalize this test instance."""
        self.rmtree(self.home_dir)
        os.environ['HOME'] = self._old_home
        super(ZeitgeistUDFsTestCase, self).tearDown()

    def _create_udf(self, id, node_id, suggested_path, subscribed=True):
        """Create an UDF and returns it and the volume."""
        path = get_udf_path(suggested_path)
        # make sure suggested_path is unicode
        if isinstance(suggested_path, str):
            suggested_path = suggested_path.decode('utf-8')
        udf = UDF(str(id), str(node_id), suggested_path, path, subscribed)
        return udf

    @defer.inlineCallbacks
    def test_udf_create_is_logged(self):
        """Test for Folders.create."""
        folder_name = u'ñoño'.encode('utf-8')
        path = os.path.join(self.home_dir, folder_name)
        id = uuid.uuid4()
        node_id = uuid.uuid4()

        def create_udf(path, name, marker):
            """Fake create_udf."""
            # check that the marker is the full path to the udf
            expanded_path = os.path.expanduser(path.encode('utf-8'))
            udf_path = os.path.join(expanded_path, name.encode('utf-8'))
            if str(marker) != udf_path:
                d.errback(ValueError("marker != path - "
                                   "marker: %r path: %r" % (marker, udf_path)))
            self.main.event_q.push("AQ_CREATE_UDF_OK", **dict(volume_id=id,
                                                       node_id=node_id,
                                                       marker=marker))

        self.patch(self.main.action_q, "create_udf", create_udf)

        d = defer.Deferred()
        self.listener.zg.deferreds.append(d)
        self.vm.create_udf(path)
        yield d

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          EVENT_INTERPRETATION_U1_UDF_CREATED)
        self.assertEqual(event.manifestation,
                          Manifestation.USER_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        folder = event.subjects[0]
        self.assertTrue(folder.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(folder.manifestation,
                          Manifestation.REMOTE_DATA_OBJECT)
        self.assertTrue(folder.origin.endswith(path))
        self.assertEqual(folder.text, folder_name)
        self.assertEqual(folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(folder.storage, STORAGE_NETWORK)

    @defer.inlineCallbacks
    def test_udf_delete_is_logged(self):
        """Test for Folders.delete."""
        id = uuid.uuid4()
        node_id = uuid.uuid4()
        folder_name = u'ñoño'.encode('utf-8')
        path = os.path.join(self.home_dir, folder_name)

        d = defer.Deferred()
        self.listener.zg.deferreds.append(d)

        def create_udf(path, name, marker):
            """Fake create_udf."""
            # check that the marker is the full path to the udf
            expanded_path = os.path.expanduser(path.encode('utf-8'))
            udf_path = os.path.join(expanded_path, name.encode('utf-8'))
            if str(marker) != udf_path:
                d.errback(ValueError("marker != path - "
                                   "marker: %r path: %r" % (marker, udf_path)))
            self.main.event_q.push("AQ_CREATE_UDF_OK", **dict(volume_id=id,
                                                       node_id=node_id,
                                                       marker=marker))

        self.patch(self.main.action_q, "create_udf", create_udf)

        self.vm.create_udf(path)
        yield d

        def delete_volume(volume_id, path):
            """Fake delete_volume."""
            self.main.event_q.push("AQ_DELETE_VOLUME_OK", volume_id=id)

        self.patch(self.main.action_q, "delete_volume", delete_volume)

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        d2 = defer.Deferred()
        self.listener.zg.deferreds.append(d2)
        self.vm.delete_volume(str(id))
        yield d2

        self.assertEqual(len(self.listener.zg.events), 2)
        event = self.listener.zg.events[1]

        self.assertEqual(event.interpretation,
                          EVENT_INTERPRETATION_U1_UDF_DELETED)
        self.assertEqual(event.manifestation,
                          Manifestation.USER_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        folder = event.subjects[0]
        self.assertTrue(folder.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(folder.manifestation,
                          Manifestation.DELETED_RESOURCE)
        self.assertTrue(folder.origin.endswith(path))
        self.assertEqual(folder.text, folder_name)
        self.assertEqual(folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(folder.storage, STORAGE_DELETED)

    @defer.inlineCallbacks
    def test_udf_subscribe_is_logged(self):
        """Test for Folders.subscribe."""
        folder_name = u"ñoño"
        suggested_path = u'~/' + folder_name
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=False)
        yield self.main.vm.add_udf(udf)
        d = defer.Deferred()
        self.listener.zg.deferreds.append(d)
        self.vm.subscribe_udf(udf.volume_id)
        yield d

        self.assertEqual(len(self.listener.zg.events), 2)
        event = self.listener.zg.events[1]

        self.assertEqual(event.interpretation,
                          EVENT_INTERPRETATION_U1_UDF_SUBSCRIBED)
        self.assertEqual(event.manifestation,
                          Manifestation.USER_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        folder = event.subjects[0]
        self.assertTrue(folder.uri.endswith(udf.path))
        self.assertEqual(folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(folder.manifestation,
                          Manifestation.FILE_DATA_OBJECT)
        self.assertTrue(folder.origin.startswith(URI_PROTOCOL_U1))
        self.assertEqual(folder.text, folder_name)
        self.assertEqual(folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(folder.storage, STORAGE_LOCAL)

    @defer.inlineCallbacks
    def test_udf_unsubscribe_is_logged(self):
        """Test for Folders.unsubscribe."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=True)
        folder_name = basename(udf.path)
        yield self.main.vm.add_udf(udf)
        d = defer.Deferred()
        self._listen_for('VM_UDF_UNSUBSCRIBED', d.callback, 1, collect=True)
        self.vm.unsubscribe_udf(udf.volume_id)
        yield d

        self.assertEqual(len(self.listener.zg.events), 2)
        event = self.listener.zg.events[1]

        self.assertEqual(event.interpretation,
                          EVENT_INTERPRETATION_U1_UDF_UNSUBSCRIBED)
        self.assertEqual(event.manifestation,
                          Manifestation.USER_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        folder = event.subjects[0]
        self.assertTrue(folder.uri.endswith(udf.path))
        self.assertEqual(folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(folder.manifestation,
                          Manifestation.DELETED_RESOURCE)
        self.assertTrue(folder.origin.startswith(URI_PROTOCOL_U1))
        self.assertEqual(folder.text, folder_name)
        self.assertEqual(folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(folder.storage, STORAGE_DELETED)


class ZeitgeistRemoteFileSyncTestCase(ConnectedBaseTestCase):
    """File sync events are logged into Zeitgeist."""

    def setUp(self):
        """Initialize this test instance."""
        ConnectedBaseTestCase.setUp(self)
        self.rq = request_queue = RequestQueue(action_queue=self.action_queue)
        self.rq.transfers_semaphore = FakeSemaphore()

        class MyUpload(Upload):
            """Just to allow monkeypatching."""

        self.share_id = ""
        self.command = MyUpload(request_queue, share_id=self.share_id,
                                node_id='a_node_id', previous_hash='prev_hash',
                                hash='yadda', crc32=0, size=0, path='path',
                                fileobj_factory=lambda: None,
                                tempfile_factory=lambda: None)
        self.command.make_logger()
        self.fsm = self.action_queue.main.fs
        self.vm = self.action_queue.main.vm
        self.patch(zglog, "ZeitgeistLogger", MockLogger)
        self.listener = ZeitgeistListener(self.fsm, self.vm)
        self.action_queue.event_queue.subscribe(self.listener)
        self.root_id = "roootid"
        self.sync = Sync(main=self.main)

    def tearDown(self):
        """Finalize this test instance."""
        ConnectedBaseTestCase.tearDown(self)

    def test_syncdaemon_creates_file_on_server_is_logged(self):
        """Files created by SyncDaemon on the server are logged."""
        filename = "filename.mp3"
        path = os.path.join(self.vm.root.path, filename)
        self.fsm.create(path, "")
        self.fsm.set_node_id(path, "a_node_id")

        request = client.MakeFile(self.action_queue.client, self.share_id,
                                  'parent', filename)
        request.new_id = 'a_node_id'
        request.new_generation = 13

        # create a command and trigger it success
        cmd = MakeFile(self.rq, self.share_id, 'parent', filename,
                       'marker', path)
        cmd.handle_success(request)

        # create a request and fill it with succesful information
        request = client.PutContent(self.action_queue.client, self.share_id,
                                    'node', 'prvhash', 'newhash', 'crc32',
                                    'size', 'deflated', 'fd')
        request.new_generation = 13

        # trigger success in the command
        self.command.handle_success(request)

        # check for successful event
        kwargs = dict(share_id=self.command.share_id, node_id='a_node_id',
                      hash='yadda', new_generation=13)

        info = dict(marker='marker', new_id='a_node_id', new_generation=13,
                    volume_id=self.share_id)
        events = [
            ('AQ_FILE_NEW_OK', info),
            ('AQ_UPLOAD_FINISHED', kwargs),
        ]
        self.assertEqual(events, self.command.action_queue.event_queue.events)

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          Interpretation.CREATE_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.SCHEDULED_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        remote_file = event.subjects[0]
        self.assertTrue(remote_file.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(remote_file.interpretation, Interpretation.AUDIO)
        self.assertEqual(remote_file.manifestation,
                          Manifestation.REMOTE_DATA_OBJECT)
        self.assertTrue(remote_file.origin.endswith(filename))
        self.assertEqual(remote_file.text, filename)
        self.assertEqual(remote_file.mimetype, "audio/mpeg")
        self.assertEqual(remote_file.storage, STORAGE_NETWORK)

    def test_syncdaemon_creates_dir_on_server_is_logged(self):
        """Dirs created by SyncDaemon on the server are logged."""
        dirname = "dirname"
        path = os.path.join(self.vm.root.path, dirname)
        self.fsm.create(path, "")
        self.fsm.set_node_id(path, "a_node_id")

        request = client.MakeDir(self.action_queue.client, self.share_id,
                                  'parent', dirname)
        request.new_id = 'a_node_id'
        request.new_generation = 13

        # create a command and trigger it success
        cmd = MakeDir(self.rq, self.share_id, 'parent',
                      dirname, 'marker', path)
        cmd.handle_success(request)

        # check for successful event
        info = dict(marker='marker', new_id='a_node_id', new_generation=13,
                    volume_id=self.share_id)
        events = [('AQ_DIR_NEW_OK', info)]
        self.assertEqual(events, self.command.action_queue.event_queue.events)

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          Interpretation.CREATE_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.SCHEDULED_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        remote_folder = event.subjects[0]
        self.assertTrue(remote_folder.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(remote_folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(remote_folder.manifestation,
                          Manifestation.REMOTE_DATA_OBJECT)
        self.assertTrue(remote_folder.origin.endswith(dirname))
        self.assertEqual(remote_folder.text, dirname)
        self.assertEqual(remote_folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(remote_folder.storage, STORAGE_NETWORK)

    def test_syncdaemon_modifies_on_server_is_logged(self):
        """Files modified by SyncDaemon on the server are logged."""
        filename = "filename.mp3"
        path = os.path.join(self.vm.root.path, filename)
        self.fsm.create(path, "")
        self.fsm.set_node_id(path, "a_node_id")

        # create a request and fill it with succesful information
        request = client.PutContent(self.action_queue.client, self.share_id,
                                    'node', 'prvhash', 'newhash', 'crc32',
                                    'size', 'deflated', 'fd')
        request.new_generation = 13

        # trigger success in the command
        self.command.handle_success(request)

        # check for successful event
        kwargs = dict(share_id=self.command.share_id, node_id='a_node_id',
                      hash='yadda', new_generation=13)

        events = [('AQ_UPLOAD_FINISHED', kwargs)]
        self.assertEqual(events, self.command.action_queue.event_queue.events)

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          Interpretation.MODIFY_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.SCHEDULED_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        remote_file = event.subjects[0]
        self.assertTrue(remote_file.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(remote_file.interpretation, Interpretation.AUDIO)
        self.assertEqual(remote_file.manifestation,
                          Manifestation.REMOTE_DATA_OBJECT)
        self.assertTrue(remote_file.origin.endswith(filename))
        self.assertEqual(remote_file.text, filename)
        self.assertEqual(remote_file.mimetype, "audio/mpeg")
        self.assertEqual(remote_file.storage, STORAGE_NETWORK)

    @defer.inlineCallbacks
    def test_syncdaemon_deletes_file_on_server_is_logged(self):
        """Files deleted by SD on the server are logged."""
        file_name = "filename.mp3"
        d = defer.Deferred()
        listen_for(self.main.event_q, 'AQ_UNLINK_OK', d.callback)

        path = os.path.join(self.main.vm.root.path, "filename.mp3")
        self.main.event_q.push("AQ_UNLINK_OK", share_id="",
                               parent_id="parent_id",
                               node_id="node_id", new_generation=13,
                               was_dir=False, old_path=path)
        yield d

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          Interpretation.DELETE_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.SCHEDULED_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        remote_file = event.subjects[0]
        self.assertTrue(remote_file.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(remote_file.interpretation, Interpretation.AUDIO)
        self.assertEqual(remote_file.manifestation,
                          Manifestation.DELETED_RESOURCE)
        self.assertTrue(remote_file.origin.endswith(file_name))
        self.assertEqual(remote_file.text, file_name)
        self.assertEqual(remote_file.mimetype, "audio/mpeg")
        self.assertEqual(remote_file.storage, STORAGE_DELETED)

    @defer.inlineCallbacks
    def test_syncdaemon_deletes_dir_on_server_is_logged(self):
        """Directories deleted by SD on the server are logged."""
        folder_name = "folder name"
        d = defer.Deferred()
        listen_for(self.main.event_q, 'AQ_UNLINK_OK', d.callback)

        path = os.path.join(self.main.vm.root.path, "folder name")
        self.main.event_q.push("AQ_UNLINK_OK", share_id="",
                               parent_id="parent_id",
                               node_id="node_id", new_generation=13,
                               was_dir=True, old_path=path)
        yield d

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          Interpretation.DELETE_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.SCHEDULED_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        remote_folder = event.subjects[0]
        self.assertTrue(remote_folder.uri.startswith(URI_PROTOCOL_U1))
        self.assertEqual(remote_folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(remote_folder.manifestation,
                          Manifestation.DELETED_RESOURCE)
        self.assertTrue(remote_folder.origin.endswith(folder_name))
        self.assertEqual(remote_folder.text, folder_name)
        self.assertEqual(remote_folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(remote_folder.storage, STORAGE_DELETED)


class ZeitgeistLocalFileSyncTestCase(BaseTwistedTestCase):
    """Zeitgeist events coming from the server."""
    timeout = 5

    def setUp(self):
        """Initialize this instance."""
        BaseTwistedTestCase.setUp(self)
        self.root = self.mktemp('root')
        self.shares = self.mktemp('shares')
        self.data = self.mktemp('data')
        self.partials_dir = self.mktemp('partials_dir')
        self.handler = MementoHandler()
        self.handler.setLevel(logging.ERROR)
        FakeMain._sync_class = Sync
        self.main = FakeMain(root_dir=self.root, shares_dir=self.shares,
                             data_dir=self.data,
                             partials_dir=self.partials_dir)
        self._logger = logging.getLogger('ubuntuone.SyncDaemon')
        self._logger.addHandler(self.handler)

        self.root_id = root_id = "roootid"
        self.main.vm._got_root(root_id)
        self.filemp3delta = delta.FileInfoDelta(
            generation=5, is_live=True, file_type=delta.FILE,
            parent_id=self.root_id, share_id=ROOT, node_id=uuid.uuid4(),
            name=u"fileñ.mp3", is_public=False, content_hash="hash",
            crc32=1, size=10, last_modified=0)

        self.dirdelta = delta.FileInfoDelta(
            generation=6, is_live=True, file_type=delta.DIRECTORY,
            parent_id=root_id, share_id=ROOT, node_id=uuid.uuid4(),
            name=u"directory_ñ", is_public=False, content_hash="hash",
            crc32=1, size=10, last_modified=0)

        self.patch(zglog, "ZeitgeistLogger", MockLogger)
        self.listener = ZeitgeistListener(self.main.fs, self.main.vm)
        self.main.event_q.subscribe(self.listener)

    def tearDown(self):
        """Clean up this instance."""
        self._logger.removeHandler(self.handler)
        self.main.shutdown()
        FakeMain._sync_class = None
        shutil.rmtree(self.root)
        shutil.rmtree(self.shares)
        shutil.rmtree(self.data)
        for record in self.handler.records:
            exc_info = getattr(record, 'exc_info', None)
            if exc_info is not None:
                raise exc_info[0], exc_info[1], exc_info[2]
        BaseTwistedTestCase.tearDown(self)

    @defer.inlineCallbacks
    def test_syncdaemon_creates_file_locally_is_logged(self):
        """Files created locally by SyncDaemon are logged."""
        file_name = self.filemp3delta.name.encode('utf8')
        d = defer.Deferred()
        d2 = defer.Deferred()
        listen_for(self.main.event_q, 'SV_FILE_NEW', d.callback)
        listen_for(self.main.event_q, 'AQ_DOWNLOAD_FINISHED', d2.callback)

        deltas = [ self.filemp3delta ]
        kwargs = dict(volume_id=ROOT, delta_content=deltas, end_generation=11,
                      full=True, free_bytes=10)
        self.main.sync.handle_AQ_DELTA_OK(**kwargs)

        # check that the file is created
        node = self.main.fs.get_by_node_id(ROOT, self.filemp3delta.node_id)
        self.assertEqual(node.path, file_name)
        self.assertEqual(node.is_dir, False)
        self.assertEqual(node.generation, self.filemp3delta.generation)

        yield d # wait for SV_FILE_NEW

        dlargs = dict(
            share_id=self.filemp3delta.share_id,
            node_id=self.filemp3delta.node_id,
            server_hash="server hash")
        self.main.event_q.push("AQ_DOWNLOAD_FINISHED", **dlargs)

        yield d2 # wait for AQ_DOWNLOAD_FINISHED

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          Interpretation.CREATE_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.WORLD_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        local_file = event.subjects[0]
        self.assertTrue(local_file.uri.endswith(file_name))
        self.assertEqual(local_file.interpretation, Interpretation.AUDIO)
        self.assertEqual(local_file.manifestation,
                          Manifestation.FILE_DATA_OBJECT)
        self.assertTrue(local_file.origin.startswith(URI_PROTOCOL_U1))
        self.assertEqual(local_file.text, file_name)
        self.assertEqual(local_file.mimetype, "audio/mpeg")
        self.assertEqual(local_file.storage, STORAGE_LOCAL)

    @defer.inlineCallbacks
    def test_syncdaemon_creates_dir_locally_is_logged(self):
        """Dirs created locally by SyncDaemon are logged."""
        folder_name = self.dirdelta.name.encode('utf8')
        d = defer.Deferred()
        listen_for(self.main.event_q, 'SV_DIR_NEW', d.callback)

        deltas = [ self.dirdelta ]
        kwargs = dict(volume_id=ROOT, delta_content=deltas, end_generation=11,
                      full=True, free_bytes=10)
        self.main.sync.handle_AQ_DELTA_OK(**kwargs)

        # check that the dir is created
        node = self.main.fs.get_by_node_id(ROOT, self.dirdelta.node_id)
        self.assertEqual(node.path, folder_name)
        self.assertEqual(node.is_dir, True)
        self.assertEqual(node.generation, self.dirdelta.generation)

        yield d # wait for SV_DIR_NEW

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          Interpretation.CREATE_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.WORLD_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        local_folder = event.subjects[0]
        self.assertTrue(local_folder.uri.endswith(folder_name))
        self.assertEqual(local_folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(local_folder.manifestation,
                          Manifestation.FILE_DATA_OBJECT)
        self.assertTrue(local_folder.origin.startswith(URI_PROTOCOL_U1))
        self.assertEqual(local_folder.text, folder_name)
        self.assertEqual(local_folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(local_folder.storage, STORAGE_LOCAL)

    @defer.inlineCallbacks
    def test_syncdaemon_modifies_locally_is_logged(self):
        """Files modified locally by SyncDaemon are logged."""
        file_name = self.filemp3delta.name.encode('utf8')
        d = defer.Deferred()
        d2 = defer.Deferred()
        listen_for(self.main.event_q, 'SV_FILE_NEW', d.callback)
        listen_for(self.main.event_q, 'AQ_DOWNLOAD_FINISHED', d2.callback)

        deltas = [ self.filemp3delta ]
        kwargs = dict(volume_id=ROOT, delta_content=deltas, end_generation=11,
                      full=True, free_bytes=10)
        self.main.sync.handle_AQ_DELTA_OK(**kwargs)

        # check that the file is modified
        node = self.main.fs.get_by_node_id(ROOT, self.filemp3delta.node_id)
        self.assertEqual(node.path, file_name)
        self.assertEqual(node.is_dir, False)
        self.assertEqual(node.generation, self.filemp3delta.generation)

        yield d # wait for SV_FILE_NEW

        # remove from the recent list
        local_file_id = (self.filemp3delta.share_id, self.filemp3delta.node_id)
        self.listener.newly_created_local_files.remove(local_file_id)

        dlargs = dict(
            share_id=self.filemp3delta.share_id,
            node_id=self.filemp3delta.node_id,
            server_hash="server hash")
        self.main.event_q.push("AQ_DOWNLOAD_FINISHED", **dlargs)

        yield d2 # wait for AQ_DOWNLOAD_FINISHED

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          Interpretation.MODIFY_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.WORLD_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        local_file = event.subjects[0]
        self.assertTrue(local_file.uri.endswith(file_name))
        self.assertEqual(local_file.interpretation, Interpretation.AUDIO)
        self.assertEqual(local_file.manifestation,
                          Manifestation.FILE_DATA_OBJECT)
        self.assertTrue(local_file.origin.startswith(URI_PROTOCOL_U1))
        self.assertEqual(local_file.text, file_name)
        self.assertEqual(local_file.mimetype, "audio/mpeg")
        self.assertEqual(local_file.storage, STORAGE_LOCAL)

    @defer.inlineCallbacks
    def test_syncdaemon_deletes_file_locally_is_logged(self):
        """Files deleted locally by SyncDaemon are logged."""
        file_name = self.filemp3delta.name.encode("utf-8")
        d = defer.Deferred()
        listen_for(self.main.event_q, 'SV_FILE_DELETED', d.callback)

        filename = self.filemp3delta.name.encode("utf-8")
        path = os.path.join(self.main.vm.root.path, filename)
        self.main.event_q.push("SV_FILE_DELETED", volume_id="",
                               node_id="node_id", was_dir=False,
                               old_path=path)
        yield d

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          Interpretation.DELETE_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.WORLD_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        local_file = event.subjects[0]
        self.assertTrue(local_file.uri.endswith(file_name))
        self.assertEqual(local_file.interpretation, Interpretation.AUDIO)
        self.assertEqual(local_file.manifestation,
                          Manifestation.DELETED_RESOURCE)
        self.assertTrue(local_file.origin.startswith(URI_PROTOCOL_U1))
        self.assertEqual(local_file.text, file_name)
        self.assertEqual(local_file.mimetype, "audio/mpeg")
        self.assertEqual(local_file.storage, STORAGE_DELETED)

    @defer.inlineCallbacks
    def test_syncdaemon_deletes_dir_locally_is_logged(self):
        """Dirs deleted locally by SyncDaemon are logged."""
        folder_name = "folder name"
        d = defer.Deferred()
        listen_for(self.main.event_q, 'SV_FILE_DELETED', d.callback)

        path = os.path.join(self.main.vm.root.path, "folder name")
        self.main.event_q.push("SV_FILE_DELETED", volume_id="",
                               node_id="node_id", was_dir=True,
                               old_path=path)

        yield d

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          Interpretation.DELETE_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.WORLD_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        local_folder = event.subjects[0]
        self.assertTrue(local_folder.uri.endswith(folder_name))
        self.assertEqual(local_folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(local_folder.manifestation,
                          Manifestation.DELETED_RESOURCE)
        self.assertTrue(local_folder.origin.startswith(URI_PROTOCOL_U1))
        self.assertEqual(local_folder.text, folder_name)
        self.assertEqual(local_folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(local_folder.storage, STORAGE_DELETED)

    @defer.inlineCallbacks
    def test_file_sync_conflict_is_logged(self):
        """Files renamed because of conflict are logged."""
        file_name = "sample.mp3"
        d = defer.Deferred()
        listen_for(self.main.event_q, 'FSM_FILE_CONFLICT', d.callback)

        testfile = os.path.join(self.main.vm.root.path, file_name)
        mdid = self.main.fs.create(testfile, "")
        self.main.fs.set_node_id(testfile, "uuid")
        with open(testfile, "w") as fh:
            fh.write("this is music!")

        self.main.fs.move_to_conflict(mdid)

        yield d

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          EVENT_INTERPRETATION_U1_CONFLICT_RENAME)
        self.assertEqual(event.manifestation,
                          Manifestation.WORLD_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        local_file = event.subjects[0]
        new_name = testfile + self.main.fs.CONFLICT_SUFFIX
        self.assertTrue(local_file.uri.endswith(new_name))
        self.assertEqual(local_file.interpretation, Interpretation.AUDIO)
        self.assertEqual(local_file.manifestation,
                          Manifestation.FILE_DATA_OBJECT)
        self.assertTrue(local_file.origin.endswith(testfile))
        self.assertEqual(local_file.text,
                         file_name + self.main.fs.CONFLICT_SUFFIX)
        self.assertEqual(local_file.mimetype, "audio/mpeg")
        self.assertEqual(local_file.storage, STORAGE_LOCAL)

    @defer.inlineCallbacks
    def test_dir_sync_conflict_is_logged(self):
        """Dirs renamed because of conflict are logged."""
        folder_name = "sampledir"
        d = defer.Deferred()
        listen_for(self.main.event_q, 'FSM_DIR_CONFLICT', d.callback)

        testdir = os.path.join(self.main.vm.root.path, folder_name)
        mdid = self.main.fs.create(testdir, "", is_dir=True)
        self.main.fs.set_node_id(testdir, "uuid")
        os.mkdir(testdir)

        self.main.fs.move_to_conflict(mdid)

        yield d

        self.assertEqual(len(self.listener.zg.events), 1)
        event = self.listener.zg.events[0]

        self.assertEqual(event.interpretation,
                          EVENT_INTERPRETATION_U1_CONFLICT_RENAME)
        self.assertEqual(event.manifestation,
                          Manifestation.WORLD_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        local_folder = event.subjects[0]
        new_name = testdir + self.main.fs.CONFLICT_SUFFIX
        self.assertTrue(local_folder.uri.endswith(new_name))
        self.assertEqual(local_folder.interpretation, Interpretation.FOLDER)
        self.assertEqual(local_folder.manifestation,
                          Manifestation.FILE_DATA_OBJECT)
        self.assertTrue(local_folder.origin.endswith(testdir))
        self.assertEqual(local_folder.text,
                         folder_name + self.main.fs.CONFLICT_SUFFIX)
        self.assertEqual(local_folder.mimetype, DIRECTORY_MIMETYPE)
        self.assertEqual(local_folder.storage, STORAGE_LOCAL)

class ZeitgeistPublicFilesTestCase(ZeitgeistListenerTestCase):
    """Public files events are logged into Zeitgeist."""

    @defer.inlineCallbacks
    def test_publish_url_is_logged(self):
        """Publishing a file with a url is logged."""
        share_id = "share"
        node_id = "node_id"
        is_public = True
        public_url = 'http://example.com/foo.mp3'

        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share',
                                           other_username='other username'))
        path = os.path.join(share_path, "foo.mp3")
        self.main.fs.create(path, str(share_id))
        self.main.fs.set_node_id(path, str(node_id))

        d = defer.Deferred()
        self._listen_for('AQ_CHANGE_PUBLIC_ACCESS_OK', d.callback)
        self.main.event_q.push('AQ_CHANGE_PUBLIC_ACCESS_OK',
                               share_id=share_id, node_id=node_id,
                               is_public=is_public, public_url=public_url)
        yield d

        self.assertEqual(len(self.listener.zg.events), 2)
        event = self.listener.zg.events[1]

        self.assertEqual(event.interpretation,
                          Interpretation.CREATE_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.USER_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        public_file = event.subjects[0]
        self.assertEqual(public_file.uri, public_url)
        self.assertEqual(public_file.interpretation, Interpretation.AUDIO)
        self.assertEqual(public_file.manifestation,
                          Manifestation.REMOTE_DATA_OBJECT)
        self.assertTrue(public_file.origin.endswith(node_id))
        self.assertEqual(public_file.text, public_url)
        self.assertEqual(public_file.mimetype, "audio/mpeg")
        self.assertEqual(public_file.storage, STORAGE_NETWORK)

    @defer.inlineCallbacks
    def test_unpublish_url_is_logged(self):
        """Unpublishing a file with a url is logged."""
        share_id = "share"
        node_id = "node_id"
        is_public = False
        public_url = 'http://example.com/foo.mp3'

        share_path = os.path.join(self.shares_dir, 'share')
        yield self.main.vm.add_share(Share(path=share_path, volume_id='share',
                                           other_username='other username'))
        path = os.path.join(share_path, "foo.mp3")
        self.main.fs.create(path, str(share_id))
        self.main.fs.set_node_id(path, str(node_id))

        d = defer.Deferred()
        self._listen_for('AQ_CHANGE_PUBLIC_ACCESS_OK', d.callback)
        self.main.event_q.push('AQ_CHANGE_PUBLIC_ACCESS_OK',
                               share_id=share_id, node_id=node_id,
                               is_public=is_public, public_url=public_url)
        yield d

        self.assertEqual(len(self.listener.zg.events), 2)
        event = self.listener.zg.events[1]

        self.assertEqual(event.interpretation,
                          Interpretation.DELETE_EVENT)
        self.assertEqual(event.manifestation,
                          Manifestation.USER_ACTIVITY)
        self.assertEqual(event.actor, ACTOR_UBUNTUONE)

        public_file = event.subjects[0]
        self.assertEqual(public_file.uri, public_url)
        self.assertEqual(public_file.interpretation, Interpretation.AUDIO)
        self.assertEqual(public_file.manifestation,
                          Manifestation.DELETED_RESOURCE)
        self.assertTrue(public_file.origin.endswith(node_id))
        self.assertEqual(public_file.text, public_url)
        self.assertEqual(public_file.mimetype, "audio/mpeg")
        self.assertEqual(public_file.storage, STORAGE_DELETED)


def test_suite():
    """Collect these tests only on linux."""
    import sys
    if sys.platform == 'linux2':
        tests = unittest.TestLoader().loadTestsFromName(__name__)
    else:
        tests = []
    return tests
