# ubuntuone.syncdaemon.action_queue - Action queue
#
# Author: John Lenton <john.lenton@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
The ActionQueue is where actions to be performed on the server are
queued up and then executed. The idea is that there are two queues,
one for metadata and another for content; the metadata queue has
priority over the content queue.
"""
from collections import deque, defaultdict
from functools import wraps, partial
import itertools
import logging
import os
import random
import tempfile
import traceback
import zlib

from zope.interface import implements
from twisted.internet import reactor, defer
from twisted.names import client as dns_client
from twisted.python.failure import Failure

import uuid
import re
from urllib import urlencode
from urllib2 import urlopen, Request, HTTPError
from twisted.internet import threads

from oauth import oauth
from ubuntuone.storageprotocol.context import get_ssl_context
from ubuntuone.storageprotocol import protocol_pb2, request
from ubuntuone.storageprotocol.client import ThrottlingStorageClient, \
    ThrottlingStorageClientFactory
from ubuntuone.syncdaemon.logger import mklog, TRACE
from ubuntuone.syncdaemon.interfaces import IActionQueue, \
    IMarker

logger = logging.getLogger("ubuntuone.SyncDaemon.ActionQueue")

# I want something which repr() is "---" *without* the quotes :)
UNKNOWN = type('', (), {'__repr__': lambda _: '---'})()

# Regular expression to validate an e-mail address
EREGEX = "^.+\\@(\\[?)[a-zA-Z0-9\\-\\.]+\\.([a-zA-Z]{2,3}|[0-9]{1,3})(\\]?)$"


def passit(func):
    """
    Pass the value on for the next deferred, while calling func with
    it.
    """
    @wraps(func)
    def wrapper(a):
        """
        Do it.
        """
        func(a)
        return a
    return wrapper


class UploadCompressionCancelled(Exception):
    """Compression of a file for upload cancelled."""


class RequestCleanedUp(Exception):
    """
    The request was cancelled by ActionQueue.cleanup()
    """


class NamedTemporaryFile(object):
    """
    Like tempfile.NamedTemporaryFile, but working in 2.5 also WRT the
    delete argument. Actually, one of these NamedTemporaryFile()s is
    the same as a tempfile.NamedTemporaryFile(delete=False) from 2.6.

    Or so the theory goes.
    """
    def __init__(self):
        fileno, self.name = tempfile.mkstemp()
        self._fd = os.fdopen(fileno, 'r+w')

    def __getattr__(self, attr):
        """proxy everything else (other than .name) on to self._fd"""
        return getattr(self._fd, attr)


class MultiProxy(list):
    """
    Proxy many objects of the same kind, like this:

    >>> m = MultiProxy(['foo', 'bar', 'baz'])
    >>> m.upper()
    MultiProxy(['FOO', 'BAR', 'BAZ'])
    """

    def __getattr__(self, attr):
        return MultiProxy(getattr(i, attr) for i in self)

    def __call__(self, *args, **kwargs):
        return MultiProxy(i(*args, **kwargs) for i in self)

    def __repr__(self):
        return 'MultiProxy(%s)' % (super(MultiProxy, self).__repr__(),)


class LoggingStorageClient(ThrottlingStorageClient):
    """ A subclass of StorageClient that adds logging to
    processMessage and sendMessage.
    """

    def __init__(self):
        """ setup logging and create the instance. """
        ThrottlingStorageClient.__init__(self)
        self.log = logging.getLogger('ubuntuone.SyncDaemon.StorageClient')
        # configure the handler level to be < than DEBUG
        self.log.setLevel(TRACE)
        self.log.debug = partial(self.log.log, TRACE)

    def processMessage(self, message):
        """ wrapper that logs the message and result. """
        # don't log the full message if it's of type BYTES
        if message.type == protocol_pb2.Message.BYTES:
            self.log.debug('start - processMessage: id: %s, type: %s',
                           message.id, message.type)
        else:
            self.log.debug('start - processMessage: %s',
                          str(message).replace("\n", " "))
        if message.id in self.requests:
            req = self.requests[message.id]
            req.deferred.addCallbacks(self.log_success, self.log_error)
        result = ThrottlingStorageClient.processMessage(self, message)
        self.log.debug('end - processMessage: id: %s - result: %s',
                       message.id, result)
        return result

    def log_error(self, failure):
        """ logging errback for requests """
        self.log.debug('request error: %s', failure)
        return failure

    def log_success(self, result):
        """ logging callback for requests """
        self.log.debug('request finished: %s', result)
        if getattr(result, '__dict__', None):
            self.log.debug('result.__dict__: %s', result.__dict__)
        return result

    def sendMessage(self, message):
        """ wrapper that logs the message and result. """
        # don't log the full message if it's of type BYTES
        if message.type == protocol_pb2.Message.BYTES:
            self.log.debug('start - sendMessage: id: %s, type: %s',
                           message.id, message.type)
        else:
            self.log.debug('start - sendMessage: %s',
                          str(message).replace("\n", " "))
        result = ThrottlingStorageClient.sendMessage(self, message)
        self.log.debug('end - sendMessage: id: %s', message.id)
        return result


class ActionQueueProtocol(LoggingStorageClient):
    """
    This is the Action Queue version of the StorageClient protocol.
    """
    connection_state = 'disconnected'
    factory = None
    _finished = False

    def connectionMade(self):
        """
        Called when a new connection is made.
        All the state is saved in the factory.
        """
        if self._finished:
            # the factory is no longer interested in what we have to say
            return

        if self.factory.client is not None:
            # this is a connectionMade for a second client while we still have
            # a previous one alive: discard!
            self._finished = True
            logger.debug("discarding client, duplicated connectionMade")
            if self.transport is not None:
                self.transport.loseConnection()
            return

        logger.debug("connection made")
        LoggingStorageClient.connectionMade(self)
        self.connection_state = 'connected'
        # pylint: disable-msg=W0212
        self.set_node_state_callback(self.factory._node_state_callback)
        self.set_share_change_callback(self.factory._share_change_callback)
        self.set_share_answer_callback(self.factory._share_answer_callback)
        self.set_free_space_callback(self.factory._free_space_callback)
        self.set_account_info_callback(self.factory._account_info_callback)
        self.factory.event_queue.push('SYS_CONNECTION_MADE')

    def disconnect(self):
        """
        Close down the sockets
        """
        self._finished = True
        logger.debug("disconnected")
        if self.transport is not None:
            self.transport.loseConnection()
        else:
            # fake the event
            self.factory.connectionLost(self, 'Transport is None')

    def connectionLost(self, reason=None):
        """
        The connection went down, for some reason (which might or
        might not be described in failure).
        """
        logger.warning('connection lost: %s' % reason.getErrorMessage())
        self.factory.connectionLost(self, reason)
        LoggingStorageClient.connectionLost(self, reason)


class Marker(str):
    """
    A uuid4-based marker class
    """
    implements(IMarker)
    def __new__(cls):
        return super(Marker, cls).__new__(cls, uuid.uuid4())


class ZipQueue(object):
    """
    A queue of files to be compressed for upload

    Parts of this were shamelessly copied from
    twisted.internet.defer.DeferredSemaphore

    see bug #373984
    """
    def __init__(self):
        self.waiting = deque()
        self.tokens = self.limit = 10

    def acquire(self):
        """
        return a deferred which fires on token acquisition.
        """
        assert self.tokens >= 0, "Tokens should never be negative"
        d = defer.Deferred()
        if not self.tokens:
            self.waiting.append(d)
        else:
            self.tokens = self.tokens - 1
            d.callback(self)
        return d

    def release(self):
        """
        Release the token.

        Should be called by whoever did the acquire() when the shared
        resource is free.
        """
        assert self.tokens < self.limit, "Too many tokens!"
        self.tokens = self.tokens + 1
        if self.waiting:
            # someone is waiting to acquire token
            self.tokens = self.tokens - 1
            d = self.waiting.popleft()
            d.callback(self)

    def _compress(self, deferred, upload):
        """Compression background task."""
        try:
            fileobj = upload.fileobj_factory()
        except StandardError:
            # presumably the user deleted the file before we got to
            # upload it. Logging a warning just in case.
            upload.log.warn('unable to build fileobj'
                            ' (user deleted the file, maybe?)'
                            ' so cancelling the upload.')
            upload.cancel()
            fileobj = None

        filename = getattr(fileobj, 'name', '<?>')

        try:
            if upload.cancelled:
                raise UploadCompressionCancelled("Cancelled")
            upload.log.debug('compressing', filename)
            # we need to compress the file completely to figure out its
            # compressed size. So streaming is out :(
            if upload.tempfile_factory is None:
                f = NamedTemporaryFile()
            else:
                f = upload.tempfile_factory()
            zipper = zlib.compressobj()
            while not upload.cancelled:
                data = fileobj.read(4096)
                if not data:
                    f.write(zipper.flush())
                    # no flush/sync because we don't need this to persist
                    # on disk; if the machine goes down, we'll lose it
                    # anyway (being in /tmp and all)
                    break
                f.write(zipper.compress(data))
            if upload.cancelled:
                raise UploadCompressionCancelled("Cancelled")
            upload.deflated_size = f.tell()
            # close the compressed file (thus, if you actually want to stream
            # it out, it must have a name so it can be reopnened)
            f.close()
            upload.tempfile = f
        except Exception, e: # pylint: disable-msg=W0703
            reactor.callFromThread(deferred.errback, e)
        else:
            reactor.callFromThread(deferred.callback, True)

    def zip(self, upload):
        """
        Acquire, do the compression in a thread, release.
        """
        d_zip = defer.Deferred()
        d_lck = self.acquire()
        d_lck.addCallback(
            lambda _: reactor.callInThread(self._compress,
                                           d_zip, upload) or d_zip)
        d_lck.addBoth(passit(lambda _: self.release()))

        return d_lck


class RequestQueue(object):
    """
    RequestQueue is a queue that ensures that there is at most one
    request at a time 'on the wire', and that uses the action queue's
    states for its syncrhonization.
    """
    commutative = False

    def __init__(self, name, action_queue):
        super(RequestQueue, self).__init__()
        self.name = name
        self.action_queue = action_queue
        self.waiting = deque()
        self._head = None

    def __len__(self):
        """return the length of the queue"""
        return len(self.waiting)

    def cleanup_and_retry(self):
        """call cleanup_and_retry on the head"""
        if self._head is not None:
            self._head.cleanup_and_retry()

    def schedule_next(self, share_id, node_id):
        """
        Promote the newest command on the given share and node to
        the front of the queue, if it is possible to do so.
        """
        if not self.commutative:
            return
        for cmd in reversed(self.waiting):
            try:
                if cmd.share_id == share_id and cmd.node_id == node_id:
                    break
            except AttributeError:
                # not a command we can reschedule
                pass
        else:
            raise KeyError, "No waiting command for that share and node"
        self.waiting.remove(cmd)
        self.waiting.appendleft(cmd)

    def queue(self, command):
        """
        Add a command to the queue.
        """
        self.waiting.append(command)
        if len(self.waiting) == 1 and not self._head:
            self.action_queue.event_queue.push('SYS_' + self.name
                                               + '_WAITING')
        if command.is_runnable():
            self.check_conditions()

    def queue_top(self, command):
        """
        Add a command to the head of the queue.

        This is *only* for using during cleanup
        """
        self.waiting.appendleft(command)
        self.action_queue.event_queue.push('SYS_' + self.name + '_WAITING')

    def done(self):
        """
        A command must call done() on the queue when it has finished.
        """
        self._head = None
        # pushing both events makes the queue 'tick'
        self.action_queue.event_queue.push('SYS_' + self.name + '_DONE')
        if self.waiting:
            self.action_queue.event_queue.push('SYS_' + self.name + '_WAITING')

    def _is_empty_or_runnable_waiting(self):
        """Returns True if there is an immediately runnable command in the
        queue, or if the queue is empty.
        """
        if not self.waiting:
            return True
        if self.commutative:
            for command in self.waiting:
                if command.is_runnable():
                    return True
            return False
        else:
            return self.waiting[0].is_runnable()

    def _get_next_runnable_command(self):
        """Returns the next runnable command, if there is one."""
        if self.commutative:
            n_skipped = 0
            for command in self.waiting:
                if command.is_runnable():
                    break
                n_skipped += 1
            if n_skipped < len(self.waiting):
                # we found a runnable command
                if n_skipped > 0:
                    # move the skipped commands to the end
                    self.waiting.rotate(-n_skipped)
                command = self.waiting.popleft()
            else:
                command = None
        else:
            if self.waiting and self.waiting[0].is_runnable():
                command = self.waiting.popleft()
            else:
                command = None
        if command is None:
            command = WaitForCondition(self, self._is_empty_or_runnable_waiting)
            command.start_unqueued()
        return command

    def check_conditions(self):
        """Checks conditions on which the currently running command may be
        waiting.
        """
        if self._head is not None:
            self._head.check_conditions()

    def run(self):
        """
        Run the next available command in the queue.
        """
        if self.waiting:
            self._head = command = self._get_next_runnable_command()
            d = command.run()
            d.addBoth(passit(lambda _: self.done()))
            return d
        else:
            self.action_queue.event_queue.push('SYS_' + self.name + '_DONE')

    def node_is_queued(self, command, share_id, node_id):
        '''True if a command is queued for that node.'''
        for cmd in itertools.chain((self._head,), self.waiting):
            if isinstance(cmd, command):
                if getattr(cmd, "node_id", None) == node_id and \
                   getattr(cmd, "share_id", None) == share_id:
                    return True

    def full_queue(self):
        """
        Get the full queue (head + waiting)
        """
        return [self._head] + list(self.waiting)


class UniqueRequestQueue(RequestQueue):
    '''
    A RequestQueue that only ever queues one command for each
    (share_id, node_id) pair.
    '''
    def queue(self, command):
        """
        Add a command to the queue. If there are commands in the queue for the
        same node, they will be dropped on the floor and laughed at.
        """
        for wc in iter(self.waiting):
            wc_share = self.action_queue.resolve_uuid_maybe(wc.share_id)
            wc_node =  self.action_queue.resolve_uuid_maybe(wc.node_id)
            if wc_share == command.share_id and wc_node == command.node_id:
                self.waiting.remove(wc)
                break
        return super(UniqueRequestQueue, self).queue(command)


class NoisyRequestQueue(RequestQueue):
    """
    A RequestQueue that notifies when it's changed
    """
    def __init__(self, name, action_queue, change_notification_cb=None):
        self.set_change_notification_cb(change_notification_cb)
        super(NoisyRequestQueue, self).__init__(name, action_queue)

    def set_change_notification_cb(self, change_notification_cb):
        '''
        Set the change notification callback.
        '''
        if change_notification_cb is None:
            self.change_notification_cb = lambda *_: None
        else:
            self.change_notification_cb = change_notification_cb

    def queue(self, command):
        '''
        Add a command to the queue.
        Calls the change notification callback
        '''
        super(NoisyRequestQueue, self).queue(command)
        self.change_notification_cb(self._head, self.waiting)

    def queue_top(self, command):
        '''
        Add a command to the head of the queue.
        Calls the change notification callback
        '''
        super(NoisyRequestQueue, self).queue_top(command)
        self.change_notification_cb(self._head, self.waiting)

    def get_head(self):
        """
        Getter for the queue's head
        """
        return self.__head

    def set_head(self, head):
        """
        Setter for the queue's head. Calls he change notification callback.
        """
        self.change_notification_cb(head, self.waiting)
        self.__head = head

    _head = property(get_head, set_head)


class ContentQueue(NoisyRequestQueue, UniqueRequestQueue):
    '''
    The content queue is a commutative (uniquified), noisy request
    queue.
    '''
    commutative = True


class DeferredMap(object):
    """
    A mapping of deferred values. Or a deferred map of values. Or a
    mapping that returns deferreds and then fires them when it has the
    value.
    """
    def __init__(self):
        self.waiting = defaultdict(list)
        self.failed = {}
        self.map = {}

    def get(self, key):
        """
        Get the value for the given key.

        This always returns a deferred; when we already know the value
        we return a `succeed`, and if we don't know the value because
        it failed we return a `fail`; otherwise we return a plain
        unfired `Deferred`, and add it to the list of deferreds to
        call when we actually get the value.
        """
        if key in self.map:
            return defer.succeed(self.map[key])
        if key in self.failed:
            return defer.fail(Exception(self.failed[key]))
        d = defer.Deferred()
        self.waiting[key].append(d)
        return d

    def set(self, key, value):
        """
        We've got the value for a key! Write it down in the map, and
        fire the waiting deferreds.
        """
        if key not in self.map:
            self.map[key] =  value
            for d in self.waiting.pop(key, ()):
                d.callback(value)
        elif self.map[key] != value:
            if key in self.map:
                raise KeyError("key is taken -- dunno what to do")

    def err(self, key, failure):
        """
        Something went terribly wrong in the process of getting a
        value. Break the news to the waiting deferreds.
        """
        self.failed[key] = failure.getErrorMessage()
        for d in self.waiting.pop(key, ()):
            d.errback(failure)

    def resolve_maybe(self, key):
        """
        Return either the mapping of key (if key has been resolved),
        or key itself (if not).
        """
        return self.map.get(key, key)


class UploadProgressWrapper(object):
    """
    A wrapper around the file-like object used for Uploads, with which
    we can keep track of the number of bytes that have been written to
    the store.
    """
    __slots__ = ('fd', 'data_dict', 'n_bytes_read')
    def __init__(self, fd, data_dict):
        """
        fd is the file-like object used for uploads. data_dict is the
        entry in the uploading dictionary.
        """
        self.fd = fd
        self.data_dict = data_dict
        self.n_bytes_read = 0

    def read(self, size=None):
        """
        read at most size bytes from the file-like object.

        Keep track of the number of bytes that have been read, and the
        number of bytes that have been written (assumed to be equal to
        the number of bytes read on the previews call to read). The
        latter is done directly in the data_dict.
        """
        self.data_dict['n_bytes_written'] = self.n_bytes_read
        data = self.fd.read(size)
        self.n_bytes_read += len(data)
        return data

    def __getattr__(self, attr):
        """
        Proxy all the rest.
        """
        return getattr(self.fd, attr)


class ConfigurableThrottlingFactory(ThrottlingStorageClientFactory):
    """ThrottlingStorageClientFactory that support configuring it at runtime"""

    def __init__(self, read_limit=None, write_limit=None,
                 throttling_enabled=False):
        """Create the instance"""
        ThrottlingStorageClientFactory.__init__(self, read_limit=read_limit,
                                                write_limit=write_limit)
        self.throttling = (read_limit is not None or write_limit is not None) \
                and throttling_enabled
        self.enable_throttling(self.throttling)

    def registerWritten(self, length):
        """Only call registerWritten if we are throttling"""
        # only do something if throttling is enabled
        if self.throttling:
            ThrottlingStorageClientFactory.registerWritten(self, length)

    def registerRead(self, length):
        """Only call registerRead if we are throttling"""
        # only do something if throttling is enabled
        if self.throttling:
            ThrottlingStorageClientFactory.registerRead(self, length)

    def enable_throttling(self, enable):
        """Enable/Disable throttling. Start the counter reset loops or cancel
        them if throttling is being disabled.
        """
        if enable:
            # check if we need to start the reset loops
            if self.resetReadThisSecondID is None:
                self._resetReadThisSecond()
            if self.resetWriteThisSecondID is None:
                self._resetWrittenThisSecond()
        else:
            if self.throttling:
                # if it's currently enabled, stop the reset loops
                if self.resetReadThisSecondID is not None:
                    self._cancel_delayed_call(self.resetReadThisSecondID)
                if self.resetWriteThisSecondID is not None:
                    self._cancel_delayed_call(self.resetWriteThisSecondID)
            # else leave it as it is
        self.throttling = enable


class ActionQueue(ConfigurableThrottlingFactory, object):
    """
    This is the ActionQueue itself.
    """
    implements(IActionQueue)
    protocol = ActionQueueProtocol

    def __init__(self, event_queue, host, port, dns_srv,
                 use_ssl=False, disable_ssl_verify=False,
                 read_limit=None, write_limit=None, throttling_enabled=False):
        ConfigurableThrottlingFactory.__init__(self, read_limit=read_limit,
                                      write_limit=write_limit,
                                      throttling_enabled=throttling_enabled)
        self.event_queue = event_queue
        self.host = host
        self.port = port
        self.dns_srv = dns_srv
        self.use_ssl = use_ssl
        self.disable_ssl_verify = disable_ssl_verify

        self.token = None
        self.client = None
        self.deferred = None

        self.content_queue = ContentQueue('CONTENT_QUEUE', self)
        self.meta_queue = RequestQueue('META_QUEUE', self)
        self.uuid_map = DeferredMap()
        self.zip_queue = ZipQueue()

        self.estimated_free_space = {}

        self.uploading = {}
        self.downloading = {}

        event_queue.subscribe(self)

    def connectionLost(self, protocol, reason):
        """
        The connection went down, for some reason (which might or
        might not be described in reason).
        """
        if protocol is self.client:
            self.event_queue.push('SYS_CONNECTION_LOST')

    def check_conditions(self):
        """Poll conditions on which running actions may be waiting."""
        self.content_queue.check_conditions()
        self.meta_queue.check_conditions()

    def have_sufficient_space_for_upload(self, share_id, upload_size):
        """Returns True if we have sufficient space for the given upload."""
        return self.estimated_free_space.get(share_id, 0) >= upload_size

    def handle_SV_FREE_SPACE(self, share_id, free_bytes):
        """Update estimated free space for a share."""
        share_id = str(share_id)
        self.estimated_free_space[share_id] = free_bytes
        self.check_conditions()

    def handle_SV_SHARE_CHANGED(self, message, info):
        """Drop disappearing shares from estimated free space."""
        if message == 'deleted':
            share_id = str(info.share_id)
            if share_id in self.estimated_free_space:
                del self.estimated_free_space[share_id]

    def handle_SYS_CONNECT(self, access_token):
        """
        Stow the access token away for later use
        """
        self.token = access_token

    def cleanup(self):
        """
        Cancel, clean up, and reschedule things that were in progress
        when a disconnection happened
        """
        self.event_queue.push('SYS_CLEANUP_STARTED')
        if self.client is not None:
            self.client.disconnect()
        self.meta_queue.cleanup_and_retry()
        self.content_queue.cleanup_and_retry()
        self.event_queue.push('SYS_CLEANUP_FINISHED')

    def _node_state_callback(self, share_id, node_id, hash):
        """
        Called by the client when notified that node changed.
        """
        self.event_queue.push('SV_HASH_NEW',
                              share_id=share_id, node_id=node_id, hash=hash)

    def _share_change_callback(self, message, info):
        """
        Called by the client when notified that a share changed.
        """
        self.event_queue.push('SV_SHARE_CHANGED',
                              message=message, info=info)

    def _share_answer_callback(self, share_id, answer):
        """
        Called by the client when it gets a share answer notification.
        """
        self.event_queue.push('SV_SHARE_ANSWERED',
                              share_id=str(share_id), answer=answer)

    def _free_space_callback(self, share_id, free_bytes):
        """
        Called by the client when it gets a free space notification.
        """
        self.event_queue.push('SV_FREE_SPACE',
                              share_id=str(share_id), free_bytes=free_bytes)

    def _account_info_callback(self, account_info):
        """
        Called by the client when it gets an account info notification.
        """
        self.event_queue.push('SV_ACCOUNT_CHANGED',
                              account_info=account_info)

    def _lookup_srv(self):
        """ do the SRV lookup and return a deferred whose callback is going
        to be called with (host, port). If we can't do the lookup, the default
        host, port is used.
        """
        def on_lookup_ok(results):
            """Get a random host from the SRV result."""
            logger.debug('SRV lookup done, choosing a server')
            # pylint: disable-msg=W0612
            records, auth, add = results
            if not records:
                raise ValueError("No available records.")
            # pick a random server
            record = random.choice(records)
            logger.debug('Using record: %r', record)
            if record.payload:
                return record.payload.target.name, record.payload.port
            else:
                logger.info('Empty SRV record, fallback to %r:%r',
                            self.host, self.port)
                return self.host, self.port

        def on_lookup_error(failure):
            """ return the default host/post on a DNS SRV lookup failure. """
            logger.info("SRV lookup error, fallback to %r:%r \n%s",
                        self.host, self.port, failure.getTraceback())
            return self.host, self.port

        if self.dns_srv:
            # lookup the DNS SRV records
            d = dns_client.lookupService(self.dns_srv, timeout=[3, 2])
            d.addCallback(on_lookup_ok)
            d.addErrback(on_lookup_error)
            return d
        else:
            return defer.succeed((self.host, self.port))

    def connect(self):
        """
        Start the circus going.
        """
        self.client = None
        self.deferred = defer.Deferred()
        d = self._lookup_srv()
        def _connect(result):
            """ do the real thing """
            host, port = result
            ssl_context = get_ssl_context(self.disable_ssl_verify)
            if self.use_ssl:
                reactor.connectSSL(host, port, self, ssl_context)
            else:
                reactor.connectTCP(host, port, self)
        d.addCallback(_connect)
        return self.deferred

    def disconnect(self):
        """
        Shut down the client, if it isn't already.
        """
        if self.client is not None:
            self.client.disconnect()
            self.client = None
        else:
            # fake it
            self.connectionLost(None, 'client is None')

    def conectionFailed(self, reason=None):
        """
        Called when the connect() call fails
        """
        self.deferred.errback(reason)

    @defer.inlineCallbacks
    def check_version(self):
        """
        Check the client protocol version matches that of the
        server. Call callback on success, errback on failure.
        """
        # if the client changes while we're waiting, this message is
        # old news and should be discarded (the message would
        # typically be a failure: timeout or disconnect). So keep the
        # original client around for comparison.
        client = self.client
        try:
            yield client.protocol_version()
        except Exception, e:
            if client is self.client:
                logger.error("Protocol version error")
                logger.debug('traceback follows:\n\n' + traceback.format_exc())
                if e.message == 'UNSUPPORTED_VERSION':
                    self.event_queue.push('SYS_PROTOCOL_VERSION_ERROR',
                                          error=e.message)
                else:
                    self.event_queue.push('SYS_UNKNOWN_ERROR')
                # it looks like we won't be authenticating, so hook up the
                # for-testing deferred now
                self.deferred.callback(Failure(e))
        else:
            if client is self.client:
                logger.info("Protocol version OK")
                self.event_queue.push('SYS_PROTOCOL_VERSION_OK')

    @defer.inlineCallbacks
    def set_capabilities(self, caps):
        """Set the capabilities with the server"""
        client = self.client
        is_failed = None
        try:
            req = (yield client.query_caps(caps))
        except Exception, e:
            is_failed = e
        else:
            if not req.accepted:
                is_failed = StandardError("The server doesn't have"
                                          " the requested capabilities")

        if client is self.client:
            if is_failed is not None:
                logger.error("Capabilities query failed: %s" % is_failed)
                # Push the error to the event queue,
                self.event_queue.push('SYS_SET_CAPABILITIES_ERROR',
                                      error=is_failed.message)
                # it looks like we won't be authenticating, so hook up the
                # for-testing deferred now
                self.deferred.callback(Failure(is_failed))
                return

            is_failed = None
            try:
                req = (yield client.set_caps(caps))
            except Exception, e:
                if client is self.client:
                    is_failed = e
            else:
                if client is self.client:
                    if req.accepted:
                        self.event_queue.push('SYS_SET_CAPABILITIES_OK')
                        defer.returnValue(self.client)
                    else:
                        is_failed = StandardError("The server denied setting"
                                                  " %s capabilities" % req.caps)
            if is_failed is not None:
                logger.error("Capabilities set failed: %s" % is_failed)
                # Push the error to the event queue,
                self.event_queue.push('SYS_SET_CAPABILITIES_ERROR',
                                      error=is_failed.message)
                # it looks like we won't be authenticating, so hook up the
                # for-testing deferred now
                self.deferred.callback(Failure(is_failed))

    @defer.inlineCallbacks
    def authenticate(self, oauth_consumer):
        client = self.client
        try:
            yield client.oauth_authenticate(oauth_consumer, self.token)
        except request.StorageRequestError, e:
            if client is not self.client:
                return
            if e.error_message.error.type == \
                                    protocol_pb2.Error.AUTHENTICATION_FAILED:
                logger.error("OAuth failed: %s", e)
                self.event_queue.push('SYS_OAUTH_ERROR', error=str(e))
            else:
                logger.error("StorageRequestError during OAuth: %s", e)
                self.event_queue.push('SYS_UNKNOWN_ERROR')
            self.deferred.callback(Failure(e))
        except Exception, e:
            if client is not self.client:
                return
            logger.error("Generic error during OAuth: %s", e)
            self.event_queue.push('SYS_UNKNOWN_ERROR')
            self.deferred.callback(Failure(e))
        else:
            if client is not self.client:
                return
            logger.info("Oauth OK")
            self.event_queue.push('SYS_OAUTH_OK')
            self.deferred.callback(client)

    @defer.inlineCallbacks
    def server_rescan(self, data_gen):
        """
        Do the server rescan
        """
        client = self.client
        yield self.get_root(object())
        if client is not self.client:
            return
        self.event_queue.push('SYS_SERVER_RESCAN_STARTING')
        data = data_gen()
        logger.info("server rescan: will query %d objects" % len(data))
        # we check we're going to actually log, because this could be expensive
        if logger.isEnabledFor(logging.DEBUG):
            for share, node, hash in data:
                logger.debug("server rescan: share: %s, node: %s, hash: %s"
                                  % (share or '/root/', node, hash))
            logger.debug("server rescan: all data shown")
        yield client.query(data)
        if client is not self.client:
            return
        logger.info("server rescan: done")
        self.event_queue.push('SYS_SERVER_RESCAN_DONE')

    def get_root(self, marker):
        """
        Get the user's root uuid. Use the uuid_map, so the caller can
        use the marker in followup operations.
        """
        log = mklog(logger, 'get_root', '', marker, marker=marker)
        log.debug('starting')
        d = self.client.get_root()
        d.addCallbacks(*log.callbacks())
        d.addCallbacks(passit(lambda root: self.uuid_map.set(marker, root)),
                       passit(lambda f: self.uuid_map.err(marker, f)))

        return d

    def make_file(self, share_id, parent_id, name, marker):
        """
        See .interfaces.IMetaQueue
        """
        return MakeFile(self.meta_queue, share_id, parent_id,
                        name, marker).start()

    def make_dir(self, share_id, parent_id, name, marker):
        """
        See .interfaces.IMetaQueue
        """
        return MakeDir(self.meta_queue, share_id, parent_id,
                       name, marker).start()

    def move(self, share_id, node_id, old_parent_id, new_parent_id, new_name):
        """
        See .interfaces.IMetaQueue
        """
        return Move(self.meta_queue, share_id, node_id, old_parent_id,
                    new_parent_id, new_name).start()

    def unlink(self, share_id, parent_id, node_id):
        """
        See .interfaces.IMetaQueue
        """
        return Unlink(self.meta_queue, share_id, parent_id, node_id).start()

    def query(self, items):
        """
        See .interfaces.IMetaQueue
        """
        return Query(self.meta_queue, items).start()

    def inquire_free_space(self, share_id):
        """
        See .interfaces.IMetaQueue
        """
        return FreeSpaceInquiry(self.meta_queue, share_id).start()

    def inquire_account_info(self):
        """
        see .interfaces.IMetaQueue
        """
        return AccountInquiry(self.meta_queue).start()

    def list_shares(self):
        """
        List the shares; put the result on the event queue
        """
        return ListShares(self.meta_queue).start()

    def answer_share(self, share_id, answer):
        """
        Answer the offer of a share.
        """
        return AnswerShare(self.meta_queue, share_id, answer).start()

    def create_share(self, node_id, share_to, name, access_level, marker):
        """
        Share a node with somebody.
        """
        return CreateShare(self.meta_queue, node_id, share_to, name,
                           access_level, marker).start()

    def listdir(self, share_id, node_id, server_hash, fileobj_factory):
        """
        See .interfaces.IMetaQueue.listdir
        """
        return ListDir(self.meta_queue, share_id, node_id, server_hash,
                       fileobj_factory).start()

    def download(self, share_id, node_id, server_hash, fileobj_factory):
        """
        See .interfaces.IContentQueue.download
        """
        return Download(self.content_queue, share_id, node_id, server_hash,
                        fileobj_factory).start()

    def upload(self, share_id, node_id, previous_hash, hash, crc32,
               size, fileobj_factory, tempfile_factory=None):
        """
        See .interfaces.IContentQueue
        """
        return Upload(self.content_queue, share_id, node_id, previous_hash,
                      hash, crc32, size,
                      fileobj_factory, tempfile_factory).start()

    def _cancel_op(self, share_id, node_id, cmdclass, logstr):
        """
        Generalized form of cancel_upload and cancel_download
        """
        log = mklog(logger, logstr, share_id, node_id,
                    share=share_id, node=node_id)
        log.debug('starting')
        for queue in self.meta_queue, self.content_queue:
            for cmd in queue.full_queue():
                if isinstance(cmd, cmdclass) \
                        and share_id == cmd.share_id \
                        and node_id in (cmd.marker_maybe, cmd.node_id):
                    log.debug('cancelling')
                    cmd.cancel()
        log.debug('finished')

    def cancel_upload(self, share_id, node_id):
        """
        See .interfaces.IContentQueue
        """
        self._cancel_op(share_id, node_id, Upload, 'cancel_upload')

    def cancel_download(self, share_id, node_id):
        """
        See .interfaces.IContentQueue
        """
        self._cancel_op(share_id, node_id, GetContentMixin, 'cancel_download')

    def node_is_with_queued_move(self, share_id, node_id):
        '''True if a Move is queued for that node.'''
        return self.meta_queue.node_is_queued(Move, share_id, node_id)

    def resolve_uuid_maybe(self, marker_maybe):
        '''
        Resolve the maybe_marker if it is a marker and has been
        resolved. Otherwise just return the marker_maybe back again.
        '''
        return self.uuid_map.resolve_maybe(marker_maybe)


SKIP_THIS_ITEM = object()

# pylint: disable-msg=W0231

class ActionQueueCommand(object):
    """
    Base of all the action queue commands
    """

    # protobuf doesn't seem to have much introspectionable stuff
    # without going into private attributes
    # pylint: disable-msg=W0212
    known_error_messages = (set(protocol_pb2._ERROR_ERRORTYPE.values_by_name)
                            | set(['CANCELLED']))
    suppressed_error_messages = (known_error_messages
                                 - set(['INTERNAL_ERROR'])
                                 | set(['Cleaned up',
                                        'Connection was closed cleanly.']))
    retryable_errors = set([
            'Cleaned up',
            'TRY_AGAIN',
            'Connection was closed cleanly.',
            'Connection to the other side was lost in a non-clean fashion.',
            ])

    logged_attrs = ()

    __slots__ = ('_queue', 'start_done', 'running', 'log')
    def __init__(self, request_queue):
        """Initialize a command instance."""
        self._queue = request_queue
        self.start_done = False
        self.running = False
        self.log = None

    def make_logger(self):
        """Create a logger for this object."""
        share_id = getattr(self, "share_id", UNKNOWN)
        node_id = getattr(self, "node_id", None) or \
                      getattr(self, "marker", UNKNOWN)
        attr_dict = dict([(n, getattr(self, n)) for n in self.logged_attrs])
        return mklog(logger, self.__class__.__name__,
                     share_id, node_id, **attr_dict)

    def is_runnable(self):
        """Returns True if the command is eligible to run."""
        return self._is_runnable()

    def _is_runnable(self):
        """Hook for subclasses to supply a predicate for runnability."""
        return True

    def check_conditions(self):
        """Check conditions on which the command may be waiting."""
        pass

    def demark(self, *maybe_markers):
        """
        Arrange to have maybe_markers realized
        """
        l = []
        for marker in maybe_markers:
            if IMarker.providedBy(marker):
                self.log.debug('waiting until we know the real value of %s'
                               % marker)
                d = self.action_queue.uuid_map.get(marker)
                d.addCallbacks(passit(lambda _:
                                          self.log.debug('got %s' % marker)),
                               passit(lambda f:
                                          self.log.error('failed %s' % marker)))
            else:
                d = defer.succeed(marker)
            l.append(d)
        dl = defer.DeferredList(l, fireOnOneErrback=True, consumeErrors=True)
        dl.addCallbacks(self.unwrap,
                        lambda f: f.value.subFailure)
        return dl

    @staticmethod
    def unwrap(results):
        """
        Unpack the values from the result of a DeferredList. If
        there's a failure, return it instead.
        """
        values = []
        for result in results:
            # result can be none if one of the callbacks failed
            # before the others were ready
            if result is not None:
                is_ok, value = result
                if not is_ok:
                    # a failure!
                    return value
                if value is not SKIP_THIS_ITEM:
                    values.append(value)
        return values

    def end_callback(self, arg):
        """
        It worked!
        """
        if self.running:
            self.log.debug('success')
            return self.handle_success(arg)
        else:
            self.log.debug('not running, so no success')

    def end_errback(self, failure):
        """
        It failed!
        """
        error_message = failure.getErrorMessage()
        if error_message not in self.suppressed_error_messages:
            self.log.error('failure', error_message)
            self.log.debug('traceback follows:\n\n' + failure.getTraceback())
        else:
            self.log.warn('failure', error_message)
        self.cleanup()
        if error_message in self.retryable_errors:
            if self.running:
                reactor.callLater(0.1, self.retry)
        else:
            return self.handle_failure(failure)

    def start_unqueued(self):
        """Do basic pre-start setup."""
        self.log = self.make_logger()

    def start(self, _=None):
        """
        Queue the command.
        """
        self.start_unqueued()
        self.log.debug('queueing in the %s' % self._queue.name)
        self._queue.queue(self)

    def cleanup(self):
        """
        Do whatever is needed to clean up from a failure (such as stop
        producers and other such that aren't cleaned up appropriately
        on their own)
        """
        self.running = False
        self.log.debug('cleanup')

    def _start(self):
        """
        Do the specialized pre-run setup
        """
        return defer.succeed(None)

    def store_marker_result(self, _):
        """
        Called when all the markers are realized.
        """

    def run(self):
        """
        Do the deed.
        """
        self.running = True
        if self.start_done:
            self.log.debug('retrying')
            d = defer.succeed(None)
        else:
            self.log.debug('starting')
            d = self._start()
            d.addCallback(self.store_marker_result)
        d.addCallbacks(self._ready_to_run, self.end_errback)
        d.addBoth(self._done_running)
        return d

    def _done_running(self, x):
        self.running = False
        return x

    def _ready_to_run(self, _):
        self.log.debug('running')
        if self.running:
            d = self._run()
        else:
            d = defer.succeed(None)
        d.addCallbacks(self.end_callback, self.end_errback)
        return d

    def handle_success(self, success):
        """
        Do anthing that's needed to handle success of the operation.
        """
        return success

    def handle_failure(self, failure):
        """
        Do anthing that's needed to handle failure of the operation.
        Note that cancellation and TRY_AGAIN are already handled.
        """
        return failure

    def cleanup_and_retry(self):
        """
        First, cleanup; then, retry :)
        """
        self.log.debug('cleanup and retry')
        self.cleanup()
        return self.retry()

    def retry(self):
        """
        Request cancelled or TRY_AGAIN. Well then, try again!
        """
        self.running = False
        self.log.debug('will retry')
        return self._queue.queue_top(self)

    @property
    def action_queue(self):
        """Returns the action queue."""
        return self._queue.action_queue


class WaitForCondition(ActionQueueCommand):
    """A command which waits for some condition to be satisfied."""

    __slots__ = ('_condition', '_deferred')
    def __init__(self, request_queue, condition):
        """Initializes the command instance."""
        ActionQueueCommand.__init__(self, request_queue)
        self._condition = condition
        self._deferred = None

    def _run(self):
        """Returns a deferred which blocks until the condition is satisfied."""
        if self._condition():
            d = defer.succeed(None)
            self._deferred = None
        else:
            d = defer.Deferred()
            self._deferred = d
            event_name = 'SYS_' + self._queue.name + '_BLOCKED'
            self.action_queue.event_queue.push(event_name)
        return d

    def check_conditions(self):
        """Poll the action's condition."""
        if self._deferred is not None and self._condition():
            try:
                self._deferred.callback(None)
                self._deferred = None
            except defer.AlreadyCalledError:
                return

    def cleanup(self):
        """Does cleanup."""
        self._deferred = None
        ActionQueueCommand.cleanup(self)


class MakeThing(ActionQueueCommand):
    """
    Base of MakeFile and MakeDir
    """
    __slots__ = ('share_id', 'parent_id', 'name', 'marker')
    logged_attrs = __slots__

    def __init__(self, request_queue, share_id, parent_id, name, marker):
        super(MakeThing, self).__init__(request_queue)
        self.share_id = share_id
        self.parent_id = parent_id
        # Unicode boundary! the name is Unicode in protocol and server, but
        # here we use bytes for paths
        self.name = name.decode("utf8")
        self.marker = marker

    def _start(self):
        """
        Do the specialized pre-run setup
        """
        return self.demark(self.share_id, self.parent_id)

    def store_marker_result(self, (share_id, parent_id)):
        """
        Called when all the markers are realized.
        """
        self.share_id = share_id
        self.parent_id = parent_id
        self.start_done = True

    def _run(self):
        """
        Do the actual running
        """
        maker = getattr(self.action_queue.client, self.client_method)
        return maker(self.share_id,
                     self.parent_id,
                     self.name)

    def handle_success(self, success):
        """
        It worked! Push the event, and update the uuid map.
        """
        # note that we're not getting the new name from the answer
        # message, if we would get it, we would have another Unicode
        # boundary with it
        self.action_queue.event_queue.push(self.ok_event_name,
                                           marker=self.marker,
                                           new_id=success.new_id)
        if IMarker.providedBy(self.marker):
            self.action_queue.uuid_map.set(self.marker, success.new_id)
        return success

    def handle_failure(self, failure):
        """
        It didn't work! Push the event, and update the uuid map.
        """
        self.action_queue.event_queue.push(self.error_event_name,
                                           marker=self.marker,
                                           error=failure.getErrorMessage())
        if IMarker.providedBy(self.marker):
            self.action_queue.uuid_map.err(self.marker,
                                           failure)


class MakeFile(MakeThing):
    """
    Make a file
    """
    __slots__ = ()
    ok_event_name = 'AQ_FILE_NEW_OK'
    error_event_name = 'AQ_FILE_NEW_ERROR'
    client_method = 'make_file'


class MakeDir(MakeThing):
    """
    Make a directory
    """
    __slots__ = ()
    ok_event_name = 'AQ_DIR_NEW_OK'
    error_event_name = 'AQ_DIR_NEW_ERROR'
    client_method = 'make_dir'


class Move(ActionQueueCommand):
    """
    Move a file or directory
    """
    __slots__ = ('share_id', 'node_id', 'old_parent_id',
                 'new_parent_id', 'new_name')
    logged_attrs =  __slots__

    def __init__(self, request_queue, share_id, node_id, old_parent_id,
                 new_parent_id, new_name):
        super(Move, self).__init__(request_queue)
        self.share_id = share_id
        self.node_id = node_id
        self.old_parent_id = old_parent_id
        self.new_parent_id = new_parent_id
        # Unicode boundary! the name is Unicode in protocol and server, but
        # here we use bytes for paths
        self.new_name = new_name.decode("utf8")

    def _start(self):
        """
        Do the specialized pre-run setup
        """
        return self.demark(self.share_id, self.node_id, self.new_parent_id)

    def store_marker_result(self, (share_id, node_id, new_parent_id)):
        """
        Called when all the markers are realized.
        """
        self.share_id = share_id
        self.node_id = node_id
        self.new_parent_id = new_parent_id
        self.start_done = True

    def _run(self):
        """
        Do the actual running
        """
        return self.action_queue.client.move(self.share_id,
                                             self.node_id,
                                             self.new_parent_id,
                                             self.new_name)
    def handle_success(self, success):
        """
        It worked! Push the event.
        """
        self.action_queue.event_queue.push('AQ_MOVE_OK',
                                           share_id=self.share_id,
                                           node_id=self.node_id)
        return success

    def handle_failure(self, failure):
        """
        It didn't work! Push the event.
        """
        self.action_queue.event_queue.push('AQ_MOVE_ERROR',
                                           error=failure.getErrorMessage(),
                                           share_id=self.share_id,
                                           node_id=self.node_id,
                                           old_parent_id=self.old_parent_id,
                                           new_parent_id=self.new_parent_id,
                                           new_name=self.new_name)


class Unlink(ActionQueueCommand):
    """
    Unlink a file or dir
    """
    __slots__ = ('share_id', 'node_id', 'parent_id')
    logged_attrs = __slots__

    def __init__(self, request_queue, share_id, parent_id, node_id):
        super(Unlink, self).__init__(request_queue)
        self.share_id = share_id
        self.node_id = node_id
        self.parent_id = parent_id

    def _start(self):
        """
        Do the specialized pre-run setup
        """
        return self.demark(self.share_id, self.node_id, self.parent_id)

    def store_marker_result(self, (share_id, node_id, parent_id)):
        """
        Called when all the markers are realized.
        """
        self.share_id = share_id
        self.node_id = node_id
        self.parent_id = parent_id
        self.start_done = True

    def _run(self):
        """
        Do the actual running
        """
        return self.action_queue.client.unlink(self.share_id, self.node_id)

    def handle_success(self, success):
        """
        It worked! Push the event.
        """
        self.action_queue.event_queue.push('AQ_UNLINK_OK',
                                           share_id=self.share_id,
                                           parent_id=self.parent_id,
                                           node_id=self.node_id)
        return success

    def handle_failure(self, failure):
        """
        It didn't work! Push the event.
        """
        self.action_queue.event_queue.push('AQ_UNLINK_ERROR',
                                           error=failure.getErrorMessage(),
                                           share_id=self.share_id,
                                           parent_id=self.parent_id,
                                           node_id=self.node_id)


class Query(ActionQueueCommand):
    """
    Ask about the freshness of server hashes
    """
    __slots__ = ('items')
    def __init__(self, request_queue, items):
        super(Query, self).__init__(request_queue)
        self.items = items

    def make_logger(self):
        """Make the magic logger for queries."""
        return MultiProxy(
            [mklog(logger, '(unrolled) query', share, node,
                   share=share, node=node, hash=hash, index=i)
             for (i, (share, node, hash)) in enumerate(self.items)])

    def store_marker_result(self, items):
        """
        Called when all the markers are realized.
        """
        self.items = items
        self.start_done = True

    def _start(self):
        """
        Do the specialized pre-run setup
        """
        # node_hash will (should?) never be a marker, but it's the
        # easiest way to keep the trio together: send it along for the
        # trip
        dl = []
        for item in self.items:
            d = self.demark(*item)
            d.addErrback(self.handle_single_failure, item)
            dl.append(d)
        d = defer.DeferredList(dl, fireOnOneErrback=True, consumeErrors=True)
        d.addCallbacks(self.unwrap)
        return d

    def handle_failure(self, failure):
        """
        It didn't work! Never mind.
        """
        pass

    def handle_single_failure(self, failure, item):
        """
        The only failure mode of a Query is for a query to be done
        using a marker that fails to realize.
        """
        self.action_queue.event_queue.push('AQ_QUERY_ERROR', item=item,
                                           error=failure.getErrorMessage())
        return SKIP_THIS_ITEM

    def _run(self):
        """
        Do the actual running
        """
        return self.action_queue.client.query(self.items)


class ListShares(ActionQueueCommand):
    """
    List shares shared to me
    """
    __slots__ = ()

    def _run(self):
        """
        Do the actual running
        """
        return self.action_queue.client.list_shares()

    def handle_success(self, success):
        """
        It worked! Push the event.
        """
        self.action_queue.event_queue.push('AQ_SHARES_LIST',
                                           shares_list=success)

    def handle_failure(self, failure):
        """
        It didn't work! Push the event.
        """
        self.action_queue.event_queue.push('AQ_LIST_SHARES_ERROR',
                                           error=failure.getErrorMessage())


class FreeSpaceInquiry(ActionQueueCommand):
    """Inquire about free space."""

    def __init__(self, request_queue, share_id):
        """Initialize the instance."""
        super(FreeSpaceInquiry, self).__init__(request_queue)
        self.share_id = share_id

    def _run(self):
        """Do the query."""
        return self.action_queue.client.get_free_space(self.share_id)

    def handle_success(self, success):
        """Publish the free space information."""
        self.action_queue.event_queue.push('SV_FREE_SPACE',
                                           share_id=success.share_id,
                                           free_bytes=success.free_bytes)

    def handle_failure(self, failure):
        """Publish the error."""
        self.action_queue.event_queue.push('AQ_FREE_SPACE_ERROR',
                                           error=failure.getErrorMessage())


class AccountInquiry(ActionQueueCommand):
    """Query user account information."""
    def _run(self):
        """Make the actual request."""
        return self.action_queue.client.get_account_info()

    def handle_success(self, success):
        """Publish the account information to the event queue."""
        self.action_queue.event_queue.push('SV_ACCOUNT_CHANGED',
                                           account_info=success)

    def handle_failure(self, failure):
        """Publish the error."""
        self.action_queue.event_queue.push('AQ_ACCOUNT_ERROR',
                                           error=failure.getErrorMessage())


class AnswerShare(ActionQueueCommand):
    """
    Answer a share offer
    """
    __slots__ = ('share_id', 'answer')
    def __init__(self, request_queue, share_id, answer):
        super(AnswerShare, self).__init__(request_queue)
        self.share_id = share_id
        self.answer = answer

    def _run(self):
        """
        Do the actual running
        """
        return self.action_queue.client.accept_share(self.share_id, self.answer)

    def handle_success(self, success):
        """
        It worked! Push the event.
        """
        self.action_queue.event_queue.push('AQ_ANSWER_SHARE_OK',
                                           share_id=self.share_id,
                                           answer=self.answer)
        return success

    def handle_failure(self, failure):
        """
        It didn't work. Push the event.
        """
        self.action_queue.event_queue.push('AQ_ANSWER_SHARE_ERROR',
                                           share_id=self.share_id,
                                           answer=self.answer,
                                           error=failure.getErrorMessage())
        return failure


class CreateShare(ActionQueueCommand):
    """
    Offer a share to somebody
    """
    __slots__ = ('node_id', 'share_to', 'name', 'access_level',
                 'marker', 'use_http')
    def __init__(self, request_queue, node_id, share_to, name, access_level,
                 marker):
        super(CreateShare, self).__init__(request_queue)
        self.node_id = node_id
        self.share_to = share_to
        self.name = name
        self.access_level = access_level
        self.marker = marker
        self.use_http = False

        if share_to and re.match(EREGEX, share_to):
            self.use_http = True

    def store_marker_result(self, (node_id,)):
        """
        Called when all the markers are realized.
        """
        self.node_id = node_id
        self.start_done = True

    def _start(self):
        """
        Do the specialized pre-run setup
        """
        return self.demark(self.node_id)

    def _create_share_http(self, node_id, user, name, modify, deferred):
        """Create a share using the HTTP Web API method."""

        consumer = oauth.OAuthConsumer("ubuntuone", "hammertime")
        url = "https://one.ubuntu.com/files/api/offer_share/"
        method = oauth.OAuthSignatureMethod_PLAINTEXT()
        request = oauth.OAuthRequest.from_consumer_and_token(
            http_url=url,
            http_method="POST",
            oauth_consumer=consumer,
            token=self.action_queue.token)
        request.sign_request(method, consumer, self.action_queue.token)
        data = dict(offer_to_email = user,
                    read_only = modify != True,
                    node_id = node_id,
                    share_name = name)
        pdata = urlencode(data)
        headers = request.to_header()
        req = Request(url, pdata, headers)
        try:
            urlopen(req)
        except HTTPError, e:
            deferred.errback(Failure(e))

        deferred.callback(None)

    def _run(self):
        """
        Do the actual running
        """
        if self.use_http:
            # External user, do the HTTP REST method
            deferred = defer.Deferred()
            d = threads.deferToThread(self._create_share_http,
                                      self.node_id, self.share_to,
                                      self.name, self.access_level,
                                      deferred)
            d.addErrback(deferred.errback)
            return deferred
        else:
            return self.action_queue.client.create_share(self.node_id,
                                                         self.share_to,
                                                         self.name,
                                                         self.access_level)

    def handle_success(self, success):
        """
        It worked! Push the event.
        """
        # We don't get a share_id back from the HTTP REST method
        if not self.use_http:
            self.action_queue.event_queue.push('AQ_CREATE_SHARE_OK',
                                               share_id=success.share_id,
                                               marker=self.marker)
        return success

    def handle_failure(self, failure):
        """
        It didn't work! Push the event.
        """
        self.action_queue.event_queue.push('AQ_CREATE_SHARE_ERROR',
                                           marker=self.marker,
                                           error=failure.getErrorMessage())


class GetContentMixin(ActionQueueCommand):
    """
    Base for ListDir and Download. It's a mixin (ugh) because
    otherwise things would be even more confusing
    """
    __slots__ = ('share_id', 'node_id', 'server_hash', 'fileobj_factory',
                 'fileobj', 'gunzip', 'marker_maybe', 'cancelled',
                 'download_req')
    logged_attrs = ('share_id', 'node_id', 'server_hash', 'fileobj_factory')
    def __init__(self, request_queue, share_id, node_id, server_hash,
                 fileobj_factory):
        super(GetContentMixin, self).__init__(request_queue)
        self.share_id = share_id
        self.marker_maybe = self.node_id = node_id
        self.server_hash = server_hash
        self.fileobj_factory = fileobj_factory
        self.fileobj = None
        self.gunzip = zlib.decompressobj()
        self.cancelled = False
        self.download_req = None
        self.action_queue.cancel_download(self.share_id, self.node_id)

    def _start(self):
        """
        Do the specialized pre-run setup
        """
        return self.demark(self.node_id)

    def cancel(self):
        """
        Cancel the download.
        """
        self.cancelled = True
        if self.download_req is not None:
            self.download_req.cancel()
        self.cleanup()

    def store_marker_result(self, (node_id,)):
        """
        Called when all the markers are realized.
        """
        self.node_id = node_id
        self.start_done = True

    def _run(self):
        """
        Do the actual running
        """
        if self.cancelled:
            return defer.fail(RuntimeError('CANCELLED'))
        if self.fileobj is None:
            try:
                self.fileobj = self.fileobj_factory()
            except StandardError:
                self.log.debug(traceback.format_exc())
                return defer.fail(Failure('unable to build fileobj'
                                          ' (file went away?)'
                                          ' so aborting the download.'))
        downloading = self.action_queue.downloading
        if (self.share_id, self.node_id) not in downloading:
            downloading[self.share_id, self.node_id] = {'n_bytes_read': 0,
                                                        'command': self}
        assert downloading[self.share_id, self.node_id]['command'] is self
        offset = downloading[self.share_id, self.node_id]['n_bytes_read']

        self.action_queue.event_queue.push('AQ_DOWNLOAD_STARTED',
                                           share_id=self.share_id,
                                           node_id=self.node_id,
                                           server_hash=self.server_hash)

        req = self.action_queue.client.get_content_request(
            self.share_id, self.node_id, self.server_hash,
            offset=offset,
            callback=self.cb, node_attr_callback=self.nacb)
        self.download_req = req
        downloading[self.share_id, self.node_id]['req'] = req
        d = req.deferred
        d.addBoth(lambda x: defer.fail(RuntimeError('CANCELLED'))
                  if self.cancelled else x)
        d.addCallback(passit(
                lambda _: downloading.pop((self.share_id, self.node_id))))
        return d

    def handle_success(self, _):
        """
        It worked! Push the event.
        """
        self.sync()
        # for directories, a FINISHED; for files, a COMMIT (the Nanny
        # will issue the FINISHED if it's ok)
        if self.is_dir:
            event = 'AQ_DOWNLOAD_FINISHED'
        else:
            event = 'AQ_DOWNLOAD_COMMIT'
        self.action_queue.event_queue.push(event,
                                           share_id=self.share_id,
                                           node_id=self.node_id,
                                           server_hash=self.server_hash)

    def handle_failure(self, failure):
        """
        It didn't work! Push the event.
        """
        downloading = self.action_queue.downloading
        if (self.share_id, self.node_id) in downloading \
                and downloading[self.share_id, self.node_id]['command'] is self:
            del downloading[self.share_id, self.node_id]
        self.reset_fileobj()
        if failure.getErrorMessage() == 'CANCELLED':
            self.action_queue.event_queue.push('AQ_DOWNLOAD_CANCELLED',
                                               share_id=self.share_id,
                                               node_id=self.node_id,
                                               server_hash=self.server_hash)
        else:
            self.action_queue.event_queue.push('AQ_DOWNLOAD_ERROR',
                                               error=failure.getErrorMessage(),
                                               share_id=self.share_id,
                                               node_id=self.node_id,
                                               server_hash=self.server_hash)

    def reset_fileobj(self):
        """
        Rewind and empty the file (i.e. get it ready to try again if
        necessary)
        """
        self.log.debug('reset fileobj')
        if self.fileobj is not None:
            self.fileobj.seek(0, 0)
            self.fileobj.truncate(0)

    def cb(self, bytes):
        """
        A streaming decompressor
        """
        dloading = self.action_queue.downloading[self.share_id,
                                                 self.node_id]
        dloading['n_bytes_read'] += len(bytes)
        self.fileobj.write(self.gunzip.decompress(bytes))
        self.fileobj.flush()     # not strictly necessary but nice to
                                 # see the downloaded size

    def nacb(self, **kwargs):
        """
        set the node attrs in the 'currently downloading' dict
        """
        self.action_queue.downloading[self.share_id,
                                      self.node_id].update(kwargs)

    def sync(self):
        """
        Flush the buffers and sync them to disk if possible
        """
        remains = self.gunzip.flush()
        if remains:
            self.fileobj.write(remains)
        self.fileobj.flush()
        if getattr(self.fileobj, 'fileno', None) is not None:
            # it's a real file, with a fileno! Let's sync its data
            # out to disk
            os.fsync(self.fileobj.fileno())
        self.fileobj.close()


class ListDir(GetContentMixin, ActionQueueCommand):
    """
    Get a listing of a directory's contents
    """
    __slots__ = ()
    is_dir = True

class Download(GetContentMixin, ActionQueueCommand):
    """
    Get the contents of a file.
    """
    __slots__ = ()
    is_dir = False

class Upload(ActionQueueCommand):
    """
    Upload stuff to a file
    """
    __slots__ = ('share_id', 'node_id', 'previous_hash', 'hash', 'crc32',
                 'size', 'fileobj_factory', 'tempfile_factory', 'deflated_size',
                 'tempfile', 'cancelled', 'upload_req', 'marker_maybe')
    logged_attrs = ('share_id', 'node_id', 'previous_hash', 'hash', 'crc32',
                    'size', 'fileobj_factory')
    retryable_errors = (ActionQueueCommand.retryable_errors
                        | set(['UPLOAD_IN_PROGRESS']))

    def __init__(self, request_queue, share_id, node_id, previous_hash, hash,
                 crc32, size, fileobj_factory, tempfile_factory):
        super(Upload, self).__init__(request_queue)
        self.share_id = share_id
        self.node_id = node_id
        self.previous_hash = previous_hash
        self.hash = hash
        self.crc32 = crc32
        self.size = size
        self.fileobj_factory = fileobj_factory
        self.tempfile_factory = tempfile_factory
        self.deflated_size = None
        self.tempfile = None
        self.cancelled = False
        self.upload_req = None
        self.marker_maybe = None
        if (self.share_id, self.node_id) in self.action_queue.uploading:
            self.action_queue.cancel_upload(self.share_id, self.node_id)

    def _is_runnable(self):
        """Returns True if there is sufficient space available to complete
        the upload."""
        return self.action_queue.have_sufficient_space_for_upload(self.share_id,
                                                                  self.size)

    def cancel(self):
        """Cancel the upload."""
        self.cancelled = True
        if self.upload_req is not None:
            self.upload_req.cancel()
        self.cleanup()

    def cleanup(self):
        """
        Cleanup: stop the producer.
        """
        super(Upload, self).cleanup()
        if self.upload_req is not None and self.upload_req.producer is not None:
            self.log.debug('stopping the producer')
            self.upload_req.producer.stopProducing()

    def _start(self):
        """
        Do the specialized pre-run setup
        """
        d = defer.Deferred()

        uploading = {"hash": self.hash, "req": self}
        self.action_queue.uploading[self.share_id, self.node_id] = uploading

        d = self.action_queue.zip_queue.zip(self)
        d.addCallback(lambda _: self.demark(self.node_id))
        d.addBoth(lambda x: defer.fail(RuntimeError('CANCELLED'))
                  if self.cancelled else x)
        return d

    def store_marker_result(self, (node_id,)):
        """
        Called when all the markers are realized.
        """
        # update action_queue.uploading with the real node_id
        uploading = self.action_queue.uploading.pop((self.share_id,
                                                     self.node_id))
        self.node_id = node_id
        self.action_queue.uploading[self.share_id, node_id] = uploading
        self.start_done = True

    def _run(self):
        """
        Do the actual running
        """
        if self.cancelled:
            return defer.fail(RuntimeError('CANCELLED'))
        uploading = {"hash": self.hash, "deflated_size": self.deflated_size,
                     "req": self}
        self.action_queue.uploading[self.share_id, self.node_id] = uploading

        self.action_queue.event_queue.push('AQ_UPLOAD_STARTED',
                                           share_id=self.share_id,
                                           node_id=self.node_id,
                                           hash=self.hash)

        if getattr(self.tempfile, 'name', None) is not None:
            self.tempfile = open(self.tempfile.name)
        f = UploadProgressWrapper(self.tempfile, uploading)
        req = self.action_queue.client.put_content_request(
            self.share_id, self.node_id, self.previous_hash, self.hash,
            self.crc32, self.size, self.deflated_size, f)
        self.upload_req = req
        d = req.deferred
        d.addBoth(lambda x: defer.fail(RuntimeError('CANCELLED'))
                  if self.cancelled else x)
        d.addBoth(passit(lambda _:
                             self.action_queue.uploading.pop((self.share_id,
                                                              self.node_id))))
        d.addBoth(passit(lambda _: self.tempfile.close()))
        return d

    def handle_success(self, _):
        """
        It worked! Push the event.
        """
        if getattr(self.tempfile, 'name', None) is not None:
            os.unlink(self.tempfile.name)
        self.action_queue.event_queue.push('AQ_UPLOAD_FINISHED',
                                           share_id=self.share_id,
                                           node_id=self.node_id,
                                           hash=self.hash)

    def handle_failure(self, failure):
        """
        It didn't work! Push the event.
        """
        if getattr(self.tempfile, 'name', None) is not None:
            os.unlink(self.tempfile.name)
        self.action_queue.event_queue.push('AQ_UPLOAD_ERROR',
                                           error=failure.getErrorMessage(),
                                           share_id=self.share_id,
                                           node_id=self.node_id,
                                           hash=self.hash)
