# -*- coding: utf-8 -*-
#
# Authors: Natalia B. Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Windows-specific tests for the XDG constants."""

import os
import sys

from twisted.trial.unittest import TestCase
from ubuntuone.devtools.testcase import skipIfOS


# pylint: disable=E1101, E0611, F0401
if sys.platform == "win32":
    import win32com.shell
    from ubuntu_sso.xdg_base_directory.windows import (
         get_config_dirs,
         get_data_dirs,
         get_special_folders,
    )


# pylint: disable=C0103
class FakeShellConModule(object):
    """Override CSIDL_ constants."""
    CSIDL_PROFILE = 0
    CSIDL_LOCAL_APPDATA = 1
    CSIDL_COMMON_APPDATA = 2


class FakeShellModule(object):

    """Fake Shell Module."""

    def __init__(self):
        """Set the proper mapping between CSIDL_ consts."""
        self.values = {
            0: u'c:\\path\\to\\users\\home',
            1: u'c:\\path\\to\\users\\home\\appData\\local',
            2: u'c:\\programData',
        }

    def SHGetFolderPath(self, dummy0, shellconValue, dummy2, dummy3):
        """Override SHGetFolderPath functionality."""
        return self.values[shellconValue]


@skipIfOS('linux2', 'Windows-specific tests.')
class TestBaseDirectoryWindows(TestCase):
    """Tests for the BaseDirectory module."""

    def test_get_special_folders(self):
        """Make sure we can import the platform module."""

        shellModule = FakeShellModule()
        self.patch(win32com.shell, "shell", shellModule)
        self.patch(win32com.shell, "shellcon", FakeShellConModule())
        special_folders = get_special_folders()
        self.assertTrue('Personal' in special_folders)
        self.assertTrue('Local AppData' in special_folders)
        self.assertTrue('AppData' in special_folders)
        self.assertTrue('Common AppData' in special_folders)

        self.assertTrue(special_folders['Personal'] == \
            shellModule.values[FakeShellConModule.CSIDL_PROFILE])
        self.assertTrue(special_folders['Local AppData'] == \
            shellModule.values[FakeShellConModule.CSIDL_LOCAL_APPDATA])
        self.assertTrue(special_folders['Local AppData'].startswith(
            special_folders['AppData']))
        self.assertTrue(special_folders['Common AppData'] == \
            shellModule.values[FakeShellConModule.CSIDL_COMMON_APPDATA])

        # Can't use assert_syncdaemon_path
        for val in special_folders.itervalues():
            self.assertIsInstance(val, str)
            val.decode('utf8')
        # Should not raise exceptions

    def test_get_data_dirs(self):
        """Check thet get_data_dirs uses pathsep correctly."""
        bad_sep = filter(lambda x: x not in os.pathsep, ":;")
        dir_list = ["A", "B", bad_sep, "C"]
        self.patch(os, "environ",
            dict(XDG_DATA_DIRS=os.pathsep.join(
                dir_list)))
        dirs = get_data_dirs()
        self.assertEqual(dirs, dir_list)

    def test_get_config_dirs(self):
        """Check thet get_data_dirs uses pathsep correctly."""
        bad_sep = filter(lambda x: x not in os.pathsep, ":;")
        dir_list = ["A", "B", bad_sep, "C"]
        self.patch(os, "environ",
            dict(XDG_CONFIG_DIRS=os.pathsep.join(
                dir_list)))
        dirs = get_config_dirs()[1:]
        self.assertEqual(dirs, dir_list)
