# -*- coding: utf-8 -*-
# Author: Manuel de la Pena <manuel@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Test the ui controllers."""

from twisted.trial.unittest import TestCase

from mocker import ANY, MATCH, MockerTestCase

from PyQt4.QtGui import QWizard
from ubuntu_sso.qt.controllers import (
    ChooseSignInController,
    CurrentUserController,
    EmailVerificationController,
    ErrorController,
    ForgottenPasswordController,
    FAKE_URL,
    ResetPasswordController,
    SetUpAccountController,
    SuccessController,
    TosController,
    UbuntuSSOWizardController)
from ubuntu_sso.utils.ui import (
    CAPTCHA_SOLUTION_ENTRY,
    CAPTCHA_REQUIRED_ERROR,
    EMAIL1_ENTRY,
    EMAIL2_ENTRY,
    EMAIL_MISMATCH,
    ERROR,
    EMAIL_LABEL,
    EXISTING_ACCOUNT_CHOICE_BUTTON,
    FORGOTTEN_PASSWORD_BUTTON,
    JOIN_HEADER_LABEL,
    LOGIN_PASSWORD_LABEL,
    NAME_ENTRY,
    PASSWORD1_ENTRY,
    PASSWORD2_ENTRY,
    PASSWORD_HELP,
    PASSWORD_TOO_WEAK,
    PASSWORD_MISMATCH,
    RESET_PASSWORD,
    RESET_CODE_ENTRY,
    REQUEST_PASSWORD_TOKEN_LABEL,
    SET_UP_ACCOUNT_CHOICE_BUTTON,
    SET_UP_ACCOUNT_BUTTON,
    SIGN_IN_BUTTON,
    SUCCESS,
    REQUEST_PASSWORD_TOKEN_WRONG_EMAIL,
    REQUEST_PASSWORD_TOKEN_TECH_ERROR,
    TC_BUTTON,
    TOS_LABEL,
    TRY_AGAIN_BUTTON,
    VERIFY_EMAIL_TITLE,
    VERIFY_EMAIL_CONTENT,
    YES_TO_TC,
    is_min_required_password,
    is_correct_email)

#ignore the comon mocker issues with lint
# pylint: disable=W0212,W0104,W0106,E1103


class ChooseSignInControllerTestCase(MockerTestCase):
    """Test the choose sign in controller."""

    def setUp(self):
        """Set tests."""
        super(ChooseSignInControllerTestCase, self).setUp()
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.controller = ChooseSignInController()
        self.controller.view = self.view
        self.controller.backend = self.backend
        self.title = 'title'
        self.subtitle = 'sub'

    def test_setup_ui(self):
        """Test the set up of the ui."""
        self.controller._title = self.title
        self.controller._subtitle = self.subtitle
        self.controller._set_up_translated_strings()
        self.view.header.set_title(self.title)
        self.view.header.set_subtitle(self.subtitle)
        self.controller._connect_buttons()
        self.mocker.replay()
        self.controller.setupUi(self.view)

    def test_set_up_translated_strings(self):
        """Ensure that the translations are used."""
        self.view.ui.existing_account_button.setText(
                                            EXISTING_ACCOUNT_CHOICE_BUTTON)
        self.view.ui.setup_account_button.setText(SET_UP_ACCOUNT_CHOICE_BUTTON)
        self.mocker.replay()
        self.controller._set_up_translated_strings()

    def test_connect_buttons(self):
        """Ensure that all the buttons are correcly connected."""
        self.view.ui.existing_account_button.clicked.connect(
                                            self.controller._set_next_existing)
        self.view.ui.setup_account_button.clicked.connect(
                                                 self.controller._set_next_new)
        self.mocker.replay()
        self.controller._connect_buttons()

    def test_set_next_existing(self):
        """Test the execution of the callback."""
        next_id = 4
        self.view.wizard().current_user_page_id
        self.mocker.result(next_id)
        self.view.next = next_id
        self.view.wizard().next()
        self.mocker.replay()
        self.controller._set_next_existing()

    def test_set_next_new(self):
        """Test the execution of the callback."""
        next_id = 5
        self.view.wizard().setup_account_page_id
        self.mocker.result(next_id)
        self.view.next = next_id
        self.view.wizard().next()
        self.mocker.replay()
        self.controller._set_next_new()


class CurrentUserControllerTestCase(MockerTestCase):
    """Test the current user controller."""

    def setUp(self):
        """Setup tests."""
        super(CurrentUserControllerTestCase, self).setUp()
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.connect = self.mocker.mock()
        self.signal = self.mocker.replace('PyQt4.QtCore.SIGNAL')
        self.controller = CurrentUserController(title='the title',
            subtitle='the subtitle')
        self.controller.view = self.view
        self.controller.backend = self.backend

    def test_translated_strings(self):
        """test that the ui is correctly set up."""
        self.view.ui.email_label.setText(EMAIL_LABEL)
        self.view.ui.email_edit.setPlaceholderText(EMAIL1_ENTRY)
        self.view.ui.password_label.setText(LOGIN_PASSWORD_LABEL)
        self.view.ui.password_edit.setPlaceholderText(PASSWORD1_ENTRY)
        self.view.ui.forgot_password_label.setText(
                                        FAKE_URL % FORGOTTEN_PASSWORD_BUTTON)
        self.view.ui.sign_in_button.setText(SIGN_IN_BUTTON)
        self.mocker.replay()
        self.controller._set_translated_strings()

    def test_connect_ui(self):
        """test that the ui is correctly set up."""
        self.view.ui.sign_in_button.clicked.connect(self.controller.login)
        self.backend.on_login_error_cb = MATCH(callable)
        self.backend.on_logged_in_cb = MATCH(callable)
        self.view.ui.forgot_password_label.linkActivated.connect(
                                                        MATCH(callable))
        self.view.ui.email_edit.textChanged.connect(MATCH(callable))
        self.view.ui.password_edit.textChanged.connect(MATCH(callable))
        self.mocker.replay()
        self.controller._connect_ui()

    def test_title_subtitle(self):
        """Ensure we are storing the title and subtitle correctly."""
        self.mocker.replay()
        self.assertEqual(self.controller._title, 'the title')
        self.assertEqual(self.controller._subtitle, 'the subtitle')


class FakeLineEdit(object):

    """A fake QLineEdit."""

    def __init__(self, *args):
        """Initialize."""
        self._text = u""

    def text(self):
        """Save text."""
        return self._text

    # setText is inherited
    # pylint: disable=C0103
    def setText(self, text):
        """Return saved text."""
        self._text = text


class FakePage(object):

    """A fake wizard page."""

    app_name = "APP"

    def wizard(self):
        """Fake wizard."""
        return self


class FakeCurrentUserPage(FakePage):

    """Fake CurrentuserPage."""

    def __init__(self, *args):
        """Initialize."""
        super(FakeCurrentUserPage, self).__init__(*args)
        self.ui = self
        self.ui.email_edit = FakeLineEdit()
        self.ui.password_edit = FakeLineEdit()
        self.sign_in_button = self
        self._enabled = False

    # setEnabled is inherited
    # pylint: disable=C0103
    def setEnabled(self, value):
        """Fake setEnabled."""
        self._enabled = value

    def enabled(self):
        """Fake enabled."""
        return self._enabled


class CurrentUserControllerValidationMockTest(MockerTestCase):
    """Test the choose sign in controller."""

    def setUp(self):
        """Set tests."""
        super(CurrentUserControllerValidationMockTest, self).setUp()
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.controller = CurrentUserController()
        self.controller.view = self.view
        self.controller.backend = self.backend
        self.title = 'title'
        self.subtitle = 'sub'

    def test_setup_ui(self):
        """Test the set up of the ui."""
        self.controller._title = self.title
        self.controller._subtitle = self.subtitle
        self.backend = yield self.controller.get_backend()
        self.view.header.set_title(self.title)
        self.view.header.set_subtitle(self.subtitle)
        self.controller._set_translated_strings()
        self.mocker.replay()
        self.controller.setupUi(self.view)


class CurrentUserControllerErrorTestCase(TestCase):

    """Tests for CurrentUserController's error handler."""

    on_error_method_name = "on_login_error"
    controller_class = CurrentUserController

    def setUp(self):
        """Setup test."""
        super(CurrentUserControllerErrorTestCase, self).setUp()
        self.message_box = FakeMessageBox()
        self.controller = self.controller_class(
            message_box=self.message_box)
        self.controller.view = FakePage()
        self._called = False
        self.on_error_method = getattr(
            self.controller, self.on_error_method_name)

    def test_error_message_key(self):
        """Test that on_login_error reacts to errors with "error_message"."""
        self.on_error_method({"error_message": "WORRY!"})
        self.assertEqual(self.message_box.critical_args, (('WORRY!',), {}))

    def test_message_key(self):
        """Test that on_login_error reacts to errors with "message"."""
        self.on_error_method({"message": "WORRY!"})
        self.assertEqual(self.message_box.critical_args, (('WORRY!',), {}))

    def test_broken_error(self):
        """Test that on_login_error reacts to broken errors."""
        self.on_error_method({"boo!": "WORRY!"})
        self.assertEqual(self.message_box.critical_args,
            (("Error: {'boo!': 'WORRY!'}",), {}))

    def test_all_and_message(self):
        """Test that on_login_error reacts to broken errors."""
        self.on_error_method(
            {"message": "WORRY!", "__all__": "MORE!"})
        self.assertEqual(self.message_box.critical_args,
            (('MORE!\nWORRY!',), {}))

    def test_all_and_error_message(self):
        """Test that on_login_error reacts to broken errors."""
        self.on_error_method(
            {"error_message": "WORRY!", "__all__": "MORE!"})
        self.assertEqual(self.message_box.critical_args,
            (('MORE!\nWORRY!',), {}))

    def test_only_all(self):
        """Test that on_login_error reacts to broken errors."""
        self.on_error_method(
            {"__all__": "MORE!"})
        self.assertEqual(self.message_box.critical_args,
            (('MORE!',), {}))


class EmailVerificationControllerErrorTestCase(
    CurrentUserControllerErrorTestCase):

    """Tests for EmailVerificationController's error handler."""

    on_error_method_name = "on_email_validation_error"
    controller_class = EmailVerificationController

    def setUp(self):
        """Setup test."""
        super(EmailVerificationControllerErrorTestCase, self).setUp()
        # This error handler takes one extra argument.
        self.on_error_method = lambda error: getattr(
            self.controller, self.on_error_method_name)('APP', error)


class SetUpAccountControllerErrorTestCase(
    EmailVerificationControllerErrorTestCase):

    """Tests for SetUpAccountController's error handler."""

    on_error_method_name = "on_user_registration_error"
    controller_class = SetUpAccountController

    def setUp(self):
        """Setup test."""
        super(SetUpAccountControllerErrorTestCase, self).setUp()
        self.patch(self.controller, "_refresh_captcha", lambda *args: None)


class ResetPasswordControllerErrorTestCase(
    EmailVerificationControllerErrorTestCase):

    """Tests for ResetPasswordController's error handler."""

    on_error_method_name = "on_password_change_error"
    controller_class = ResetPasswordController


class CurrentUserControllerValidationTest(TestCase):

    """Tests for CurrentUserController, but without Mocker."""

    def setUp(self):
        """Setup test."""
        super(CurrentUserControllerValidationTest, self).setUp()
        self.message_box = FakeMessageBox()
        self.controller = CurrentUserController(
            message_box=self.message_box)
        self.controller.view = FakeCurrentUserPage()
        self._called = False

    def _set_called(self, *args, **kwargs):
        """Store 'args' and 'kwargs' for test assertions."""
        self._called = (args, kwargs)

    def test_valid(self):
        """Enable the button with a valid email/password."""
        self.controller.view.email_edit.setText("a@b")
        self.controller.view.password_edit.setText("pass")
        self.controller._validate()
        self.assertTrue(self.controller.view.sign_in_button.enabled())

    def test_invalid_email(self):
        """The submit button should be disabled with an invalid email."""
        self.controller.view.email_edit.setText("ab")
        self.controller.view.password_edit.setText("pass")
        self.controller._validate()
        self.assertFalse(self.controller.view.sign_in_button.enabled())

    def test_invalid_password(self):
        """The submit button should be disabled with an invalid password."""
        self.controller.view.email_edit.setText("a@b")
        self.controller.view.password_edit.setText("")
        self.controller._validate()
        self.assertFalse(self.controller.view.sign_in_button.enabled())

    def test_invalid_both(self):
        """The submit button should be disabled with invalid data."""
        self.controller.view.email_edit.setText("ab")
        self.controller.view.password_edit.setText("")
        self.controller._validate()
        self.assertFalse(self.controller.view.sign_in_button.enabled())


class SetUpAccountControllerTestCase(MockerTestCase):
    """test the controller used to setup a new account."""

    def setUp(self):
        """Set the different tests."""
        super(SetUpAccountControllerTestCase, self).setUp()
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.get_password_strength = self.mocker.replace(
                                 'ubuntu_sso.utils.ui.get_password_strength')
        self.app_name = 'test'
        self.help = 'help'
        self.message_box = self.mocker.mock()
        self.controller = SetUpAccountController(message_box=self.message_box)
        self.controller.backend = self.backend
        self.controller.view = self.view

    def test_set_translated_strings(self):
        """Ensure all the strings are set."""
        self.view.wizard().app_name
        self.mocker.result(self.app_name)
        self.view.ui.name_label.setText(NAME_ENTRY)
        self.view.ui.email_label.setText(EMAIL1_ENTRY)
        self.view.ui.confirm_email_label.setText(EMAIL2_ENTRY)
        self.view.ui.password_label.setText(PASSWORD1_ENTRY)
        self.view.ui.confirm_password_label.setText(PASSWORD2_ENTRY)
        self.view.ui.password_info_label.setText(PASSWORD_HELP)
        self.view.ui.captcha_solution_edit.setPlaceholderText(
                                                    CAPTCHA_SOLUTION_ENTRY)
        self.view.ui.terms_checkbox.setText(
                                       YES_TO_TC % {'app_name': self.app_name})
        self.view.ui.terms_button.setText(TC_BUTTON)
        self.view.ui.set_up_button.setText(SET_UP_ACCOUNT_BUTTON)
        self.mocker.replay()
        self.controller._set_translated_strings()

    def test_set_titles(self):
        """Test how the different titles are set."""
        self.view.wizard().app_name
        self.mocker.result(self.app_name)
        self.view.wizard().help_text
        self.mocker.result(self.help)
        self.view.header.set_title(
            JOIN_HEADER_LABEL % {'app_name': self.app_name})
        self.view.header.set_subtitle(self.help)
        self.mocker.replay()
        self.controller._set_titles()

    def test_connect_ui_elements(self):
        """Test that the ui elements are correctly connect."""
        self.view.ui.terms_button.clicked.connect(self.controller.set_next_tos)
        self.view.ui.set_up_button.clicked.connect(
                                           self.controller.set_next_validation)
        self.controller._enable_setup_button
        self.mocker.result(lambda: None)
        self.view.ui.name_edit.textEdited.connect(MATCH(callable))
        self.view.ui.email_edit.textEdited.connect(MATCH(callable))
        self.view.ui.confirm_email_edit.textEdited.connect(MATCH(callable))
        self.view.ui.password_edit.textEdited.connect(MATCH(callable))
        self.view.ui.confirm_password_edit.textEdited.connect(MATCH(callable))
        self.view.ui.captcha_solution_edit.textEdited.connect(MATCH(callable))
        self.view.ui.terms_checkbox.stateChanged.connect(MATCH(callable))
        self.view.ui.refresh_label.linkActivated.connect(MATCH(callable))
        # set the callbacks for the captcha generation
        self.backend.on_captcha_generated_cb = MATCH(callable)
        self.backend.on_captcha_generation_error_cb = MATCH(callable)
        self.backend.on_user_registration_error_cb = MATCH(callable)
        self.backend.on_user_registered_cb = MATCH(callable)
        self.mocker.replay()
        self.controller._connect_ui_elements()

    def test_is_correct_password_confirmation_false(self):
        """Test when the password is not correct."""
        password = 'test'
        self.view.ui.password_edit.text()
        self.mocker.result('other')
        self.mocker.replay()
        self.assertFalse(
                    self.controller.is_correct_password_confirmation(password))

    def test_is_correct_password_confirmation_true(self):
        """Test when the password is correct."""
        password = 'test'
        self.view.ui.password_edit.text()
        self.mocker.result(password)
        self.mocker.replay()
        self.assertTrue(
                    self.controller.is_correct_password_confirmation(password))

    def test_is_correct_email_confirmation_false(self):
        """Test when the emails are not the same."""
        email_address = 'address'
        self.view.ui.email_edit.text()
        self.mocker.result('other')
        self.mocker.replay()
        self.assertFalse(
                self.controller.is_correct_email_confirmation(email_address))

    def test_is_correct_email_confirmation_true(self):
        """Test when the emails are the same."""
        email_address = 'address'
        self.view.ui.email_edit.text()
        self.mocker.result(email_address)
        self.mocker.replay()
        self.assertTrue(
                self.controller.is_correct_email_confirmation(email_address))

    def test_set_next_tos(self):
        """Test the callback."""
        next_id = 2
        self.view.wizard().tos_page_id
        self.mocker.result(next_id)
        self.view.next = next_id
        self.view.wizard().next()
        self.mocker.replay()
        self.controller.set_next_tos()

    def test_set_next_validation(self):
        """Test the callback."""
        name = 'name'
        email = 'email@example.com'
        password = 'Qwerty9923'
        captcha_id = 'captcha_id'
        captcha_solution = 'captcha_solution'
        self.view.wizard().app_name
        self.mocker.result(self.app_name)
        self.view.ui.name_edit.text()
        self.mocker.result(name)
        self.view.ui.email_edit.text()
        self.mocker.result(email)
        self.view.ui.email_edit.text()
        self.mocker.result(email)
        self.view.ui.confirm_email_edit.text()
        self.mocker.result(email)
        self.view.ui.password_edit.text()
        self.mocker.result(password)
        self.view.ui.password_edit.text()
        self.mocker.result(password)
        self.view.ui.confirm_password_edit.text()
        self.mocker.result(password)
        self.view.captcha_id
        self.mocker.result(captcha_id)
        self.view.ui.captcha_solution_edit.text()
        self.mocker.result(captcha_solution)
        self.backend.register_user(self.app_name, email, password, name,
                                   captcha_id, captcha_solution)
        self.view.wizard().email_verification_page_id
        self.view.wizard().page(None).controller.set_titles()
        self.view.ui.captcha_solution_edit.text()
        self.mocker.replay()
        self.controller.set_next_validation()

    def test_set_next_validation_wrong_email(self):
        """Test the callback when there is a wrong email."""
        email = 'email@example.com'
        password = 'Qwerty9923'
        captcha_solution = 'captcha'
        self.view.ui.email_edit.text()
        self.mocker.result(email)
        self.view.ui.confirm_email_edit.text()
        self.mocker.result('email2')
        self.view.ui.password_edit.text()
        self.mocker.result(password)
        self.view.ui.confirm_password_edit.text()
        self.mocker.result(password)
        self.view.ui.captcha_solution_edit.text()
        self.mocker.result(captcha_solution)
        self.message_box.critical(EMAIL_MISMATCH)
        self.mocker.replay()
        self.assertFalse(self.controller.validate_form())

    def test_set_next_validation_not_min_password(self):
        """Test the callback with a weak password."""
        weak_password = 'weak'
        email = 'email@example.com'
        captcha_solution = 'captcha'
        self.view.ui.email_edit.text()
        self.mocker.result(email)
        self.view.ui.confirm_email_edit.text()
        self.mocker.result(email)
        self.view.ui.password_edit.text()
        self.mocker.result(weak_password)
        self.view.ui.captcha_solution_edit.text()
        self.mocker.result(captcha_solution)
        self.view.ui.confirm_password_edit.text()
        self.mocker.result(weak_password)
        self.message_box.critical(PASSWORD_TOO_WEAK)
        self.mocker.replay()
        self.assertFalse(self.controller.validate_form())

    def test_set_next_validation_wrong_password(self):
        """Test the callback where the password is wrong."""
        password = 'Qwerty9923'
        email = 'email@example.com'
        captcha_solution = 'captcha'
        self.view.ui.email_edit.text()
        self.mocker.result(email)
        self.view.ui.confirm_email_edit.text()
        self.mocker.result(email)
        self.view.ui.password_edit.text()
        self.mocker.result(password)
        self.view.ui.confirm_password_edit.text()
        self.mocker.result('other_password')
        self.view.ui.captcha_solution_edit.text()
        self.mocker.result(captcha_solution)
        self.message_box.critical(PASSWORD_MISMATCH)
        self.mocker.replay()
        self.assertFalse(self.controller.validate_form())

    def test_set_next_validation_empty_captcha(self):
        """Test the callback where the password is wrong."""
        password = 'Qwerty9923'
        email = 'email@example.com'
        captcha_solution = ''
        self.view.ui.email_edit.text()
        self.mocker.result(email)
        self.view.ui.confirm_email_edit.text()
        self.mocker.result(email)
        self.view.ui.password_edit.text()
        self.mocker.result(password)
        self.view.ui.confirm_password_edit.text()
        self.mocker.result(password)
        self.view.ui.captcha_solution_edit.text()
        self.mocker.result(captcha_solution)
        self.message_box.critical(CAPTCHA_REQUIRED_ERROR)
        self.mocker.replay()
        self.assertFalse(self.controller.validate_form())

    def test_set_line_edits_validations(self):
        """Set the validations to be performed on the edits."""
        self.view.ui.email_edit
        self.mocker.result(self.view)
        self.view.set_line_edit_validation_rule(self.view, is_correct_email)
        self.view.ui.confirm_email_edit
        self.mocker.result(self.view)
        self.view.set_line_edit_validation_rule(self.view, MATCH(callable))
        self.view.ui.confirm_email_edit.textChanged.emit
        self.mocker.result(lambda: None)
        self.view.ui.email_edit.textChanged.connect(MATCH(callable))
        self.view.ui.password_edit
        self.mocker.result(self.view)
        self.view.set_line_edit_validation_rule(self.view,
                                                is_min_required_password)
        self.view.ui.confirm_password_edit
        self.mocker.result(self.view)
        self.view.set_line_edit_validation_rule(self.view, MATCH(callable))
        self.view.ui.confirm_password_edit.textChanged.emit
        self.mocker.result(lambda: None)
        self.view.ui.password_edit.textChanged.connect(MATCH(callable))
        self.mocker.replay()
        self.controller._set_line_edits_validations()


class FakeMessageBox(object):

    """A fake message box."""

    args = None
    critical_args = None

    def __init__(self, *args, **kwargs):
        self.args = (args, kwargs)

    def critical(self, *args, **kwargs):
        """Fake critical popup."""
        self.critical_args = (args, kwargs)


class FakeView(object):

    """A fake view."""

    app_name = "TestApp"

    def wizard(self):
        """Use itself as a fake wizard, too."""
        return self


class SetupAccountControllerValidationTest(TestCase):

    """Tests for SetupAccountController, but without Mocker."""

    def setUp(self):
        """Set the different tests."""
        super(SetupAccountControllerValidationTest, self).setUp()
        self.message_box = FakeMessageBox()
        self.controller = SetUpAccountController(message_box=self.message_box)
        self.patch(self.controller, '_refresh_captcha', self._set_called)
        self.patch(self.controller, 'view', FakeView())
        self._called = False

    def _set_called(self, *args, **kwargs):
        """Store 'args' and 'kwargs' for test assertions."""
        self._called = (args, kwargs)

    def test_on_user_registration_refresh_captcha(self):
        """If there is a user reg. error, captcha should refresh."""
        self.controller.on_user_registration_error('TestApp', {})
        self.assertEqual(self._called, ((), {}))

    def test_on_user_registration_all_only(self):
        """Pass only a __all__ error key."""
        self.controller.on_user_registration_error('TestApp',
            {'__all__': "Error in All"})
        self.assertEqual(self.message_box.critical_args, ((
            "Error in All", ), {}))

    def test_on_user_registration_all_fields(self):
        """Pass all known error keys, plus unknown one."""
        self.controller.on_user_registration_error('TestApp',
            {'__all__': "Error in All",
             'email': "Error in email",
             'pasword': "Error in password",
             'unknownfield': "Error in unknown",
            })
        self.assertEqual(self.message_box.critical_args, ((
            "Error in All", ), {}))


class TosControllerTestCase(MockerTestCase):
    """Test the tos controller."""

    def setUp(self):
        """Set the tests."""
        super(TosControllerTestCase, self).setUp()
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.title = 'title'
        self.subtitle = 'sub'
        self.url = 'url'
        self.controller = TosController(title=self.title,
                                        subtitle=self.subtitle,
                                        tos_url=self.url)
        self.controller.view = self.view

    def test_setup_ui(self):
        """Test the set up of the ui."""
        self.view.header.set_title(self.title)
        self.view.header.set_subtitle(self.subtitle)
        self.view.ui.terms_webkit.load(ANY)
        self.view.ui.tos_link_label.setText(
            TOS_LABEL %
            {'url': self.url})
        self.mocker.replay()
        self.controller.setupUi(self.view)


class EmailVerificationControllerTestCase(MockerTestCase):
    """Test the controller."""

    def setUp(self):
        """Set tests."""
        super(EmailVerificationControllerTestCase, self).setUp()
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.controller = EmailVerificationController(
            message_box=self.mocker.mock())
        self.controller.view = self.view
        self.controller.backend = self.backend

    def test_set_translated_strings(self):
        """Test that strings are translated."""
        self.view.ui.verification_code_edit.setPlaceholderText(
                                                            VERIFY_EMAIL_TITLE)
        self.mocker.replay()
        self.controller._set_translated_strings()

    def test_connect_ui_elements(self):
        """Set the ui connections."""
        self.view.next_button.clicked.connect(self.controller.validate_email)
        self.backend.on_email_validated_cb = MATCH(callable)
        self.backend.on_email_validation_error_cb = \
                                    self.controller.on_email_validation_error
        self.mocker.replay()
        self.controller._connect_ui_elements()

    def test_set_titles(self):
        """Test that the titles are set."""
        self.view.header.set_title(VERIFY_EMAIL_CONTENT)
        self.view.header.set_subtitle(VERIFY_EMAIL_CONTENT % {
            "app_name": None,
            "email": None,
            })
        self.mocker.replay()
        self.view.header.set_title(VERIFY_EMAIL_CONTENT)
        self.view.header.set_subtitle(VERIFY_EMAIL_CONTENT % {
            "app_name": None,
            "email": None,
            })

    def test_validate_email(self):
        """Test the callback."""
        email_address = 'email@test.com'
        password = 'password'
        app_name = 'app_name'
        code = 'code'
        self.view.wizard().field('email_address').toString()
        self.mocker.result(email_address)
        self.view.wizard().field('password').toString()
        self.mocker.result(password)
        self.view.wizard().app_name
        self.mocker.result(app_name)
        self.view.ui.verification_code_edit.text()
        self.mocker.result(code)
        self.backend.validate_email(app_name, email_address, password, code)
        self.mocker.replay()
        self.controller.validate_email()


class EmailVerificationControllerValidationTestCase(TestCase):
    """Tests for EmailVerificationController, but without Mocker."""

    def setUp(self):
        """Set the different tests."""
        super(EmailVerificationControllerValidationTestCase, self).setUp()
        self.message_box = FakeMessageBox()
        self.controller = EmailVerificationController(
            message_box=self.message_box)
        self.patch(self.controller, 'view', FakeView())
        self._called = False

    def test_on_email_validation_error(self):
        """Test that on_email_validation_error callback works as expected."""
        error = dict(errtype='BadTokenError')
        app_name = 'app_name'
        self.controller.on_email_validation_error(app_name, error)
        self.assertEqual(self.message_box.critical_args,
            (("Error: {'errtype': 'BadTokenError'}",), {}))


class ErrorControllerTestCase(MockerTestCase):
    """Test the success page controller."""

    def setUp(self):
        """Set the tests."""
        super(ErrorControllerTestCase, self).setUp()
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.controller = ErrorController()
        self.controller.view = self.view
        self.controller.backend = self.backend

    def test_set_ui(self):
        """Test the process that sets the ui."""
        self.controller._title = ERROR
        self.controller._subtitle = ERROR
        self.view.next = -1
        self.view.ui.error_message_label.setText(ERROR)
        self.view.header.set_title(ERROR)
        self.view.header.set_subtitle(ERROR)
        self.mocker.replay()
        self.controller.setupUi(self.view)

    def test_hide_titles(self):
        """Test how the different titles are set."""
        self.view.next = -1
        self.view.ui.error_message_label.setText(ERROR)
        self.view.header.set_title('')
        self.view.header.set_subtitle('')
        self.mocker.replay()
        self.controller.setupUi(self.view)


class SuccessControllerTestCase(MockerTestCase):
    """Test the success page controller."""

    def setUp(self):
        """Set the tests."""
        super(SuccessControllerTestCase, self).setUp()
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.controller = SuccessController()
        self.controller.view = self.view
        self.controller.backend = self.backend

    def test_set_ui(self):
        """Test the process that sets the ui."""
        self.controller._title = SUCCESS
        self.controller._subtitle = SUCCESS
        self.view.next = -1
        self.view.ui.success_message_label.setText(SUCCESS)
        self.view.header.set_title(SUCCESS)
        self.view.header.set_subtitle(SUCCESS)
        self.mocker.replay()
        self.controller.setupUi(self.view)

    def test_hide_titles(self):
        """Test how the different titles are set."""
        self.view.next = -1
        self.view.ui.success_message_label.setText(SUCCESS)
        self.view.header.set_title('')
        self.view.header.set_subtitle('')
        self.mocker.replay()
        self.controller.setupUi(self.view)


class UbuntuSSOWizardControllerTestCase(MockerTestCase):
    """Test the wizard controller."""

    def setUp(self):
        """Set tests."""
        super(UbuntuSSOWizardControllerTestCase, self).setUp()
        self.app_name = 'test'
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.callback = self.mocker.mock()
        self.controller = UbuntuSSOWizardController()
        self.controller.view = self.view
        self.controller.backend = self.backend

    def test_on_user_cancelation(self):
        """Test that the callback is indeed called."""
        self.view.app_name
        self.mocker.result(self.app_name)
        self.controller.user_cancellation_callback = self.callback
        self.callback(self.app_name)
        self.view.close()
        self.mocker.replay()
        self.controller.on_user_cancelation()

    def test_on_login_success(self):
        """Test that the callback is indeed called."""
        app_name = 'app'
        email = 'email'
        self.controller.login_success_callback = self.callback
        self.callback(app_name, email)
        self.mocker.replay()
        self.controller.login_success_callback(app_name, email)

    def test_on_registration_success(self):
        """Test that the callback is indeed called."""
        app_name = 'app'
        email = 'email'
        self.controller.registration_success_callback = self.callback
        self.callback(app_name, email)
        self.mocker.replay()
        self.controller.registration_success_callback(app_name, email)

    def test_show_success_message(self):
        """Test that the correct page will be shown."""
        success_page_id = 0
        # the buttons layout we expect to have
        buttons_layout = []
        buttons_layout.append(QWizard.Stretch)
        buttons_layout.append(QWizard.FinishButton)
        self.view.success_page_id
        self.mocker.result(success_page_id)
        self.view.currentPage()
        self.mocker.result(self.view)
        self.view.next = success_page_id
        self.view.next()
        self.view.setButtonLayout(buttons_layout)
        self.mocker.replay()
        self.controller.show_success_message()

    def test_show_error_message(self):
        """Test that the correct page will be shown."""
        error_page_id = 0
        # the buttons layout we expect to have
        buttons_layout = []
        buttons_layout.append(QWizard.Stretch)
        buttons_layout.append(QWizard.FinishButton)
        self.view.error_page_id
        self.mocker.result(error_page_id)
        self.view.currentPage()
        self.mocker.result(self.view)
        self.view.next = error_page_id
        self.view.next()
        self.view.setButtonLayout(buttons_layout)
        self.mocker.replay()
        self.controller.show_error_message()

    def test_setup_ui(self):
        """Test that the ui is connect."""
        self.view.setWizardStyle(QWizard.ModernStyle)
        self.view.button(QWizard.CancelButton)
        self.mocker.result(self.view)
        self.view.clicked.connect(self.controller.on_user_cancelation)
        self.view.loginSuccess.connect(self.controller.on_login_success)
        self.view.registrationSuccess.connect(
                                    self.controller.on_registration_success)
        self.mocker.replay()
        self.controller.setupUi(self.view)


class FakeForgottenPasswordPage(FakeView):
    """Fake the page."""

    def __init__(self):
        self.email_address_line_edit = self
        self.send_button = self
        self._text = ""
        self._enabled = False
        super(FakeForgottenPasswordPage, self).__init__()

    def text(self):
        """Return text."""
        return self._text

    # setText, setEnabled are inherited
    # pylint: disable=C0103
    def setText(self, text):
        """Save text."""
        self._text = text

    def setEnabled(self, value):
        """Fake setEnabled."""
        self._enabled = value

    def enabled(self):
        """Fake enabled."""
        return self._enabled


class ForgottenPasswordControllerValidationTest(TestCase):

    """Tests for ForgottenPasswordController, but without Mocker."""

    def setUp(self):
        """Set the different tests."""
        super(ForgottenPasswordControllerValidationTest, self).setUp()
        self.message_box = FakeMessageBox()
        self.controller = ForgottenPasswordController(
            message_box=self.message_box)
        self.controller.view = FakeForgottenPasswordPage()
        self._called = False

    def _set_called(self, *args, **kwargs):
        """Store 'args' and 'kwargs' for test assertions."""
        self._called = (args, kwargs)

    def test_valid(self):
        """The submit button should be enabled with a valid email."""
        self.controller.view.email_address_line_edit.setText("a@b")
        self.assertNotEqual(unicode(
            self.controller.view.email_address_line_edit.text()), u"")
        self.controller._validate()
        self.assertTrue(self.controller.view.send_button.enabled())

    def test_invalid(self):
        """The submit button should be disabled with an invalid email."""
        self.controller.view.email_address_line_edit.setText("ab")
        self.assertNotEqual(
            unicode(self.controller.view.email_address_line_edit.text()), u"")
        self.controller._validate()
        self.assertFalse(self.controller.view.send_button.enabled())

    def test_empty(self):
        """The submit button should be disabled without email."""
        self.assertEqual(
            unicode(self.controller.view.email_address_line_edit.text()), u"")
        self.assertFalse(self.controller.view.send_button.enabled())


class ForgottenPasswordControllerTestCase(MockerTestCase):
    """Test the controller of the fogotten password page."""

    def setUp(self):
        """Setup the tests."""
        super(ForgottenPasswordControllerTestCase, self).setUp()
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.controller = ForgottenPasswordController()
        self.controller.view = self.view
        self.controller.backend = self.backend
        self.app_name = 'app_name'

    def test_register_fields(self):
        """Ensure that all the diff fields are registered."""
        line_edit = 'line_edit'
        self.view.email_address_line_edit
        self.mocker.result(line_edit)
        self.view.registerField('email_address', line_edit)
        self.mocker.replay()
        self.controller._register_fields()

    def test_set_translated_strings(self):
        """Ensure that the correct strings are translated."""
        self.view.wizard().app_name
        self.mocker.result(self.app_name)
        self.view.forgotted_password_intro_label.setText(
                                    REQUEST_PASSWORD_TOKEN_LABEL % {'app_name':
                                    self.app_name})
        self.view.email_address_label.setText(EMAIL_LABEL)
        self.view.send_button.setText(RESET_PASSWORD)
        self.view.try_again_button.setText(TRY_AGAIN_BUTTON)
        self.mocker.replay()
        self.controller._set_translated_strings()

    def test_set_enhanced_line_edit(self):
        """Test that the correct line enhancements have been added."""
        line_edit = 'line_edit'
        self.view.email_address_line_edit
        self.mocker.result(line_edit)
        self.view.set_line_edit_validation_rule(line_edit, is_correct_email)
        self.mocker.replay()
        self.controller._set_enhanced_line_edit()

    def test_connect_ui(self):
        """Test that the correct ui signals are connected."""
        self.view.email_address_line_edit.textChanged.connect(MATCH(callable))
        self.view.send_button.clicked.connect(MATCH(callable))
        self.view.try_again_button.clicked.connect(
                                                  self.controller.on_try_again)
        self.backend.on_password_reset_token_sent_cb = MATCH(callable)
        self.backend.on_password_reset_error_cb = \
                                        self.controller.on_password_reset_error
        self.mocker.replay()
        self.controller._connect_ui()

    def test_on_try_again(self):
        """Test that the on_try_again callback does work as expected."""
        self.view.error_label.setVisible(False)
        self.view.try_again_widget.setVisible(False)
        self.view.email_widget.setVisible(True)
        self.mocker.replay()
        self.controller.on_try_again()

    def test_on_password_reset_token_sent(self):
        """Test that the on_password_token_sent callback works as expected."""
        page_id = 3
        self.view.wizard().reset_password_page_id
        self.mocker.result(page_id)
        self.view.next = page_id
        self.view.wizard().next()
        self.mocker.replay()
        self.controller.on_password_reset_token_sent()

    def test_on_password_reset_error_token_error(self):
        """Test that the on_password_reset_error callback works as expected."""
        error = dict(errtype='ResetPasswordTokenError')
        self.view.error_label.setText(REQUEST_PASSWORD_TOKEN_WRONG_EMAIL)
        self.view.error_label.setVisible(True)
        self.mocker.replay()
        self.controller.on_password_reset_error(self.app_name, error)

    def test_on_password_reset_error_general_error(self):
        """Test that the on_password_reset_error callback works as expected."""
        error = dict(errtype='RandomError')
        self.view.email_widget.setVisible(False)
        self.view.forgotted_password_intro_label.setVisible(False)
        self.view.try_again_wisget.setVisible(True)
        self.view.error_label.setText(REQUEST_PASSWORD_TOKEN_TECH_ERROR)
        self.mocker.replay()
        self.controller.on_password_reset_error(self.app_name, error)


class ResetPasswordControllerTestCase(MockerTestCase):
    """Ensure that the reset password works as expected."""

    def setUp(self):
        """Setup the tests."""
        super(ResetPasswordControllerTestCase, self).setUp()
        self.view = self.mocker.mock()
        self.backend = self.mocker.mock()
        self.controller = ResetPasswordController()
        self.controller.view = self.view
        self.controller.backend = self.backend

    def test_set_translated_strings(self):
        """Ensure that the correct strings are set."""
        self.controller._subtitle = PASSWORD_HELP
        self.view.ui.reset_code_line_edit.setPlaceholderText(RESET_CODE_ENTRY)
        self.view.ui.password_line_edit.setPlaceholderText(PASSWORD1_ENTRY)
        self.view.ui.confirm_password_line_edit.setPlaceholderText(
                                                            PASSWORD2_ENTRY)
        self.view.ui.reset_password_button.setText(RESET_PASSWORD)
        self.view.setSubTitle(PASSWORD_HELP)
        self.mocker.replay()
        self.controller._set_translated_strings()

    def test_connect_ui(self):
        """Ensure that the diffent signals from the ui are connected."""
        self.view.ui.reset_password_button.clicked.connect(
                                            self.controller.set_new_password)
        self.backend.on_password_changed_cb = \
                                        self.controller.on_password_changed
        self.backend.on_password_change_error_cb = \
                                    self.controller.on_password_change_error
        self.view.ui.reset_code_line_edit.textChanged.connect(
            MATCH(callable))
        self.view.ui.password_line_edit.textChanged.connect(
            MATCH(callable))
        self.view.ui.confirm_password_line_edit.textChanged.connect(
            MATCH(callable))

        self.mocker.replay()
        self.controller._connect_ui()

    def test_add_line_edits_validations(self):
        """Ensure that the line validation have been added."""
        line_edit = 'line_edit'
        emit = 'emit'
        self.view.ui.password_line_edit
        self.mocker.result('line_edit')
        self.view.set_line_edit_validation_rule(line_edit,
                                                is_min_required_password)
        self.view.ui.confirm_password_line_edit
        self.mocker.result('line_edit')
        self.view.set_line_edit_validation_rule(line_edit, MATCH(callable))
        self.view.ui.confirm_password_line_edit.textChanged.emit
        self.mocker.result(emit)
        self.view.ui.password_line_edit.textChanged.connect(emit)
        self.mocker.replay()
        self.controller._add_line_edits_validations()

    def test_set_new_password(self):
        """Test that the correct action is performed."""
        app_name = 'app_name'
        email = 'email'
        code = 'code'
        password = 'password'
        self.view.wizard().app_name
        self.mocker.result(app_name)
        self.view.wizard().field('email_address').toString()
        self.mocker.result(email)
        self.view.ui.reset_code_line_edit.text()
        self.mocker.result(code)
        self.view.ui.password_line_edit.text()
        self.mocker.result(password)
        self.backend.set_new_password(app_name, email, code, password)
        self.mocker.replay()
        self.controller.set_new_password()

    def test_is_correct_password_confirmation_true(self):
        """Test that the correct password confirmation is used."""
        password = 'password'
        self.view.ui.password_line_edit.text()
        self.mocker.result(password)
        self.mocker.replay()
        self.assertTrue(self.controller.is_correct_password_confirmation(
                                                                    password))

    def test_is_correct_password_confirmation_false(self):
        """Test that the correct password confirmation is used."""
        password = 'password'
        self.view.ui.password_line_edit.text()
        self.mocker.result(password + password)
        self.mocker.replay()
        self.assertFalse(self.controller.is_correct_password_confirmation(
                                                                    password))


class FakeResetPasswordPage(object):

    """Fake ResetPasswordPage."""

    def __init__(self, *args):
        """Initialize."""
        self.ui = self
        self.ui.reset_code_line_edit = FakeLineEdit()
        self.ui.password_line_edit = FakeLineEdit()
        self.ui.confirm_password_line_edit = FakeLineEdit()
        self.reset_password_button = self
        self._enabled = 9  # Intentionally wrong.

    # setEnabled is inherited
    # pylint: disable=C0103
    def setEnabled(self, value):
        """Fake setEnabled."""
        self._enabled = value

    def enabled(self):
        """Fake enabled."""
        return self._enabled


class ResetPasswordControllerValidationTest(TestCase):

    """Tests for ResetPasswordController, but without Mocker."""

    def setUp(self):
        """Setup test."""
        super(ResetPasswordControllerValidationTest, self).setUp()
        self.controller = ResetPasswordController()
        self.controller.view = FakeResetPasswordPage()
        self._called = False

    def _set_called(self, *args, **kwargs):
        """Store 'args' and 'kwargs' for test assertions."""
        self._called = (args, kwargs)

    def test_valid(self):
        """Enable the button with valid data."""
        self.controller.view.reset_code_line_edit.setText("ABCD")
        self.controller.view.password_line_edit.setText("1234567A")
        self.controller.view.confirm_password_line_edit.setText("1234567A")
        self.controller._validate()
        self.assertTrue(self.controller.view.reset_password_button.enabled())

    def test_invalid_code(self):
        """Disable the button with an invalid code."""
        self.controller.view.reset_code_line_edit.setText("")
        self.controller.view.password_line_edit.setText("1234567A")
        self.controller.view.confirm_password_line_edit.setText("1234567A")
        self.controller._validate()
        self.assertFalse(self.controller.view.reset_password_button.enabled())

    def test_invalid_password(self):
        """Disable the button with an invalid password."""
        self.controller.view.reset_code_line_edit.setText("")
        self.controller.view.password_line_edit.setText("1234567")
        self.controller.view.confirm_password_line_edit.setText("1234567")
        self.controller._validate()
        self.assertFalse(self.controller.view.reset_password_button.enabled())

    def test_invalid_confirm(self):
        """Disable the button with an invalid password confirm."""
        self.controller.view.reset_code_line_edit.setText("")
        self.controller.view.password_line_edit.setText("1234567A")
        self.controller.view.confirm_password_line_edit.setText("1234567")
        self.controller._validate()
        self.assertFalse(self.controller.view.reset_password_button.enabled())
