# -*- coding: utf-8 -*-
# Author: Manuel de la Pena <manuel@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Qt implementation of the UI."""

from PyQt4.QtCore import pyqtSignal
from PyQt4.QtCore import Qt
from PyQt4.QtGui import (
    QApplication,
    QWidget,
    QCursor,
    QHBoxLayout,
    QVBoxLayout,
    QPixmap,
    QStyle,
    QWizard,
    QWizardPage,
    QLabel)

from ubuntu_sso.logger import setup_logging
# pylint: disable=F0401,E0611
from ubuntu_sso.qt.choose_sign_in_ui import Ui_ChooseSignInPage
from ubuntu_sso.qt.current_user_sign_in_ui import Ui_CurrentUserSignInPage
from ubuntu_sso.qt.email_verification_ui import Ui_EmailVerificationPage
from ubuntu_sso.qt.error_message_ui import Ui_ErrorPage
from ubuntu_sso.qt.setup_account_ui import Ui_SetUpAccountPage
from ubuntu_sso.qt.terms_and_conditions_ui import Ui_TosPage
from ubuntu_sso.qt.success_message_ui import Ui_SuccessPage
from ubuntu_sso.qt.forgotten_password_ui import Ui_ForgottenPasswordPage
from ubuntu_sso.qt.reset_password_ui import Ui_ResetPasswordPage
# pylint: enable=F0401,E0611
from ubuntu_sso.qt.controllers import (
    ChooseSignInController,
    CurrentUserController,
    EmailVerificationController,
    ErrorController,
    ForgottenPasswordController,
    ResetPasswordController,
    SetUpAccountController,
    SuccessController,
    TosController,
    UbuntuSSOWizardController)


logger = setup_logging('ubuntu_sso.gui')


class Header(QWidget):
    """Header Class for Title and Subtitle in all wizard pages."""

    def __init__(self):
        """Create a new instance."""
        QWidget.__init__(self)
        vbox = QVBoxLayout(self)
        self.title_label = QLabel()
        self.title_label.setWordWrap(True)
        self.title_label.setObjectName('title_label')
        self.subtitle_label = QLabel()
        self.subtitle_label.setWordWrap(True)
        vbox.addWidget(self.title_label)
        vbox.addWidget(self.subtitle_label)
        self.title_label.setVisible(False)
        self.subtitle_label.setVisible(False)

    def set_title(self, title):
        """Set the Title of the page or hide it otherwise"""
        if title:
            self.title_label.setText(title)
            self.title_label.setVisible(True)
        else:
            self.title_label.setVisible(False)

    def set_subtitle(self, subtitle):
        """Set the Subtitle of the page or hide it otherwise"""
        if subtitle:
            self.subtitle_label.setText(subtitle)
            self.subtitle_label.setVisible(True)
        else:
            self.subtitle_label.setVisible(False)


class SSOWizardPage(QWizardPage):
    """Root class for all wizard pages."""

    def __init__(self, ui, controller, parent=None):
        """Create a new instance."""
        QWizardPage.__init__(self, parent)
        self.ui = ui
        self.ui.setupUi(self)
        self.header = Header()
        self.layout().insertWidget(0, self.header)
        if controller:
            self.controller = controller
            self.controller.setupUi(self)
        self.next = -1

    # pylint: disable=C0103
    def nextId(self):
        """Provide the next id."""
        return self.next
    # pylint: enable=C0103

    # pylint: disable=C0103
    def initializePage(self):
        """Called to prepare the page just before it is shown."""
        if self.controller:
            self.controller.pageInitialized()
    # pylint: enable=C0103

    # pylint: disable=C0103
    def setTitle(self, title=''):
        """Set the Wizard Page Title."""
        self.header.set_title(title)
    # pylint: enable=C0103

    # pylint: disable=C0103
    def setSubTitle(self, subtitle=''):
        """Set the Wizard Page Subtitle."""
        self.header.set_subtitle(subtitle)
    # pylint: enable=C0103


class EnhancedLineEdit(object):
    """Represents and enhanced lineedit.

    This class works on an already added lineedit to the widget so
    that we are just adding extra items to it.
    """

    def __init__(self, line_edit, valid_cb=lambda x: False,
                 warning_sign=False):
        """Create an instance."""
        self._line_edit = line_edit
        layout = QHBoxLayout(self._line_edit)
        layout.setMargin(0)
        self._line_edit.setLayout(layout)
        self.valid_cb = valid_cb
        layout.addStretch()
        self.clear_label = QLabel(self._line_edit)
        self.clear_label.setMargin(2)
        self.clear_label.setProperty("lineEditWarning", True)
        layout.addWidget(self.clear_label)
        self.clear_label.setMinimumSize(16, 16)
        self.clear_label.setVisible(False)
        self.clear_label.setCursor(QCursor(Qt.ArrowCursor))
        if warning_sign:
            icon = QApplication.style().standardIcon(
                QStyle.SP_MessageBoxWarning)
            self.clear_label.setPixmap(icon.pixmap(16, 16))
        # connect the change of text to the cation that will check if the
        # text is valid and if the icon should be shown.
        self._line_edit.textChanged.connect(self.show_button)

    def show_button(self, string):
        """Decide if we show the button or not."""
        if not self.valid_cb(string) and self.clear_label.pixmap() is not None:
            self.clear_label.setVisible(True)
        else:
            self.clear_label.setVisible(False)


class SSOWizardEnhancedEditPage(SSOWizardPage):
    """Page that contains enhanced line edits."""

    def __init__(self, ui, controller, parent=None):
        """Create a new instance."""
        SSOWizardPage.__init__(self, ui, controller, parent)
        self._enhanced_edits = {}

    def set_line_edit_validation_rule(self, edit, cb):
        """Set a new enhanced edit so that we can show an icon."""
        if edit in self._enhanced_edits:
            self._enhanced_edits[edit].valid_cb = cb
        else:
            # create a new enhanced edit
            enhanced_edit = EnhancedLineEdit(edit, cb)
            self._enhanced_edits[edit] = enhanced_edit


class ChooseSignInPage(SSOWizardPage):
    """Widget that allows the user to choose how to sign in."""

    def __init__(self, ui, controller, parent=None):
        """Create a new widget to be used."""
        SSOWizardPage.__init__(self, ui, controller, parent)


class CurrentUserSignInPage(SSOWizardPage):
    """Widget that allows to get the data of user to sign in."""

    def __init__(self, ui, controller, parent=None):
        """Create a new widget to be used."""
        SSOWizardPage.__init__(self, ui, controller, parent)


class EmailVerificationPage(SSOWizardPage):
    """Widget used to input the email verification code."""

    def __init__(self, ui, controller, parent=None):
        """Create a new widget to be used."""
        SSOWizardPage.__init__(self, ui, controller, parent)

    @property
    def verification_code(self):
        """Return the content of the verification code edit."""
        return str(self.ui.verification_code_edit.text())

    @property
    def next_button(self):
        """Return the button that move to the next stage."""
        return self.ui.next_button


class ErrorPage(SSOWizardPage):
    """Widget used to show the diff errors."""

    def __init__(self, ui, controller, parent=None):
        """Create a new widget to be used."""
        SSOWizardPage.__init__(self, ui, controller, parent)


class ForgottenPasswordPage(SSOWizardEnhancedEditPage):
    """Widget used to deal with users that forgot the password."""

    def __init__(self, ui, controller, parent=None):
        """Create a new instance."""
        SSOWizardEnhancedEditPage.__init__(self, ui, controller, parent)

    @property
    def email_widget(self):
        """Return the widget used to show the email information."""
        return self.ui.email_widget

    @property
    def forgotted_password_intro_label(self):
        """Return the intro label that lets the user know the issue."""
        return self.ui.forgotted_password_intro_label

    @property
    def error_label(self):
        """Return the label used to show error."""
        return self.ui.error_label

    @property
    def email_address_label(self):
        """Return the lable used to state the use of the line edit."""
        return self.ui.email_address_label

    @property
    def email_address(self):
        """Return the email address provided by the user."""
        return str(self.ui.email_line_edit.text())

    @property
    def email_address_line_edit(self):
        """Return the line edit with the content."""
        return self.ui.email_line_edit

    @property
    def send_button(self):
        """Return the button used to request the new password."""
        return self.ui.send_button

    @property
    def try_again_widget(self):
        """Return the widget used to display the try again button."""
        return self.ui.try_again_widget

    @property
    def try_again_button(self):
        """Return the button used to try again the reset password."""
        return self.ui.try_again_button


class ResetPasswordPage(SSOWizardEnhancedEditPage):
    """Widget used to allow the user change his password."""

    def __init__(self, ui, controller, parent=None):
        """Create a new instance."""
        SSOWizardEnhancedEditPage.__init__(self, ui, controller, parent)


class TosPage(SSOWizardPage):
    """Widget used to show the tos."""

    def __init__(self, ui, controller, parent=None):
        """Create a new instance."""
        SSOWizardPage.__init__(self, ui, controller, parent)


class SetupAccountPage(SSOWizardEnhancedEditPage):
    """Widget used to create a new account."""

    def __init__(self, ui, controller, parent=None):
        """Create a new widget to be used."""
        SSOWizardEnhancedEditPage.__init__(self, ui, controller, parent)
        self._enhanced_edits = {}
        # palettes that will be used to set the colors of the password strengh
        self.captcha_id = None
        self.captcha_file = None
        self.ui.captcha_view.setPixmap(QPixmap())

    def get_captcha_image(self):
        """Return the path to the captcha image."""
        return self.ui.captcha_view.pixmap()

    def set_captcha_image(self, pixmap_image):
        """Set the new image of the captcha."""
        # lets set the QPixmap for the label
        self.ui.captcha_view.setPixmap(pixmap_image)

    captcha_image = property(get_captcha_image, set_captcha_image)


class SuccessPage(SSOWizardPage):
    """Page used to display success message."""

    def __init__(self, ui, controller, parent=None):
        """Create a new instance."""
        SSOWizardPage.__init__(self, ui, controller, parent)


class UbuntuSSOWizard(QWizard):
    """Wizard used to create or use sso."""

    # definition of the signals raised by the widget
    recoverableError = pyqtSignal('QString', 'QString')
    loginSuccess = pyqtSignal('QString', 'QString')
    registrationSuccess = pyqtSignal('QString', 'QString')

    def __init__(self, controller, parent=None, app_name='', tos_url='',
                 help_text=''):
        """Create a new wizard."""
        QWizard.__init__(self, parent)
        # store common useful data provided by the app
        self.app_name = app_name
        self.tos_url = tos_url
        self.help_text = help_text

        # set the diff pages of the QWizard
        self.sign_in_controller = ChooseSignInController(title='Sign In')
        self.sign_in_page = ChooseSignInPage(Ui_ChooseSignInPage(),
                                             self.sign_in_controller,
                                             parent=self)
        self.setup_controller = SetUpAccountController()
        self.setup_account = SetupAccountPage(Ui_SetUpAccountPage(),
                                              self.setup_controller,
                                              parent=self)
        self.tos = TosPage(Ui_TosPage(), TosController(tos_url=tos_url),
                           parent=self)
        self.email_verification = EmailVerificationPage(
                                                Ui_EmailVerificationPage(),
                                                EmailVerificationController())
        self.current_user_controller = CurrentUserController(title='Sign in')
        self.current_user = CurrentUserSignInPage(Ui_CurrentUserSignInPage(),
                                                  self.current_user_controller,
                                                  parent=self)
        self.success_controller = SuccessController()
        self.success = SuccessPage(Ui_SuccessPage(), self.success_controller,
                                   parent=self)
        self.error_controller = ErrorController()
        self.error = ErrorPage(Ui_ErrorPage(), self.error_controller)
        self.forgotte_pwd_controller = ForgottenPasswordController()
        self.forgotten = ForgottenPasswordPage(Ui_ForgottenPasswordPage(),
                                               self.forgotte_pwd_controller,
                                               parent=self)
        self.reset_password_controller = ResetPasswordController()
        self.reset_password = ResetPasswordPage(Ui_ResetPasswordPage(),
                                                self.reset_password_controller,
                                                parent=self)
        # store the ids of the pages so that it is easier to access them later
        self._pages = {}
        for page in [self.sign_in_page, self.setup_account, self.tos,
                     self.email_verification, self.current_user, self.success,
                     self.error, self.forgotten, self.reset_password]:
            self._pages[page] = self.addPage(page)

        # set the buttons layout to only have cancel and back since the next
        # buttons are the ones used in the diff pages.
        buttons_layout = []
        buttons_layout.append(QWizard.Stretch)
        buttons_layout.append(QWizard.BackButton)
        buttons_layout.append(QWizard.CancelButton)
        self.setButtonLayout(buttons_layout)
        self.setWindowTitle(app_name)
        self.controller = controller
        self.controller.setupUi(self)

    @property
    def sign_in_page_id(self):
        """Return the id of the page used for choosing sign in type."""
        return self._pages[self.sign_in_page]

    @property
    def setup_account_page_id(self):
        """Return the id of the page used for sign in."""
        return self._pages[self.setup_account]

    @property
    def tos_page_id(self):
        """Return the id of the tos page."""
        return self._pages[self.tos]

    @property
    def email_verification_page_id(self):
        """Return the id of the verification page."""
        return self._pages[self.email_verification]

    @property
    def current_user_page_id(self):
        """Return the id used to signin by a current user."""
        return self._pages[self.current_user]

    @property
    def success_page_id(self):
        """Return the id of the success page."""
        return self._pages[self.success]

    @property
    def forgotten_password_page_id(self):
        """Return the id of the forgotten password page."""
        return self._pages[self.forgotten]

    @property
    def reset_password_page_id(self):
        """Return the id of the reset password page."""
        return self._pages[self.reset_password]

    @property
    def error_page_id(self):
        """Return the id of the error page."""
        return self._pages[self.error]


class UbuntuSSOClientGUI(object):
    """Ubuntu single sign-on GUI."""

    def __init__(self, app_name, tc_url='http://one.ubuntu.com', help_text='',
                 window_id=0, login_only=False):
        """Create a new instance."""
        # create the controller and the ui, then set the cb and call the show
        # method so that we can work
        self.controller = UbuntuSSOWizardController(app_name)
        self.view = UbuntuSSOWizard(self.controller, app_name=app_name,
                                    tos_url=tc_url, help_text=help_text)
        self.view.show()

    def get_login_success_callback(self):
        """Return the log in success cb."""
        return self.controller.login_success_callback

    def set_login_success_callback(self, cb):
        """Set log in success cb."""
        self.controller.login_success_callback = cb

    login_success_callback = property(get_login_success_callback,
                                      set_login_success_callback)

    def get_registration_success_callback(self):
        """Return the registration success cb."""
        return self.controller.registration_success_callback

    def set_registration_success_callback(self, cb):
        """Set registration success cb."""
        self.controller.registration_success_callback = cb

    registration_success_callback = property(get_registration_success_callback,
                                             set_registration_success_callback)

    def get_user_cancellation_callback(self):
        """Return the user cancellation callback."""
        return self.controller.user_cancellation_callback

    def set_user_cancellation_callback(self, cb):
        """Set the user cancellation callback."""
        self.controller.user_cancellation_callback = cb

    user_cancellation_callback = property(get_user_cancellation_callback,
                                          set_user_cancellation_callback)
