# -*- coding: utf-8 -*-
#
# test_gui - tests for ubuntu_sso.gui
#
# Author: Natalia Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program. If not, see <http://www.gnu.org/licenses/>.
"""Tests for the GUI for registration/login."""

import itertools
import logging
import os

from collections import defaultdict

import dbus
import gtk
import webkit

from twisted.trial.unittest import TestCase
from ubuntuone.devtools.handlers import MementoHandler

from ubuntu_sso.gtk import gui
from ubuntu_sso.tests import (APP_NAME, TC_URL, HELP_TEXT, CAPTCHA_ID,
    CAPTCHA_SOLUTION, EMAIL, EMAIL_TOKEN, NAME, PASSWORD, RESET_PASSWORD_TOKEN)


# Access to a protected member 'yyy' of a client class
# pylint: disable=W0212

# Instance of 'UbuntuSSOClientGUI' has no 'yyy' member
# pylint: disable=E1101,E1103


class FakedSSOBackend(object):
    """Fake a SSO Backend (acts as a dbus.Interface as well)."""

    def __init__(self, *args, **kwargs):
        self._args = args
        self._kwargs = kwargs
        self._called = {}
        for i in ('generate_captcha', 'login', 'register_user',
                  'validate_email', 'request_password_reset_token',
                  'set_new_password'):
            setattr(self, i, self._record_call(i))

    def _record_call(self, func_name):
        """Store values when calling 'func_name'."""

        def inner(*args, **kwargs):
            """Fake 'func_name'."""
            self._called[func_name] = (args, kwargs)

        return inner


class FakedDbusObject(object):
    """Fake a dbus."""

    def __init__(self, *args, **kwargs):
        """Init."""
        self._args = args
        self._kwargs = kwargs


class FakedSessionBus(object):
    """Fake the session bus."""

    def __init__(self):
        """Init."""
        self.obj = None
        self.callbacks = {}

    def add_signal_receiver(self, method, signal_name, dbus_interface):
        """Add a signal receiver."""
        self.callbacks[(dbus_interface, signal_name)] = method

    def remove_signal_receiver(self, match, signal_name, dbus_interface):
        """Remove the signal receiver."""
        assert (dbus_interface, signal_name) in self.callbacks
        del self.callbacks[(dbus_interface, signal_name)]

    def get_object(self, bus_name, object_path, introspect=True,
                   follow_name_owner_changes=False, **kwargs):
        """Return a faked proxy for the given remote object."""
        if bus_name == gui.DBUS_BUS_NAME and \
           object_path == gui.DBUS_ACCOUNT_PATH:
            assert self.obj is None
            kwargs = dict(object_path=object_path,
                          bus_name=bus_name, follow_name_owner_changes=True)
            self.obj = FakedDbusObject(**kwargs)
        return self.obj


class Settings(dict):
    """Faked embedded browser settings."""

    def get_property(self, prop_name):
        """Alias for __getitem__."""
        return self[prop_name]

    def set_property(self, prop_name, newval):
        """Alias for __setitem__."""
        self[prop_name] = newval


class FakedEmbeddedBrowser(gtk.TextView):
    """Faked an embedded browser."""

    def __init__(self):
        super(FakedEmbeddedBrowser, self).__init__()
        self._props = {}
        self._signals = defaultdict(list)
        self._settings = Settings()

    def connect(self, signal_name, callback):
        """Connect 'signal_name' with 'callback'."""
        self._signals[signal_name].append(callback)

    def load_uri(self, uri):
        """Navigate to the given 'uri'."""
        self._props['uri'] = uri

    def set_property(self, prop_name, newval):
        """Set 'prop_name' to 'newval'."""
        self._props[prop_name] = newval

    def get_property(self, prop_name):
        """Return the current value for 'prop_name'."""
        return self._props[prop_name]

    def get_settings(self,):
        """Return the current settings."""
        return self._settings

    def get_load_status(self):
        """Return the current load status."""
        return gui.WEBKIT_LOAD_FINISHED

    def show(self):
        """Show this instance."""
        self.set_property('visible', True)


class BasicTestCase(TestCase):
    """Test case with a helper tracker."""

    def setUp(self):
        """Init."""
        self._called = False  # helper

        self.memento = MementoHandler()
        self.memento.setLevel(logging.DEBUG)
        gui.logger.addHandler(self.memento)

    def tearDown(self):
        """Clean up."""
        self._called = False

    def _set_called(self, *args, **kwargs):
        """Set _called to True."""
        self._called = (args, kwargs)


class LabeledEntryTestCase(BasicTestCase):
    """Test suite for the labeled entry."""

    def setUp(self):
        """Init."""
        super(LabeledEntryTestCase, self).setUp()
        self.label = 'Test me please'
        self.entry = gui.LabeledEntry(label=self.label)

        # we need a window to be able to realize ourselves
        window = gtk.Window()
        window.add(self.entry)
        window.show_all()

    def tearDown(self):
        """Clean up."""
        self.entry = None
        super(LabeledEntryTestCase, self).tearDown()

    def grab_focus(self, focus_in=True):
        """Grab focus on widget, if None use self.entry."""
        direction = 'in' if focus_in else 'out'
        self.entry.emit('focus-%s-event' % direction, None)

    # Bad first argument 'LabeledEntry' given to super class
    # pylint: disable=E1003
    def assert_correct_label(self):
        """Check that the entry has the correct label."""
        # text content is correct
        msg = 'Text content must be "%s" (got "%s" instead).'
        expected = self.label
        actual = super(gui.LabeledEntry, self.entry).get_text()
        self.assertEqual(expected, actual, msg % (expected, actual))

        # text color is correct
        msg = 'Text color must be "%s" (got "%s" instead).'
        expected = gui.HELP_TEXT_COLOR
        actual = self.entry.style.text[gtk.STATE_NORMAL]
        self.assertEqual(expected, actual, msg % (expected, actual))

    def test_initial_text(self):
        """Entry have the correct text at startup."""
        self.assert_correct_label()

    def test_width_chars(self):
        """Entry have the correct width."""
        self.assertEqual(self.entry.get_width_chars(), gui.DEFAULT_WIDTH)

    def test_tooltip(self):
        """Entry have the correct tooltip."""
        msg = 'Tooltip must be "%s" (got "%s" instead).'
        expected = self.label
        actual = self.entry.get_tooltip_text()
        # tooltip is correct
        self.assertEqual(expected, actual, msg % (expected, actual))

    def test_clear_entry_on_focus_in(self):
        """Entry are cleared when focused."""
        self.grab_focus()

        msg = 'Entry must be cleared on focus in.'
        self.assertEqual('', self.entry.get_text(), msg)

    def test_text_defaults_to_theme_color_when_focus_in(self):
        """Entry restore its text color when focused in."""
        self.patch(self.entry, 'modify_text', self._set_called)

        self.grab_focus()

        self.assertEqual(((gtk.STATE_NORMAL, None), {}), self._called,
                         'Entry text color must be restore on focus in.')

    def test_refill_entry_on_focus_out_if_no_input(self):
        """Entry is re-filled with label when focused out if no user input."""

        self.grab_focus()  # grab focus
        self.grab_focus(focus_in=False)  # loose focus

        # Entry must be re-filled on focus out
        self.assert_correct_label()

    def test_refill_entry_on_focus_out_if_empty_input(self):
        """Entry is re-filled with label when focused out if empty input."""

        self.grab_focus()  # grab focus

        self.entry.set_text('        ')  # add empty text to the entry

        self.grab_focus(focus_in=False)  # loose focus

        # Entry must be re-filled on focus out
        self.assert_correct_label()

    def test_preserve_input_on_focus_out_if_user_input(self):
        """Entry is unmodified when focused out if user input."""
        msg = 'Entry must be left unmodified on focus out when user input.'
        expected = 'test me please'

        self.grab_focus()  # grab focus

        self.entry.set_text(expected)  # add empty text to the entry

        self.grab_focus(focus_in=False)  # loose focus

        self.assertEqual(expected, self.entry.get_text(), msg)

    def test_preserve_input_on_focus_out_and_in_again(self):
        """Entry is unmodified when focused out and then in again."""
        msg = 'Entry must be left unmodified on focus out and then in again.'
        expected = 'test me I mean it'

        self.grab_focus()  # grab focus

        self.entry.set_text(expected)  # add text to the entry

        self.grab_focus(focus_in=False)  # loose focus
        self.grab_focus()  # grab focus again!

        self.assertEqual(expected, self.entry.get_text(), msg)

    def test_get_text_ignores_label(self):
        """Entry's text is only user input (label is ignored)."""
        self.assertEqual(self.entry.get_text(), '')

    def test_get_text_ignores_empty_input(self):
        """Entry's text is only user input (empty text is ignored)."""
        self.entry.set_text('       ')
        self.assertEqual(self.entry.get_text(), '')

    def test_get_text_doesnt_ignore_user_input(self):
        """Entry's text is user input."""
        self.entry.set_text('a')
        self.assertEqual(self.entry.get_text(), 'a')

    def test_no_warning_by_default(self):
        """No secondary icon by default."""
        self.assertEqual(self.entry.warning, None)
        self.assertEqual(self.entry.get_property('secondary-icon-stock'),
                         None)
        self.assertEqual(self.entry.get_property('secondary-icon-sensitive'),
                         False)
        self.assertEqual(self.entry.get_property('secondary-icon-activatable'),
                         False)
        prop = self.entry.get_property('secondary-icon-tooltip-text')
        self.assertEqual(prop, None)

    def test_set_warning(self):
        """Setting a warning show the proper secondary icon."""
        msg = 'You failed!'
        self.entry.set_warning(msg)
        self.assertEqual(self.entry.warning, msg)
        self.assertEqual(self.entry.get_property('secondary-icon-stock'),
                         gtk.STOCK_DIALOG_WARNING)
        self.assertEqual(self.entry.get_property('secondary-icon-sensitive'),
                         True)
        self.assertEqual(self.entry.get_property('secondary-icon-activatable'),
                         False)
        prop = self.entry.get_property('secondary-icon-tooltip-text')
        self.assertEqual(prop, msg)

    def test_clear_warning(self):
        """Clearing a warning no longer show the secondary icon."""
        self.entry.clear_warning()
        self.assertEqual(self.entry.warning, None)
        self.assertEqual(self.entry.get_property('secondary-icon-stock'),
                         None)
        self.assertEqual(self.entry.get_property('secondary-icon-sensitive'),
                         False)
        self.assertEqual(self.entry.get_property('secondary-icon-activatable'),
                         False)
        prop = self.entry.get_property('secondary-icon-tooltip-text')
        self.assertEqual(prop, None)


class PasswordLabeledEntryTestCase(LabeledEntryTestCase):
    """Test suite for the labeled entry when is_password is True."""

    def setUp(self):
        """Init."""
        super(PasswordLabeledEntryTestCase, self).setUp()
        self.entry.is_password = True

    def test_password_fields_are_visible_at_startup(self):
        """Password entrys show the helping text at startup."""
        self.assertTrue(self.entry.get_visibility(),
                        'Password entry should be visible at start up.')

    def test_password_field_is_visible_if_no_input_and_focus_out(self):
        """Password entry show the label when focus out."""
        self.grab_focus()  # user cliked or TAB'd to the entry
        self.grab_focus(focus_in=False)  # loose focus
        self.assertTrue(self.entry.get_visibility(),
                        'Entry should be visible when focus out and no input.')

    def test_password_fields_are_not_visible_when_editing(self):
        """Password entrys show the hidden chars instead of the password."""
        self.grab_focus()  # user cliked or TAB'd to the entry
        self.assertFalse(self.entry.get_visibility(),
                         'Entry should not be visible when editing.')


class UbuntuSSOClientTestCase(BasicTestCase):
    """Basic setup and helper functions."""

    gui_class = gui.UbuntuSSOClientGUI
    kwargs = dict(app_name=APP_NAME, tc_url=TC_URL, help_text=HELP_TEXT)

    def setUp(self):
        """Init."""
        super(UbuntuSSOClientTestCase, self).setUp()
        self.patch(dbus, 'SessionBus', FakedSessionBus)
        self.patch(dbus, 'Interface', FakedSSOBackend)
        self.pages = ('enter_details', 'processing', 'verify_email', 'finish',
                      'tc_browser', 'login', 'request_password_token',
                      'set_new_password')
        self.ui = self.gui_class(**self.kwargs)
        self.error = {'message': self.ui.UNKNOWN_ERROR}

    def tearDown(self):
        """Clean up."""
        self.ui.bus.callbacks = {}
        self.ui = None
        super(UbuntuSSOClientTestCase, self).tearDown()

    def assert_entries_are_packed_to_ui(self, container_name, entries):
        """Every entry is properly packed in the ui 'container_name'."""
        msg = 'Entry "%s" must be packed in "%s" but is not.'
        container = getattr(self.ui, container_name)
        for kind in entries:
            name = '%s_entry' % kind
            entry = getattr(self.ui, name)
            self.assertIsInstance(entry, gui.LabeledEntry)
            self.assertIn(entry, container, msg % (name, container_name))

    def assert_warnings_visibility(self, visible=False):
        """Every warning label should be 'visible'."""
        msg = '"%s" should have %sempty content.'
        for name in self.ui.widgets:
            widget = getattr(self.ui, name)
            if 'warning' in name:
                self.assertEqual('', widget.get_text(),
                                 msg % (name, '' if visible else 'non-'))
            elif 'entry' in name:
                self.assertEqual(widget.warning, '')

    def assert_correct_label_warning(self, label, message):
        """Check that a warning is shown displaying 'message'."""
        # warning label is visible
        self.assertTrue(label.get_property('visible'))

        # warning content is correct
        actual = label.get_text()
        self.assertEqual(actual, message)

        # content color is correct
        expected = gui.WARNING_TEXT_COLOR
        actual = label.style.fg[gtk.STATE_NORMAL]
        self.assertEqual(expected, actual)  # until realized this will fail

    def assert_correct_entry_warning(self, entry, message):
        """Check that a warning is shown displaying 'message'."""
        self.assertEqual(entry.warning, message)

    def assert_pages_visibility(self, **kwargs):
        """The page 'name' is the current page for the content notebook."""
        msg = 'page %r must be self.ui.content\'s current page.'
        (name, _), = kwargs.items()
        page = getattr(self.ui, '%s_vbox' % name)
        self.assertEqual(self.ui.content.get_current_page(),
                         self.ui.content.page_num(page), msg % name)

    def click_join_with_valid_data(self):
        """Move to the next page after entering details."""
        self.ui.on_captcha_generated(app_name=APP_NAME, captcha_id=CAPTCHA_ID)

        self.ui.name_entry.set_text(NAME)
        # match emails
        self.ui.email1_entry.set_text(EMAIL)
        self.ui.email2_entry.set_text(EMAIL)
        # match passwords
        self.ui.password1_entry.set_text(PASSWORD)
        self.ui.password2_entry.set_text(PASSWORD)
        # agree to TC
        self.ui.yes_to_tc_checkbutton.set_active(True)
        # resolve captcha properly
        self.ui.captcha_solution_entry.set_text(CAPTCHA_SOLUTION)

        self.ui.join_ok_button.clicked()

    def click_verify_email_with_valid_data(self):
        """Move to the next page after entering email token."""
        self.click_join_with_valid_data()

        # resolve email token properly
        self.ui.email_token_entry.set_text(EMAIL_TOKEN)

        self.ui.verify_token_button.clicked()

    def click_connect_with_valid_data(self):
        """Move to the next page after entering login info."""
        # enter email
        self.ui.login_email_entry.set_text(EMAIL)
        # enter password
        self.ui.login_password_entry.set_text(PASSWORD)

        self.ui.login_ok_button.clicked()

    def click_request_password_token_with_valid_data(self):
        """Move to the next page after requesting for password reset token."""
        # enter email
        self.ui.reset_email_entry.set_text(EMAIL)

        self.ui.request_password_token_ok_button.clicked()

    def click_set_new_password_with_valid_data(self):
        """Move to the next page after resetting password."""
        # enter reset code
        self.ui.reset_code_entry.set_text(RESET_PASSWORD_TOKEN)
        # match passwords
        self.ui.reset_password1_entry.set_text(PASSWORD)
        self.ui.reset_password2_entry.set_text(PASSWORD)

        self.ui.set_new_password_ok_button.clicked()


class BasicUbuntuSSOClientTestCase(UbuntuSSOClientTestCase):
    """Test suite for basic functionality."""

    def test_main_window_is_visible_at_startup(self):
        """The main window is shown at startup."""
        self.assertTrue(self.ui.window.get_property('visible'))

    def test_main_window_is_resizable(self):
        """The main window can be resized."""
        self.assertTrue(self.ui.window.get_property('resizable'))

    def test_closing_main_window_calls_close_callback(self):
        """The close_callback is called when closing the main window."""
        self.ui.close_callback = self._set_called
        self.ui.on_close_clicked()
        self.assertTrue(self._called,
                        'close_callback was called when window was closed.')

    def test_close_callback_if_not_set(self):
        """The close_callback is a no op if not set."""
        self.ui.on_close_clicked()
        # no crash when close_callback is not set

    def test_app_name_is_stored(self):
        """The app_name is stored for further use."""
        self.assertIn(APP_NAME, self.ui.app_name)

    def test_session_bus_is_correct(self):
        """The session bus is created and is correct."""
        self.assertIsInstance(self.ui.bus, FakedSessionBus)

    def test_iface_name_is_correct(self):
        """The session bus is created and is correct."""
        self.assertEqual(self.ui.iface_name, gui.DBUS_IFACE_USER_NAME)

    def test_bus_object_is_created(self):
        """SessionBus.get_object is called properly."""
        self.assertIsInstance(self.ui.bus.obj, FakedDbusObject)
        self.assertEqual(self.ui.bus.obj._args, ())
        expected = dict(object_path=gui.DBUS_ACCOUNT_PATH,
                        bus_name=gui.DBUS_BUS_NAME,
                        follow_name_owner_changes=True)
        self.assertEqual(expected, self.ui.bus.obj._kwargs)

    def test_bus_interface_is_created(self):
        """dbus.Interface is called properly."""
        self.assertIsInstance(self.ui.backend, FakedSSOBackend)
        self.assertEqual(self.ui.backend._args, ())
        expected = dict(object=self.ui.bus.obj,
                        dbus_interface=gui.DBUS_IFACE_USER_NAME)
        self.assertEqual(expected, self.ui.backend._kwargs)

    def test_dbus_signals_are_removed(self):
        """The hooked signals are removed at shutdown time."""
        self.ui._setup_signals()
        assert len(self.ui.bus.callbacks) > 0  # at least one callback

        self.ui.on_close_clicked()

        self.assertEqual(self.ui.bus.callbacks, {})

    def test_pages_are_packed_into_container(self):
        """All the pages are packed in the main container."""
        children = self.ui.content.get_children()
        for page_name in self.pages:
            page = getattr(self.ui, '%s_vbox' % page_name)
            self.assertIn(page, children)

    def test_initial_text_for_entries(self):
        """Entries have the correct text at startup."""
        msg = 'Text for "%s" must be "%s" (got "%s" instead).'
        for name in self.ui.entries:
            entry = getattr(self.ui, name)
            expected = getattr(self.ui, name.upper())
            actual = entry.label
            # text content is correct
            self.assertEqual(expected, actual, msg % (name, expected, actual))

    def test_entries_activates_default(self):
        """Entries have the activates default prop set."""
        msg = '"%s" must have activates_default set to True.'
        for name in self.ui.entries:
            entry = getattr(self.ui, name)
            self.assertTrue(entry.get_activates_default(), msg % (name,))

    def test_label_size_allocated_is_connected(self):
        """Labels have the size-allocate signal connected."""
        msg = 'Label %r must have size-allocate connected.'
        labels = [i for i in self.ui.widgets if 'label' in i]
        for label in labels:
            widget = getattr(self.ui, label)
            widget.emit('size-allocate', gtk.gdk.Rectangle(1, 2, 3, 4))
            self.assertEqual(widget.get_size_request(), (3 - 2, -1),
                             msg % (label,))

    def test_password_fields_are_password(self):
        """Password fields have the is_password flag set."""
        msg = '"%s" should be a password LabeledEntry instance.'
        passwords = filter(lambda name: 'password' in name,
                           self.ui.entries)
        for name in passwords:
            widget = getattr(self.ui, name)
            self.assertTrue(widget.is_password, msg % name)

    def test_warning_fields_are_cleared(self):
        """Every warning label should be cleared."""
        self.assert_warnings_visibility()

    def test_cancel_buttons_close_window(self):
        """Every cancel button should close the window when clicked."""
        msg = '"%s" should close the window when clicked.'
        buttons = filter(lambda name: 'cancel_button' in name or
                                      'close_button' in name, self.ui.widgets)
        for name in buttons:
            self.ui.close_callback = self._set_called
            widget = getattr(self.ui, name)
            widget.clicked()
            self.assertEqual(self._called, ((widget,), {}), msg % name)
            self._called = False

    def test_window_icon(self):
        """Main window has the proper icon."""
        self.assertEqual('ubuntu-logo', self.ui.window.get_icon_name())

    def test_transient_window_is_none_if_window_id_is_zero(self):
        """The transient window is correct."""
        self.patch(gtk.gdk, 'window_foreign_new', self._set_called)
        self.gui_class(window_id=0, **self.kwargs)
        self.assertFalse(self._called, 'set_transient_for must not be called.')

    def test_transient_window_is_correct(self):
        """The transient window is correct."""
        xid = 5
        self.patch(gtk.gdk, 'window_foreign_new', self._set_called)
        self.gui_class(window_id=xid, **self.kwargs)
        self.assertTrue(self.memento.check(logging.ERROR, 'set_transient_for'))
        self.assertTrue(self.memento.check(logging.ERROR, str(xid)))
        self.assertEqual(self._called, ((xid,), {}))

    def test_transient_window_accepts_negative_id(self):
        """The transient window accepts a negative window id."""
        xid = -5
        self.patch(gtk.gdk, 'window_foreign_new', self._set_called)
        self.gui_class(window_id=xid, **self.kwargs)
        self.assertEqual(self._called, ((xid,), {}))

    def test_finish_success_shows_success_page(self):
        """When calling 'finish_success' the success page is shown."""
        self.ui.finish_success()
        self.assert_pages_visibility(finish=True)
        self.assertEqual(self.ui.SUCCESS, self.ui.finish_vbox.label.get_text())

    def test_finish_error_shows_error_page(self):
        """When calling 'finish_error' the error page is shown."""
        self.ui.finish_error()
        self.assert_pages_visibility(finish=True)
        self.assertEqual(self.ui.ERROR, self.ui.finish_vbox.label.get_text())


class EnterDetailsTestCase(UbuntuSSOClientTestCase):
    """Test suite for the user registration (enter details page)."""

    def test_initial_text_for_header_label(self):
        """The header must have the correct text at startup."""
        msg = 'Text for the header must be "%s" (got "%s" instead).'
        expected = self.ui.JOIN_HEADER_LABEL % {'app_name': APP_NAME}
        actual = self.ui.header_label.get_text()
        # text content is correct
        self.assertEqual(expected, actual, msg % (expected, actual))

    def test_entries_are_packed_to_ui(self):
        """Every entry is properly packed in the ui."""
        for kind in ('email', 'password'):
            container_name = '%ss_hbox' % kind
            entries = ('%s%s' % (kind, i) for i in xrange(1, 3))
            self.assert_entries_are_packed_to_ui(container_name, entries)

        self.assert_entries_are_packed_to_ui('enter_details_vbox', ('name',))
        self.assert_entries_are_packed_to_ui('captcha_solution_vbox',
                                             ('captcha_solution',))
        self.assert_entries_are_packed_to_ui('verify_email_details_vbox',
                                             ('email_token',))

    def test_initial_texts_for_checkbuttons(self):
        """Check buttons have the correct text at startup."""
        msg = 'Text for "%s" must be "%s" (got "%s" instead).'
        expected = self.ui.YES_TO_UPDATES % {'app_name': APP_NAME}
        actual = self.ui.yes_to_updates_checkbutton.get_label()
        self.assertEqual(expected, actual, msg % ('yes_to_updates_checkbutton',
                                                  expected, actual))
        expected = self.ui.YES_TO_TC % {'app_name': APP_NAME}
        actual = self.ui.yes_to_tc_checkbutton.get_label()
        self.assertEqual(expected, actual,
                         msg % ('yes_to_tc_checkbutton', expected, actual))

    def test_checkbuttons_are_checked_at_startup(self):
        """Checkbuttons are checked by default."""
        msg = '"%s" is checked by default.'
        for name in ('yes_to_updates_checkbutton', 'yes_to_tc_checkbutton'):
            widget = getattr(self.ui, name)
            self.assertTrue(widget.get_active(), msg % name)

    def test_vboxes_visible_properties(self):
        """Only 'enter_details' vbox is visible at start up."""
        self.assert_pages_visibility(enter_details=True)

    def test_join_ok_button_clicked(self):
        """Clicking 'join_ok_button' sends info to backend using 'register'."""
        self.click_join_with_valid_data()

        # assert register_user was called
        expected = 'register_user'
        self.assertIn(expected, self.ui.backend._called)
        self.assertEqual(self.ui.backend._called[expected],
                         ((APP_NAME, EMAIL, PASSWORD, NAME, CAPTCHA_ID,
                           CAPTCHA_SOLUTION),
                          dict(reply_handler=gui.NO_OP,
                               error_handler=gui.NO_OP)))

    def test_join_ok_button_clicked_morphs_to_processing_page(self):
        """Clicking 'join_ok_button' presents the processing vbox."""
        self.click_join_with_valid_data()
        self.assert_pages_visibility(processing=True)

    def test_processing_vbox_displays_an_active_spinner(self):
        """When processing the registration, an active spinner is shown."""
        self.click_join_with_valid_data()

        self.assertTrue(self.ui.processing_vbox.get_property('visible'),
                        'the processing box should be visible.')

        box = self.ui.processing_vbox.get_children()[0].get_children()[0]
        self.assertEqual(2, len(box.get_children()),
                         'processing_vbox must have two children.')

        spinner, label = box.get_children()
        self.assertIsInstance(spinner, gtk.Spinner)
        self.assertIsInstance(label, gtk.Label)

        self.assertTrue(spinner.get_property('visible'),
                        'the processing spinner should be visible.')
        self.assertTrue(spinner.get_property('active'),
                        'the processing spinner should be active.')
        self.assertTrue(label.get_property('visible'),
                        'the processing label should be visible.')
        self.assertEqual(label.get_text(), self.ui.ONE_MOMENT_PLEASE,
                        'the processing label text must be correct.')

    def test_captcha_image_is_not_visible_at_startup(self):
        """Captcha image is not shown at startup."""
        self.assertFalse(self.ui.captcha_image.get_property('visible'),
                        'the captcha_image should not be visible.')

    def test_captcha_filename_is_different_each_time(self):
        """The captcha image is different each time."""
        ui = self.gui_class(**self.kwargs)
        self.assertNotEqual(self.ui._captcha_filename, ui._captcha_filename)

    def test_captcha_image_is_removed_when_exiting(self):
        """The captcha image is removed at shutdown time."""
        open(self.ui._captcha_filename, 'w').close()
        assert os.path.exists(self.ui._captcha_filename)
        self.ui.on_close_clicked()

        self.assertFalse(os.path.exists(self.ui._captcha_filename),
                         'captcha image must be removed when exiting.')

    def test_captcha_image_is_a_spinner_at_first(self):
        """Captcha image shows a spinner until the image is downloaded."""
        self.assertTrue(self.ui.captcha_loading.get_property('visible'),
                        'the captcha_loading box should be visible.')

        box = self.ui.captcha_loading.get_children()[0].get_children()[0]
        self.assertEqual(2, len(box.get_children()),
                         'captcha_loading must have two children.')

        spinner, label = box.get_children()
        self.assertIsInstance(spinner, gtk.Spinner)
        self.assertIsInstance(label, gtk.Label)

        self.assertTrue(spinner.get_property('visible'),
                        'the captcha_loading spinner should be visible.')
        self.assertTrue(spinner.get_property('active'),
                        'the captcha_loading spinner should be active.')
        self.assertTrue(label.get_property('visible'),
                        'the captcha_loading label should be visible.')
        self.assertEqual(label.get_text(), self.ui.LOADING,
                        'the captcha_loading label text must be correct.')

    def test_join_ok_button_is_disabled_until_captcha_is_available(self):
        """The join_ok_button is not sensitive until captcha is available."""
        self.assertFalse(self.ui.join_ok_button.is_sensitive())

    def test_join_ok_button_is_enabled_when_captcha_is_available(self):
        """The join_ok_button is sensitive when captcha is available."""
        self.ui.on_captcha_generated(app_name=APP_NAME, captcha_id=CAPTCHA_ID)
        self.assertTrue(self.ui.join_ok_button.is_sensitive())

    def test_captcha_loading_is_hid_when_captcha_is_available(self):
        """The captcha_loading is hid when captcha is available."""
        self.ui.on_captcha_generated(app_name=APP_NAME, captcha_id=CAPTCHA_ID)
        self.assertFalse(self.ui.captcha_loading.get_property('visible'),
                         'captcha_loading is not visible.')

    def test_captcha_id_is_stored_when_captcha_is_available(self):
        """The captcha_id is stored when captcha is available."""
        self.ui.on_captcha_generated(app_name=APP_NAME, captcha_id=CAPTCHA_ID)
        self.assertEqual(CAPTCHA_ID, self.ui._captcha_id)

    def test_captcha_image_is_requested_as_startup(self):
        """The captcha image is requested at startup."""
        # assert generate_captcha was called
        expected = 'generate_captcha'
        self.assertIn(expected, self.ui.backend._called)
        self.assertEqual(self.ui.backend._called[expected],
                         ((APP_NAME, self.ui._captcha_filename),
                          dict(reply_handler=gui.NO_OP,
                               error_handler=gui.NO_OP)))

    def test_captcha_is_shown_when_available(self):
        """The captcha image is shown when available."""
        self.patch(self.ui.captcha_image, 'set_from_file', self._set_called)
        self.ui.on_captcha_generated(app_name=APP_NAME, captcha_id=CAPTCHA_ID)
        self.assertTrue(self.ui.captcha_image.get_property('visible'))
        self.assertEqual(self._called, ((self.ui._captcha_filename,), {}))

    def test_on_captcha_generated_logs_captcha_id_when_none(self):
        """If the captcha id is None, a warning is logged."""
        self.ui.on_captcha_generated(app_name=APP_NAME, captcha_id=None)
        self.assertTrue(self.memento.check(logging.WARNING, APP_NAME))
        self.assertTrue(self.memento.check(logging.WARNING,
                                           'captcha_id is None'))

    def test_captcha_reload_button_visible(self):
        """The captcha reload button is initially visible."""
        self.assertTrue(self.ui.captcha_reload_button.get_visible(),
                        "The captcha button is not visible")

    def test_captcha_reload_button_reloads_captcha(self):
        """The captcha reload button loads a new captcha."""
        self.ui.on_captcha_generated(app_name=APP_NAME, captcha_id=CAPTCHA_ID)
        self.patch(self.ui, '_generate_captcha', self._set_called)
        self.ui.captcha_reload_button.clicked()
        self.assertEqual(self._called, ((), {}))

    def test_captcha_reload_button_has_tooltip(self):
        """The captcha reload button has a tooltip."""
        self.assertEqual(self.ui.captcha_reload_button.get_tooltip_text(),
                         self.ui.CAPTCHA_RELOAD_TOOLTIP)

    def test_login_button_has_correct_wording(self):
        """The sign in button has the proper wording."""
        actual = self.ui.login_button.get_label()
        self.assertEqual(self.ui.LOGIN_BUTTON_LABEL, actual)

    def test_join_ok_button_does_nothing_if_clicked_but_disabled(self):
        """The join form can only be submitted if the button is sensitive."""
        self.patch(self.ui.email1_entry, 'get_text', self._set_called)

        self.ui.join_ok_button.set_sensitive(False)
        self.ui.join_ok_button.clicked()
        self.assertFalse(self._called)

        self.ui.join_ok_button.set_sensitive(True)
        self.ui.join_ok_button.clicked()
        self.assertTrue(self._called)

    def test_user_and_pass_are_cached(self):
        """Username and password are temporarly cached for further use."""
        self.click_join_with_valid_data()
        self.assertEqual(self.ui.user_email, EMAIL)
        self.assertEqual(self.ui.user_password, PASSWORD)

    def test_on_captcha_generation_error(self):
        """on_captcha_generation_error shows an error and reloads captcha."""
        self.patch(self.ui, '_generate_captcha', self._set_called)
        self.ui.on_captcha_generation_error(APP_NAME, error=self.error)
        self.assert_correct_label_warning(self.ui.warning_label,
                                          self.ui.CAPTCHA_LOAD_ERROR)
        self.assertEqual(self._called, ((), {}))

    def test_captcha_success_after_error(self):
        """When captcha was retrieved after error, the warning is removed."""
        self.ui.on_captcha_generation_error(APP_NAME, error=self.error)
        self.ui.on_captcha_generated(app_name=APP_NAME, captcha_id=CAPTCHA_ID)
        self.assertEqual(self.ui.warning_label.get_text(), '')


class NoTermsAndConditionsTestCase(UbuntuSSOClientTestCase):
    """Test suite for the user registration (with no t&c link)."""

    kwargs = dict(app_name=APP_NAME, tc_url='', help_text=HELP_TEXT)

    def test_no_tc_link(self):
        """The T&C button and checkbox are not shown if no link is provided"""
        self.assertEqual(self.ui.tc_vbox.get_visible(), False)


class TermsAndConditionsTestCase(UbuntuSSOClientTestCase):
    """Test suite for the user registration (terms & conditions page)."""

    def test_has_tc_link(self):
        """The T&C button and checkbox are shown if the link is provided"""
        self.assertEqual(self.ui.tc_button.get_visible(), True)
        self.assertEqual(self.ui.yes_to_tc_checkbutton.get_visible(), True)


class TermsAndConditionsBrowserTestCase(UbuntuSSOClientTestCase):
    """Test suite for the terms & conditions browser."""

    def setUp(self):
        super(TermsAndConditionsBrowserTestCase, self).setUp()
        self.patch(webkit, 'WebView', FakedEmbeddedBrowser)

        self.ui.tc_button.clicked()
        children = self.ui.tc_browser_window.get_children()
        assert len(children) == 1
        self.browser = children[0]

    def tearDown(self):
        self.ui.tc_browser_vbox.hide()
        super(TermsAndConditionsBrowserTestCase, self).tearDown()

    def test_tc_browser_is_created_when_tc_page_is_shown(self):
        """The browser is created when the TC button is clicked."""
        self.ui.on_tc_browser_notify_load_status(self.browser)

        children = self.ui.tc_browser_window.get_children()
        self.assertEqual(1, len(children))

    def test_is_visible(self):
        """The browser is visible."""
        self.assertIsInstance(self.browser, FakedEmbeddedBrowser)
        self.assertTrue(self.browser.get_property('visible'))

    def test_settings(self):
        """The browser settings are correct."""
        settings = self.browser.get_settings()
        self.assertFalse(settings.get_property('enable-plugins'))
        self.assertFalse(settings.get_property('enable-default-context-menu'))

    def test_tc_browser_is_destroyed_when_tc_page_is_hid(self):
        """The browser is destroyed when the TC page is hid."""
        self.ui.on_tc_browser_notify_load_status(self.browser)
        self.patch(self.browser, 'destroy', self._set_called)
        self.ui.tc_browser_vbox.hide()
        self.assertEqual(self._called, ((), {}))

    def test_tc_browser_is_removed_when_tc_page_is_hid(self):
        """The browser is removed when the TC page is hid."""
        self.ui.on_tc_browser_notify_load_status(self.browser)

        self.ui.tc_browser_vbox.hide()

        children = self.ui.tc_browser_window.get_children()
        self.assertEqual(0, len(children))

    def test_tc_button_clicked_morphs_into_processing_page(self):
        """Clicking the T&C button morphs into processing page."""
        self.assert_pages_visibility(processing=True)

    def test_tc_back_clicked_returns_to_previous_page(self):
        """Terms & Conditions back button return to previous page."""
        self.ui.on_tc_browser_notify_load_status(self.browser)
        self.ui.tc_back_button.clicked()
        self.assert_pages_visibility(enter_details=True)

    def test_tc_button_has_the_proper_wording(self):
        """Terms & Conditions has the proper wording."""
        self.assertEqual(self.ui.tc_button.get_label(), self.ui.TC_BUTTON)

    def test_tc_has_no_help_text(self):
        """The help text is removed."""
        self.ui.on_tc_browser_notify_load_status(self.browser)
        self.assertEqual('', self.ui.help_label.get_text())

    def test_tc_browser_opens_the_proper_url(self):
        """Terms & Conditions browser shows the proper uri."""
        self.assertEqual(self.browser.get_property('uri'), TC_URL)

    def test_notify_load_status_connected(self):
        """The 'notify::load-status' signal is connected."""
        expected = [self.ui.on_tc_browser_notify_load_status]
        self.assertEqual(self.browser._signals['notify::load-status'],
                         expected)

    # Unused variable 'skip'
    # pylint: disable=W0612
    test_notify_load_status_connected.skip = \
        'Connecting to notify::load-status makes U1 terms navigation fail.'

    def test_notify_load_finished_connected(self):
        """The 'load-finished' signal is connected."""
        expected = [self.ui.on_tc_browser_notify_load_status]
        self.assertEqual(self.browser._signals['load-finished'],
                         expected)

    def test_tc_loaded_morphs_into_tc_browser_vbox(self):
        """When the Terms & Conditions is loaded, show the browser window."""
        self.ui.on_tc_browser_notify_load_status(self.browser)
        self.assert_pages_visibility(tc_browser=True)

    def test_navigation_requested_connected(self):
        """The 'navigation-policy-decision-requested' signal is connected."""
        actual = self.browser._signals['navigation-policy-decision-requested']
        expected = [self.ui.on_tc_browser_navigation_requested]
        self.assertEqual(actual, expected)

    def test_navigation_requested_succeeds_for_no_clicking(self):
        """The navigation request succeeds when user hasn't clicked a link."""
        action = webkit.WebNavigationAction()
        action.set_reason(gui.WEBKIT_WEB_NAVIGATION_REASON_OTHER)

        decision = webkit.WebPolicyDecision()
        decision.use = self._set_called

        kwargs = dict(browser=self.browser, frame=None, request=None,
                      action=action, decision=decision)
        self.ui.on_tc_browser_navigation_requested(**kwargs)
        self.assertEqual(self._called, ((), {}))

    def test_navigation_requested_ignores_clicked_links(self):
        """The navigation request is ignored if a link was clicked."""
        action = webkit.WebNavigationAction()
        action.set_reason(gui.WEBKIT_WEB_NAVIGATION_REASON_LINK_CLICKED)

        decision = webkit.WebPolicyDecision()
        decision.ignore = self._set_called

        self.patch(gui.webbrowser, 'open', lambda *args, **kwargs: None)

        kwargs = dict(browser=self.browser, frame=None, request=None,
                      action=action, decision=decision)
        self.ui.on_tc_browser_navigation_requested(**kwargs)
        self.assertEqual(self._called, ((), {}))

    def test_navigation_requested_ignores_for_none(self):
        """The navigation request is ignoref the request if params are None."""
        kwargs = dict(browser=None, frame=None, request=None,
                      action=None, decision=None)
        self.ui.on_tc_browser_navigation_requested(**kwargs)

    def test_navigation_requested_opens_links_when_clicked(self):
        """The navigation request is opened on user's default browser

        (If the user opened a link by clicking into it).

        """
        url = 'http://something.com/yadda'
        action = webkit.WebNavigationAction()
        action.set_reason(gui.WEBKIT_WEB_NAVIGATION_REASON_LINK_CLICKED)
        action.set_original_uri(url)

        decision = webkit.WebPolicyDecision()
        decision.ignore = gui.NO_OP

        self.patch(gui.webbrowser, 'open', self._set_called)

        kwargs = dict(browser=self.browser, frame=None, request=None,
                      action=action, decision=decision)
        self.ui.on_tc_browser_navigation_requested(**kwargs)
        self.assertEqual(self._called, ((url,), {}))


class RegistrationErrorTestCase(UbuntuSSOClientTestCase):
    """Test suite for the user registration error handling."""

    def setUp(self):
        """Init."""
        super(RegistrationErrorTestCase, self).setUp()
        self.click_join_with_valid_data()

    def test_previous_page_is_shown(self):
        """On UserRegistrationError the previous page is shown."""
        self.ui.on_user_registration_error(app_name=APP_NAME, error=self.error)
        self.assert_pages_visibility(enter_details=True)

    def test_captcha_is_reloaded(self):
        """On UserRegistrationError the captcha is reloaded."""
        self.patch(self.ui, '_generate_captcha', self._set_called)
        self.ui.on_user_registration_error(app_name=APP_NAME, error=self.error)
        self.assertEqual(self._called, ((), {}))

    def test_warning_label_is_shown(self):
        """On UserRegistrationError the warning label is shown."""
        self.ui.on_user_registration_error(app_name=APP_NAME, error=self.error)
        self.assert_correct_label_warning(self.ui.warning_label,
                                          self.ui.UNKNOWN_ERROR)

    def test_specific_errors_from_backend_are_shown(self):
        """Specific errors from backend are used."""
        error = {'errtype': 'RegistrationError',
                 'message': 'We\'re so doomed.',
                 'email': 'Enter a valid e-mail address.',
                 'password': 'I don\'t like your password.',
                 '__all__': 'Wrong captcha solution.'}

        self.ui.on_user_registration_error(app_name=APP_NAME, error=error)

        expected = '\n'.join((error['__all__'], error['message']))
        self.assert_correct_label_warning(self.ui.warning_label, expected)
        self.assert_correct_entry_warning(self.ui.email1_entry,
                                          error['email'])
        self.assert_correct_entry_warning(self.ui.email2_entry,
                                          error['email'])
        self.assert_correct_entry_warning(self.ui.password1_entry,
                                          error['password'])
        self.assert_correct_entry_warning(self.ui.password2_entry,
                                          error['password'])


class VerifyEmailTestCase(UbuntuSSOClientTestCase):
    """Test suite for the user registration (verify email page)."""

    def setUp(self):
        """Init."""
        super(VerifyEmailTestCase, self).setUp()
        self.ui.on_user_registered(app_name=APP_NAME, email=EMAIL)
        self.click_verify_email_with_valid_data()

    def test_registration_successful_shows_verify_email_vbox(self):
        """Receiving 'registration_successful' shows the verify email vbox."""
        self.ui.on_user_registered(app_name=APP_NAME, email=EMAIL)
        self.assert_pages_visibility(verify_email=True)

    def test_help_label_display_correct_wording(self):
        """The help_label display VERIFY_EMAIL_LABEL."""
        msg = 'help_label must read "%s" (got "%s" instead).'
        actual = self.ui.help_label.get_label()
        expected = self.ui.VERIFY_EMAIL_LABEL % {'app_name': APP_NAME,
                                                 'email': EMAIL}
        self.assertEqual(expected, actual, msg % (expected, actual))

    def test_on_verify_token_button_clicked_calls_validate_email(self):
        """Verify token button triggers call to backend."""
        self.click_verify_email_with_valid_data()
        expected = 'validate_email'
        self.assertIn(expected, self.ui.backend._called)
        self.assertEqual(self.ui.backend._called[expected],
                         ((APP_NAME, EMAIL, PASSWORD, EMAIL_TOKEN),
                          dict(reply_handler=gui.NO_OP,
                               error_handler=gui.NO_OP)))

    def test_on_verify_token_button_clicked(self):
        """Verify token uses cached user_email and user_password."""
        self.ui.user_email = 'test@me.com'
        self.ui.user_password = 'yadda-yedda'
        self.ui.on_verify_token_button_clicked()
        self.assertEqual(self.ui.backend._called['validate_email'],
                         ((APP_NAME, self.ui.user_email,
                           self.ui.user_password, EMAIL_TOKEN),
                          dict(reply_handler=gui.NO_OP,
                               error_handler=gui.NO_OP)))

    def test_on_verify_token_button_shows_processing_page(self):
        """Verify token button triggers call to backend."""
        self.click_verify_email_with_valid_data()
        self.assert_pages_visibility(processing=True)

    def test_no_warning_messages_if_valid_data(self):
        """No warning messages are shown if the data is valid."""
        # this will certainly NOT generate warnings
        self.click_verify_email_with_valid_data()
        self.assert_warnings_visibility()

    def test_on_email_validated_shows_finish_page(self):
        """On email validated the finish page is shown."""
        self.ui.on_email_validated(app_name=APP_NAME, email=EMAIL)
        self.assert_pages_visibility(finish=True)

    def test_on_email_validated_does_not_clear_the_help_text(self):
        """On email validated the help text is not removed."""
        self.ui.on_email_validated(app_name=APP_NAME, email=EMAIL)
        self.assertEqual(self.ui.verify_email_vbox.help_text,
                         self.ui.help_label.get_label())

    def test_on_email_validation_error_verify_email_is_shown(self):
        """On email validation error, the verify_email page is shown."""
        self.ui.on_email_validation_error(app_name=APP_NAME, error=self.error)
        self.assert_pages_visibility(verify_email=True)
        self.assert_correct_label_warning(self.ui.warning_label,
                                          self.ui.UNKNOWN_ERROR)

    def test_specific_errors_from_backend_are_shown(self):
        """Specific errors from backend are used."""
        error = {'errtype': 'EmailValidationError',
                 'message': 'We\'re so doomed.',
                 'email_token': 'Enter a valid e-mail address.',
                 '__all__': 'We all are gonna die.'}

        self.ui.on_email_validation_error(app_name=APP_NAME, error=error)

        expected = '\n'.join((error['__all__'], error['message']))
        self.assert_correct_label_warning(self.ui.warning_label, expected)
        self.assert_correct_entry_warning(self.ui.email_token_entry,
                                          error['email_token'])

    def test_success_label_is_correct(self):
        """The success message is correct."""
        self.assertEqual(self.ui.SUCCESS,
                         self.ui.success_vbox.label.get_text())
        markup = self.ui.success_vbox.label.get_label()
        self.assertTrue('<span size="x-large">' in markup)

    def test_error_label_is_correct(self):
        """The error message is correct."""
        self.assertEqual(self.ui.ERROR,
                         self.ui.error_vbox.label.get_text())
        markup = self.ui.error_vbox.label.get_label()
        self.assertTrue('<span size="x-large">' in markup)

    def test_on_finish_close_button_clicked_closes_window(self):
        """When done the window is closed."""
        self.ui.finish_close_button.clicked()
        self.assertFalse(self.ui.window.get_property('visible'))

    def test_verify_token_button_does_nothing_if_clicked_but_disabled(self):
        """The email token can only be submitted if the button is sensitive."""
        self.patch(self.ui.email_token_entry, 'get_text', self._set_called)

        self.ui.verify_token_button.set_sensitive(False)
        self.ui.verify_token_button.clicked()
        self.assertFalse(self._called)

        self.ui.verify_token_button.set_sensitive(True)
        self.ui.verify_token_button.clicked()
        self.assertTrue(self._called)

    def test_after_registration_success_finish_success(self):
        """After REGISTRATION_SUCCESS is called, finish_success is called.

        Only when REGISTRATION_SUCCESS returns 0.

        """
        self.patch(self.ui, 'finish_success', self._set_called)
        self.ui.registration_success_callback = lambda app, email: 0

        self.ui.on_email_validated(app_name=APP_NAME, email=EMAIL)

        self.assertEqual(self._called, ((), {}))

    def test_after_registration_error_finish_error(self):
        """After REGISTRATION_SUCCESS is called, finish_error is called.

        Only when REGISTRATION_SUCCESS returns a non-zero value.

        """
        self.patch(self.ui, 'finish_error', self._set_called)
        self.ui.registration_success_callback = lambda app, email: -1

        self.ui.on_email_validated(app_name=APP_NAME, email=EMAIL)

        self.assertEqual(self._called, ((), {}))


class VerifyEmailValidationTestCase(UbuntuSSOClientTestCase):
    """Test suite for the user registration validation (verify email page)."""

    def setUp(self):
        """Init."""
        super(VerifyEmailValidationTestCase, self).setUp()
        self.ui.on_user_registered(app_name=APP_NAME, email=EMAIL)

    def test_warning_is_shown_if_empty_email_token(self):
        """A warning message is shown if email token is empty."""
        self.ui.email_token_entry.set_text('')

        self.ui.verify_token_button.clicked()

        self.assert_correct_entry_warning(self.ui.email_token_entry,
                                          self.ui.FIELD_REQUIRED)
        self.assertNotIn('validate_email', self.ui.backend._called)

    def test_no_warning_messages_if_valid_data(self):
        """No warning messages are shown if the data is valid."""
        # this will certainly NOT generate warnings
        self.click_verify_email_with_valid_data()

        self.assert_warnings_visibility()

    def test_no_warning_messages_if_valid_data_after_invalid_data(self):
        """No warnings if the data is valid (with prior invalid data)."""
        # this will certainly generate warnings
        self.ui.verify_token_button.clicked()

        # this will certainly NOT generate warnings
        self.click_verify_email_with_valid_data()

        self.assert_warnings_visibility()


class VerifyEmailLoginOnlyTestCase(VerifyEmailTestCase):
    """Test suite for the user login (verify email page)."""

    kwargs = dict(app_name=APP_NAME, tc_url=TC_URL, help_text=HELP_TEXT,
                  login_only=True)


class VerifyEmailValidationLoginOnlyTestCase(VerifyEmailValidationTestCase):
    """Test suite for the user login validation (verify email page)."""

    kwargs = dict(app_name=APP_NAME, tc_url=TC_URL, help_text=HELP_TEXT,
                  login_only=True)


class RegistrationValidationTestCase(UbuntuSSOClientTestCase):
    """Test suite for the user registration  validations."""

    def setUp(self):
        """Init."""
        super(RegistrationValidationTestCase, self).setUp()
        self.ui.join_ok_button.set_sensitive(True)

    def test_warning_is_shown_if_name_empty(self):
        """A warning message is shown if name is empty."""
        self.ui.name_entry.set_text('')

        self.ui.join_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.name_entry,
                                          self.ui.FIELD_REQUIRED)
        self.assertNotIn('register_user', self.ui.backend._called)

    def test_warning_is_shown_if_empty_email(self):
        """A warning message is shown if emails are empty."""
        self.ui.email1_entry.set_text('')
        self.ui.email2_entry.set_text('')

        self.ui.join_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.email1_entry,
                                          self.ui.FIELD_REQUIRED)
        self.assert_correct_entry_warning(self.ui.email2_entry,
                                          self.ui.FIELD_REQUIRED)
        self.assertNotIn('register_user', self.ui.backend._called)

    def test_warning_is_shown_if_email_mismatch(self):
        """A warning message is shown if emails doesn't match."""
        self.ui.email1_entry.set_text(EMAIL)
        self.ui.email2_entry.set_text(EMAIL * 2)

        self.ui.join_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.email1_entry,
                                          self.ui.EMAIL_MISMATCH)
        self.assert_correct_entry_warning(self.ui.email2_entry,
                                          self.ui.EMAIL_MISMATCH)
        self.assertNotIn('register_user', self.ui.backend._called)

    def test_warning_is_shown_if_invalid_email(self):
        """A warning message is shown if email is invalid."""
        self.ui.email1_entry.set_text('q')
        self.ui.email2_entry.set_text('q')

        self.ui.join_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.email1_entry,
                                          self.ui.EMAIL_INVALID)
        self.assert_correct_entry_warning(self.ui.email2_entry,
                                          self.ui.EMAIL_INVALID)
        self.assertNotIn('register_user', self.ui.backend._called)

    def test_password_help_is_always_shown(self):
        """Password help text is correctly displayed."""
        self.assertTrue(self.ui.password_help_label.get_property('visible'),
                        'password help text is visible.')
        self.assertEqual(self.ui.password_help_label.get_text(),
                         self.ui.PASSWORD_HELP)
        self.assertNotIn('register_user', self.ui.backend._called)

    def test_warning_is_shown_if_password_mismatch(self):
        """A warning message is shown if password doesn't match."""
        self.ui.password1_entry.set_text(PASSWORD)
        self.ui.password2_entry.set_text(PASSWORD * 2)

        self.ui.join_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.password1_entry,
                                          self.ui.PASSWORD_MISMATCH)
        self.assert_correct_entry_warning(self.ui.password2_entry,
                                          self.ui.PASSWORD_MISMATCH)
        self.assertNotIn('register_user', self.ui.backend._called)

    def test_warning_is_shown_if_password_too_weak(self):
        """A warning message is shown if password is too weak."""
        # password will match but will be too weak
        for pwd in ('', 'h3lloWo', PASSWORD.lower(), 'helloWorld'):
            self.ui.password1_entry.set_text(pwd)
            self.ui.password2_entry.set_text(pwd)

            self.ui.join_ok_button.clicked()

            self.assert_correct_entry_warning(self.ui.password1_entry,
                                              self.ui.PASSWORD_TOO_WEAK)
            self.assert_correct_entry_warning(self.ui.password2_entry,
                                              self.ui.PASSWORD_TOO_WEAK)
        self.assertNotIn('register_user', self.ui.backend._called)

    def test_warning_is_shown_if_tc_not_accepted(self):
        """A warning message is shown if TC are not accepted."""
        # don't agree to TC
        self.ui.yes_to_tc_checkbutton.set_active(False)

        self.ui.join_ok_button.clicked()

        self.assert_correct_label_warning(self.ui.tc_warning_label,
                                          self.ui.TC_NOT_ACCEPTED)
        self.assertNotIn('register_user', self.ui.backend._called)

    def test_warning_is_shown_if_not_captcha_solution(self):
        """A warning message is shown if TC are not accepted."""
        # captcha solution will be empty
        self.ui.captcha_solution_entry.set_text('')

        self.ui.join_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.captcha_solution_entry,
                                          self.ui.FIELD_REQUIRED)
        self.assertNotIn('register_user', self.ui.backend._called)

    def test_no_warning_messages_if_valid_data(self):
        """No warning messages are shown if the data is valid."""
        # this will certainly NOT generate warnings
        self.click_join_with_valid_data()

        self.assert_warnings_visibility()

    def test_no_warning_messages_if_valid_data_after_invalid_data(self):
        """No warnings if the data is valid (with prior invalid data)."""
        # this will certainly generate warnings
        self.ui.join_ok_button.clicked()

        # this will certainly NOT generate warnings
        self.click_join_with_valid_data()

        self.assert_warnings_visibility()


class LoginTestCase(UbuntuSSOClientTestCase):
    """Test suite for the user login pages."""

    def setUp(self):
        """Init."""
        super(LoginTestCase, self).setUp()
        self.ui.login_button.clicked()

    def test_login_button_clicked_morphs_to_login_page(self):
        """Clicking sig_in_button morphs window into login page."""
        self.assert_pages_visibility(login=True)

    def test_initial_text_for_header_label(self):
        """The header must have the correct text when logging in."""
        msg = 'Text for the header must be "%s" (got "%s" instead).'
        expected = self.ui.LOGIN_HEADER_LABEL % {'app_name': APP_NAME}
        actual = self.ui.header_label.get_text()
        self.assertEqual(expected, actual, msg % (expected, actual))

    def test_initial_text_for_help_label(self):
        """The help must have the correct text at startup."""
        msg = 'Text for the help must be "%s" (got "%s" instead).'
        expected = self.ui.CONNECT_HELP_LABEL % {'app_name': APP_NAME}
        actual = self.ui.help_label.get_text()
        self.assertEqual(expected, actual, msg % (expected, actual))

    def test_entries_are_packed_to_ui_for_login(self):
        """Every entry is properly packed in the ui for the login page."""
        entries = ('login_email', 'login_password')
        self.assert_entries_are_packed_to_ui('login_details_vbox', entries)

    def test_entries_are_packed_to_ui_for_set_new_password(self):
        """Every entry is packed in the ui for the reset password page."""
        entries = ('reset_code', 'reset_password1', 'reset_password2')
        self.assert_entries_are_packed_to_ui('set_new_password_details_vbox',
                                             entries)

    def test_entries_are_packed_to_ui_for_request_password_token(self):
        """Every entry is packed in the ui for the reset email page."""
        container_name = 'request_password_token_details_vbox'
        entries = ('reset_email',)
        self.assert_entries_are_packed_to_ui(container_name, entries)

    def test_on_login_back_button_clicked(self):
        """Clicking login_back_button show registration page."""
        self.ui.login_back_button.clicked()
        self.assert_pages_visibility(enter_details=True)

    def test_on_login_connect_button_clicked(self):
        """Clicking login_ok_button calls backend.login."""
        self.click_connect_with_valid_data()

        expected = 'login'
        self.assertIn(expected, self.ui.backend._called)
        self.assertEqual(self.ui.backend._called[expected],
                         ((APP_NAME, EMAIL, PASSWORD),
                          dict(reply_handler=gui.NO_OP,
                               error_handler=gui.NO_OP)))

    def test_on_login_connect_button_clicked_morphs_to_processing_page(self):
        """Clicking login_ok_button morphs to the processing page."""
        self.click_connect_with_valid_data()
        self.assert_pages_visibility(processing=True)

    def test_on_logged_in_morphs_to_finish_page(self):
        """When user logged in the finish page is shown."""
        self.click_connect_with_valid_data()
        self.ui.on_logged_in(app_name=APP_NAME, email=EMAIL)
        self.assert_pages_visibility(finish=True)

    def test_on_login_error_morphs_to_login_page(self):
        """On user login error, the previous page is shown."""
        self.click_connect_with_valid_data()
        self.ui.on_login_error(app_name=APP_NAME, error=self.error)
        self.assert_pages_visibility(login=True)

    def test_on_user_not_validated_morphs_to_verify_page(self):
        """On user not validated, the verify page is shown."""
        self.click_connect_with_valid_data()
        self.ui.on_user_not_validated(app_name=APP_NAME, email=EMAIL)
        self.assert_pages_visibility(verify_email=True)

    def test_on_login_error_a_warning_is_shown(self):
        """On user login error, a warning is shown with proper wording."""
        self.click_connect_with_valid_data()
        self.ui.on_login_error(app_name=APP_NAME, error=self.error)
        self.assert_correct_label_warning(self.ui.warning_label,
                                          self.ui.UNKNOWN_ERROR)

    def test_specific_errors_from_backend_are_shown(self):
        """Specific errors from backend are used."""
        error = {'errtype': 'AuthenticationError',
                 'message': 'We\'re so doomed.',
                 '__all__': 'We all are gonna die.'}

        self.ui.on_login_error(app_name=APP_NAME, error=error)

        expected = '\n'.join((error['__all__'], error['message']))
        self.assert_correct_label_warning(self.ui.warning_label, expected)

    def test_back_to_registration_hides_warning(self):
        """After user login error, warning is hidden when clicking 'Back'."""
        self.click_connect_with_valid_data()
        self.ui.on_login_error(app_name=APP_NAME, error=self.error)
        self.ui.login_back_button.clicked()
        self.assert_warnings_visibility()

    def test_login_ok_button_does_nothing_if_clicked_but_disabled(self):
        """The join form can only be submitted if the button is sensitive."""
        self.patch(self.ui.login_email_entry, 'get_text', self._set_called)

        self.ui.login_ok_button.set_sensitive(False)
        self.ui.login_ok_button.clicked()
        self.assertFalse(self._called)

        self.ui.login_ok_button.set_sensitive(True)
        self.ui.login_ok_button.clicked()
        self.assertTrue(self._called)

    def test_user_and_pass_are_cached(self):
        """Username and password are temporarly cached for further use."""
        self.click_connect_with_valid_data()
        self.assertEqual(self.ui.user_email, EMAIL)
        self.assertEqual(self.ui.user_password, PASSWORD)

    def test_after_login_success_finish_success(self):
        """After LOGIN_SUCCESSFULL is called, finish_success is called.

        Only when LOGIN_SUCCESSFULL returns 0.

        """
        self.patch(self.ui, 'finish_success', self._set_called)
        self.ui.login_success_callback = lambda app, email: 0

        self.ui.on_logged_in(app_name=APP_NAME, email=EMAIL)

        self.assertEqual(self._called, ((), {}))

    def test_after_login_error_finish_error(self):
        """After LOGIN_SUCCESSFULL is called, finish_error is called.

        Only when LOGIN_SUCCESSFULL returns a non-zero value.

        """
        self.patch(self.ui, 'finish_error', self._set_called)
        self.ui.login_success_callback = lambda app, email: -1

        self.ui.on_logged_in(app_name=APP_NAME, email=EMAIL)

        self.assertEqual(self._called, ((), {}))


class LoginValidationTestCase(UbuntuSSOClientTestCase):
    """Test suite for the user login validation."""

    def setUp(self):
        """Init."""
        super(LoginValidationTestCase, self).setUp()
        self.ui.login_button.clicked()

    def test_warning_is_shown_if_empty_email(self):
        """A warning message is shown if email is empty."""
        self.ui.login_email_entry.set_text('')

        self.ui.login_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.login_email_entry,
                                          self.ui.FIELD_REQUIRED)
        self.assertNotIn('login', self.ui.backend._called)

    def test_warning_is_shown_if_invalid_email(self):
        """A warning message is shown if email is invalid."""
        self.ui.login_email_entry.set_text('q')

        self.ui.login_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.login_email_entry,
                                          self.ui.EMAIL_INVALID)
        self.assertNotIn('login', self.ui.backend._called)

    def test_warning_is_shown_if_empty_password(self):
        """A warning message is shown if password is empty."""
        self.ui.login_password_entry.set_text('')

        self.ui.login_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.login_password_entry,
                                          self.ui.FIELD_REQUIRED)
        self.assertNotIn('login', self.ui.backend._called)

    def test_no_warning_messages_if_valid_data(self):
        """No warning messages are shown if the data is valid."""
        # this will certainly NOT generate warnings
        self.click_connect_with_valid_data()

        self.assert_warnings_visibility()

    def test_no_warning_messages_if_valid_data_after_invalid_data(self):
        """No warnings if the data is valid (with prior invalid data)."""
        # this will certainly generate warnings
        self.ui.login_ok_button.clicked()

        # this will certainly NOT generate warnings
        self.click_connect_with_valid_data()

        self.assert_warnings_visibility()


class ResetPasswordTestCase(UbuntuSSOClientTestCase):
    """Test suite for the reset password functionality."""

    def setUp(self):
        """Init."""
        super(ResetPasswordTestCase, self).setUp()
        self.ui.login_button.clicked()
        self.ui.forgotten_password_button.clicked()

    def test_forgotten_password_button_has_the_proper_wording(self):
        """The forgotten_password_button has the proper wording."""
        self.assertEqual(self.ui.forgotten_password_button.get_label(),
                         self.ui.FORGOTTEN_PASSWORD_BUTTON)

    def test_on_forgotten_password_button_clicked_help_text(self):
        """Clicking forgotten_password_button the help is properly changed."""
        wanted = self.ui.REQUEST_PASSWORD_TOKEN_LABEL % {'app_name': APP_NAME}
        self.assertEqual(self.ui.help_label.get_text(), wanted)

    def test_on_forgotten_password_button_clicked_header_label(self):
        """Clicking forgotten_password_button the title is properly changed."""
        self.assertEqual(self.ui.header_label.get_text(),
                         self.ui.RESET_PASSWORD)

    def test_on_forgotten_password_button_clicked_ok_button(self):
        """Clicking forgotten_password_button the ok button reads 'Next'."""
        self.assertEqual(self.ui.request_password_token_ok_button.get_label(),
                         self.ui.NEXT)

    def test_on_forgotten_password_button_clicked_morphs_window(self):
        """Clicking forgotten_password_button the proper page is shown."""
        self.assert_pages_visibility(request_password_token=True)

    def test_on_request_password_token_back_button_clicked(self):
        """Clicking request_password_token_back_button show login screen."""
        self.ui.request_password_token_back_button.clicked()
        self.assert_pages_visibility(login=True)

    def test_request_password_token_ok_button_disabled_until_email_added(self):
        """The button is disabled until email added."""
        is_sensitive = self.ui.request_password_token_ok_button.get_sensitive
        self.assertFalse(is_sensitive())

        self.ui.reset_email_entry.set_text('a')
        self.assertTrue(is_sensitive())

        self.ui.reset_email_entry.set_text('')
        self.assertFalse(is_sensitive())

        self.ui.reset_email_entry.set_text('         ')
        self.assertFalse(is_sensitive())

    def test_on_request_password_token_ok_button_clicked_morphs_window(self):
        """Clicking request_password_token_ok_button morphs processing page."""
        self.click_request_password_token_with_valid_data()
        self.assert_pages_visibility(processing=True)

    def test_on_request_password_token_ok_button_clicked_calls_backend(self):
        """Clicking request_password_token_ok_button the backend is called."""
        self.click_request_password_token_with_valid_data()
        expected = 'request_password_reset_token'
        self.assertIn(expected, self.ui.backend._called)
        self.assertEqual(self.ui.backend._called[expected],
                         ((APP_NAME, EMAIL),
                          dict(reply_handler=gui.NO_OP,
                               error_handler=gui.NO_OP)))

    def test_on_password_reset_token_sent_morphs_window(self):
        """When the reset token was sent, the reset password page is shown."""
        self.click_request_password_token_with_valid_data()
        self.ui.on_password_reset_token_sent(app_name=APP_NAME, email=EMAIL)
        self.assert_pages_visibility(set_new_password=True)

    def test_on_password_reset_token_sent_help_text(self):
        """Clicking request_password_token_ok_button changes the help text."""
        self.click_request_password_token_with_valid_data()
        self.ui.on_password_reset_token_sent(app_name=APP_NAME, email=EMAIL)

        self.assertEqual(self.ui.help_label.get_text(),
                         self.ui.SET_NEW_PASSWORD_LABEL % {'email': EMAIL})

    def test_on_password_reset_token_sent_ok_button(self):
        """After request_password_token_ok_button the ok button is updated."""
        self.click_request_password_token_with_valid_data()
        self.ui.on_password_reset_token_sent(app_name=APP_NAME, email=EMAIL)

        self.assertEqual(self.ui.set_new_password_ok_button.get_label(),
                         self.ui.RESET_PASSWORD)

    def test_on_password_reset_error_shows_login_page(self):
        """When reset token wasn't successfuly sent the login page is shown."""
        self.ui.on_password_reset_error(app_name=APP_NAME, error=self.error)
        self.assert_correct_label_warning(self.ui.warning_label,
                                          self.ui.UNKNOWN_ERROR)
        self.assert_pages_visibility(login=True)

    def test_specific_errors_from_backend_are_shown(self):
        """Specific errors from backend are used."""
        error = {'errtype': 'ResetPasswordTokenError',
                 'message': 'We\'re so doomed.',
                 '__all__': 'We all are gonna die.'}

        self.ui.on_password_reset_error(app_name=APP_NAME, error=error)

        expected = '\n'.join((error['__all__'], error['message']))
        self.assert_correct_label_warning(self.ui.warning_label, expected)

    def test_ok_button_does_nothing_if_clicked_but_disabled(self):
        """The password token can be requested if the button is sensitive."""
        self.patch(self.ui.reset_email_entry, 'get_text', self._set_called)

        self.ui.request_password_token_ok_button.set_sensitive(False)
        self.ui.request_password_token_ok_button.clicked()
        self.assertFalse(self._called)

        self.ui.request_password_token_ok_button.set_sensitive(True)
        self.ui.request_password_token_ok_button.clicked()
        self.assertTrue(self._called)


class ResetPasswordValidationTestCase(UbuntuSSOClientTestCase):
    """Test suite for the password reset validations."""

    def test_warning_is_shown_if_empty_email(self):
        """A warning message is shown if emails are empty."""
        self.ui.reset_email_entry.set_text(' ')

        self.ui.request_password_token_ok_button.set_sensitive(True)
        self.ui.request_password_token_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.reset_email_entry,
                                          self.ui.FIELD_REQUIRED)
        self.assertNotIn('request_password_reset_token',
                         self.ui.backend._called)

    def test_warning_is_shown_if_invalid_email(self):
        """A warning message is shown if email is invalid."""
        self.ui.reset_email_entry.set_text('q')

        self.ui.request_password_token_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.reset_email_entry,
                                          self.ui.EMAIL_INVALID)
        self.assertNotIn('request_password_reset_token',
                         self.ui.backend._called)

    def test_no_warning_messages_if_valid_data(self):
        """No warning messages are shown if the data is valid."""
        # this will certainly NOT generate warnings
        self.click_request_password_token_with_valid_data()

        self.assert_warnings_visibility()

    def test_no_warning_messages_if_valid_data_after_invalid_data(self):
        """No warnings if the data is valid (with prior invalid data)."""
        # this will certainly generate warnings
        self.ui.request_password_token_ok_button.clicked()

        # this will certainly NOT generate warnings
        self.click_request_password_token_with_valid_data()

        self.assert_warnings_visibility()


class SetNewPasswordTestCase(UbuntuSSOClientTestCase):
    """Test suite for setting a new password functionality."""

    def setUp(self):
        """Init."""
        super(SetNewPasswordTestCase, self).setUp()
        self.click_request_password_token_with_valid_data()
        self.ui.on_password_reset_token_sent(app_name=APP_NAME, email=EMAIL)

    def test_on_set_new_password_ok_button_disabled(self):
        """The set_new_password_ok_button is disabled until values added."""
        self.click_request_password_token_with_valid_data()
        self.assertFalse(self.ui.set_new_password_ok_button.get_sensitive())

        msg = 'set_new_password_ok_button must be sensitive (%s) for %r.'
        entries = (self.ui.reset_code_entry,
                   self.ui.reset_password1_entry,
                   self.ui.reset_password2_entry)
        for values in itertools.product(('', ' ', 'a'), repeat=3):
            expected = True
            for entry, val in zip(entries, values):
                entry.set_text(val)
                expected &= bool(val and not val.isspace())

            actual = self.ui.set_new_password_ok_button.get_sensitive()
            self.assertEqual(expected, actual, msg % (expected, values))

    def test_on_set_new_password_ok_button_clicked_morphs_window(self):
        """Clicking set_new_password_ok_button the processing page is shown."""
        self.click_set_new_password_with_valid_data()
        self.assert_pages_visibility(processing=True)

    def test_on_set_new_password_ok_button_clicked_calls_backend(self):
        """Clicking set_new_password_ok_button the backend is called."""
        self.click_set_new_password_with_valid_data()
        expected = 'set_new_password'
        self.assertIn(expected, self.ui.backend._called)
        self.assertEqual(self.ui.backend._called[expected],
                         ((APP_NAME, EMAIL, RESET_PASSWORD_TOKEN, PASSWORD),
                          dict(reply_handler=gui.NO_OP,
                               error_handler=gui.NO_OP)))

    def test_on_password_changed_shows_login_page(self):
        """When password was successfuly changed the login page is shown."""
        self.ui.on_password_changed(app_name=APP_NAME, email=EMAIL)
        self.assert_correct_label_warning(self.ui.warning_label,
                                          self.ui.PASSWORD_CHANGED)
        self.assert_pages_visibility(login=True)

    def test_on_password_change_error_shows_login_page(self):
        """When password wasn't changed the reset password page is shown."""
        self.ui.on_password_change_error(app_name=APP_NAME, error=self.error)
        self.assert_correct_label_warning(self.ui.warning_label,
                                          self.ui.UNKNOWN_ERROR)
        self.assert_pages_visibility(request_password_token=True)

    def test_specific_errors_from_backend_are_shown(self):
        """Specific errors from backend are used."""
        error = {'errtype': 'NewPasswordError',
                 'message': 'We\'re so doomed.',
                 '__all__': 'We all are gonna die.'}

        self.ui.on_password_change_error(app_name=APP_NAME, error=error)

        expected = '\n'.join((error['__all__'], error['message']))
        self.assert_correct_label_warning(self.ui.warning_label, expected)

    def test_ok_button_does_nothing_if_clicked_but_disabled(self):
        """The new passwrd can only be set if the button is sensitive."""
        self.patch(self.ui.reset_code_entry, 'get_text', self._set_called)

        self.ui.set_new_password_ok_button.set_sensitive(False)
        self.ui.set_new_password_ok_button.clicked()
        self.assertFalse(self._called)

        self.ui.set_new_password_ok_button.set_sensitive(True)
        self.ui.set_new_password_ok_button.clicked()
        self.assertTrue(self._called)


class SetNewPasswordValidationTestCase(UbuntuSSOClientTestCase):
    """Test suite for validations for setting a new password."""

    def test_warning_is_shown_if_reset_code_empty(self):
        """A warning message is shown if reset_code is empty."""
        self.ui.reset_code_entry.set_text('')

        self.ui.set_new_password_ok_button.set_sensitive(True)
        self.ui.set_new_password_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.reset_code_entry,
                                          self.ui.FIELD_REQUIRED)
        self.assertNotIn('set_new_password', self.ui.backend._called)

    def test_password_help_is_always_shown(self):
        """Password help text is correctly displayed."""
        visible = self.ui.reset_password_help_label.get_property('visible')
        self.assertTrue(visible, 'password help text is visible.')
        self.assertEqual(self.ui.reset_password_help_label.get_text(),
                         self.ui.PASSWORD_HELP)
        self.assertNotIn('set_new_password', self.ui.backend._called)

    def test_warning_is_shown_if_password_mismatch(self):
        """A warning message is shown if password doesn't match."""
        self.ui.reset_password1_entry.set_text(PASSWORD)
        self.ui.reset_password2_entry.set_text(PASSWORD * 2)

        self.ui.set_new_password_ok_button.set_sensitive(True)
        self.ui.set_new_password_ok_button.clicked()

        self.assert_correct_entry_warning(self.ui.reset_password1_entry,
                                          self.ui.PASSWORD_MISMATCH)
        self.assert_correct_entry_warning(self.ui.reset_password2_entry,
                                          self.ui.PASSWORD_MISMATCH)
        self.assertNotIn('set_new_password', self.ui.backend._called)

    def test_warning_is_shown_if_password_too_weak(self):
        """A warning message is shown if password is too weak."""
        # password will match but will be too weak
        for pwd in ('', 'h3lloWo', PASSWORD.lower(), 'helloWorld'):
            self.ui.reset_password1_entry.set_text(pwd)
            self.ui.reset_password2_entry.set_text(pwd)

            self.ui.set_new_password_ok_button.set_sensitive(True)
            self.ui.set_new_password_ok_button.clicked()

            self.assert_correct_entry_warning(self.ui.reset_password1_entry,
                                              self.ui.PASSWORD_TOO_WEAK)
            self.assert_correct_entry_warning(self.ui.reset_password2_entry,
                                              self.ui.PASSWORD_TOO_WEAK)
        self.assertNotIn('set_new_password', self.ui.backend._called)

    def test_no_warning_messages_if_valid_data(self):
        """No warning messages are shown if the data is valid."""
        # this will certainly NOT generate warnings
        self.click_set_new_password_with_valid_data()

        self.assert_warnings_visibility()

    def test_no_warning_messages_if_valid_data_after_invalid_data(self):
        """No warnings if the data is valid (with prior invalid data)."""
        # this will certainly generate warnings
        self.ui.set_new_password_ok_button.clicked()

        # this will certainly NOT generate warnings
        self.click_set_new_password_with_valid_data()

        self.assert_warnings_visibility()


class DbusTestCase(UbuntuSSOClientTestCase):
    """Test suite for the dbus calls."""

    def test_all_the_signals_are_listed(self):
        """All the backend signals are listed to be binded."""
        for sig in ('CaptchaGenerated', 'CaptchaGenerationError',
                    'UserRegistered', 'UserRegistrationError',
                    'LoggedIn', 'LoginError', 'UserNotValidated',
                    'EmailValidated', 'EmailValidationError',
                    'PasswordResetTokenSent', 'PasswordResetError',
                    'PasswordChanged', 'PasswordChangeError'):
            self.assertIn(sig, self.ui._signals)

    def test_signal_receivers_are_connected(self):
        """Callbacks are connected to signals of interest."""
        msg1 = 'callback %r for signal %r must be added to the internal bus.'
        msg2 = 'callback %r for signal %r must be added to the ui log.'
        dbus_iface = self.ui.iface_name
        for signal, method in self.ui._signals.iteritems():
            actual = self.ui.bus.callbacks.get((dbus_iface, signal))
            self.assertEqual(method, actual, msg1 % (method, signal))
            actual = self.ui._signals_receivers.get((dbus_iface, signal))
            self.assertEqual(method, actual, msg2 % (method, signal))

    def test_callbacks_only_log_when_app_name_doesnt_match(self):
        """Callbacks do nothing but logging when app_name doesn't match."""
        mismatch_app_name = self.ui.app_name * 2
        for method in self.ui._signals.itervalues():
            msgs = ('ignoring', method.__name__, mismatch_app_name)
            method(mismatch_app_name, 'dummy')
            self.assertTrue(self.memento.check(logging.INFO, *msgs))
            self.memento.records = []

    def test_on_captcha_generated_is_not_called(self):
        """on_captcha_generated is not called if incorrect app_name."""
        self.patch(self.ui, 'on_captcha_generated', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['CaptchaGenerated'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_captcha_generation_error_is_not_called(self):
        """on_captcha_generation_error is not called if incorrect app_name."""
        self.patch(self.ui, 'on_captcha_generation_error', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['CaptchaGenerationError'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_user_registered_is_not_called(self):
        """on_user_registered is not called if incorrect app_name."""
        self.patch(self.ui, 'on_user_registered', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['UserRegistered'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_user_registration_error_is_not_called(self):
        """on_user_registration_error is not called if incorrect app_name."""
        self.patch(self.ui, 'on_user_registration_error', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['UserRegistrationError'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_email_validated_is_not_called(self):
        """on_email_validated is not called if incorrect app_name."""
        self.patch(self.ui, 'on_email_validated', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['EmailValidated'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_email_validation_error_is_not_called(self):
        """on_email_validation_error is not called if incorrect app_name."""
        self.patch(self.ui, 'on_email_validation_error', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['EmailValidationError'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_logged_in_is_not_called(self):
        """on_logged_in is not called if incorrect app_name."""
        self.patch(self.ui, 'on_logged_in', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['LoggedIn'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_login_error_is_not_called(self):
        """on_login_error is not called if incorrect app_name."""
        self.patch(self.ui, 'on_login_error', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['LoginError'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_user_not_validated_is_not_called(self):
        """on_user_not_validated is not called if incorrect app_name."""
        self.patch(self.ui, 'on_user_not_validated', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['UserNotValidated'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_password_reset_token_sent_is_not_called(self):
        """on_password_reset_token_sent is not called if incorrect app_name."""
        self.patch(self.ui, 'on_password_reset_token_sent', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['PasswordResetTokenSent'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_password_reset_error_is_not_called(self):
        """on_password_reset_error is not called if incorrect app_name."""
        self.patch(self.ui, 'on_password_reset_error', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['PasswordResetError'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_password_changed_is_not_called(self):
        """on_password_changed is not called if incorrect app_name."""
        self.patch(self.ui, 'on_password_changed', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['PasswordChanged'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)

    def test_on_password_change_error_is_not_called(self):
        """on_password_change_error is not called if incorrect app_name."""
        self.patch(self.ui, 'on_password_change_error', self._set_called)
        mismatch_app_name = self.ui.app_name * 2
        self.ui._signals['PasswordChangeError'](mismatch_app_name, 'dummy')
        self.assertFalse(self._called)


class LoginOnlyTestCase(UbuntuSSOClientTestCase):
    """Test suite for the login only GUI."""

    kwargs = dict(app_name=APP_NAME, tc_url=None, help_text=HELP_TEXT,
                  login_only=True)

    def test_login_is_first_page(self):
        """When starting, the login page is the first one."""
        self.assert_pages_visibility(login=True)

    def test_no_back_button(self):
        """There is no back button in the login screen."""
        self.assertFalse(self.ui.login_back_button.get_property('visible'))

    def test_login_ok_button_has_the_focus(self):
        """The login_ok_button has the focus."""
        self.assertTrue(self.ui.login_ok_button.is_focus())

    def test_help_text_is_used(self):
        """The passed help_text is used."""
        self.assertEqual(self.ui.help_label.get_text(), HELP_TEXT)


class CallbacksTestCase(UbuntuSSOClientTestCase):
    """Test the GTK callback calls."""

    LOGIN_SUCCESSFULL = 'login_success_callback'
    REGISTRATION_SUCCESS = 'registration_success_callback'
    USER_CANCELLATION = 'user_cancellation_callback'

    def setUp(self):
        """Init."""
        super(CallbacksTestCase, self).setUp()
        self._called = {}
        for name in (self.LOGIN_SUCCESSFULL,
                     self.REGISTRATION_SUCCESS,
                     self.USER_CANCELLATION):
            setattr(self.ui, name, self._set_called(name))

    def _set_called(self, callback_name):
        """Keep trace of callbacks calls."""

        # pylint: disable=W0221

        def inner(*args, **kwargs):
            """Store arguments used in this call."""
            self._called[callback_name] = args

        return inner

    def test_closing_main_window(self):
        """When closing the main window, USER_CANCELLATION is called."""
        self.ui.window.emit('delete-event', gtk.gdk.Event(gtk.gdk.DELETE))
        self.assertEqual(self._called[self.USER_CANCELLATION], (APP_NAME,))

    def test_every_cancel_calls_proper_callback(self):
        """When any cancel button is clicked, USER_CANCELLATION is called."""
        msg = 'user_cancellation_callback is called when "%s" is clicked.'
        buttons = filter(lambda name: 'cancel_button' in name, self.ui.widgets)
        for name in buttons:
            widget = getattr(self.ui, name)
            widget.clicked()
            self.assertEqual(self._called[self.USER_CANCELLATION], (APP_NAME,),
                             msg % name)
            self._called = {}

    def test_on_user_registration_error_proper_callback_is_called(self):
        """On UserRegistrationError, USER_CANCELLATION is called."""
        self.ui.on_user_registration_error(app_name=APP_NAME, error=self.error)
        self.ui.on_close_clicked()

        self.assertEqual(self._called[self.USER_CANCELLATION], (APP_NAME,))

    def test_on_email_validated_proper_callback_is_called(self):
        """On EmailValidated, REGISTRATION_SUCCESS is called."""
        self.ui.on_email_validated(app_name=APP_NAME, email=EMAIL)
        self.ui.on_close_clicked()

        self.assertEqual(self._called[self.REGISTRATION_SUCCESS],
                         (APP_NAME, EMAIL))

    def test_on_email_validation_error_proper_callback_is_called(self):
        """On EmailValidationError, USER_CANCELLATION is called."""
        self.ui.on_email_validation_error(app_name=APP_NAME, error=self.error)
        self.ui.on_close_clicked()

        self.assertEqual(self._called[self.USER_CANCELLATION], (APP_NAME,))

    def test_on_logged_in_proper_callback_is_called(self):
        """On LoggedIn, LOGIN_SUCCESSFULL is called."""
        self.ui.on_logged_in(app_name=APP_NAME, email=EMAIL)
        self.ui.on_close_clicked()

        self.assertEqual(self._called[self.LOGIN_SUCCESSFULL],
                         (APP_NAME, EMAIL))

    def test_on_login_error_proper_callback_is_called(self):
        """On LoginError, USER_CANCELLATION is called."""
        self.click_connect_with_valid_data()
        self.ui.on_login_error(app_name=APP_NAME, error=self.error)
        self.ui.on_close_clicked()

        self.assertEqual(self._called[self.USER_CANCELLATION], (APP_NAME,))

    def test_registration_success_even_if_prior_registration_error(self):
        """Only one callback is called with the final outcome.

        When the user successfully registers, REGISTRATION_SUCCESS is
        called even if there were errors before.

        """
        self.click_join_with_valid_data()
        self.ui.on_user_registration_error(app_name=APP_NAME, error=self.error)
        self.click_join_with_valid_data()
        self.ui.on_email_validated(app_name=APP_NAME, email=EMAIL)
        self.ui.on_close_clicked()

        self.assertEqual(self._called[self.REGISTRATION_SUCCESS],
                         (APP_NAME, EMAIL))

    def test_login_success_even_if_prior_login_error(self):
        """Only one callback is called with the final outcome.

        When the user successfully logins, LOGIN_SUCCESSFULL is called even if
        there were errors before.

        """
        self.click_connect_with_valid_data()
        self.ui.on_login_error(app_name=APP_NAME, error=self.error)
        self.click_connect_with_valid_data()
        self.ui.on_logged_in(app_name=APP_NAME, email=EMAIL)
        self.ui.on_close_clicked()

        self.assertEqual(self._called[self.LOGIN_SUCCESSFULL],
                         (APP_NAME, EMAIL))

    def test_user_cancelation_even_if_prior_registration_error(self):
        """Only one callback is called with the final outcome.

        When the user closes the window, USER_CANCELLATION is called even if
        there were registration errors before.

        """
        self.click_join_with_valid_data()
        self.ui.on_user_registration_error(app_name=APP_NAME, error=self.error)
        self.ui.join_cancel_button.clicked()

        self.assertEqual(self._called[self.USER_CANCELLATION], (APP_NAME,))

    def test_user_cancelation_even_if_prior_login_error(self):
        """Only one callback is called with the final outcome.

        When the user closes the window, USER_CANCELLATION is called even if
        there were login errors before.

        """
        self.click_connect_with_valid_data()
        self.ui.on_login_error(app_name=APP_NAME, error=self.error)
        self.ui.login_cancel_button.clicked()

        self.assertEqual(self._called[self.USER_CANCELLATION], (APP_NAME,))


class DefaultButtonsTestCase(UbuntuSSOClientTestCase):
    """Each UI page has a default button when visible."""

    def setUp(self):
        """Init."""
        super(DefaultButtonsTestCase, self).setUp()
        self.mapping = (
            ('enter_details_vbox', 'join_ok_button'),
            ('tc_browser_vbox', 'tc_back_button'),
            ('verify_email_vbox', 'verify_token_button'),
            ('login_vbox', 'login_ok_button'),
            ('request_password_token_vbox',
             'request_password_token_ok_button'),
            ('set_new_password_vbox', 'set_new_password_ok_button'),
            ('success_vbox', 'finish_close_button'),
            ('error_vbox', 'finish_close_button'),
            ('processing_vbox', None))

    def test_pages_have_default_widget_set(self):
        """Each page has a proper button as default widget."""
        msg = 'Page %r must have %r as default_widget (got %r instead).'
        for pname, bname in self.mapping:
            page = getattr(self.ui, pname)
            button = bname and getattr(self.ui, bname)
            self.assertTrue(page.default_widget is button,
                            msg % (pname, bname, page.default_widget))

    def test_default_widget_can_default(self):
        """Each default button can default."""
        msg = 'Button %r must have can-default enabled.'
        for _, bname in self.mapping:
            if bname is not None:
                button = getattr(self.ui, bname)
                self.assertTrue(button.get_property('can-default'),
                                msg % bname)

    def test_set_current_page_grabs_focus_for_default_button(self):
        """Setting the current page marks the default widget as default."""
        msg = '%r "has_default" must be True when %r if the current page.'
        for pname, bname in self.mapping:
            if bname is not None:
                page = getattr(self.ui, pname)
                self.patch(page.default_widget, 'grab_default',
                           self._set_called)
                self.ui._set_current_page(page)
                self.assertEqual(self._called, ((), {}), msg % (bname, pname))
                self._called = False
