#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
# Author: Natalia B. Bidart <natalia.bidart@canonical.com>
#
# Copyright 2009-2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Base tests cases and test utilities."""

import logging

import dbus

from dbus.mainloop.glib import DBusGMainLoop
from twisted.internet import defer
from twisted.python import failure
from twisted.trial.unittest import TestCase


class DBusTestCase(TestCase):
    """Test the DBus event handling."""

    def setUp(self):
        """Setup the infrastructure fo the test (dbus service)."""
        self.loop = DBusGMainLoop(set_as_default=True)
        self.bus = dbus.bus.BusConnection(mainloop=self.loop)
        # monkeypatch busName.__del__ to avoid errors on gc
        # we take care of releasing the name in shutdown
        dbus.service.BusName.__del__ = lambda _: None
        self.bus.set_exit_on_disconnect(False)
        self.signal_receivers = set()

    def tearDown(self):
        """Cleanup the test."""
        d = self.cleanup_signal_receivers(self.signal_receivers)
        d.addBoth(self._tear_down)
        return d

    def _tear_down(self, _):
        """Shutdown."""
        self.bus.flush()
        self.bus.close()

    def error_handler(self, error):
        """Default error handler for DBus calls."""
        if isinstance(error, failure.Failure):
            self.fail(error.getErrorMessage())

    def cleanup_signal_receivers(self, signal_receivers):
        """Cleanup self.signal_receivers and returns a deferred."""
        deferreds = []
        for match in signal_receivers:
            d = defer.Deferred()

            def callback(*args):
                """Callback that accepts *args."""
                if not d.called:
                    d.callback(args)

            self.bus.call_async(dbus.bus.BUS_DAEMON_NAME,
                                dbus.bus.BUS_DAEMON_PATH,
                                dbus.bus.BUS_DAEMON_IFACE, 'RemoveMatch', 's',
                                (str(match),), callback, self.error_handler)
            deferreds.append(d)
        if deferreds:
            return defer.DeferredList(deferreds)
        else:
            return defer.succeed(True)


class MementoHandler(logging.Handler):
    """A handler class which store logging records in a list."""

    def __init__(self, *args, **kwargs):
        """Create the instance, and add a records attribute."""
        logging.Handler.__init__(self, *args, **kwargs)
        self.records = []

    def emit(self, record):
        """Just add the record to self.records."""
        self.records.append(record)

    def check(self, level, *msgs):
        """Verifies that the msgs are logged in the specified level."""
        for rec in self.records:
            if rec.levelno == level and all(m in rec.message for m in msgs):
                return True
        return False
