/*
A GeoClue provider to use the Ubuntu GeoIP web service

Copyright 2010 Canonical Ltd.

Authors:
    Ted Gould <ted@canonical.com>

This program is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License version 3, as published 
by the Free Software Foundation.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranties of 
MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along 
with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <glib.h>

#include "ubuntu-geoip-provider.h"

/* GeoClue */
#include <geoclue/gc-iface-position.h>
#include <geoclue/gc-iface-address.h>

/* Soup */
#include <libsoup/soup.h>

/* Network Manager */
#include <nm-client.h>

#define REQUEUE_MIN_SEC   15
#define REQUEUE_MAX_SEC   (60 * 60)
#define REQUEUE_SCALE     2

typedef enum _MarkupField MarkupField;
enum _MarkupField {
	/* Address */
	MARKUP_FIELD_COUNTRY_CODE,
	MARKUP_FIELD_COUNTRY_NAME,
	MARKUP_FIELD_REGION_NAME,
	MARKUP_FIELD_CITY,
	MARKUP_FIELD_POSTAL_CODE,
	MARKUP_FIELD_TIMEZONE,
	/* Position */
	MARKUP_FIELD_LONGITUDE,
	MARKUP_FIELD_LATITUDE,
	/* End of list */
	MARKUP_FIELD_CNT
};

typedef struct _MarkupData MarkupData;
struct _MarkupData {
	UbuntuGeoipProvider * provider;
	MarkupField field;

	gboolean position_changed;
	gboolean address_changed;

	gboolean valid_header;
};

struct _UbuntuGeoipProviderPrivate
{
	SoupSession * soup;
	NMClient * client;

	/* Markup thingies */
	GMarkupParseContext * context;
	MarkupData markup_data;

	/* Position Variables */
	double latitude;
	double longitude;

	/* Address variables */
	gchar * country_code;
	gchar * country_name;
	gchar * region_name;
	gchar * city;
	gchar * postal_code;
	gchar * timezone;

	/* State variables */
	gboolean connected;
	gboolean gotdata;

	/* Requeue variables */
	guint requeue_source;
	guint requeue_time;
};

#define UBUNTU_GEOIP_PROVIDER_GET_PRIVATE(o) \
(G_TYPE_INSTANCE_GET_PRIVATE ((o), UBUNTU_GEOIP_PROVIDER_TYPE, UbuntuGeoipProviderPrivate))

static GMainLoop * mainloop = NULL;
static const gchar * field_names[MARKUP_FIELD_CNT] = {
	"CountryCode",
	"CountryName",
	"RegionName",
	"City",
	"ZipPostalCode",
	"TimeZone",
	"Longitude",
	"Latitude"
};

static void ubuntu_geoip_provider_class_init (UbuntuGeoipProviderClass *klass);
static void ubuntu_geoip_provider_init       (UbuntuGeoipProvider *self);
static void ubuntu_geoip_provider_dispose    (GObject *object);
static void ubuntu_geoip_provider_finalize   (GObject *object);
static void gc_shutdown                      (GcProvider *provider);
static gboolean get_status                   (GcIfaceGeoclue *geoclue,
                                              GeoclueStatus  *status,
                                              GError        **error);
static void position_iface_init              (GcIfacePositionClass *iface);
static void address_iface_init               (GcIfaceAddressClass *iface);
static void send_message                     (UbuntuGeoipProvider *provider);
static void queue_message                    (UbuntuGeoipProvider * provider);
static void nm_state_change                  (NMClient *client,
                                              const GParamSpec *pspec,
                                              gpointer user_data);
static gboolean get_address                  (GcIfaceAddress *gc,
                                              int *timestamp,
                                              GHashTable **address,
                                              GeoclueAccuracy **accuracy,
                                              GError **error);
static void markup_start                     (GMarkupParseContext * context,
                                              const gchar * element_name,
                                              const gchar ** attribute_names,
                                              const gchar ** attribute_values,
                                              gpointer user_data,
                                              GError ** error);
static void markup_end                       (GMarkupParseContext * context,
                                              const gchar * element_name,
                                              gpointer user_data,
                                              GError ** error);
static void markup_text                      (GMarkupParseContext * context,
                                              const gchar * text,
                                              gsize text_len,
                                              gpointer user_data,
                                              GError ** error);

static GMarkupParser parser_struct = {
	start_element: markup_start,
	end_element: markup_end,
	text: markup_text,
	passthrough: NULL,
	error: NULL
};

G_DEFINE_TYPE_WITH_CODE (UbuntuGeoipProvider, ubuntu_geoip_provider, GC_TYPE_PROVIDER,
                         G_IMPLEMENT_INTERFACE (GC_TYPE_IFACE_POSITION, position_iface_init)
                         G_IMPLEMENT_INTERFACE (GC_TYPE_IFACE_ADDRESS,  address_iface_init)
                         );

/* Discuss our relationship with other classes */
static void
ubuntu_geoip_provider_class_init (UbuntuGeoipProviderClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (UbuntuGeoipProviderPrivate));

	object_class->dispose = ubuntu_geoip_provider_dispose;
	object_class->finalize = ubuntu_geoip_provider_finalize;

	GcProviderClass * gc_class = GC_PROVIDER_CLASS(klass);

	gc_class->get_status = get_status;
	gc_class->shutdown = gc_shutdown;

	return;
}

/* Make everything consistent */
static void
ubuntu_geoip_provider_init (UbuntuGeoipProvider *self)
{
	self->priv = UBUNTU_GEOIP_PROVIDER_GET_PRIVATE(self);

	/* Init */
	self->priv->soup = NULL;
	self->priv->client = NULL;

	self->priv->latitude = 0.0;
	self->priv->longitude = 0.0;
	self->priv->connected = FALSE;
	self->priv->gotdata = FALSE;

	self->priv->country_code = NULL;
	self->priv->country_name = NULL;
	self->priv->region_name = NULL;
	self->priv->city = NULL;
	self->priv->postal_code = NULL;
	self->priv->timezone = NULL;

	self->priv->context = NULL;
	self->priv->markup_data.provider = self;
	self->priv->markup_data.field = MARKUP_FIELD_CNT;

	self->priv->requeue_source = 0;
	self->priv->requeue_time = REQUEUE_MIN_SEC;

	/* Build Parser */
	self->priv->context = g_markup_parse_context_new(&parser_struct, 0, &self->priv->markup_data, NULL);

	/* Allocate */
	self->priv->soup = soup_session_async_new();
	self->priv->client = nm_client_new();
	g_signal_connect(G_OBJECT(self->priv->client), "notify::" NM_CLIENT_STATE, G_CALLBACK(nm_state_change), self);

	/* Check things out */
	nm_state_change(self->priv->client, NULL, self);

	gc_provider_set_details (GC_PROVIDER (self),
	                         "org.freedesktop.Geoclue.Providers.UbuntuGeoIP",
	                         "/org/freedesktop/Geoclue/Providers/UbuntuGeoIP",
	                         "Ubuntu GeoIP",
	                         "Provides location using IP and the Ubuntu IP Geolocation Service");

	return;
}

/* Loose references */
static void
ubuntu_geoip_provider_dispose (GObject *object)
{
	UbuntuGeoipProvider * provider = UBUNTU_GEOIP_PROVIDER(object);

	if (provider->priv->requeue_source != 0) {
		g_source_remove(provider->priv->requeue_source);
		provider->priv->requeue_source = 0;
	}

	if (provider->priv->soup != NULL) {
		g_object_unref(provider->priv->soup);
		provider->priv->soup = NULL;
	}

	if (provider->priv->client != NULL) {
		g_object_unref(provider->priv->client);
		provider->priv->client = NULL;
	}

	if (provider->priv->context != NULL) {
		g_object_unref(provider->priv->context);
		provider->priv->context = NULL;
	}

	G_OBJECT_CLASS (ubuntu_geoip_provider_parent_class)->dispose (object);
	return;
}

/* Free Memory */
static void
ubuntu_geoip_provider_finalize (GObject *object)
{
	UbuntuGeoipProvider * provider = UBUNTU_GEOIP_PROVIDER(object);

	if (provider->priv->country_code != NULL) {
		g_free(provider->priv->country_code);
		provider->priv->country_code = NULL;
	}

	if (provider->priv->country_name != NULL) {
		g_free(provider->priv->country_name);
		provider->priv->country_name = NULL;
	}

	if (provider->priv->region_name != NULL) {
		g_free(provider->priv->region_name);
		provider->priv->region_name = NULL;
	}

	if (provider->priv->city != NULL) {
		g_free(provider->priv->city);
		provider->priv->city = NULL;
	}

	if (provider->priv->postal_code != NULL) {
		g_free(provider->priv->postal_code);
		provider->priv->postal_code = NULL;
	}

	if (provider->priv->timezone != NULL) {
		g_free(provider->priv->timezone);
		provider->priv->timezone = NULL;
	}


	G_OBJECT_CLASS (ubuntu_geoip_provider_parent_class)->finalize (object);
	return;
}

/* Respond to a request to shutdown from the GeoClue provider
   interface. */
static void
gc_shutdown (GcProvider *provider)
{
	g_debug("Shutdown by provider");
	g_main_loop_quit(mainloop);
	return;
}

/* Get teh status of our position */
static gboolean
get_status (GcIfaceGeoclue *geoclue, GeoclueStatus * status, GError ** error)
{
	UbuntuGeoipProvider *provider = UBUNTU_GEOIP_PROVIDER(geoclue);
	g_return_val_if_fail(provider != NULL, FALSE);

	if (provider->priv->connected == FALSE) {
		*status = GEOCLUE_STATUS_UNAVAILABLE;
	} else if (provider->priv->gotdata == FALSE) {
		*status = GEOCLUE_STATUS_ACQUIRING;
	} else {
		*status = GEOCLUE_STATUS_AVAILABLE;
	}
	return TRUE;
}

/* Get the current position as we see it. */
static gboolean
get_position (GcIfacePosition *gc, GeocluePositionFields * fields, int * timestamp, double *latitude, double *longitude, double *altitude, GeoclueAccuracy **accuracy, GError **error)
{
	UbuntuGeoipProvider *provider = UBUNTU_GEOIP_PROVIDER(gc);
	g_return_val_if_fail(provider != NULL, FALSE);

	*timestamp = time (NULL);

	if (provider->priv->gotdata == FALSE) {
		*fields = GEOCLUE_POSITION_FIELDS_NONE;
		*accuracy = geoclue_accuracy_new (GEOCLUE_ACCURACY_LEVEL_NONE, 0.0, 0.0);
	} else {
		*fields = GEOCLUE_POSITION_FIELDS_LATITUDE | GEOCLUE_POSITION_FIELDS_LONGITUDE;
		*latitude = provider->priv->latitude;
		*longitude = provider->priv->longitude;
		*accuracy = geoclue_accuracy_new (GEOCLUE_ACCURACY_LEVEL_LOCALITY, 0.0, 0.0);
	}

	return TRUE;
}

/* Bring up the position interface */
static void
position_iface_init (GcIfacePositionClass *iface)
{
	iface->get_position = get_position;
	return;
}

/* Put the details into the hash table correctly */
static void
address_details_build (UbuntuGeoipProvider * provider, GHashTable * details)
{
	if (provider->priv->country_code != NULL) {
		geoclue_address_details_insert(details, GEOCLUE_ADDRESS_KEY_COUNTRYCODE, provider->priv->country_code);
	}

	if (provider->priv->country_name != NULL) {
		geoclue_address_details_insert(details, GEOCLUE_ADDRESS_KEY_COUNTRY, provider->priv->country_name);
	}

	if (provider->priv->region_name != NULL) {
		geoclue_address_details_insert(details, GEOCLUE_ADDRESS_KEY_REGION, provider->priv->region_name);
	}

	if (provider->priv->city != NULL) {
		geoclue_address_details_insert(details, GEOCLUE_ADDRESS_KEY_LOCALITY, provider->priv->city);
	}

	if (provider->priv->postal_code != NULL) {
		geoclue_address_details_insert(details, GEOCLUE_ADDRESS_KEY_POSTALCODE, provider->priv->postal_code);
	}

	if (provider->priv->timezone != NULL) {
		geoclue_address_details_insert(details, "timezone", provider->priv->timezone);
	}

	return;
}

/* Get the address as we have it */
static gboolean
get_address (GcIfaceAddress *gc, int *timestamp, GHashTable **address, GeoclueAccuracy **accuracy, GError **error)
{
	UbuntuGeoipProvider *provider = UBUNTU_GEOIP_PROVIDER(gc);
	g_return_val_if_fail(provider != NULL, FALSE);

	*timestamp = time (NULL);

	if (provider->priv->gotdata == FALSE) {
		*address = geoclue_address_details_new();
		*accuracy = geoclue_accuracy_new (GEOCLUE_ACCURACY_LEVEL_NONE, 0.0, 0.0);
	} else {
		*address = geoclue_address_details_new();
		address_details_build(provider, *address);
		*accuracy = geoclue_accuracy_new (GEOCLUE_ACCURACY_LEVEL_LOCALITY, 0.0, 0.0);
	}

	return TRUE;
}

/* Bring up the address interface */
static void
address_iface_init (GcIfaceAddressClass *iface)
{
	iface->get_address = get_address;
	return;
}

/* Called when the message is finished in its response */
static void
message_finished (SoupMessage * message, gpointer user_data)
{
	UbuntuGeoipProvider *provider = UBUNTU_GEOIP_PROVIDER(user_data);
	g_return_if_fail(provider != NULL);

	g_debug("Message Finished");

	guint statuscode = 0;
	g_object_get(G_OBJECT(message), SOUP_MESSAGE_STATUS_CODE, &statuscode, NULL);
	if (statuscode != 200) {
		g_warning("Unable to connect to geoip service, status: %d", statuscode);
		queue_message(provider);
		return;
	}

	provider->priv->markup_data.position_changed = FALSE;
	provider->priv->markup_data.address_changed = FALSE;
	provider->priv->markup_data.valid_header = FALSE;

	if (g_markup_parse_context_parse(provider->priv->context, message->response_body->data, message->response_body->length, NULL) == FALSE) {
		g_warning("Unable to parse the data from the geoip provider");
		queue_message(provider);
		return;
	}

	if (!provider->priv->markup_data.valid_header) {
		g_warning("We didn't get valid XML, that's not cool.");
		queue_message(provider);
		return;
	}

	provider->priv->requeue_time = REQUEUE_MIN_SEC;

	if (provider->priv->gotdata == FALSE) {
		gc_iface_geoclue_emit_status_changed(GC_IFACE_GEOCLUE(provider), GEOCLUE_STATUS_AVAILABLE);
	}
	provider->priv->gotdata = TRUE;

	if (provider->priv->markup_data.position_changed) {
		g_debug("New position:");
		g_debug("Latitude:  %f", provider->priv->latitude);
		g_debug("Longitude: %f", provider->priv->longitude);

		GeoclueAccuracy * accuracy = geoclue_accuracy_new (GEOCLUE_ACCURACY_LEVEL_LOCALITY, 0.0, 0.0);
		gc_iface_position_emit_position_changed(GC_IFACE_POSITION(provider), 
		                                        time(NULL),
		                                        GEOCLUE_POSITION_FIELDS_LATITUDE | GEOCLUE_POSITION_FIELDS_LONGITUDE,
		                                        provider->priv->latitude,
		                                        provider->priv->longitude,
		                                        0, /* Altitude */
		                                        accuracy);

		geoclue_accuracy_free(accuracy);
	}

	if (provider->priv->markup_data.address_changed) {
		GeoclueAccuracy * accuracy = geoclue_accuracy_new (GEOCLUE_ACCURACY_LEVEL_LOCALITY, 0.0, 0.0);
		GHashTable * address = geoclue_address_details_new();

		address_details_build(provider, address);

		gc_iface_address_emit_address_changed(GC_IFACE_ADDRESS(provider), 
		                                        time(NULL),
		                                        address,
		                                        accuracy);

		g_hash_table_destroy(address);
		geoclue_accuracy_free(accuracy);
	}

	return;
}

/* Called when the message is entirely complete */
static void
message_complete (SoupSession * session, SoupMessage * message, gpointer user_data)
{
	UbuntuGeoipProvider *provider = UBUNTU_GEOIP_PROVIDER(user_data);
	g_return_if_fail(provider != NULL);

	g_debug("Message Complete");
	return;
}

/* Sends the message to the GeoIP service and sets up some
   callbacks for fun */
static void
send_message (UbuntuGeoipProvider *provider)
{
	g_return_if_fail(provider->priv->soup != NULL);

	if (provider->priv->requeue_source != 0) {
		g_source_remove(provider->priv->requeue_source);
		provider->priv->requeue_source = 0;
	}

	SoupMessage * message = soup_message_new("get", "http://geoip.ubuntu.com/lookup");
	g_signal_connect(G_OBJECT(message), "finished", G_CALLBACK(message_finished), provider);

	soup_session_queue_message(provider->priv->soup, message,
	                           message_complete, provider);

	return;
}

/* It's now time to send the delayed message */
static gboolean
queue_message_timeout (gpointer user_data)
{
	UbuntuGeoipProvider * provider = UBUNTU_GEOIP_PROVIDER(user_data);
	g_return_val_if_fail(provider != NULL, FALSE);

	provider->priv->requeue_source = 0;

	send_message(provider);
	return FALSE;
}

/* Queue up a timeout to send a message */
static void
queue_message (UbuntuGeoipProvider * provider)
{
	if (provider->priv->requeue_source != 0) {
		return;
	}

	provider->priv->requeue_time *= REQUEUE_SCALE;
	if (provider->priv->requeue_time > REQUEUE_MAX_SEC) {
		provider->priv->requeue_time = REQUEUE_MAX_SEC;
	}

	provider->priv->requeue_source = g_timeout_add_seconds(provider->priv->requeue_time, queue_message_timeout, provider);

	return;
}

/* Callback for when the Network Manger state changes */
static void
nm_state_change (NMClient *client, const GParamSpec *pspec, gpointer user_data)
{
	UbuntuGeoipProvider *provider = UBUNTU_GEOIP_PROVIDER(user_data);
	g_return_if_fail(provider != NULL);

	NMState state = nm_client_get_state (provider->priv->client);
	gboolean connected = !(state == NM_STATE_CONNECTING || state == NM_STATE_DISCONNECTED);

	if (provider->priv->connected != connected) {
		/* Either way we want to refresh */
		provider->priv->gotdata = FALSE;
		if (connected) {
			g_debug("Network connected");
			gc_iface_geoclue_emit_status_changed(GC_IFACE_GEOCLUE(provider), GEOCLUE_STATUS_ACQUIRING);
			provider->priv->connected = TRUE;
			send_message(provider);
		} else {
			g_debug("Network lost");
			gc_iface_geoclue_emit_status_changed(GC_IFACE_GEOCLUE(provider), GEOCLUE_STATUS_UNAVAILABLE);
			provider->priv->connected = FALSE;
		}
	}

	return;
}

/* Handle the start of tags */
static void
markup_start (GMarkupParseContext * context, const gchar * element_name, const gchar ** attribute_names, const gchar ** attribute_values, gpointer user_data, GError ** error)
{
	MarkupData * markup_data = (MarkupData *)user_data;

	if (markup_data->valid_header == FALSE) {
		if (g_strcmp0(element_name, "Response") == 0) {
			markup_data->valid_header = TRUE;
			markup_data->field = MARKUP_FIELD_CNT;
		}
		return;
	}

	if (markup_data->field != MARKUP_FIELD_CNT) {
		g_warning("New start tag without clearing old one.");
	}

	int i;
	for (i = 0; i < MARKUP_FIELD_CNT; i++) {
		if (g_strcmp0(field_names[i], element_name) == 0) {
			break;
		}
	}

	markup_data->field = i;

	return;
}

/* Look at teh end of tags */
static void
markup_end (GMarkupParseContext * context, const gchar * element_name, gpointer user_data, GError ** error)
{
	MarkupData * markup_data = (MarkupData *)user_data;
	markup_data->field = MARKUP_FIELD_CNT;
	return;
}

/* Handle the text inside a tag */
static void
markup_text (GMarkupParseContext * context, const gchar * text, gsize text_len, gpointer user_data, GError ** error)
{
	MarkupData * markup_data = (MarkupData *)user_data;
	gchar ** field = NULL;
	double * val = NULL;

	if (markup_data->valid_header == FALSE) {
		return;
	}

	/* Decode the enum into an offset */
	switch (markup_data->field) {
	case MARKUP_FIELD_COUNTRY_CODE:
		field = &(markup_data->provider->priv->country_code);
		break;
	case MARKUP_FIELD_COUNTRY_NAME:
		field = &(markup_data->provider->priv->country_name);
		break;
	case MARKUP_FIELD_REGION_NAME:
		field = &(markup_data->provider->priv->region_name);
		break;
	case MARKUP_FIELD_CITY:
		field = &(markup_data->provider->priv->city);
		break;
	case MARKUP_FIELD_POSTAL_CODE:
		field = &(markup_data->provider->priv->postal_code);
		break;
	case MARKUP_FIELD_TIMEZONE:
		field = &(markup_data->provider->priv->timezone);
		break;
	case MARKUP_FIELD_LATITUDE:
		val = &(markup_data->provider->priv->latitude);
		break;
	case MARKUP_FIELD_LONGITUDE:
		val = &(markup_data->provider->priv->longitude);
		break;
	case MARKUP_FIELD_CNT:
	default:
		break;
	}

	/* Make the text null terminated */
	gchar * newstr = g_memdup(text, text_len + 1);
	if (newstr != NULL) {
		newstr[text_len] = '\0';
	}

	/* If we've got a string field, let's deal with that */
	if (field != NULL && newstr != NULL) {
		/* We've got new data? */
		if (g_strcmp0(*field, newstr) != 0) {
			if (*field != NULL) {
				g_free(*field);
				*field = NULL;
			}
			*field = newstr;
			newstr = NULL; /* Steal the memory */
			markup_data->address_changed = TRUE;
		}
	}

	/* We got a double field let's do that */
	if (val != NULL && newstr != NULL) {
		gdouble newval = strtod(newstr, NULL);
		if (newval != *val) {
			*val = newval;
			markup_data->position_changed = TRUE;
		}
	}

	/* Free the allocated memory */
	if (newstr != NULL) {
		g_free(newstr);
	}

	return;
}


/* Function to start everything */
int
main (int argv, char * argc[])
{
	g_type_init();

	UbuntuGeoipProvider * provider = g_object_new(UBUNTU_GEOIP_PROVIDER_TYPE, NULL);
	g_return_val_if_fail(provider != NULL, 1);

	mainloop = g_main_loop_new(NULL, TRUE);

	g_main_loop_run(mainloop);

	g_main_loop_unref(mainloop);
	g_object_unref(provider);

	return 0;
}
