/*
 * Copyright(C) 2011 Canonical Ltd.
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

#include <glib.h>
#include <gio/gdesktopappinfo.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <libindicate/server.h>
#include <libindicate/indicator.h>
#include <libindicate-gtk/indicator.h>
#include <telepathy-glib/telepathy-glib.h>
#include <telepathy-glib/simple-approver.h>

#include <unity.h>

static GHashTable *indicators = NULL;
static gboolean has_interest = FALSE;
UnityLauncherEntry *launcher = NULL;
GList *conns;

static void update_launcher ()
{
	if (launcher != NULL)
	{
		gint count = g_hash_table_size (indicators);
		g_debug ("unity launcher: count is %d", count);
		if (count > 0)
		{
			g_debug ("unity launcher: setting count to %d", count);
			unity_launcher_entry_set_count (launcher, count);
			unity_launcher_entry_set_count_visible (launcher, TRUE);
		} else {
			unity_launcher_entry_set_count (launcher, count);
			g_debug ("unity launcher: hiding count");
			unity_launcher_entry_set_count_visible (launcher, FALSE);
		}
	}
}

static IndicateIndicator *
find_indicator_by_id (const gchar *id)
{
	if (!check_indicator (id))
	{
		return NULL;
	}

	const gchar *channel_id;
	g_debug ("find_indicator_by_id: %s", id);

	GHashTableIter iter;
	gpointer data, indicator;

	g_hash_table_iter_init (&iter, indicators);
	while (g_hash_table_iter_next (&iter, &indicator, &data)) {
		channel_id = indicate_indicator_get_property (INDICATE_INDICATOR (indicator), "channel-id");
		g_debug ("indicator for channel %s", channel_id);
		if (g_strcmp0 (id, channel_id) == 0)
		{
			return INDICATE_INDICATOR (indicator);
		}
	}

	return NULL;
}

static void 
indicator_remove_by_indicator (IndicateIndicator *indicator)
{
	g_debug ("indicator_remove_by_indicator");
	if (indicator != NULL)
	{
		if (g_hash_table_lookup (indicators, indicator) != NULL)
		{
			g_debug ("removing indicator");
			indicate_indicator_hide (INDICATE_INDICATOR (indicator));
			g_hash_table_remove (indicators, indicator);
		}
		update_launcher ();
		g_object_unref (indicator);
	}
}

static void
indicator_remove_by_id (const gchar *id)
{
	g_debug ("indicator_remove_by_id: %s", id);
	IndicateIndicator *indicator = find_indicator_by_id (id);

	if (indicator != NULL)
	{
		indicator_remove_by_indicator (INDICATE_INDICATOR (indicator));
	}
}


static void
dispatch_op_finished_cb (TpProxy *self, guint domain, gint code, gchar *message, IndicateIndicator *indicator)
{
	g_return_if_fail (INDICATE_IS_INDICATOR(indicator));

	g_debug ("DispatchOperation invalidated, removing indicator");
	indicator_remove_by_indicator (INDICATE_INDICATOR (indicator));
}

static void
handle_with_cb (GObject *self, GAsyncResult *result, gpointer user_data)
{
	IndicateIndicator *indicator = INDICATE_INDICATOR (user_data);
	g_return_if_fail (INDICATE_IS_INDICATOR(indicator));

	TpChannelDispatchOperation *dispatch_op = TP_CHANNEL_DISPATCH_OPERATION (self);
	GError *error = NULL;


	if (!tp_channel_dispatch_operation_handle_with_time_finish (dispatch_op, result, &error))
	{
                if (error)
		{
			g_warning ("Failed to handle operations: %s\n", error->message);
			g_error_free (error);
			g_debug ("Activated, removing indicator");
			indicator_remove_by_indicator (INDICATE_INDICATOR (indicator));
		}
	}
}


static void
indicator_display (IndicateIndicator *indicator, guint timestamp, TpChannelDispatchOperation * dispatch_op)
{
	g_debug ("indicator_display");
	tp_channel_dispatch_operation_handle_with_time_async (dispatch_op, NULL, timestamp, handle_with_cb, indicator);
}

static void
observer_handle_with_cb (GObject *self, GAsyncResult *result, gpointer user_data)
{
	g_debug ("observer_handle_with_cb");
	IndicateIndicator *indicator = INDICATE_INDICATOR (user_data);
	g_return_if_fail (INDICATE_IS_INDICATOR(indicator));

	TpChannelDispatcher *cd = TP_CHANNEL_DISPATCHER (self);
	GError *error = NULL;

	g_debug ("Activated, removing indicator");
	indicator_remove_by_indicator (INDICATE_INDICATOR (indicator));
	
	tp_channel_dispatcher_present_channel_finish (cd, result, &error);
	if (error)
	{
		g_warning ("Failed to handle operations: %s\n", error->message);
		g_error_free (error);
	}
}

static void
observer_indicator_display (IndicateIndicator *indicator, guint timestamp, gpointer user_data)
{
        g_debug ("observer_indicator_display");
	TpChannel *channel = TP_CHANNEL (user_data);
	g_return_if_fail (TP_IS_CHANNEL (channel));

	g_object_ref (indicator);
	GError *error = NULL;

	TpDBusDaemon *bus = tp_dbus_daemon_dup (&error);

	TpChannelDispatcher *cd = tp_channel_dispatcher_new (bus);
	g_object_unref (bus);

	if (error)
	{
		g_warning ("Failed to get Daemon: %s\n", error->message);
		g_error_free (error);
	}

	tp_channel_dispatcher_present_channel_async (cd, TP_CHANNEL (channel), timestamp, observer_handle_with_cb, INDICATE_INDICATOR (indicator));
}

gboolean check_indicator (const gchar *id)
{
	const gchar *channel_id;
	g_debug ("check_indicator: %s", id);
	GHashTableIter iter;
	gpointer data, ind;

	g_hash_table_iter_init (&iter, indicators);
	while (g_hash_table_iter_next (&iter, &ind, &data)) {
		channel_id = indicate_indicator_get_property (INDICATE_INDICATOR (ind), "channel-id");
		g_debug ("indicator for channel %s", channel_id);
		if (g_strcmp0 (id, channel_id) == 0)
		{
			return TRUE;
		}
	}

	return FALSE;
}

static TpContact *
get_contact_from_channel (TpChannel * channel)
{
	g_return_val_if_fail (TP_IS_CHANNEL (channel), NULL);
	TpContact *contact = NULL;
	if (tp_channel_get_requested (TP_CHANNEL (channel)))
		contact = tp_channel_get_target_contact (TP_CHANNEL (channel));
	else
		contact = tp_channel_get_initiator_contact (TP_CHANNEL (channel));
	return TP_CONTACT (contact);
}
	

static void
handle_contacts_pending_add_indicator_cb (TpConnection *connection, guint n_contacts, TpContact * const *contacts, guint n_failed, const TpHandle *failed, const GError *err, gpointer user_data, GObject *weak_object)
{
	g_debug ("handle_contacts_pending_add_indicator_cb");
	TpChannel *channel = TP_CHANNEL (user_data);
	GError *error = NULL;
	const gchar *id = "";
	const gchar *alias = "";
	GFile *avatar = NULL;
	GdkPixbuf *pixbuf = NULL;
	TpContact *contact = NULL;

	contact = get_contact_from_channel (channel);

	if (contact)
	{
		alias = tp_contact_get_alias (contact);
		avatar = tp_contact_get_avatar_file (contact);
		g_debug ("handle_contacts_pending_add_indicator_cb HAS CONTACT alias: %s", alias);
	} else
	{
		alias = tp_channel_get_initiator_identifier (channel);
		g_debug ("handle_contacts_pending_add_indicator_cb NO CONTACT alias: %s", alias);
	}
	g_debug ("handle_contacts_pending_add_indicator_cb alias: %s", alias);

	id = tp_channel_get_identifier (TP_CHANNEL (channel));
	IndicateIndicator *indicator;
	indicator = indicate_indicator_new ();
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					"subtype", "im");
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					"sender", alias);
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					"draw-attention", "true");
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					"channel-id", id);
	g_debug ("added channel-id: %s", id);
	g_signal_connect (G_OBJECT (indicator),
		INDICATE_INDICATOR_SIGNAL_DISPLAY,
		G_CALLBACK (observer_indicator_display), channel);

	GTimeVal time;
	g_get_current_time (&time);
	indicate_indicator_set_property_time (INDICATE_INDICATOR (indicator),
						"time", &time);
	if (avatar)
	{
		pixbuf = gdk_pixbuf_new_from_file_at_scale (g_file_get_path (avatar), 22, 22, TRUE, &error);
	}

	if (error)
	{
		g_warning ("Failed to create pixbuf: %s", error->message);
		g_error_free (error);
	}

	if (pixbuf)
	{
		indicate_gtk_indicator_set_property_icon (indicator, "icon", pixbuf);
		g_object_unref(G_OBJECT(pixbuf));
	}

	indicate_indicator_show (INDICATE_INDICATOR (indicator));
	g_hash_table_insert(indicators, indicator, channel);
	update_launcher ();
}

gboolean check_pending_messages (gpointer data)
{
	TpTextChannel *channel = TP_TEXT_CHANNEL (data);
	TpConnection *connection;
	GList *pending;
	const gchar *id;
	gboolean has_indicator = FALSE;
	TpHandle handle;

	static TpContactFeature features[] = {
		TP_CONTACT_FEATURE_ALIAS,
		TP_CONTACT_FEATURE_AVATAR_DATA,
	};

	pending = tp_text_channel_get_pending_messages (TP_TEXT_CHANNEL (channel));
	g_debug ("pending: %u", g_list_length (pending));
	if (g_list_length (pending) > 0)
	{
		id = tp_channel_get_identifier (TP_CHANNEL (channel));

		if (!check_indicator (id))
		{
			connection = tp_channel_borrow_connection (TP_CHANNEL (channel));
			handle = tp_channel_get_initiator_handle (TP_CHANNEL (channel));
			tp_connection_get_contacts_by_handle (connection, 1, &handle, 2, features, handle_contacts_pending_add_indicator_cb, channel, NULL, NULL);
		}
	}
        g_list_free (pending);
	return FALSE;
}

static void
pending_message_removed_cb (TpTextChannel *channel, TpMessage *message, gpointer user_data)
{
	const gchar *id;
	g_debug ("pending_message_removed_cb: %s", tp_channel_get_identifier (TP_CHANNEL (channel)));
	id = tp_channel_get_identifier (TP_CHANNEL (channel));
	indicator_remove_by_id (id);
}

static void
message_received_cb (TpTextChannel *channel, TpMessage *message, gpointer user_data)
{
	g_debug ("message_received_cb: %s", tp_channel_get_identifier (TP_CHANNEL (channel)));
	g_timeout_add (100, check_pending_messages, channel);
}

static void
handle_contacts_add_indicator_cb (TpConnection *connection, guint n_contacts, TpContact * const *contacts, guint n_failed, const TpHandle *failed, const GError *err, gpointer user_data, GObject *weak_object)
{
	IndicateIndicator *indicator = NULL;
	TpContact *contact = NULL;
	const gchar *id = "";
	const gchar *alias = "";
	GTimeVal time;
	GFile *avatar = NULL;
	GdkPixbuf *pixbuf = NULL;
	GError *error = NULL;
	
	TpChannelDispatchOperation *dispatch_op = TP_CHANNEL_DISPATCH_OPERATION (user_data);

	TpChannel *channel;
	GPtrArray *channels;
	channels = tp_channel_dispatch_operation_borrow_channels (dispatch_op);
	channel = g_ptr_array_index (channels, 0);

	contact = get_contact_from_channel (channel);

	if (contact)
	{
		alias = tp_contact_get_alias (contact);
		avatar = tp_contact_get_avatar_file (contact);
	} else
	{
		alias = tp_channel_get_initiator_identifier (channel);
	}
	g_debug ("handle_contacts_add_indicator_cb alias: %s", alias);

	id = tp_channel_get_identifier (TP_CHANNEL (channel));

	if (check_indicator (id))
	{
		indicator_remove_by_id (id);
	}

	indicator = indicate_indicator_new ();
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					 "subtype", "im");
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					 "sender", alias);
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					 "draw-attention", "true");
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					 "channel-id", id);
	g_debug ("added channel-id: %s", id);

	if (avatar)
	{
		pixbuf = gdk_pixbuf_new_from_file_at_scale (g_file_get_path (avatar), 22, 22, TRUE, &error);
	}

	if (error)
	{
		g_warning ("Failed to create pixbuf: %s", error->message);
		g_error_free (error);
	}

	if (pixbuf)
	{
		indicate_gtk_indicator_set_property_icon (indicator, "icon", pixbuf);
		g_object_unref(G_OBJECT(pixbuf));
	}

	g_signal_connect (G_OBJECT (indicator),
			INDICATE_INDICATOR_SIGNAL_DISPLAY,
			G_CALLBACK (indicator_display), dispatch_op);

	g_get_current_time (&time);
	indicate_indicator_set_property_time (INDICATE_INDICATOR (indicator),
					"time", &time);
	indicate_indicator_show (INDICATE_INDICATOR (indicator));
	g_hash_table_insert(indicators, indicator, dispatch_op);
	update_launcher ();
	g_signal_connect (dispatch_op, "invalidated", G_CALLBACK (dispatch_op_finished_cb), indicator);
}

static void
add_dispatch_operation_cb (TpSimpleApprover *self, TpAccount *account, TpConnection *connection,
    GList *channels, TpChannelDispatchOperation *dispatch_op, TpAddDispatchOperationContext *context,
    gpointer user_data)
{
	if (has_interest == FALSE)
	{
		g_debug ("add_dispatch_operation_cb but there are no listeners with interest");
		tp_add_dispatch_operation_context_accept (context);
		return;
	}

	GList *itr;
	TpChannel *channel;
	TpHandle handle;

	static TpContactFeature features[] = {
		TP_CONTACT_FEATURE_ALIAS,
		TP_CONTACT_FEATURE_AVATAR_DATA,
	};

	g_object_ref (dispatch_op);
	for (itr = channels; itr != NULL; itr = g_list_next (itr))
	{
		channel = itr->data;
		handle = tp_channel_get_initiator_handle (channel);
		tp_connection_get_contacts_by_handle (connection, 1, &handle, 2, features, handle_contacts_add_indicator_cb, dispatch_op, NULL, NULL);
	}
	g_list_free(itr);
	tp_add_dispatch_operation_context_accept (context);
}

static void
observer_add_dispatch_operation_cb (TpSimpleObserver *self, TpAccount *account, TpConnection *connection,
    GList *channels, TpChannelDispatchOperation *dispatch_op, GList *requests, TpObserveChannelsContext *context,
    gpointer user_data)
{
	if (has_interest == FALSE)
	{
		g_debug ("observer_add_dispatch_operation_cb but there are no listeners with interest");
		tp_observe_channels_context_accept (context);
		return;
	}
	GList *itr;
	TpChannel *channel;

	for (itr = channels; itr != NULL; itr = g_list_next (itr))
	{
		channel = itr->data;
		if (TP_IS_TEXT_CHANNEL (channel))
		{
			g_debug ("Handling text channel: %s", tp_channel_get_identifier (channel));

			g_object_ref (channel);

			g_signal_connect (channel, "message-received",
				G_CALLBACK (message_received_cb), NULL);
			g_signal_connect (channel, "pending-message-removed",
				G_CALLBACK (pending_message_removed_cb), NULL);

			check_pending_messages (TP_CHANNEL (channel));
		}
	}
	g_list_free(itr);
	tp_observe_channels_context_accept (context);
}

static void
server_display (IndicateServer * indicate_server, guint timestamp, gpointer user_data)
{
	g_debug ("server_display at %u", timestamp);

	if (has_interest == FALSE)
	{
		g_debug ("server_display but there are no listeners with interest");
		return;
	}

	GDesktopAppInfo *app;
	GError *error = NULL;

	g_unsetenv ("DESKTOP_AUTOSTART_ID");

	app = g_desktop_app_info_new ("empathy.desktop");
	g_app_info_launch ((GAppInfo*)app, NULL, NULL, &error);

	if (error)
	{
		g_warning ("Failed to launch empathy: %s", error->message);
		g_error_free (error);
	}

	g_object_unref (app);
}

static void
indicate_server_show_interest (IndicateServer * server, IndicateInterests interest, gpointer data)
{
	g_debug ("indicate_server_show_interest");
	if (interest == INDICATE_INTEREST_SERVER_SIGNAL)
	{
		if (has_interest == FALSE)
		{
			has_interest = TRUE;
		}
	}
	return;
}

static void
indicate_server_remove_interest (IndicateServer * server, IndicateInterests interest, gpointer data)
{
	g_debug ("indicate_server_remove_interest");
	if (interest == INDICATE_INTEREST_SERVER_SIGNAL)
	{
		if (has_interest == TRUE)
		{
			has_interest = FALSE;
		}
	}
	return;
}

static IndicateServer *
indicate_server_setup ()
{
	IndicateServer *indicate_server = NULL;

	const gchar *desktop_file = "/usr/share/applications/empathy.desktop";
	indicate_server = indicate_server_ref_default ();
	indicate_server_set_type (indicate_server, "message.im");

	indicate_server_set_desktop_file (indicate_server, desktop_file);
	indicate_server_show (indicate_server);
	indicators = g_hash_table_new (g_direct_hash, g_direct_equal);
	launcher = unity_launcher_entry_get_for_desktop_id ("empathy.desktop");

	g_signal_connect (G_OBJECT (indicate_server),
		INDICATE_SERVER_SIGNAL_SERVER_DISPLAY,
		G_CALLBACK (server_display), NULL);

	g_signal_connect (G_OBJECT (indicate_server), 
		INDICATE_SERVER_SIGNAL_INTEREST_ADDED, 
		G_CALLBACK (indicate_server_show_interest), NULL);
	g_signal_connect (G_OBJECT (indicate_server), 
		INDICATE_SERVER_SIGNAL_INTEREST_REMOVED, 
		G_CALLBACK (indicate_server_remove_interest), NULL);

	has_interest = indicate_server_check_interest (indicate_server, 
				INDICATE_INTEREST_SERVER_SIGNAL);
	
	return indicate_server;
}

static void
contact_request_subscription_cb (GObject *source, GAsyncResult *result, gpointer user_data)
{ 
	TpContact *contact = (TpContact *) source;
	GError *error = NULL;

	if (!tp_contact_request_subscription_finish (contact, result, &error))
	{
		g_debug ("Failed to request_subscription on %s\n", tp_contact_get_identifier (contact));
		g_error_free (error);
	}
}

static void
contact_authorize_publication_cb (GObject *source, GAsyncResult *result, gpointer user_data) 
{ 
        TpContact *contact = (TpContact *) source;
        GError *error = NULL; 

        if (!tp_contact_authorize_publication_finish (contact, result, &error))
        {
                g_debug ("Failed to authorize_publication on %s\n", tp_contact_get_identifier (contact)); 
                g_error_free (error);
        }
}

static void
contact_unblock_cb (GObject *source, GAsyncResult *result, gpointer user_data) 
{ 
        TpContact *contact = (TpContact *) source;
        GError *error = NULL; 

        if (!tp_contact_unblock_finish (contact, result, &error))
        {
                g_debug ("Failed to unblock on %s\n", tp_contact_get_identifier (contact)); 
                g_error_free (error);
        }
}

static void 
contact_remove_cb (GObject *source, GAsyncResult *result, gpointer user_data)             
{
        TpContact *contact = (TpContact *) source;
        GError *error = NULL; 

	g_debug ("contact_remove_cb");

        if (!tp_contact_remove_finish (contact, result, &error))
        {
                g_debug ("Failed to remove on %s\n", tp_contact_get_identifier (contact));
                g_error_free (error);
        }
}

void
contact_add_to_contact_list (TpContact *contact)
{
	g_return_if_fail (contact != NULL);
	g_debug ("contact_add_to_contact_list");
	const gchar *message;
	message = tp_contact_get_publish_request (contact);
	tp_contact_request_subscription_async (contact, message, contact_request_subscription_cb, NULL);
	tp_contact_authorize_publication_async (contact, contact_authorize_publication_cb, NULL);
	tp_contact_unblock_async (contact, contact_unblock_cb, NULL);
}

void
contact_remove_from_contact_list (TpContact *contact)
{
	g_return_if_fail (contact != NULL);
	g_debug ("contact_remove_from_contact_list");
	tp_contact_remove_async (contact, contact_remove_cb, NULL);
}

static void
contact_add_response_cb (GtkWidget *dialog, gint response, TpContact *contact)
{
	g_debug ("contact_add_response_cb");

	gtk_widget_destroy (dialog);
	dialog = NULL;

	if (response != GTK_RESPONSE_ACCEPT)
		contact_remove_from_contact_list (contact);

	else
		contact_add_to_contact_list (contact);
}

static void
handle_contacts_subscription_add_indicator_cb (TpConnection *connection, guint n_contacts, TpContact * const *contacts, const GError *err, gpointer user_data, GObject *weak_object)
{
	GtkWidget *dialog;
	GtkWidget *button;
	const gchar *alias = "";
	gchar *title;
	const gchar *message;
	GFile *avatar = NULL;
	GdkPixbuf *pixbuf = NULL;
	GtkWidget *image = NULL;
	GError *error = NULL;

	IndicateIndicator *indicator = INDICATE_INDICATOR (user_data);

	message = tp_contact_get_publish_request (contacts[0]);

	alias = tp_contact_get_alias (contacts[0]);

	g_debug ("Getting avatar for %s", alias);

	avatar = tp_contact_get_avatar_file (contacts[0]);

	if (avatar)
	{
		pixbuf = gdk_pixbuf_new_from_file_at_scale (g_file_get_path (avatar), 48, 48, TRUE, &error);
	}

	if (error)
        {
		g_warning ("Failed to create pixbuf: %s", error->message);
		g_error_free (error);
	}

	if (pixbuf)
        {
		image = gtk_image_new_from_pixbuf (pixbuf);
		g_object_unref(G_OBJECT(pixbuf));
	}


	dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE, "%s", message);

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog), _("%s would like permission to see when you are online"), alias);

	title = g_strdup_printf (_("Subscription Request"));
	gtk_window_set_title (GTK_WINDOW (dialog), title);
	g_free (title);

	if (image)
	{
		gtk_message_dialog_set_image (GTK_MESSAGE_DIALOG (dialog), image);
		gtk_widget_show (image);
	}

	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);

	button = gtk_dialog_add_button (GTK_DIALOG (dialog), _("_Decline"), GTK_RESPONSE_REJECT);

	button = gtk_dialog_add_button (GTK_DIALOG (dialog), _("_Accept"), GTK_RESPONSE_ACCEPT);

	g_signal_connect (dialog, "response", G_CALLBACK (contact_add_response_cb), contacts[0]);

	gtk_widget_show (GTK_WIDGET (dialog));

	if (indicator != NULL)
	{
		if (g_hash_table_lookup (indicators, indicator) != NULL)
		{
			g_debug ("Activated, removing indicator");
			indicate_indicator_hide (INDICATE_INDICATOR (indicator));
			g_hash_table_remove (indicators, indicator);
		}
		g_object_unref (indicator);
	}
}

static void
subscription_indicator_display (IndicateIndicator *indicator, guint timestamp, TpContact * contact)
{
	if (TP_IS_CONTACT (contact))
	{
		TpConnection *connection = tp_contact_get_connection (contact);
		static TpContactFeature features[] = {
			TP_CONTACT_FEATURE_ALIAS,
			TP_CONTACT_FEATURE_AVATAR_DATA,
		};

		TpContact *contacts[] = {
			contact,
		};

		tp_connection_upgrade_contacts (connection, 1, contacts, 2, features, handle_contacts_subscription_add_indicator_cb, indicator, NULL, NULL);
	}
}

static void
conn_invalidated_cb (TpConnection *conn, guint domain, gint code, gchar *message, gpointer user_data)
{
	g_debug ("conn_invalidated_cb");
	conns = g_list_remove (conns, conn);
	g_object_unref (conn);
}

static void
contact_list_changed_cb (TpConnection *conn, GPtrArray *added, GPtrArray *removed, gpointer user_data)
{
	g_debug ("contact_list_changed_cb");
	GTimeVal time;
	const gchar *alias = "";
	guint i;
        GFile *avatar = NULL;
        GdkPixbuf *pixbuf = NULL;
        GtkWidget *image = NULL;
        GError *error = NULL;

	for (i = 0; i < added->len; i++)
	{
		TpContact *contact = g_ptr_array_index (added, i);
		if (TP_IS_CONTACT (contact))
		{
			TpSubscriptionState state = tp_contact_get_publish_state (contact);
			if (state == TP_SUBSCRIPTION_STATE_ASK)
			{
				alias = tp_contact_get_alias (contact);
	
				IndicateIndicator *indicator = NULL;
				indicator = indicate_indicator_new ();
				indicate_indicator_set_property (INDICATE_INDICATOR (indicator), "subtype", "im");
				indicate_indicator_set_property (INDICATE_INDICATOR (indicator), "sender", alias);
				indicate_indicator_set_property (INDICATE_INDICATOR (indicator), "draw-attention", "true");


				g_debug ("Getting avatar for %s", alias);

				avatar = tp_contact_get_avatar_file (contact);

				if (avatar)
				{
					g_debug ("avatar");
					pixbuf = gdk_pixbuf_new_from_file_at_scale (g_file_get_path (avatar), 22, 22, TRUE, &error);
				}

				if (error)
				{
					g_warning ("Failed to create pixbuf: %s", error->message);
					g_error_free (error);
				}

				if (pixbuf)
				{
					indicate_gtk_indicator_set_property_icon (indicator, "icon", pixbuf);
					g_object_unref(G_OBJECT(pixbuf));
				}

				g_signal_connect (G_OBJECT (indicator), INDICATE_INDICATOR_SIGNAL_DISPLAY, G_CALLBACK (subscription_indicator_display), contact);
	
				g_get_current_time (&time);
				indicate_indicator_set_property_time (INDICATE_INDICATOR (indicator), "time", &time);

				g_debug ("contact_list_changed_cb: adding %s", alias);

				indicate_indicator_show (INDICATE_INDICATOR (indicator));
				g_hash_table_insert (indicators, indicator, contact);
			}
		}
	}

	for (i = 0; i < removed->len; i++)
	{
		TpContact *contact = g_ptr_array_index (removed, i);
		if (TP_IS_CONTACT (contact))
		{
			alias = tp_contact_get_alias (contact);
			g_debug ("contact_list_changed_cb: removing %s", alias);
			contact_remove_from_contact_list (contact);
		}

		GHashTableIter iter;
		gpointer data, indicator;
		g_hash_table_iter_init (&iter, indicators);
		while (g_hash_table_iter_next (&iter, &indicator, &data)) 
		{
			if (!TP_IS_CONTACT (data))
				continue;
			if (data == contact)
			{
				g_debug ("contact_list_changed_cb: found indicator for contact");
				indicate_indicator_hide (INDICATE_INDICATOR (indicator));
				g_hash_table_remove (indicators, indicator);
				break;
			}
		}
	}
}


void
check_account (TpAccount *account, TpAccountManager *manager)
{
	g_debug ("check_account");
	TpConnection *connection = tp_account_get_connection (account);
	if (connection == NULL)
		return;

	if (g_list_find (conns, connection) != NULL)
		return;

	g_debug ("account_manager_prepared_cb %s", tp_connection_get_connection_manager_name (connection));

	conns = g_list_prepend (conns, g_object_ref (connection));

	tp_g_signal_connect_object (connection, "contact-list-changed",
		G_CALLBACK (contact_list_changed_cb), manager, 0);
	tp_g_signal_connect_object (connection, "invalidated",
		G_CALLBACK (conn_invalidated_cb), manager, 0);
}

static void
account_conn_changed_cb (TpAccount *account, GParamSpec *spec, TpAccountManager *manager)
{
	g_debug ("account_conn_changed_cb");
	g_return_if_fail (TP_IS_ACCOUNT (account));
	check_account (account, manager);
}

void
add_account (TpAccount *account, TpAccountManager *manager)
{
	g_debug ("add_account");
	tp_g_signal_connect_object (account, "notify::connection",
		G_CALLBACK (account_conn_changed_cb), manager, 0);
	check_account (account, manager);
}

static void
account_enabled_cb (TpAccountManager *manager, TpAccount *account, gboolean valid, gpointer user_data)
{
	g_debug ("account_enabled_cb");

	add_account (account, manager);
}

static void
account_manager_prepared_cb (GObject *object, GAsyncResult *res, gpointer user_data)
{
	TpAccountManager *manager = TP_ACCOUNT_MANAGER (object);
        TpAccountManager *account_manager = TP_ACCOUNT_MANAGER (user_data);
	GList *accounts;
	GError *error = NULL;

	g_debug ("account_manager_prepared_cb");

	if (!tp_proxy_prepare_finish (object, res, &error))
	{
		g_print ("Error preparing AM: %s\n", error->message);
	}

	for (accounts = tp_account_manager_get_valid_accounts (manager); accounts != NULL; accounts = g_list_delete_link (accounts, accounts))
	{
		TpAccount *account = accounts->data;
		g_debug ("account_manager_prepared_cb %s", tp_account_get_path_suffix (account));
		guint i;

		add_account (account, manager);
	}

	tp_g_signal_connect_object (manager, "account-validity-changed",
		G_CALLBACK (account_enabled_cb), manager, 0);
}

void
contact_list_setup (TpAccountManager *tp_am)
{
	g_return_if_fail (TP_IS_ACCOUNT_MANAGER (tp_am));

	TpSimpleClientFactory *factory;
	factory = tp_proxy_get_factory (tp_am);
	tp_simple_client_factory_add_account_features_varargs (factory,
		TP_ACCOUNT_FEATURE_CONNECTION,
		0);
	tp_simple_client_factory_add_connection_features_varargs (factory,
		TP_CONNECTION_FEATURE_CONTACT_LIST,
		0);
	tp_simple_client_factory_add_contact_features_varargs (factory,
		TP_CONTACT_FEATURE_ALIAS,
		TP_CONTACT_FEATURE_CONTACT_GROUPS,
		TP_CONTACT_FEATURE_INVALID);

	tp_proxy_prepare_async (tp_am, NULL, account_manager_prepared_cb,  tp_am);
}

void
most_available_presence_changed_cb (TpAccountManager *manager, guint presence, gchar *status, gchar *message, gpointer user_data)
{
	contact_list_setup (manager);
}

static TpBaseClient *
approver_setup (TpAccountManager *tp_am)
{
	TpBaseClient *approver;
	GError *error = NULL;

	g_return_if_fail (TP_IS_ACCOUNT_MANAGER (tp_am));

	approver = tp_simple_approver_new_with_am (tp_am, "IndicatorApprover",
				FALSE, add_dispatch_operation_cb, NULL, NULL);

	/* contact text chat */
	tp_base_client_take_approver_filter (approver, tp_asv_new (
		TP_PROP_CHANNEL_CHANNEL_TYPE, G_TYPE_STRING, TP_IFACE_CHANNEL_TYPE_TEXT,
		TP_PROP_CHANNEL_TARGET_HANDLE_TYPE, G_TYPE_UINT, TP_HANDLE_TYPE_CONTACT,
		NULL));

	/* room text chat */
	tp_base_client_take_approver_filter (approver, tp_asv_new (
		TP_PROP_CHANNEL_CHANNEL_TYPE, G_TYPE_STRING, TP_IFACE_CHANNEL_TYPE_TEXT,
		TP_PROP_CHANNEL_TARGET_HANDLE_TYPE, G_TYPE_UINT, TP_HANDLE_TYPE_ROOM,
		NULL));

	if (!tp_base_client_register (approver, &error))
	{
                if (error)
		{
			g_warning ("Failed to register the approver: %s", error->message);
			g_error_free (error);
		}
		g_object_unref (tp_am);
		g_object_unref (approver);
		return NULL;
	}
	return approver;
}

static TpBaseClient *
observer_setup (TpAccountManager *tp_am)
{
	TpBaseClient *observer;
	GError *error = NULL;

	g_return_if_fail (TP_IS_ACCOUNT_MANAGER (tp_am));

	observer = tp_simple_observer_new_with_am (tp_am, TRUE, "IndicatorObserver",
				FALSE, observer_add_dispatch_operation_cb, NULL, NULL);

	/* contact text chat */
	tp_base_client_take_observer_filter (observer, tp_asv_new (
		TP_PROP_CHANNEL_CHANNEL_TYPE, G_TYPE_STRING, TP_IFACE_CHANNEL_TYPE_TEXT,
		TP_PROP_CHANNEL_TARGET_HANDLE_TYPE, G_TYPE_UINT, TP_HANDLE_TYPE_CONTACT,
		NULL));

	if (!tp_base_client_register (observer, &error))
	{
		if (error)
		{
			g_warning ("Failed to register the observer: %s", error->message);
			g_error_free (error);
		}
		g_object_unref (tp_am);
		g_object_unref (observer);
		return NULL;
	}
	return observer;
}


int
main (int argc, char **argv)
{
	GMainLoop *loop = NULL;

	g_type_init ();

	gtk_init (&argc, &argv);
	TpBaseClient *approver;
	TpBaseClient *observer;
	IndicateServer *indicate_server;

	loop = g_main_loop_new (NULL, FALSE);

	TpAccountManager *tp_am = tp_account_manager_dup ();

	approver = approver_setup (tp_am);
	if (approver == NULL)
	{
		g_error ("Approver registration failed, exiting");
		return 1;
	}
	observer = observer_setup (tp_am);
	if (observer == NULL)
	{
		g_error ("Observer registration failed, exiting");
		return 1;
	}


	indicate_server = indicate_server_setup ();

	contact_list_setup (tp_am);

	g_debug ("Telepathy Indicator started");
	g_main_loop_run (loop);

	g_main_loop_unref (loop);
	return 0;
}
