/*  Screem:  screem-site.c
 *
 *  The ScreemSite object
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <string.h>
#include <errno.h>

#include <sys/types.h>
#include <unistd.h>
#include <dirent.h>

#include <glib/gfileutils.h>
#include <glib/gi18n.h>
#include <libgnome/gnome-exec.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <libxml/tree.h>
#include <libxml/parser.h>

#include "fileops.h"

#include "screem-application.h"

#include "screem-site.h"
#include "screem-site-model.h"

#include "screem-todo.h"

#include "screem-markup.h" /* for the fix link code,
			      should it be moved else where? */

#include "support.h"

const gchar *permission_strings[] = {
	"p_exec",
	"p_ignore",
	"p_all"
};
const gchar *symlink_strings[] = {
	"s_follow",
	"s_ignore",
	"s_maintain"
};
const gchar *upload_strings[] = {
	"local",
	"ftp",
	"webdav",
	"ssh",
	""
};
const gchar *upload_schemes[] = {
	"file://",
	"ftp://",
	"dav://",
	"sftp://",
	""
};

enum {
	PROP_0,
	PROP_APP,
	PROP_BROWSER,
	PROP_NAME
};

enum {
	CTAGS_SIGNAL,
	LAST_SIGNAL
};

static guint screem_site_signals[ LAST_SIGNAL ] = { 0 };

static void screem_site_class_init( ScreemSiteClass *klass );
static void screem_site_init( ScreemSite *site );
static void screem_site_set_prop( GObject *object, guint prop_id,
				  const GValue *value, GParamSpec *spec );
static void screem_site_get_prop( GObject *object, guint prop_id,
				  GValue *value, GParamSpec *spec );
static void screem_site_finalize( GObject *object );

static void screem_site_model_initialise( ScreemSite *site );

static void screem_site_create_from_template( ScreemSite *site, 
					      const gchar *template,
					      const gchar *base,
					      xmlNodePtr node );

static void screem_site_check_template( ScreemPage *page,
					ScreemSite *site );
static gboolean screem_site_manage_present( ScreemSite *site,
					    const gchar *path,
					    GtkListStore *store );
static void screem_site_manage_add( ScreemSite *site, const gchar *pattern,
				    GtkListStore *store );
static void screem_site_manage_remove( ScreemSite *site, const gchar *pattern,
				    	GtkListStore *store );
static GList *screem_site_manage_get( ScreemSite *site,
					GtkListStore *store );

ScreemSite *screem_site_new( GObject *application )
{
	ScreemSite *site;
	GType type;

	type = screem_site_get_type();

	site = SCREEM_SITE( g_object_new( type, "app", application, NULL ) );

	return site;
}

void screem_site_set_name( ScreemSite *site, const gchar *name )
{
	g_object_set( G_OBJECT( site ), "name", name, NULL );
}

const gchar* screem_site_get_name( ScreemSite *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return site->priv->name;
}

void screem_site_set_pathname( ScreemSite *site, const gchar *pathname )
{
	ScreemSitePrivate *priv;
	GnomeVFSURI *uri;
	gchar *tmp;
	gsize len;

	priv = site->priv;
	
	/* does the site already have a pathname set? */
	if( priv->pathname ) {
		g_free( priv->pathname );
	}

	uri = gnome_vfs_uri_new( pathname );
	if( uri ) {
		tmp = gnome_vfs_uri_to_string( uri, 
				GNOME_VFS_URI_HIDE_NONE );
		len = strlen( tmp ) - 1 ;
		if( tmp[ len ] == G_DIR_SEPARATOR ) {
			tmp[ len ] = '\0';
		}
		priv->pathname = gnome_vfs_unescape_string( tmp, "/" );
		if( ! priv->pathname ) {
			/* invalid uri */
			g_warning( "Invalid uri to screm_site_set_pathname\n" );	
			priv->pathname = g_strdup( pathname );
		}
		gnome_vfs_uri_unref( uri );
	} else {
		g_warning( "Invalid uri to screem_site_set_pathname\n" );	
		priv->pathname = g_strdup( pathname );
	}
}

const gchar* screem_site_get_pathname( ScreemSite *site )
{
	return site->priv->pathname;
}

void screem_site_set_preview_static( ScreemSite *site, gboolean flag )
{
	g_return_if_fail( SCREEM_IS_SITE( site ) );

	site->priv->preview_static = flag;
}

gboolean screem_site_get_preview_static( ScreemSite *site )
{
	g_return_val_if_fail( SCREEM_IS_SITE( site ), FALSE );

	return site->priv->preview_static;
}

void screem_site_set_preview_dynamic( ScreemSite *site, gboolean flag )
{
	g_return_if_fail( SCREEM_IS_SITE( site ) );

	site->priv->preview_dynamic = flag;
}

gboolean screem_site_get_preview_dynamic( ScreemSite *site )
{
	g_return_val_if_fail( SCREEM_IS_SITE( site ), FALSE );

	return site->priv->preview_dynamic;
}


void screem_site_set_remote_url( ScreemSite *site, const gchar *remote_url )
{
	/* does the site already have a remote url set? */
	if( site->priv->remote_url ) {
		g_free( site->priv->remote_url );
	}

	site->priv->remote_url = g_strdup( remote_url );
}

const gchar* screem_site_get_remote_url( ScreemSite *site )
{
	return site->priv->remote_url;
}

void screem_site_set_remote_method( ScreemSite *site, UploadMethods remote_method )
{
	site->priv->remote_method = remote_method;
}

UploadMethods screem_site_get_remote_method( ScreemSite *site )
{
	return site->priv->remote_method;
}

void screem_site_set_remote_path( ScreemSite *site, const gchar *remote_path )
{
	ScreemSitePrivate *priv;
	gint len;
	
	g_return_if_fail( SCREEM_IS_SITE( site ) );
	
	priv = site->priv;
	
	g_free( priv->remote_path );
	priv->remote_path = NULL;
	
	if( remote_path ) {
		len = strlen( remote_path );
		if( remote_path[ len - 1  ] == '/' ) {
			priv->remote_path = g_strdup( remote_path );
		} else {
			priv->remote_path = g_strconcat( remote_path, "/", NULL );
		}
	}
}

const gchar* screem_site_get_remote_path( ScreemSite *site )
{
	return site->priv->remote_path;
}

void screem_site_set_remote_user( ScreemSite *site, const gchar *remote_user )
{
	/* does the site already have a remote usernmae set? */
	if( site->priv->remote_user ) {
		g_free( site->priv->remote_user );
	}

	site->priv->remote_user = g_strdup( remote_user );
}

const gchar* screem_site_get_remote_user( ScreemSite *site )
{
	return site->priv->remote_user;
}

void screem_site_set_remote_pass( ScreemSite *site, const gchar *remote_pass )
{
	/* does the site already have a remote password set? */
	if( site->priv->remote_pass ) {
		g_free( site->priv->remote_pass );
	}

	site->priv->remote_pass = g_strdup( remote_pass );
}

const gchar* screem_site_get_remote_pass( ScreemSite *site )
{
	return site->priv->remote_pass;
}


void screem_site_set_cvs_root( ScreemSite *site, const gchar *cvs_root )
{
	/* does the site already have a cvs root set? */
	if( site->priv->cvs_root ) {
		g_free( site->priv->cvs_root );
	}

	if( cvs_root && strlen( cvs_root ) ) {
		site->priv->cvs_root = g_strdup( cvs_root );
	} else {
		site->priv->cvs_root = NULL;
	}
}

const gchar* screem_site_get_cvs_root( ScreemSite *site )
{
	return site->priv->cvs_root;
}

void screem_site_set_auto_update_ask( ScreemSite *site, gboolean value )
{
	site->priv->auto_update_ask = value;
}

gboolean screem_site_get_auto_update_ask( ScreemSite *site ) 
{
	return site->priv->auto_update_ask;
}

void screem_site_set_auto_update( ScreemSite *site, gboolean value )
{
	site->priv->auto_update = value;
}

gboolean screem_site_get_auto_update( ScreemSite *site ) 
{
	return site->priv->auto_update;
}

void screem_site_set_use_ctags( ScreemSite *site, gboolean flag )
{
	site->priv->ctags = flag;
}

gboolean screem_site_get_use_ctags( const ScreemSite *site )
{
	return site->priv->ctags;
}

void screem_site_update_ctags_file( ScreemSite *site )
{
	const gchar *command = "ctags -R";
	const gchar *pathname;
	gchar *path;
	
	g_return_if_fail( SCREEM_IS_SITE( site ) );
	g_return_if_fail( ! screem_site_get_fake_flag( site ) );

	pathname = screem_site_get_pathname( site );

	if( screem_site_get_use_ctags( site ) ) {
		path = gnome_vfs_get_local_path_from_uri( pathname );
		if( path ) {
			if( gnome_execute_shell( path, command ) != -1 ) {
				/* execute suceeded */
				g_signal_emit( G_OBJECT( site ),
		       		screem_site_signals[ CTAGS_SIGNAL ], 0 );
			}
			g_free( path );
		}
	}
}

void screem_site_set_site_template( ScreemSite *site, const gchar *path )
{
	if( site->priv->site_template_path ) {
		g_free( site->priv->site_template_path );
	}

	if( path && strlen( path ) ) {
		site->priv->site_template_path = g_strdup( path );
	} else {
		site->priv->site_template_path = NULL;
	}
}

const gchar *screem_site_get_site_template( ScreemSite *site )
{
	return site->priv->site_template_path;
}

void screem_site_set_template_path( ScreemSite *site, const gchar *template_path )
{
	/* does the site already have a template path set? */
	if( site->priv->template_path ) {
		g_free( site->priv->template_path );
	}

	if( template_path && strlen( template_path ) ) {
		site->priv->template_path = g_strdup( template_path );
	} else {
		site->priv->template_path = NULL;
	}
}

const gchar* screem_site_get_template_path( ScreemSite *site )
{
	return site->priv->template_path;
}

void screem_site_set_http_url( ScreemSite *site, const gchar *http_url )
{
	/* does the site already have a remote usernmae set? */
	if( site->priv->http_url ) {
		g_free( site->priv->http_url );
	}

	site->priv->http_url = g_strdup( http_url );
}

const gchar* screem_site_get_http_url( ScreemSite *site )
{
	return site->priv->http_url;
}

gboolean screem_site_get_fake_flag( ScreemSite *site )
{
	return ( site->priv->pathname == NULL );
}

void screem_site_set_no_delete( ScreemSite *site, gboolean flag )
{
	site->priv->no_delete = flag;
}

gboolean screem_site_get_no_delete( ScreemSite *site )
{
	return site->priv->no_delete;
}

void screem_site_set_check_moved( ScreemSite *site, gboolean flag )
{
	site->priv->check_moved = flag;
}

gboolean screem_site_get_check_moved( ScreemSite *site )
{
	return site->priv->check_moved;
}

void screem_site_set_no_overwrite( ScreemSite *site, gboolean flag )
{
	site->priv->no_overwrite = flag;
}

gboolean screem_site_get_no_overwrite( ScreemSite *site )
{
	return site->priv->no_overwrite;
}

void screem_site_set_permissions( ScreemSite *site, ScreemSitePerms permissions )
{
	site->priv->permissions = permissions;
}

ScreemSitePerms screem_site_get_permissions( ScreemSite *site )
{
	return site->priv->permissions;
}

void screem_site_set_symlinks( ScreemSite *site, ScreemSiteSymlinks symlinks )
{
	site->priv->symlinks = symlinks;
}

ScreemSiteSymlinks screem_site_get_symlinks( ScreemSite *site )
{
	return site->priv->symlinks;
}


void screem_site_set_is_import( ScreemSite *site, gboolean val )
{
	g_return_if_fail( SCREEM_IS_SITE( site ) );
	
	site->priv->is_import = val;
}

gboolean screem_site_get_is_import( ScreemSite *site )
{
	g_return_val_if_fail( SCREEM_IS_SITE( site ), FALSE );

	return site->priv->is_import;
}

gboolean screem_site_is_excluded( ScreemSite *site, const gchar *path )
{
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_SITE( site ), FALSE );
	g_return_val_if_fail( path != NULL, FALSE );
	
	ret = ! screem_site_get_fake_flag( site );
	
	if( ret ) {
		ret = screem_site_manage_present( site, path,
					   site->priv->excludes );
	}

	return ret;
}

void screem_site_add_exclude( ScreemSite *site, const gchar *pattern )
{
	screem_site_manage_add( site, pattern,
			   	 site->priv->excludes );
}
		
void screem_site_remove_exclude( ScreemSite *site, const gchar *pattern )
{
	screem_site_manage_remove( site, pattern,
				   site->priv->excludes );
}

gboolean screem_site_is_ignored( ScreemSite *site, const gchar *path )
{
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_SITE( site ), FALSE );
	g_return_val_if_fail( path != NULL, FALSE );

	ret = ! screem_site_get_fake_flag( site );
	
	if( ret ) {
		ret = screem_site_manage_present( site, path,
					   site->priv->ignores );
	}

	return ret;
}

void screem_site_add_ignore( ScreemSite *site, const gchar *pattern )
{
	screem_site_manage_add( site, pattern,
				site->priv->ignores );
}
void screem_site_remove_ignore( ScreemSite *site, const gchar *pattern )
{
	screem_site_manage_remove( site, pattern,
				   site->priv->ignores );
}

void screem_site_save( ScreemSite *site )
{
	g_return_if_fail( site );

	if( ! screem_site_get_fake_flag( site ) ) {
		/* first the project file */
		screem_site_write_project_file( site );
	}
}

gboolean screem_site_create( ScreemSite *site )
{
	ScreemSiteModel *model;
	const gchar *pathname;
	const gchar *template;

	g_return_val_if_fail( site != NULL, FALSE );

	model = site->priv->model;
	
	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	/* try and make the path, will just switch to it
	   if it already exists */
	if( ! mkdir_recursive( pathname,
			       GNOME_VFS_PERM_USER_ALL |
			       GNOME_VFS_PERM_GROUP_ALL |
			       GNOME_VFS_PERM_OTHER_READ |
			       GNOME_VFS_PERM_OTHER_EXEC,
			       screem_site_model_file_op,
			       model ) ) {
		return FALSE;
	}

	screem_file_browser_set_mode( model->browser,
				      FILE_BROWSE_RECURSE | FILE_BROWSE_ROOT );
    
	/* write the initial project file */
	screem_site_write_project_file( site );

	/* process template */
	template = screem_site_get_site_template( site );
	if( template ) {
		screem_site_create_from_template( site, template,
						  pathname, NULL );
	}

	screem_file_browser_scan_directory( model->browser,
					    pathname, -1 );

	return TRUE;
}

gboolean screem_site_load( ScreemSite *site )
{
	gchar *project_file;
	const gchar *pathname;
	gchar *mime_type;

	GList *project_files = NULL;
	GList *file;
    
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSFileInfoOptions options;
	GnomeVFSFileInfo *info;

	ScreemSiteModel *model;
	
	g_return_val_if_fail( site != NULL, FALSE );

	model = site->priv->model;
	
	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	options = GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
		GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE |
		GNOME_VFS_FILE_INFO_FOLLOW_LINKS;

	if( gnome_vfs_directory_open( &handle, pathname, options ) != 
	    GNOME_VFS_OK ) {
		return FALSE;
	}
		
	info = gnome_vfs_file_info_new();

	/* build a list of *.screem files */
	while( gnome_vfs_directory_read_next( handle,info ) == GNOME_VFS_OK ){
		gchar *name;
	
		name = g_build_filename( pathname, info->name, NULL );
		mime_type = screem_get_mime_type( name, FALSE );
		
		if( mime_type &&
		! strcmp( "application/x-screem", mime_type ) ) {
			/* add it to the list */
		
			project_files = g_list_append( project_files,
						g_strdup( info->name ) );
		}
		g_free( name );
		g_free( mime_type );
	}
	gnome_vfs_directory_close( handle );
	
	gnome_vfs_file_info_unref( info );

	/* now check the list.  The priority we give is the following:
	   .project.screem
	   project.screem

	   Perhaps the project file could be named  .<sitename>.screem
	*/
	file = g_list_find_custom( project_files, (gpointer)".project.screem",
				   (GCompareFunc)strcmp );
	if( ! file ) {
		file = g_list_find_custom( project_files, 
					   (gpointer)"project.screem",
					   (GCompareFunc)strcmp );
	}

	project_file = NULL;
	if( file ) {
		project_file = g_build_filename( pathname, file->data, NULL );
	}
	g_list_foreach( project_files, (GFunc)g_free, NULL );

	site->priv->loading = TRUE;
	if( ( ! project_file )  ||
	    ( ! screem_site_parse_project_file( site, project_file ) ) ) {
		screem_site_set_is_import( site, TRUE );
	}
	site->priv->loading = FALSE;
	if( project_file ) {
		g_free( project_file );
	}
	screem_file_browser_set_mode( model->browser,
				      FILE_BROWSE_RECURSE | FILE_BROWSE_ROOT );
	screem_file_browser_scan_directory( model->browser,
					    pathname, -1 );

	screem_site_update_ctags_file( site );
	
	return TRUE;
}

gboolean screem_site_write_project_file( ScreemSite *site )
{
	gchar *project_file;
	xmlDocPtr doc;
	xmlNsPtr ns;
	xmlNodePtr tree;
	xmlNodePtr subtree;
	gint retval;
	GList *list;
	GList *tmp;

	const gchar *pathname;
	const gchar *name;
	const gchar *remote_url;
	UploadMethods remote_method;
	const gchar *remote_user;
	const gchar *remote_pass;
	const gchar *remote_path;
	gboolean no_delete;
	gboolean check_moved;
	gboolean no_overwrite;

	ScreemSitePerms permissions;
	ScreemSiteSymlinks symlinks;

	const gchar *http_url;
	const gchar *cvs_root;
	const gchar *template_path;

	ScreemSitePrivate *priv;
	ScreemTodo *todo;

	const xmlChar *truestr = (const xmlChar*)"TRUE";
	const xmlChar *falsestr = (const xmlChar*)"FALSE";
		

	g_return_val_if_fail( site != NULL, FALSE );

	/* if we are a fake site pretend we wrote the file with no problems */
	if( screem_site_get_fake_flag( site ) ) {
		return TRUE;
	}

	priv = site->priv;

	/* if we are loading still, don't save */
	if( priv->loading ) {
		return TRUE;
	}
	
	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	name = screem_site_get_name( site );
	remote_url = screem_site_get_remote_url( site );
	remote_method = screem_site_get_remote_method( site );
	remote_user = screem_site_get_remote_user( site );
	remote_pass = screem_site_get_remote_pass( site );
	remote_path = screem_site_get_remote_path( site );
	no_delete = screem_site_get_no_delete( site );
	check_moved = screem_site_get_check_moved( site );
	no_overwrite = screem_site_get_no_overwrite( site );
	permissions = screem_site_get_permissions( site );
	symlinks = screem_site_get_symlinks( site );
	http_url = screem_site_get_http_url( site );
	cvs_root = screem_site_get_cvs_root( site );
	template_path = screem_site_get_template_path( site );

	doc = xmlNewDoc( (const xmlChar*)XML_DEFAULT_VERSION );
	xmlDocSetRootElement( doc, 
			      xmlNewDocNode( doc, NULL, 
				      (const xmlChar*)"SCREEM_PROJECT",
				      NULL ) );
	ns = xmlNewNs( xmlDocGetRootElement( doc ),
			(const xmlChar*)"http://www.screem.org/", 
			(const xmlChar*)"screem" );
	xmlNewChild( xmlDocGetRootElement( doc ), ns, 
			(const xmlChar*)"title", (xmlChar*)name );

	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, 
			(const xmlChar*)"remote", 
			(xmlChar*)remote_url );
	xmlSetProp( tree, (const xmlChar*)"method", 
			(xmlChar*)upload_strings[ remote_method ] );

	xmlSetProp( tree, (const xmlChar*)"username", 
			(xmlChar*)remote_user );
	xmlSetProp( tree, (const xmlChar*)"password", 
			(xmlChar*)remote_pass );
	xmlSetProp( tree, (const xmlChar*)"path", 
			(xmlChar*)remote_path );

	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns,
			    (const xmlChar*)"preview", NULL );
	xmlSetProp( tree, (const xmlChar*)"static_publish",
		    screem_site_get_preview_static( site ) ? truestr : falsestr );
	xmlSetProp( tree, (const xmlChar*)"dynamic_publish",
		    screem_site_get_preview_dynamic( site ) ? truestr : falsestr );

	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, 
			(const xmlChar*)"options", NULL);
	xmlSetProp( tree, (const xmlChar*)"no_delete", 
			no_delete ? truestr : falsestr );
	xmlSetProp( tree, (const xmlChar*)"check_moved", 
			check_moved ? truestr : falsestr );
	xmlSetProp( tree, (const xmlChar*)"no_overwrite", 
			no_overwrite ? truestr : falsestr );

	xmlSetProp( tree, (const xmlChar*)"permissions",  
			(xmlChar*)permission_strings[ permissions ] );

	xmlSetProp( tree, (const xmlChar*)"symlinks",  
			(xmlChar*)symlink_strings[ symlinks ] );

	xmlNewChild( xmlDocGetRootElement( doc ), ns, 
			(const xmlChar*)"http", (xmlChar*)http_url );

	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, 
			(const xmlChar*)"ctags", NULL );
	if( screem_site_get_use_ctags( site ) ) {
		xmlSetProp( tree, (const xmlChar*)"use",
				(const xmlChar*)"true" );
	} else {
		xmlSetProp( tree, (const xmlChar*)"use", 
				(const xmlChar*)"false" );
	}
	
	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, 
			(const xmlChar*)"cvs", (xmlChar*)cvs_root );
	if( cvs_root ) {
		xmlSetProp( tree, (const xmlChar*)"use", 
				(const xmlChar*)"true" );
	} else {
		xmlSetProp( tree, (const xmlChar*)"use", 
				(const xmlChar*)"false" );
	}

	if( screem_site_get_auto_update( site ) ) {
		xmlSetProp( tree, (const xmlChar*)"auto_update", 
				(const xmlChar*)"true" );
	} else {
		xmlSetProp( tree, (const xmlChar*)"auto_update", 
				(const xmlChar*)"false" );
	}

	if( screem_site_get_auto_update_ask( site ) ) {
		xmlSetProp( tree, (const xmlChar*)"ask",
				(const xmlChar*)"true" );
	} else {
		xmlSetProp( tree, (const xmlChar*)"ask",
				(const xmlChar*)"false" );
	}
      	
	xmlNewChild( xmlDocGetRootElement( doc ), ns, 
			(const xmlChar*)"template", 
			(xmlChar*)template_path );

	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, 
			(const xmlChar*)"tasks", NULL );
	todo = SCREEM_TODO( screem_site_get_todo( site ) );
        for( list = screem_todo_get_tasks( todo ); list; list = list->next ) {
                ScreemTodoItem *item;
		item = (ScreemTodoItem*)list->data;
		subtree = xmlNewChild( tree, ns,
				(const xmlChar*)"task", NULL );
		xmlSetProp( subtree, (const xmlChar*)"name", 
				(xmlChar*)item->task );
		xmlSetProp( subtree, 
				(const xmlChar*)"assigned", 
				(xmlChar*)item->assigned );
		xmlSetProp( subtree,
				(const xmlChar*)"priority", 
				(xmlChar*)item->priority );
		xmlSetProp( subtree, 
				(const xmlChar*)"linkedTo", 
				(xmlChar*)item->linked_to );
		xmlSetProp( subtree, 
				(const xmlChar*)"description",
				(xmlChar*)item->description);
		if( item->complete ) {
			xmlSetProp( subtree, 
					(const xmlChar*)"completed",
					(const xmlChar*)"true" );
		} else {
			xmlSetProp( subtree, 
					(const xmlChar*)"completed", 
					(const xmlChar*)"false" );
		}
	}

	/* write exclude,ignore lists */
	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns,
			(const xmlChar*)"excludes", NULL );
	for( tmp = list = screem_site_manage_get( site, site->priv->excludes );
		list; list = list->next ) {
		subtree = xmlNewChild(tree, ns, 
				(const xmlChar*)"exclude",
				(xmlChar*)list->data);
		g_free( list->data );
	}
	g_list_free( tmp );
	/* write ignore list */
	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, 
			(const xmlChar*)"ignores", NULL );
	for( tmp = list = screem_site_manage_get( site, site->priv->ignores );
		list; list = list->next ) {
		subtree = xmlNewChild(tree, ns, 
				(const xmlChar*)"ignore", 
				(xmlChar*)list->data);
		g_free( list->data );
	}	
	g_list_free( tmp );

	if( FALSE && name && strlen( name ) ) {
		project_file = g_strconcat( pathname, name, ".screem", NULL );
	} else {
		project_file = g_build_filename( pathname,
				".project.screem", NULL );
	}


	{
		xmlChar *mem;
		gint size;

		xmlDocDumpMemory( doc, &mem, &size );

		retval = save_file( project_file, (const gchar*)mem,
				    GNOME_VFS_PERM_USER_READ | 
				    GNOME_VFS_PERM_USER_WRITE,
				    FALSE, TRUE,
				    NULL );
		xmlFree( mem );
		xmlFreeDoc( doc );
		g_free( project_file );
	}

	return (retval < 0);
}

ScreemPage* screem_site_locate_page( ScreemSite *site, const gchar *path )
{
	ScreemPage *page = NULL;
	gchar *pname;
	GnomeVFSURI *uri;

	g_return_val_if_fail( site, NULL );
	g_return_val_if_fail( SCREEM_IS_SITE( site ), NULL );
	g_return_val_if_fail( path, NULL );

	/* we want a full URI as the path */
	uri = gnome_vfs_uri_new( path );
	if( uri ) {
		pname = gnome_vfs_uri_to_string( uri, 0 );
		gnome_vfs_uri_unref( uri );
	} else {
		pname = g_strdup( path );
	}

	page = g_hash_table_lookup( site->priv->pages,
			pname );
	g_free( pname );

	return page;
}

static void screem_site_page_saving( ScreemPage *page, ScreemSite *site )
{
	ScreemApplication *app;
	const gchar *pathname;

	g_object_get( G_OBJECT( site ), "app", &app, NULL );

	pathname = screem_page_get_pathname( page );
	if( ( ! pathname ) ||
	    ! screem_site_is_ignored( site, pathname ) ) {
		screem_application_exec_helpers( app, page );
	}
	
	g_object_unref( app );
}

static void screem_site_page_saved( ScreemPage *page, ScreemSite *site )
{
	ScreemSitePrivate *priv;
	const gchar *pathname;
	const gchar *sitepath;
	gchar *dirname;
	
	priv = site->priv;
	
	sitepath = screem_site_get_pathname( site );
	
	pathname = screem_page_get_pathname( page );
	dirname = g_path_get_dirname( pathname );

	/* attempt to add to the file browser model */
	screem_file_browser_file_mod( priv->model->browser, dirname,
				pathname, GNOME_VFS_MONITOR_EVENT_CREATED );

	/* check template */
	screem_site_check_template( page, site );

	g_free( dirname );
}

static gboolean find_data_in_hash( gpointer key, gpointer value,
		gpointer data )
{
	gboolean found;

	found = ( value == data );
	if( found ) {
		g_object_set_data( G_OBJECT( data ), "hash_key",
				key );
	}
	
	return found;
}

static void screem_site_page_pathname_changed( ScreemPage *page,
		GParamSpec *pspec, ScreemSite *site )
{
	ScreemSitePrivate *priv;
	GSList *list;
	const gchar *new_pathname;
	gchar *old_key;
	gboolean add;
	
	priv = site->priv;
	
	new_pathname = screem_page_get_pathname( page );

	add = FALSE;
	list = g_slist_find( priv->untitled, page );
	if( list ) {
		/* page was untitled */
		
		if( new_pathname ) {
			priv->untitled = g_slist_remove( priv->untitled,
					page );	
			add = TRUE;
			g_object_unref( page );
		}
	} else {
		/* page had a previous pathname, find it in the
		 * hash table */
		if( g_hash_table_find( priv->pages, 
					find_data_in_hash, page ) ) {
			old_key = g_object_get_data( G_OBJECT( page ),
					"hash_key" );
			g_object_set_data( G_OBJECT( page ), "hash_key",
					NULL );

			if( ( ! new_pathname ) ||
				strcmp( old_key, new_pathname ) ) {
			
				/* this will free old_key */
				g_hash_table_remove( priv->pages, 
						old_key );
				add = TRUE;
			}
		} else {
			g_warning( "Orphan ScreemPage in site\n" );
		}
	}

	if( add ) {
		if( ! new_pathname ) {
			priv->untitled = g_slist_prepend( priv->untitled,
					page );
		} else {
			g_hash_table_insert( priv->pages, 
					g_strdup( new_pathname ), page );
		}
		g_object_ref( page );
	}
}

gboolean screem_site_add_page( ScreemSite *site, ScreemPage *page )
{
	ScreemSitePrivate *priv;
	const gchar *pathname;
	
	g_return_val_if_fail( site != NULL, FALSE );
	g_return_val_if_fail( page != NULL, FALSE );

	priv = site->priv;

	pathname = screem_page_get_pathname( page );
	if( ! pathname && ! g_slist_find( priv->untitled, page ) ) {
		/* untitled page, stick in untitled list */
		priv->untitled = g_slist_prepend( priv->untitled, page );
	} else {
		g_assert( ! screem_site_locate_page( site, pathname ) );
	
		g_hash_table_insert( priv->pages, 
				g_strdup( pathname ), page );
	}

	/* hookup to pathname change notification */
	g_signal_connect( G_OBJECT( page ), "notify::pathname",
			G_CALLBACK( screem_site_page_pathname_changed ),
			site );
	
	/* hookup to the saving signal */
	g_signal_connect( G_OBJECT( page ), "saving",
			G_CALLBACK( screem_site_page_saving ),
			site );
	
	/* hook up to the save signal */
	g_signal_connect( G_OBJECT( page ), "saved",
			G_CALLBACK( screem_site_page_saved ),
			site );

	g_object_ref( page );
	
	return TRUE;
}

void screem_site_remove_page_with_path( ScreemSite *site, const gchar *path )
{
	ScreemPage *page;
	
	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );

	page = screem_site_locate_page( site, path );
	if( page ) {
		g_hash_table_remove( site->priv->pages, path );
	}
}

void screem_site_remove_page( ScreemSite *site, ScreemPage *page )
{
	const gchar *pathname;
	
	g_return_if_fail( SCREEM_IS_SITE( site ) );
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	pathname = screem_page_get_pathname( page );
	if( ! pathname ) {
		/* untitled page */
		site->priv->untitled = g_slist_remove( site->priv->untitled, page );
		g_object_unref( page );
	} else {
		screem_site_remove_page_with_path( site, pathname );
	}
}

static void screem_site_get_pages_real( ScreemPage *page, gpointer data )
{
	GList **ret;

	ret = (GList**)data;
	*ret = g_list_prepend( *ret, page );
}

GList *screem_site_get_pages( ScreemSite *site, gboolean excludes,
		gboolean ignores )
{
	GList *ret;
	
	ret = NULL;
	screem_site_foreach_page( site,
			(GFunc)screem_site_get_pages_real,
			excludes, ignores, &ret );
	return ret;
}

static void screem_site_foreach_page_real( gpointer key,
		gpointer value, gpointer data )
{
	ScreemSite *site;
	ScreemPage *page;
	GFunc func;
	gboolean excludes;
	gboolean ignores;
	const gchar *pathname;
	
	site = SCREEM_SITE( data );
	func = g_object_get_data( G_OBJECT( site ), "func" );
	data = g_object_get_data( G_OBJECT( site ), "data" );
	excludes = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( site ),
				"excludes" ) );
	ignores = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( site ),
				"ignores" ) );

	if( ! SCREEM_IS_PAGE( value ) ) {
		g_print( "INVALID ENTRY IN PAGE TABLE: %p\t%s\n",
				value, key );
	} else {
		page = SCREEM_PAGE( value );

		pathname = screem_page_get_pathname( page );

		if( ( ! pathname ) ||
		    ( ( excludes || ! screem_site_is_excluded( site, pathname ) ) &&
		    ( ignores || ! screem_site_is_ignored( site, pathname ) ) ) ) {
			func( page, data );
		}
	}
}

void screem_site_foreach_page( ScreemSite *site,
		GFunc func, gboolean excludes, gboolean ignores,
		gpointer user_data )
{
	GSList *tmp;
	
	g_return_if_fail( SCREEM_IS_SITE( site ) );
	g_return_if_fail( func != NULL );

	g_object_set_data( G_OBJECT( site ), "func", func );
	g_object_set_data( G_OBJECT( site ), "data", user_data );
	g_object_set_data( G_OBJECT( site ), "excludes",
			GINT_TO_POINTER( excludes ) );
	g_object_set_data( G_OBJECT( site ), "excludes",
			GINT_TO_POINTER( ignores ) );
	
	g_hash_table_foreach( site->priv->pages,
			screem_site_foreach_page_real, site );
	
	for( tmp = site->priv->untitled; tmp; tmp = tmp->next ) {
		screem_site_foreach_page_real( NULL, tmp->data,
				site );
	}
}

GObject *screem_site_get_todo( ScreemSite *site )
{
	return site->priv->tasks;
}

static void screem_site_file_change_real( ScreemPage *page,
		gpointer data )
{
	const gchar *pathname;
	gchar *oldpathname;
	gchar *temp;
	gpointer *info;
	ScreemSite *site;
	const gchar *src;
	const gchar *dest;

	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	
	pathname = screem_page_get_pathname( page );

	if( ! pathname ) {
		return;
	}
	
	info = (gpointer*)data;
	site = info[ 0 ];
	src = info[ 1 ];
	dest = info[ 2 ];

		
	/* if pathname is in src then we
	   need to update the pathname to point
	   to the new location */
	oldpathname = g_strdup( pathname );
	if( ! strcmp( pathname, dest ) ) {
		
	} else if( ! strncmp( pathname, src, strlen( src ) ) ) {
		temp = g_strconcat( dest,
				    pathname + strlen( src ),
				    NULL );
		/* FIXME: do we want to do this if dest matches
		 * temp ???  can cause problems if dest has
		 * already entered the site then it gets unrefed,
		 * not setting the pathname if it matches seems
		 * to sort this out.  This only works if we are
		 * moving a single file, moving a directory we still
		 * get the crash occuring.  this is probably due to the
		 * non-reentrant nature of g_hash_table_foreach(),
		 * should we even do this? as we get pages created
		 * when the file is saved / created in the process
		 * which led to doing this */
//		if( strcmp( dest, temp ) ) {
			screem_page_set_pathname( page, temp );
//		}
		g_free( temp );
	}
	if( screem_page_load( page, NULL ) ) {
		pathname = screem_page_get_pathname( page );
		screem_markup_fix_links( site, page, pathname,
					 oldpathname,
					 src, dest );
	} else {
		/* FIXME: report error */
	}
	g_free( oldpathname );

}

void screem_site_file_change( ScreemSite *site, const gchar *src, 
			      const gchar *dest )
{
	gpointer info[ 3 ];

	GList *pages;
	GList *tmp;
	
	/* if we are a fake site we do not want to do this,
	 * we also don't want to do it if this file is excluded,
	 * we still do it for ignored files though */
	
	if( ! screem_site_get_fake_flag( site ) ) {
		info[ 0 ] = site;
		info[ 1 ] = (gpointer)src;
		info[ 2 ] = (gpointer)dest;

#if 1
		/* get a list of all pages, the hash table
		 * may end up being modified as pathnames
		 * change, which causes problems iterating
		 * over it with g_hash_table_foreach */

		pages = screem_site_get_pages( site, FALSE, TRUE );
		pages = g_list_reverse( pages );
		g_list_foreach( pages, (GFunc)g_object_ref, NULL );
		for( tmp = pages; tmp; tmp = tmp->next ) {
			screem_site_file_change_real( SCREEM_PAGE( tmp->data ), info );
		}
		g_list_foreach( pages, (GFunc)g_object_unref, NULL );
		g_list_free( pages );
#else
		screem_site_foreach_page( site,
				(GFunc)screem_site_file_change_real,
				FALSE, TRUE,
				info );
#endif
	}
}

void screem_site_set_documents( ScreemSite *site, const gchar *name, 
				GList *list )
{
	GList *prev;

	prev = g_hash_table_lookup( site->priv->documents,
					name );
	if( prev ) {
		g_hash_table_remove( site->priv->documents, name );
		g_list_free( prev );
	}
	
	g_hash_table_insert( site->priv->documents, 
			     (gchar*)name,  /* FIXME: must ensure we remove
					       the name from the table
					       before the window
					       it refers to is destroyed */
			     list );
}

GList* screem_site_get_documents( ScreemSite *site, const gchar *name )
{
	GList *documents;

	documents = g_hash_table_lookup( site->priv->documents, name );

	return documents;
}

GList* screem_site_get_excludes( ScreemSite *site )
{
	return screem_site_manage_get( site, site->priv->excludes );
}

GList* screem_site_get_ignores( ScreemSite *site )
{
	return screem_site_manage_get( site, site->priv->ignores );
}

gchar *screem_get_local_site_path( ScreemSite *site )
{
	gchar *spath;
	GnomeVFSURI *uri;
	const gchar *pathname;
	
	spath = NULL;
	if( ! screem_site_get_fake_flag( site ) ) {
		pathname = screem_site_get_pathname( site );
		uri = gnome_vfs_uri_new( pathname );
		if( ! uri ) {
			spath = NULL;
		} else if( gnome_vfs_uri_is_local( uri ) ) {
			/* strip method */
			spath = gnome_vfs_get_local_path_from_uri( pathname );
			gnome_vfs_uri_unref( uri );
		} else if( ! strncmp( "file:", pathname,
					strlen( "file:" ) ) ) {
			/* not local, still a file uri so we
			 * will hack a little as hostnames never
			 * seem to be used on file: by anything.
			 * this will occur if a path is nfs
			 * mounted etc. */
			spath = gnome_vfs_uri_to_string( uri, 
					GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD |
					GNOME_VFS_URI_HIDE_USER_NAME |
					GNOME_VFS_URI_HIDE_PASSWORD |
					GNOME_VFS_URI_HIDE_HOST_NAME |
					GNOME_VFS_URI_HIDE_HOST_PORT );
			gnome_vfs_uri_unref( uri );
		} else 	{
			spath = gnome_vfs_uri_to_string( uri, 0 );
			gnome_vfs_uri_unref( uri );
		}
	}

	return spath;
}

/* static stuff */
static void screem_site_model_initialise( ScreemSite *site )
{
	ScreemSiteModel *model;

	model = site->priv->model = g_new0( ScreemSiteModel, 1 );
	model->site = site;

}

static void screem_site_create_from_template( ScreemSite *site, 
		const gchar *template, const gchar *base,
		xmlNodePtr node )
{
	xmlDocPtr doc;
	xmlChar *name;
	gchar *dir;

	if( ! node ) {
		doc = xmlParseFile( template );
		if( doc ) {
			node = xmlDocGetRootElement( doc )->xmlChildrenNode;
		}
	}
	
	while( node ) {
		if( ! strcmp( "directory", (gchar*)node->name ) ) {
			name = xmlGetProp( node, 
					(const xmlChar*)"name" );
			dir = g_strconcat( base, G_DIR_SEPARATOR_S,
					   name, NULL );
			
			if( name ) {
				mkdir_recursive( dir,
						GNOME_VFS_PERM_USER_ALL |
						GNOME_VFS_PERM_GROUP_ALL |
						GNOME_VFS_PERM_OTHER_READ |
						GNOME_VFS_PERM_OTHER_EXEC,
						screem_site_model_file_op,
						site->priv->model );
				xmlFree( name );
			}
			name = xmlGetProp( node, 
					(const xmlChar*)"template" );
			if( name ) {
				screem_site_create_from_template( site,
						(gchar*)name, 
						dir, NULL );
				xmlFree( name );
			}
			if( node->xmlChildrenNode ) {
				screem_site_create_from_template( site,
						template, dir,
						node->xmlChildrenNode );
			}
			g_free( dir );
		} else if( ! strcmp( "file", (gchar*)node->name ) ) {
			name = xmlGetProp( node, (const xmlChar*)"name" );
			if( name ) {
				gchar *template;

				dir = g_strconcat( base, 
						G_DIR_SEPARATOR_S,
						(gchar*)name, NULL );

				template = (gchar*)xmlGetProp( node, 
						(const xmlChar*)"template" );
				if( ! template ) {
					template = (gchar*)screem_site_get_template_path( site );
					if( template ) {
						template = g_strdup(template);
					}
				}
				if( template ) {
					/* copy template to dir */
					copy_file( template, dir,
						GNOME_VFS_XFER_OVERWRITE_MODE_QUERY,
						screem_site_model_file_op,
						site->priv->model );
					g_free( template );
				} else {
					/* create empty file */
					save_file( dir, "",
						   GNOME_VFS_PERM_USER_READ |
						   GNOME_VFS_PERM_USER_WRITE |
						   GNOME_VFS_PERM_GROUP_READ |
						   GNOME_VFS_PERM_GROUP_WRITE |
						   GNOME_VFS_PERM_OTHER_READ,
						   FALSE, TRUE, NULL );
				}
				g_free( dir );
				xmlFree( name );
			}
		}
		node = node->next;
	}
}

static void screem_site_check_template_real( ScreemPage *page,
		gpointer data )
{
	ScreemSite *site;
	ScreemPage *p;
	gpointer *info;
	const gchar *tag;
	const gchar *path;
	GSList *blocks;
	
	info = (gpointer*)data;
	site = SCREEM_SITE( info[ 0 ] );
	p = SCREEM_PAGE( info[ 1 ] );
	tag = (const gchar*)info[ 2 ];
	path = (const gchar*)info[ 3 ];
	blocks = (GSList*)info[ 4 ];
	
	if( p != page ) {
		screem_markup_update_from_template( site, p, page, tag, blocks );

	}
}

static void screem_site_check_template( ScreemPage *page, ScreemSite *site )
{
	const gchar *pathname;
	const gchar *base;
	gchar *tag;
	gchar *path;
	GSList *blocks;
	gpointer data[ 5 ];
	
	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	g_return_if_fail( SCREEM_IS_SITE( site ) );
	
	tag = NULL;

	pathname = screem_page_get_pathname( page );

	/* we don't want to check if changes are being ignored for
	 * this page, or this page is excluded */
	if( pathname && 
	    ( screem_site_is_ignored( site, pathname ) ||
	    ( screem_site_is_excluded( site, pathname ) ) ) ) {
		return;
	}
	
	blocks = NULL;
	base = screem_site_get_pathname( site );
	/* base will be NULL for the fake site so no need to check
	 * for it */

	if( base && screem_page_is_template( page, base, 
		    &tag, &path, &blocks ) ) {
		data[ 0 ] = site;
		data[ 1 ] = page;
		data[ 2 ] = tag;
		data[ 3 ] = path;
		data[ 4 ] = blocks;
	
		/* don't process excluded pages */
		screem_site_foreach_page( site,
				(GFunc)screem_site_check_template_real,
				FALSE, TRUE, data );
		g_slist_foreach( blocks, (GFunc)g_free, NULL );
		g_slist_free( blocks );
		g_free( tag );
	}		
}

static gboolean screem_site_manage_present( ScreemSite *site,
		const gchar *path, GtkListStore *store )
{
	/* we want a full URI as the path */
	GnomeVFSURI *uri;
	gchar *uripath;
	GtkTreeIter it;
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_SITE( site ), FALSE );
	g_return_val_if_fail( screem_site_get_fake_flag( site ) != TRUE, FALSE );
	g_return_val_if_fail( path != NULL, FALSE );
	g_return_val_if_fail( GTK_IS_LIST_STORE( store ), FALSE );

	ret = FALSE;
	uri = gnome_vfs_uri_new( path );
	if( uri ) {
		uripath = gnome_vfs_uri_to_string( uri, 0 );
		gnome_vfs_uri_unref( uri );
	
		ret = screem_gtk_list_store_find_pattern( store, 
				&it, 0, uripath );
	
		g_free( uripath );
	} else {
	
	}
	
	return ret;
}

static void screem_site_manage_add( ScreemSite *site, 
		const gchar *pattern, GtkListStore *store )
{
	GtkTreeIter it;

	if( ! screem_gtk_list_store_find_pattern( store, &it, 
				0, pattern ) ) {
		gtk_list_store_append( store, &it );
		gtk_list_store_set( store, &it,
				    0, pattern,
				    1, TRUE, -1 );
		
		/* update the model for the site */
		if( site->priv->model->browser ) {
			screem_site_model_update( site );
		}

		if( ! site->priv->loading ) {	
			screem_site_save( site );
		}
	}
}

static void screem_site_manage_remove( ScreemSite *site, const gchar *pattern,
				    	GtkListStore *store )
{
	GtkTreeIter it;

	if( screem_gtk_list_store_find_pattern( store, &it, 
				0, pattern ) ) {
		gtk_list_store_remove( store, &it );

		/* update the model for the site */
		if( site->priv->model->browser ) {
			screem_site_model_update( site );
		}

		if( ! site->priv->loading ) {	
			screem_site_save( site );
		}
	}
}

static GList *screem_site_manage_get( ScreemSite *site,
		GtkListStore *store )
{
	GList *ret;
	GtkTreeIter it;
	gchar *val;
	
	ret = NULL;

	if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL( store ), 
				&it ) ) {
		do {
			gtk_tree_model_get( GTK_TREE_MODEL( store ), 
					&it, 0, &val, -1 );
			ret = g_list_prepend( ret, val );
		} while( gtk_tree_model_iter_next( GTK_TREE_MODEL( store ),
				 		   &it ) );
	}
	ret = g_list_reverse( ret );

	return ret;
}

static void screem_site_set_name_real( ScreemSite *site,
		const gchar *name )
{
	if( site->priv->name != name ) {
		g_free( site->priv->name );
		site->priv->name = g_strdup( name );
	}
}

/* G Object stuff */
#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_site_class_init( ScreemSiteClass *klass )
{
	GObjectClass *object_class;
	GParamSpec *spec;
	
	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	object_class->get_property = screem_site_get_prop;
	object_class->set_property = screem_site_set_prop;
	object_class->finalize = screem_site_finalize;

	spec = g_param_spec_object( "app", "Application", "The Application",
			G_TYPE_OBJECT, G_PARAM_READWRITE | G_PARAM_CONSTRUCT );
	g_object_class_install_property( object_class, PROP_APP, spec );
							
	spec = g_param_spec_object( "browser", "Browser", 
			"The site file browser",
			G_TYPE_OBJECT, G_PARAM_READABLE );
	g_object_class_install_property( object_class, PROP_BROWSER, spec );

	spec =g_param_spec_string( "name", "Site name",
			"The name of the site", "", G_PARAM_READWRITE );
	g_object_class_install_property( object_class, PROP_NAME, spec );

	screem_site_signals[ CTAGS_SIGNAL ] =
		g_signal_new( "ctags_running",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemSiteClass, 
					       ctags_running),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
}

static void screem_site_init( ScreemSite *site )
{
	GConfClient *client;
	gchar *backup_suffix;
	gchar *pattern;
		
	site->priv = g_new0( ScreemSitePrivate, 1 );

	site->priv->documents = g_hash_table_new( g_str_hash, g_str_equal );

	site->priv->tasks = G_OBJECT( screem_todo_new( site ) );

	/* all new sites are fake ones, we change this when we
	   load / create a site */
	screem_site_set_name( site, _("Individual Files") );

	screem_site_model_initialise( site );

	site->priv->excludes = gtk_list_store_new( 2, 
						      G_TYPE_STRING,
						      G_TYPE_BOOLEAN );
	site->priv->ignores = gtk_list_store_new( 2, 
						     G_TYPE_STRING,
						     G_TYPE_BOOLEAN );

	site->priv->pages = g_hash_table_new_full( g_str_hash,
			g_str_equal,
			(GDestroyNotify)g_free,
			(GDestroyNotify)g_object_unref );

	/* exclude and ignore backup files */
	client = gconf_client_get_default();
	
	backup_suffix = gconf_client_get_string( client,
			"/apps/screem/general/backup_copy_extension",
			NULL );
	
	pattern = g_strconcat( "glob:*", backup_suffix, NULL );

	site->priv->loading = TRUE;
	screem_site_add_exclude( site, pattern );
	screem_site_add_ignore( site, pattern );
	site->priv->loading = FALSE;

	g_free( pattern );

	g_free( backup_suffix );

	g_object_unref( client );
}

static void screem_site_set_prop( GObject *object, guint prop_id,
				  const GValue *value, GParamSpec *spec )
{
	ScreemSite *site;
	ScreemSitePrivate *priv;
	ScreemIconCache *cache;
	ScreemSiteModel *model;
	const gchar *home;
	
	site = SCREEM_SITE( object );
	priv = site->priv;

	switch( prop_id ) {
		case PROP_APP:
			priv->application = g_value_get_object( value );
			cache = screem_application_get_icon_cache( SCREEM_APPLICATION( priv->application ) );
			model = priv->model;
			model->browser = screem_file_browser_new( cache );
			g_object_unref( cache );

			screem_file_browser_set_mode( model->browser,
					FILE_BROWSE_NORMAL );
			g_signal_connect( G_OBJECT( model->browser ),
					 "added", 
					 G_CALLBACK( screem_site_model_added ),
					 site );
			g_signal_connect( G_OBJECT( model->browser ),
					  "removed", 
					  G_CALLBACK( screem_site_model_removed ),
					  site );
			g_signal_connect( G_OBJECT( model->browser ),
					  "icon_change", 
					  G_CALLBACK( screem_site_model_icon_change ),
					  site );
			screem_file_browser_set_sort_func( model->browser,
				(GtkTreeIterCompareFunc)screem_site_model_compare_func,
				site );
			home = g_get_home_dir();
			if( ! home ) {
				home = "/";
			}
			screem_file_browser_scan_directory( model->browser, home, -1 );
			
			break;
		case PROP_NAME:
			screem_site_set_name_real( site,
					g_value_get_string( value ) );
			break;
		default:
			break;
	}
}

static void screem_site_get_prop( GObject *object, guint prop_id,
				  GValue *value, GParamSpec *spec )
{
	ScreemSite *site;

	site = SCREEM_SITE( object );

	switch( prop_id ) {
		case PROP_APP:
			g_value_set_object( value, site->priv->application );
			break;
		case PROP_BROWSER:
			g_value_set_object( value, 
					site->priv->model->browser );
			break;
		case PROP_NAME:
			g_value_set_string( value,
					site->priv->name );
			break;
		default:
			break;
	}
}

static void screem_site_finalize( GObject *object )
{
	ScreemSite *site;
	ScreemSitePrivate *priv;
	GSList *tmp;

	site = SCREEM_SITE( object );

	priv = site->priv;

	for( tmp = priv->untitled; tmp; tmp = tmp->next ) {
		g_object_unref( G_OBJECT( tmp->data ) );
	}
	g_slist_free( priv->untitled );
	
	screem_todo_clear( SCREEM_TODO( screem_site_get_todo( site ) ) ); 
	g_object_unref( screem_site_get_todo( site ) );

	screem_site_model_destroy( site );

	g_free( priv->name );
	g_free( priv->pathname );
	g_free( priv->remote_url );
	g_free( priv->remote_path );
	g_free( priv->remote_user );
	g_free( priv->remote_pass );
	g_free( priv->http_url );
	g_free( priv->cvs_root );
	g_free( priv->template_path );

	g_hash_table_destroy( priv->documents );

	g_object_unref( priv->excludes );
	g_object_unref( priv->ignores );

	g_hash_table_destroy( priv->pages );
	
	g_free( priv );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}



GType screem_site_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSiteClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_site_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSite ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_site_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemSite",
					       &info, 0 );
	}

	return type;
}
