/*  Screem:  screem-application.c
 *
 *  The main screem application code, manages startup and generally looks
 *  after things
 *
 *  Copyright (C) 2001  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <errno.h>

#include <gmodule.h>

#include <gtk/gtk.h>

#include <gtksourceview/gtksourcelanguagesmanager.h>
#include <gtksourceview/gtksourcetag.h>

#include <gdk-pixbuf/gdk-pixbuf.h>

#include <gdk/gdkkeysyms.h>

#include <glib/gi18n.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <unistd.h>

#include <libxml/xmlreader.h>

#ifdef HAVE_PYTHON
#include <Python.h>
#endif

#include "pageUI.h"

#include "screem-application.h"
#include "screem-file-browser.h"
#include "screem-hint.h"
#include "screem-window.h"

#include "screem-window-private.h" /* YUK! */

#include "screem-macro-manager.h"
#include "screem-plugin-manager.h"
#include "screem-helper-manager.h"
#include "screem-icon-cache.h"

#include "screem-site-ui.h" /* YUK again */

#include "screem-tagtree.h"

#include "support.h"
#include "fileops.h"

#include "screem-session.h"
#include "screem-dtd-db.h"

#include "screem-encodings.h"

#include "screem-dbus.h"

struct ScreemApplicationPrivate {
	GList *window_list;
	GList *loaded_sites;
	
	ScreemSession *session;
	ScreemDTDDB *dtddb;

	GtkTreeModel *encodings;

	ScreemDbus *dbus;
	GSList *start_files;

	ScreemPluginManager *plugin_manager;
	ScreemMacroManager *macro_manager;
	ScreemHelperManager *helper_manager;

	ScreemWindow *current;

	ScreemIconCache *cache;
};

static void screem_application_destroyed_window( GObject *object,
						 ScreemApplication *application );
static gboolean screem_window_delete_event_callback( GtkWidget *widget,
						     GdkEvent *event,
						     gpointer user_data );

static void screem_application_add_icons( ScreemApplication *application );

static gboolean screem_application_load( ScreemApplication *application );

#define SPLASH_TEXT( w, t ) gtk_label_set_text( GTK_LABEL( w ),t ); \
gdk_threads_leave(); while( g_main_context_iteration(NULL,FALSE) ){} \
gdk_threads_enter();

static void screem_application_offline_notify( GConfClient *client,
					       guint cnxn_id,
					       GConfEntry *entry,
					       gpointer data );

static void screem_application_helper_add( ScreemHelperManager *manager,
		ScreemHelper *helper, ScreemWindow *window );
static void screem_application_helper_remove( ScreemHelperManager *manager,
		ScreemHelper *helper, ScreemWindow *window );

static void screem_application_helper_added( ScreemHelperManager *manager,
		ScreemHelper *helper, ScreemApplication *app );
static void screem_application_helper_removed( ScreemHelperManager *manager,
		ScreemHelper *helper, ScreemApplication *app );

GList *screem_application_get_window_list( ScreemApplication *app )
{
	GList *list;

	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );
	
	list = app->priv->window_list;

	return list;
}

void screem_application_close_all_windows( ScreemApplication *app )
{
	gboolean close;
	ScreemHelperManager *manager;

	GdkDisplay *display;
	GtkClipboard *clipboard;

	close = FALSE;
	if( app->priv->window_list ) {
		close = screem_application_save_sites( app );
	}
	
	if( close ) {
		manager = screem_application_get_helper_manager( app );

		g_object_set_data( G_OBJECT( app ), "confirmed",
				GINT_TO_POINTER( 1 ) );

		display = gtk_widget_get_display( GTK_WIDGET( app->priv->window_list->data ) );
		clipboard = gtk_clipboard_get_for_display( display, 
				GDK_NONE );
		gtk_clipboard_store( clipboard );
	
		screem_session_save( app->priv->session, SESSION_LAST );
		while( app->priv->window_list ) {
			ScreemWindow *window;

			window = SCREEM_WINDOW( app->priv->window_list->data );
			/* remove all helpers */
			screem_helper_manager_foreach( manager,
					(ScreemHelperManagerForeach)screem_application_helper_remove,
					window );
			
			
			screem_window_close( window );
		}
	}
}

static gboolean screem_window_delete_event_callback( GtkWidget *widget,
						     GdkEvent *event,
						     gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	ScreemApplicationPrivate *priv;
	GList *list;
	gboolean close;
	gboolean last;
	gboolean confirmed;

	GdkDisplay *display;
	GtkClipboard *clipboard;
	
	window = SCREEM_WINDOW( widget );

	g_object_get( G_OBJECT( window ), "app", &app, NULL);
	priv = app->priv;
	
	close = TRUE;
	last = FALSE;
		
	/* if this is the last window we should confirm saving */
	list = screem_application_get_window_list( app );
	
	confirmed = ( g_object_get_data( G_OBJECT( app ), 
					"confirmed" ) != NULL );
	if( confirmed ) {
		close = FALSE;
	}
	
	if( g_list_length( list ) == 1 ) {
		last = TRUE; 
		if( ! confirmed ) {
			close = screem_application_save_sites( app );
		}
	} 
	if( close ) {
		display = gtk_widget_get_display( widget );
		clipboard = gtk_clipboard_get_for_display( display, 
				GDK_NONE );

		if( last && ( gtk_main_level() > 0 ) ) {
			/* force storing of clipboard, otherwise
			 * we can end up with a store request being
			 * made when we have already deleted the
			 * ScreemView which owns the selection */
			gtk_clipboard_store( clipboard );
				
			/* save session */
			screem_session_save( priv->session, 
					SESSION_LAST );
		}
		screem_plugin_manager_remove_window( priv->plugin_manager,
				window );

		/* remove all helpers */
		screem_helper_manager_foreach( priv->helper_manager,
				(ScreemHelperManagerForeach)screem_application_helper_remove,
				window );
	
		screem_window_close( window );
	}
	
	g_object_unref( app );
		
	return TRUE;
}

static void
screem_application_destroyed_window( GObject *object, ScreemApplication *app )
{
	app->priv->window_list = g_list_remove( app->priv->window_list, 
						object );

	/* if all windows are closed then exit */
	if( ( gtk_main_level() > 0 ) && ! app->priv->window_list ) {
		gtk_main_quit();
	}
}

static gboolean screem_application_focus_window( ScreemWindow *window,
		GdkEventFocus *event, ScreemApplication *app )
{
	app->priv->current = window;

	return FALSE;
}

ScreemWindow *screem_application_create_window( ScreemApplication *app )
{
	ScreemApplicationPrivate *priv;
	ScreemWindow *window;
	GList *list;
	GtkAccelGroup *group;

	priv = app->priv;
	
	window = SCREEM_WINDOW( g_object_new( SCREEM_TYPE_WINDOW,
					      "app",
					      G_OBJECT( app ),
					      "app_id", "screem", NULL ) );
	g_signal_connect( G_OBJECT( window ), "focus_in_event",
			G_CALLBACK( screem_application_focus_window ),
			app );
	g_signal_connect( G_OBJECT( window ), "delete_event",
			G_CALLBACK( screem_window_delete_event_callback ),
			 NULL );
	g_signal_connect( G_OBJECT( window ), "destroy",
			G_CALLBACK( screem_application_destroyed_window ),
			app );

	if( ! priv->current ) {
		priv->current = window;
	}

	group = screem_macro_manager_get_accel_group( priv->macro_manager );
	gtk_window_add_accel_group( GTK_WINDOW( window ), group );
	
	priv->window_list = g_list_prepend( priv->window_list, window );

	list = screem_application_get_loaded_sites( app );
	for( ; list; list = list->next ) {
		screem_window_add_site( window, SCREEM_SITE( list->data ) );
	}

	screem_plugin_manager_add_window( priv->plugin_manager,
			window );

	/* add all helpers */
	screem_helper_manager_foreach( priv->helper_manager,
			(ScreemHelperManagerForeach)screem_application_helper_add,
			window );

	return window;
}

ScreemApplication *screem_application_new()
{
	ScreemApplication *application;
	GType type;

	type = screem_application_get_type();

	application = SCREEM_APPLICATION( g_object_new( type, NULL ) );

	return application;
}

static GtkWidget *screem_application_splash( ScreemApplication *application,
					     gboolean close)
{
       	GtkWidget *label;
	static GtkWidget *window;
	GtkWidget *image;
	GtkWidget *box;

	if( ! close ) {
		window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title( GTK_WINDOW( window ),
				      _( "Loading - Screem" ) );
		gtk_window_set_position( GTK_WINDOW( window ), 
					 GTK_WIN_POS_CENTER_ALWAYS );
		gtk_window_set_type_hint( GTK_WINDOW( window ),
					GDK_WINDOW_TYPE_HINT_SPLASHSCREEN );
       
		image = gtk_image_new_from_file( SPLASH_DIR"/splash.png" );
		gtk_widget_show( image );		

		label = gtk_label_new( _( "Starting..." ) );
		gtk_widget_show( label );
		
		box = gtk_vbox_new( FALSE, 0 );
		gtk_box_pack_start( GTK_BOX( box ), image, TRUE, TRUE, 0 );
		gtk_box_pack_start( GTK_BOX( box ), label, TRUE, TRUE, 0 );

		gtk_widget_show( box );
		
		gtk_container_add( GTK_CONTAINER( window ), box );
		
		gtk_widget_show_all( window );
		
		screem_set_cursor( GTK_WIDGET( window ), GDK_WATCH );
		
		return label;
	} else {
		gtk_widget_destroy( window );
		return NULL;
	}
}

void screem_application_startup( ScreemApplication *application,
				 const gchar *session,
				 const gchar **start_files )
{
	ScreemApplicationPrivate *priv;
	ScreemWindow *window;
	ScreemSite *site;
	ScreemPage *page;
	
	GtkWidget *splash_text;

	gboolean show_hints;
	GList *windows;

	gchar *dotdir;
	gchar *tmp;

	ScreemDbus *dbus;
	
	priv = application->priv;

	priv->dbus = dbus = screem_dbus_new( application );	

	if( screem_dbus_instance_exists( dbus ) ) {
		screem_dbus_open_files( dbus, start_files );
		while( g_main_context_iteration( NULL, FALSE ) ) {}
		gdk_notify_startup_complete();
		gtk_main_quit();
		return;
	} else if( start_files ) {
		while( *start_files ) {
			priv->start_files = g_slist_prepend( priv->start_files,
					g_strdup( *start_files ) );
			start_files ++;
		}
	}
	
	gdk_threads_enter();

	application->client = gconf_client_get_default();
	
	gconf_client_add_dir( application->client, "/apps/screem",
			      GCONF_CLIENT_PRELOAD_RECURSIVE, NULL );

	/* create splash screen */
	splash_text = screem_application_splash( application, FALSE );
	SPLASH_TEXT( splash_text, _( "Initialising..." ) );

#ifdef HAVE_PYTHON
	SPLASH_TEXT( splash_text, _( "Initialise Python..." ) );
	Py_Initialize();
#endif

	/* load accel map */
	dotdir = screem_get_dot_dir();
	tmp = g_build_filename( dotdir, "accels", NULL );
	gtk_accel_map_load( tmp );
	g_free( tmp );
	g_free( dotdir );
	
	/* load syntax files */
	SPLASH_TEXT( splash_text, _( "Loading Syntax files..." ) );
	screem_application_load_syntax_tables();

	/* setup offline notification / status */
	application->offline = gconf_client_get_bool( application->client,
						      "/apps/screem/general/work_offline",
						      NULL );
	application->offline_notify = 
		gconf_client_notify_add( application->client,
					 "/apps/screem/general/work_offline", 
					 screem_application_offline_notify,
					 application, NULL, NULL );
	
	application->hint = GTK_WIDGET( screem_hint_new() );
	g_object_ref( G_OBJECT( application->hint ) );
	gtk_object_sink( GTK_OBJECT( application->hint ) );

	/* add icons */
	screem_application_add_icons( application );

	/* load plugins */
	if( g_module_supported() ) {
		screem_plugin_manager_scan_plugins( priv->plugin_manager );
	}

	/* load helpers */
	SPLASH_TEXT( splash_text, _( "Loading Helpers..." ) );
	screem_helper_manager_load_helpers( priv->helper_manager );

	/* load DTDs */
	SPLASH_TEXT( splash_text, _( "Loading DTDs..." ) );
	priv->dtddb = screem_dtd_db_new();

	/* create the fake site for editing individual files */
	site = screem_site_new( G_OBJECT( application ) );
	screem_application_add_site( application, site );

	window = NULL;
	if( session ) {
		SPLASH_TEXT( splash_text, _( "Loading Session..." ) );

		screem_session_load( priv->session, session );
	}

	/* if we loaded a session that some how didn't have any windows
	 * then we still want to create a window, and an initial
	 * blank page */
	windows = screem_application_get_window_list( application );
	if( ! windows ) {
		/* create a dummy page for the fake site */
		page = screem_page_new( G_OBJECT( application ) );
		screem_page_set_data( page, NULL );

		screem_site_add_page( site, page );
		g_object_unref( page );

		/* create initial window */
		SPLASH_TEXT( splash_text, _( "Creating Interface..." ) );

		window = screem_application_create_window( application );
		screem_window_set_current( window, site );

		screem_window_set_document( window, page );

		gtk_widget_show( GTK_WIDGET( window ) );
	}
	
	screem_application_splash( application, TRUE );


	/* display hints if necessary */
	show_hints = gconf_client_get_bool( application->client,
					    "/apps/screem/general/show_tips",
					    NULL );
	if( show_hints ) {
		windows = screem_application_get_window_list( application );
		window = SCREEM_WINDOW( windows->data );
		gtk_window_set_transient_for(GTK_WINDOW(application->hint),
					     GTK_WINDOW(window));
							  
		gtk_widget_show( application->hint );
	}

	/* delay loading of tag trees so we can get the main window up
	   quicker */
	g_idle_add( (GSourceFunc)screem_tag_tree_load_trees, NULL );
	
	/* delay loading sites / files until the main loop has started up */
	g_idle_add( (GSourceFunc)screem_application_load, application );

	gdk_threads_leave();
}

void screem_application_add_files( ScreemApplication *app,
		GSList *files )
{
	ScreemApplicationPrivate *priv;
	gboolean add;
	
	g_return_if_fail( SCREEM_IS_APPLICATION( app ) );
	
	priv = app->priv;
	
	add = ( ! priv->start_files );
	
	if( priv->start_files ) {
		priv->start_files = g_slist_concat( priv->start_files, files );
	} else {
		priv->start_files = files;
	
	}		
	if( add ) {
		g_idle_add( (GSourceFunc)screem_application_load,
				app );
	}
}

void screem_application_close( ScreemApplication *app )
{
	if( app->priv->window_list ) {
		screem_session_save( app->priv->session, SESSION_LAST );
	}
}

static gboolean screem_application_load( ScreemApplication *application )
{
	ScreemApplicationPrivate *priv;
	GSList *start_files;
	gchar *sfile;
	gchar *mime_type;
	gchar *file;
	ScreemWindow *window;

	priv = application->priv;
	
	if( ! priv->window_list ) {
		/* no window created yet, this can occur when
		 * the Open dbus method has been called while
		 * screem is still starting up, don't do anything,
		 * as we add the idle handler again at the end of
		 * screem_application_startup() */
		return FALSE;
	}
       	window = priv->window_list->data;

	start_files = priv->start_files;
	
	/* process command line files / sites */
	for( start_files = priv->start_files; start_files;
			start_files = start_files->next ) {
		gboolean isdir;

		sfile = (gchar*)start_files->data;

		file = gnome_vfs_make_uri_from_shell_arg( sfile );
			
		isdir = FALSE;
		if( uri_exists( file, NULL ) ) {
			isdir = screem_uri_is_dir( file );
		}

		/* is it a project file or an html file? */
		mime_type = screem_get_mime_type( file, FALSE );
		if( isdir || 
		    ( mime_type &&
		      ! strcmp( mime_type, "application/x-screem" ) ) ) {
			gdk_threads_enter();
			screem_site_open_with_filename( window, application,
							file );
	      		gdk_threads_leave();
		} else {
			ScreemSite *site;
			gdk_threads_enter();
			site =screem_application_get_default_site(application);
			screem_page_open_with_filename( site, window, file );
	      		gdk_threads_leave();
		}
		g_free( mime_type );
		g_free( file );

		g_free( sfile );
	}
	g_slist_free( priv->start_files );
	priv->start_files = NULL;
	
	return FALSE;
}

GList *screem_application_get_loaded_sites( ScreemApplication *app )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );

	return app->priv->loaded_sites;
}

void screem_application_add_site( ScreemApplication *app, ScreemSite *site )
{
	GList *list;
	ScreemWindow *window;
	ScreemWindowDetails *details;
	const gchar *pathname;
	EggRecentItem *item;
	
	/* add to list of loaded sites */
	app->priv->loaded_sites = g_list_append( app->priv->loaded_sites, site );

	/* add to recent sites */
	if( ! screem_site_get_fake_flag( site ) ) {
		/* work on the first window, others will pick up
		   the changes via gconf */
		if( app->priv->window_list ) {
			window = app->priv->window_list->data;
			details = window->details;
			pathname = screem_site_get_pathname( site );
			item = egg_recent_item_new_from_uri( pathname );
			egg_recent_item_add_group( item,
					RECENT_SITE_GROUP );
			egg_recent_item_set_private( item, TRUE );
			egg_recent_model_add_full( details->recent_site_model,
					item );
			egg_recent_item_unref( item );
		}
	}

	for( list = screem_application_get_window_list( app ); list;
	     list = list->next ) {
		screem_window_add_site( SCREEM_WINDOW( list->data ), site );
	}
}

ScreemSite *screem_application_get_site( ScreemApplication *app,
					 const gchar *uri )
{
	ScreemSite *site;
	const GList *list;

	/* loaded_sites->next as the first one is guarenteed
	   to be the fake site */
	site = NULL;
	for( list = app->priv->loaded_sites->next; list; list = list->next ) {
		const gchar *pathname;

		site = SCREEM_SITE( list->data );
		pathname = screem_site_get_pathname( site );
		if( ! strcmp( uri, pathname ) ) {
			break;
		}
		site = NULL;
	}

	return site;
}

ScreemSite *screem_application_get_site_by_name( ScreemApplication *app,
						 const gchar *name )
{
	ScreemSite *site;
	const GList *list;

	/* loaded_sites->next as the first one is guarenteed
	   to be the fake site */
	site = NULL;
	for( list = app->priv->loaded_sites->next; list; list = list->next ) {
		const gchar *sname;

		site = SCREEM_SITE( list->data );
		sname = screem_site_get_name( site );
		if( ! strcmp( name, sname ) ) {
			break;
		}
		site = NULL;
	}

	return site;
}

ScreemSite *screem_application_get_default_site( ScreemApplication *app )
{
	/* default site is always first in the list */
	return SCREEM_SITE( app->priv->loaded_sites->data );
}

void screem_application_close_site( ScreemApplication *app, ScreemSite *site )
{
	GList *list;

	list = screem_application_get_loaded_sites( app );
	app->priv->loaded_sites = g_list_remove( list, site );

	for( list = screem_application_get_window_list( app ); list;
	     list = list->next ) {
		screem_window_remove_site( SCREEM_WINDOW( list->data ),
				site );
	}
}

void screem_application_close_all_sites( ScreemApplication *app )
{
	ScreemSite *site;

	while( app->priv->loaded_sites ) {
		site = SCREEM_SITE( app->priv->loaded_sites->data );
		screem_application_close_site( app, site );
		g_object_unref( site );
	}
}

gboolean screem_application_save_sites( ScreemApplication *app )
{
	GList *list;
	ScreemSite *site;
	gboolean ret;

	ret = TRUE;
	
	list = screem_application_get_loaded_sites( app );
	
	/* request confirmation to save any opened sites / pages */
	for( ; ret && list; list = list->next ) {
		site = SCREEM_SITE( list->data );

		/* FIXME: we need a window,
		 * doing it this way is ugly, but then
		 * again the whole save confirmation code is */
		g_assert( app->priv->window_list );

		ret = screem_site_save_confirmation( site,
				app->priv->window_list->data );
	}

	return ret;
}

static void screem_application_load_features( GHashTable *features, 
						const gchar *path )
{
	xmlTextReaderPtr reader;
	gchar *feature;
	gboolean valid;
	GHashTable *ftable;
	gchar *tmp;	
	reader = xmlNewTextReaderFilename( path );
	feature = NULL;
	valid = FALSE;
	ftable = NULL;
	if( reader ) {
		while( xmlTextReaderRead( reader ) ) {
			xmlChar *name;
			xmlChar *type;

			name = xmlTextReaderName( reader );
			if( ! valid && ! strcmp( "screem", (gchar*)name ) ) {
				type = xmlTextReaderGetAttribute( reader,
						(xmlChar*)"type" );
				valid = ( type && 
					  ! strcmp( "features", (gchar*)type ) );
				xmlFree( type );
			} else if( valid && ! strcmp( "feature", (gchar*)name ) ) {
				g_free( feature );
				type = xmlTextReaderGetAttribute( reader,
						(xmlChar*)"type" );
				if( ! type ) {
					feature = NULL;
				} else {
					feature = g_strdup( (gchar*)type );
					ftable = g_hash_table_lookup( features, feature );
					if( ! ftable ) {
						ftable = g_hash_table_new( g_str_hash, g_str_equal );
						g_hash_table_insert( features, g_strdup( feature ), ftable );
					}
					xmlFree( type );
				}
			} else if( valid && feature &&
				   ! strcmp( "mime", (gchar*)name ) ) {
				type = NULL;
				if( xmlTextReaderRead( reader ) ) {
					type = xmlTextReaderValue( reader );
					type = (xmlChar*)g_strstrip( (gchar*)type );
				}
				if( ftable && *type ) {
					tmp = g_strdup( (gchar*)type );
					g_hash_table_insert( ftable, tmp, tmp );
				}
				if( type ) {
					xmlFree( type );
				}
			} 			
			xmlFree( name );
		}
		
		xmlFreeTextReader( reader );
	}
	g_free( feature );
}

/* next 2 funcs are from gedit-languages-manager.c in gedit */
static GtkSourceTagStyle * string_to_tag_style( const gchar *string )
{
	gchar**	items;
	GtkSourceTagStyle *style;
	
	style = gtk_source_tag_style_new ();
	items = g_strsplit (string, "/", 7);

	style->is_default = FALSE;
	
	if (items == NULL)
		goto error;

	if ((items [0] == NULL) || (strlen (items [0]) != 1))
		goto error;

	style->mask = items [0][0] - '0';
	
	if ((style->mask < 0) || (style->mask > 3))
		goto error;

	if ((items [1] == NULL) || (strlen (items [1]) != 13))
		goto error;

	if (!gdk_color_parse (items [1], &style->foreground))
		goto error;

	if ((items [2] == NULL) || (strlen (items [2]) != 13))
		goto error;

	if (!gdk_color_parse (items [2], &style->background))
		goto error;

	if ((items [3] == NULL) || (strlen (items [3]) != 1))
		goto error;

	style->italic = items [3][0] - '0';
	
	if ((style->italic < 0) || (style->italic > 1))
		goto error;
	
	if ((items [4] == NULL) || (strlen (items [4]) != 1))
		goto error;

	style->bold = items [4][0] - '0';
	
	if ((style->bold < 0) || (style->bold > 1))
		goto error;

	if ((items [5] == NULL) || (strlen (items [5]) != 1))
		goto error;

	style->underline = items [5][0] - '0';
	
	if ((style->underline < 0) || (style->underline > 1))
		goto error;

	if ((items [6] == NULL) || (strlen (items [6]) != 1))
		goto error;

	style->strikethrough = items [6][0] - '0';
	
	if ((style->strikethrough < 0) || (style->strikethrough > 1))
		goto error;

	return style;

error:
	gtk_source_tag_style_free (style);

	return NULL;
}

void screem_application_init_tag_styles( GtkSourceLanguage *lang )
{
	GConfClient *client;
	GSList *l;
	GSList *tags;
	GtkSourceTag *tag;
	gchar *id;
	gchar *key;
	gchar *value;
	GtkSourceTagStyle *style;

	gchar *langid;
	
	/* don't want to init a language twice, modified from
	 * gedit version, quicker to do this than traverse
	 * a list of inited styles */
	if( ! g_object_get_data( G_OBJECT( lang ), "screem_inited" ) ) {
		client = gconf_client_get_default();

		tags = gtk_source_language_get_tags( lang );
		langid = gtk_source_language_get_id( lang );
		for( l = tags; l; l = l->next ) {
			tag = GTK_SOURCE_TAG( l->data );

			id = gtk_source_tag_get_id( tag );
			g_return_if_fail( id != NULL );

			key = g_strconcat( "/apps/screem/editor/syntax/",
					langid, id, NULL );

			value = gconf_client_get_string( client,
					key, NULL );
			if( value ) {
				style = string_to_tag_style( value );
				g_free( value );
				if( style ) {
					gtk_source_language_set_tag_style( lang, id, style );
					gtk_source_tag_style_free( style );
				} else {
					g_warning( "gconf key %s contains an invalid value", key );
				}
			}
			g_free( key );
			g_free( id );
			g_object_unref( tag );
		}
		g_slist_free( tags );
		g_free( langid );

		g_object_set_data( G_OBJECT( lang ), "screem_inited",
				GINT_TO_POINTER( 1 ) );
		g_object_unref( client );
	}
}

GtkSourceLanguagesManager* screem_application_load_syntax_tables( void )
{
	static GtkSourceLanguagesManager *lm = NULL;
	static GHashTable *features;
	
	if( ! lm ) {
		gchar *dir;
		gchar *tmp;
		
		lm = gtk_source_languages_manager_new();
	
		features = g_hash_table_new( g_str_hash, g_str_equal );
		screem_application_load_features( features, 
					DATADIR"/screem/features.xml" );
		tmp = screem_get_dot_dir();
		dir = g_strconcat( tmp, G_DIR_SEPARATOR_S, "features.xml", 
				NULL );
		g_free( tmp );
		screem_application_load_features( features, dir );
		g_free( dir );

		g_object_set_data( G_OBJECT( lm ), "features", features );
	}

	return lm;
}

GHashTable *screem_application_get_features( ScreemApplication *application, const gchar *type )
{
	GHashTable *ret;
	GtkSourceLanguagesManager *lm;
		
	ret = NULL;
	lm = screem_application_load_syntax_tables();
	if( lm ) {
		ret = g_object_get_data( G_OBJECT( lm ), "features" );
	
		if( ret && type ) {
			ret = g_hash_table_lookup( ret, type );
		}
	}

	return ret;
}

static void screem_application_add_icons( ScreemApplication *application )
{
	static struct ScreemStockPixmap {
		gchar const *stock_id;
		gchar const *filename;
		gchar const *ext;
	} const icons[] = {
		{ "Screem_Button", "button", ".xpm" },
		{ "Screem_Caption", "caption", ".xpm" },
		{ "Screem_Checkbutton", "checkbutton", ".xpm" },
		{ "Screem_Entry", "entry", ".xpm" },
		{ "Screem_FileEntry", "gnome-fileentry", ".xpm" },
		{ "Screem_Image", "stock_insert_image", ".png" },
		{ "Screem_Object", "stock_insert_object", ".png" },
		{ "Screem_Entity", "insert-symbol", ".png" },
		{ "Screem_Link", "add-link", ".png" },
		{ "Screem_Optionmenu", "optionmenu", ".xpm" },
		{ "Screem_Paragraph", "paragraphs", ".png" },
		{ "Screem_Pre", "pre", ".xpm" },
		{ "Screem_Radiobutton", "radiobutton", ".xpm" },
		{ "Screem_Sub", "sub", ".xpm" },
		{ "Screem_Sup", "sup", ".xpm" },
		{ "Screem_Table", "stock_insert_table", ".png" },
		{ "Screem_Td", "add_column", ".png" },
		{ "Screem_Text", "text", ".xpm" },
		{ "Screem_Th", "th", ".xpm" },
		{ "Screem_Tr", "add_row", ".png" },

		{ "Screem_Site", "screem_site", ".png" },
		{ "Screem_Preview", "site_preview", ".png" },
		{ "Screem_LinkView", "site_structure", ".png" },
		{ "Screem_Resources", "resources", ".png" },

		{ "Screem_Todo", "todo", ".png" },

		{ "Screem_Bookmarks", "bookmarks-preferences", ".png" },
		{ "Screem_Bookmark", "bookmarks-open", ".png" },

		{ "Screem_Online", "stock_connect", ".png" },
		{ "Screem_Offline", "stock_disconnect", ".png" },

		{ "Screem_CVSCheckout", "cvs-checkout", ".png" },
		{ "Screem_CVSUpdate", "cvs-update", ".png" },
		{ "Screem_CVSAdd", "cvs-add", ".png" },
		{ "Screem_CVSRemove", "cvs-remove", ".png" },

		{ NULL, NULL, NULL }
	};
	static const GtkStockItem stockItems[] = {
		{ "Screem_Button", "<button>", 0, 0, 0 },
		{ "Screem_Caption", "<caption>", 0, 0, 0 },
		{ "Screem_Checkbutton", "Checkbox", 0, 0, 0 },
		{ "Screem_Entry", "text entry", 0, 0, 0 },
		{ "Screem_FileEntry", "file entry", 0, 0, 0 },
		{ "Screem_Image", "<img>", 0, 0, 0 },
		{ "Screem_Object", "<object>", 0, 0, 0 },
		{ "Screem_Entity", "Entity", 0, 0, 0 },
		{ "Screem_Link", "<a>", 0, 0, 0 },
		{ "Screem_Optionmenu", "<option>", 0, 0, 0 },
		{ "Screem_Paragraph", "<p>", 0, 0, 0 },
		{ "Screem_Pre", "<pre>", 0, 0, 0 },
		{ "Screem_Radiobutton", "Radiobutton", 0, 0, 0 },
		{ "Screem_Sub", "<sub>", 0, 0, 0 },
		{ "Screem_Sup", "<sup>", 0, 0, 0 },
		{ "Screem_Table", "<table>", 0, 0, 0 },
		{ "Screem_Td", "<td>", 0, 0, 0 },
		{ "Screem_Text", "<textarea>", 0, 0, 0 },
		{ "Screem_Th", "<th>", 0, 0, 0 },
		{ "Screem_Tr", "<tr>", 0, 0, 0 },

		{ "Screem_Site", "", 0, 0, 0 },

		{ "Screem_Preview", "Preview", 0, 0, 0 },
		{ "Screem_LinkView", "Link View", 0, 0, 0 },
		{ "Screem_Resources", "Resources", 0, 0, 0 },

		{ "Screem_Todo", "Task List", 0, 0, 0 },

		{ "Screem_Bookmarks", "Edit Bookmarks...", 0, 0, 0 },
		{ "Screem_Bookmark", "", 0, 0, 0 },

		{ "Screem_Online", "", 0, 0, 0 },
		{ "Screem_Offline", "", 0, 0, 0 },

		{ "Screem_CVSCheckout", "", 0, 0, 0 },
		{ "Screem_CVSUpdate", "", 0, 0, 0 },
		{ "Screem_CVSAdd", "", 0, 0, 0 },
		{ "Screem_CVSRemove", "", 0, 0, 0 }

	};

	GtkIconFactory *factory;
	gint i;

	factory = gtk_icon_factory_new();

	for( i = 0; icons[ i ].stock_id; ++ i ) {
		GtkIconSet *set;
		GtkIconSource *src;
		GdkPixbuf *pixbuf;
		gchar *filename;
		gint j;
		gboolean wildcarded;
		static const gchar *sizes[] = {
			"", "-48", "-24", "-16", NULL
		};

		set = gtk_icon_set_new();
		wildcarded = FALSE;

		for( j = 0; sizes[ j ]; ++ j ) {
			filename = g_strconcat( PIXMAP_PATH, G_DIR_SEPARATOR_S,
						icons[ i ].filename,
						sizes[ j ],
						icons[ i ].ext,
						NULL );
			
			pixbuf = gdk_pixbuf_new_from_file( filename, NULL );
			if( pixbuf ) {
				src = gtk_icon_source_new();
				gtk_icon_source_set_pixbuf( src, pixbuf );
				if( ! wildcarded ) {
					wildcarded = TRUE;
				} else {
					gtk_icon_source_set_size_wildcarded( src, FALSE );
				}
				
				if( j == 1 ) {
					gtk_icon_source_set_size( src,
								  GTK_ICON_SIZE_DIALOG );
				} else if( j == 2 ) {
					gtk_icon_source_set_size( src,
								  GTK_ICON_SIZE_BUTTON );
				} else if( j == 3 ) {
					gtk_icon_source_set_size( src,
								  GTK_ICON_SIZE_MENU );
				}
				gtk_icon_set_add_source( set, src );
				g_object_unref( G_OBJECT( pixbuf ) );
				gtk_icon_source_free( src );
			}
			g_free( filename );
		}
		gtk_icon_factory_add( factory, icons[ i ].stock_id, set );
		
		gtk_icon_set_unref( set );
	}

	gtk_icon_factory_add_default( factory );

	gtk_stock_add( stockItems, G_N_ELEMENTS( stockItems ) );

	g_object_unref( G_OBJECT( factory ) );
}

ScreemDTDDB *screem_application_get_dtd_db( ScreemApplication *application )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( application ),
				NULL );

	return application->priv->dtddb;
}

void screem_application_set_cursor( ScreemApplication *app,
		GdkCursorType type )
{
	GList *windows;

	for( windows = screem_application_get_window_list( app );
			windows; windows = windows->next ) {
		screem_set_cursor( GTK_WIDGET( windows->data ),
				type );
	}
}

void screem_application_file_op( GnomeVFSMonitorEventType type,
				const gchar *uri, gpointer data )
{
	ScreemApplication *app;
	ScreemFileBrowser *browser;
	gchar *dirname;
	GList *list;
	
	dirname = g_path_get_dirname( uri );
	app = SCREEM_APPLICATION( data );
	
	for( list = screem_application_get_loaded_sites( app );
			list; list = list->next ) {
		g_object_get( G_OBJECT( list->data ), "browser", &browser,
				NULL );
		screem_file_browser_file_mod( browser, dirname, uri, type );
		g_object_unref( browser );
	}
	g_free( dirname );
}

ScreemSession *screem_application_get_session( ScreemApplication *app )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );

	return app->priv->session;
}

GtkTreeModel *screem_application_get_encodings( ScreemApplication *app )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );

	return app->priv->encodings;
}

ScreemPluginManager *screem_application_get_plugin_manager( ScreemApplication *app )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );

	return app->priv->plugin_manager;
}

ScreemMacroManager *screem_application_get_macro_manager( ScreemApplication *app )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );

	return app->priv->macro_manager;
}

ScreemHelperManager *screem_application_get_helper_manager( ScreemApplication *app )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );

	return app->priv->helper_manager;
}

ScreemIconCache *screem_application_get_icon_cache( ScreemApplication *app )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );

	if( app->priv->cache ) {
		g_object_ref( app->priv->cache );
	}
	
	return app->priv->cache;
}

void screem_application_exec_helpers( ScreemApplication *app, 
		ScreemPage *page )
{
	ScreemApplicationPrivate *priv;
	ScreemWindow *window;
	GList *windows;
	GList *docs;

	priv = app->priv;
	
	window = priv->current;
	
	/* window should be the last focused window, if it
	 * is NULL scan all windows to see if one has it open */
	if( ! window ) {
		for( windows = app->priv->window_list; windows;
				windows = windows->next ) {
			window = SCREEM_WINDOW( windows->data );
	
			docs = screem_window_get_documents( window );
			if( g_list_find( docs, page ) ) {
				break;
			}
			window = NULL;
		}
	}

	screem_helper_manager_exec_all_helpers( priv->helper_manager,
			page, window );
}

static void screem_application_offline_notify( GConfClient *client,
					       guint cnxn_id,
					       GConfEntry *entry,
					       gpointer data )
{
	ScreemApplication *application;

	application = SCREEM_APPLICATION( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		gboolean state;

		state = gconf_value_get_bool( entry->value );

		if( application->offline != state ) {
			GList *list;

			application->offline = state;
	
			list = screem_application_get_window_list( application );
			g_list_foreach( list, 
					(GFunc)screem_window_toggle_online_status,
					NULL );
		}
	}
}

static gboolean screem_application_free_feature_type( gpointer key,
						gpointer value,
						gpointer data )
{
	g_assert( key == value );
	g_free( key );

	return FALSE;
}

static gboolean screem_application_free_features( gpointer key,
						  gpointer value,
						  gpointer data )
{
	GHashTable *table;

	table = (GHashTable*)value;
	
	g_hash_table_foreach( table,
			(GHFunc)screem_application_free_feature_type,
			NULL );
	
	g_hash_table_destroy( table );
	g_free( key );

	return FALSE;
}

static void screem_application_helper_add( ScreemHelperManager *manager,
		ScreemHelper *helper, ScreemWindow *window )
{
	screem_helper_add_to_window( helper, window );
}

static void screem_application_helper_remove( ScreemHelperManager *manager,
		ScreemHelper *helper, ScreemWindow *window )
{
	screem_helper_remove_from_window( helper, window );
}

static void screem_application_helper_added( ScreemHelperManager *manager,
		ScreemHelper *helper,
		ScreemApplication *app )
{
	GList *list;
	
	list = screem_application_get_window_list( app );
	while( list ) {
		screem_helper_add_to_window( helper, 
				SCREEM_WINDOW( list->data ) );
		list = list->next;
	}
}

static void screem_application_helper_removed( ScreemHelperManager *manager,
		ScreemHelper *helper,
		ScreemApplication *app )
{
	GList *list;

	list = screem_application_get_window_list( app );
	while( list ) {
		screem_helper_remove_from_window( helper, 
				SCREEM_WINDOW( list->data ) );
		list = list->next;
	}
}

/* G Object stuff */

G_DEFINE_TYPE( ScreemApplication, screem_application, G_TYPE_OBJECT )

static void screem_application_finalize( GObject *object );

static void screem_application_class_init( ScreemApplicationClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_application_finalize;
}

static void screem_application_init( ScreemApplication *application )
{
	ScreemApplicationPrivate *priv;
	gint i;
	GtkTreeIter it;
	const gchar *trans;
	gchar *tmp;

	/* we properly init later by calling screem_application_startup */
	priv = application->priv = g_new0( ScreemApplicationPrivate, 1 );
	priv->session = screem_session_new( G_OBJECT( application ) );

	priv->encodings = GTK_TREE_MODEL( gtk_list_store_new( 2, 
				G_TYPE_STRING, G_TYPE_STRING ) );

	for( i = 0; i < screem_n_encodings; ++ i ) {
		trans = gettext( screem_encodings[ i ].name );
		tmp = g_strconcat( trans, " (",
				screem_encodings[ i ].charset, ")", 
				NULL );
		gtk_list_store_insert_with_values( GTK_LIST_STORE( priv->encodings ), 
				&it, i,
				0, tmp,
				1, screem_encodings[ i ].charset,
				-1 );
		g_free( tmp );
	}

	priv->plugin_manager = screem_plugin_manager_new();

	priv->macro_manager = screem_macro_manager_new();

	priv->helper_manager = screem_helper_manager_new();

	g_signal_connect( G_OBJECT( priv->helper_manager ),
			"added",
			G_CALLBACK( screem_application_helper_added ), 
			application );
	g_signal_connect( G_OBJECT( priv->helper_manager ),
			"removed",
			G_CALLBACK( screem_application_helper_removed ), 
			application );
	
	priv->cache = screem_icon_cache_new();
}

static void screem_application_finalize( GObject *object )
{
	ScreemApplication *application;
	ScreemApplicationPrivate *priv;
	GtkSourceLanguagesManager *lm;
	GHashTable *features;

	gchar *dotdir;
	gchar *tmp;
	
	application = SCREEM_APPLICATION( object );
	priv = application->priv;

	if( priv->start_files ) {
		g_slist_foreach( priv->start_files, (GFunc)g_free, NULL );
		g_slist_free( priv->start_files );
		priv->start_files = NULL;
	}

	if( priv->dbus ) {
		g_object_unref( G_OBJECT( priv->dbus ) );
	}
	
	screem_application_close_all_windows( application );

	screem_application_close_all_sites( application );

	/* save accel map */
	dotdir = screem_get_dot_dir();
	tmp = g_build_filename( dotdir, "accels", NULL );
	gtk_accel_map_save( tmp );
	g_free( tmp );
	g_free( dotdir );
	
	if( priv->plugin_manager ) {
		g_object_unref( priv->plugin_manager );
	}

	if( priv->macro_manager ) {
		g_object_unref( priv->macro_manager );
	}

	if( priv->helper_manager ) {
		g_object_unref( priv->helper_manager );
	}
	
	if( priv->cache ) {
		g_object_unref( priv->cache );
	}
	
	if( application->offline_notify ) {
		gconf_client_notify_remove( application->client, 
				application->offline_notify );
	}

	if( priv->dtddb ) {
		g_object_unref( priv->dtddb );
	}

	if( application->client ) {
		g_object_unref( application->client );
	}

	if( priv->session ) {
		g_object_unref( priv->session );
	}

	lm = screem_application_load_syntax_tables();
	features = g_object_get_data( G_OBJECT( lm ), "features" );
	if( features ) {
		g_hash_table_foreach( features,
				(GHFunc)screem_application_free_features,
				NULL );
		g_hash_table_destroy( features );
		g_object_unref( G_OBJECT( lm ) );
	}

	if( priv->encodings ) {
		g_object_unref( priv->encodings );
	}
	
	if( application->hint ) {
		g_object_unref( application->hint );
		gtk_widget_destroy( application->hint );
	}

	g_free( priv );

	/* hack to cleanup filters */
	screem_get_file_filter( (const gchar*)0x1 );

#ifdef HAVE_PYTHON
	Py_Finalize();
#endif
	
	G_OBJECT_CLASS( screem_application_parent_class )->finalize( object );

	screem_file_browser_cleanup();
}

