/*  Screem:  screem-window.c
 *
 *  The ScreemWindow widget
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <string.h>

#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-popup-menu.h>

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <gtk/gtkhbox.h>
#include <gtk/gtknotebook.h>

#include <gdk/gdktypes.h>

#include <gmodule.h>

#include <gtk/gtktreeview.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkhpaned.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkeventbox.h>

#include <gconf/gconf-client.h>

#include "screem-window.h"

#include "screem-window-private.h"

#include "screem-editor.h"
#include "screem-preview.h"
#include "screem-site.h"
#include "screem-site-view.h"
#include "screem-tagtree.h"
#include "screem-linkview.h"
#include "screem-tree-view.h"
#include "screem-search.h"

#include "support.h"

#include "pageUI.h"

#include "fileops.h"

#include "rb-sidebar.h"
#include "rb-sidebar-button.h"

#include "libegg/dock/egg-dock.h"
#include "libegg/dock/egg-dock-layout.h"
#include "libegg/dock/egg-dock-notebook.h"

enum {
	ARG_0,
	ARG_APP_ID,
	ARG_APP
};

typedef enum _Drop_types {
	TARGET_URI_LIST,
	TARGET_URL,
	TARGET_MOZ
} Drop_types;

static const GtkTargetEntry drop_types[] = {
        { "text/uri-list", 0, TARGET_URI_LIST },
	{ "text/x-moz-url", 0, TARGET_MOZ },
        { "_NETSCAPE_URL", 0, TARGET_URL },
        { "x-url/http", 0, TARGET_URL },
        { "x-url/ftp", 0, TARGET_URL },
};
static const gint num_drop_types = sizeof(drop_types) / sizeof(drop_types [0]);

static void screem_window_initialise_class( ScreemWindowClass *klass );
static void screem_window_initialise( ScreemWindow *window );
static void screem_window_finalize( GObject *object );
static void screem_window_set_prop( GObject *object, guint prop_id, 
				    const GValue *value, GParamSpec *spec );
static void screem_window_get_prop( GObject *object, guint prop_id, 
				    GValue *value, GParamSpec *spec );
static void screem_window_size_request( GtkWidget *widget,
					GtkRequisition *req );
static void screem_window_realize( GtkWidget *widget );
static void screem_window_show( GtkWidget *widget );



void screem_window_set_document( ScreemWindow *window, ScreemPage *page );


static void screem_window_constructed( ScreemWindow *window );

static gboolean screem_window_file_view_click( GtkTreeView *tree_view,
					       GtkTreePath *path,
					       GtkTreeViewColumn *column,
					       gpointer data );

static gboolean screem_window_file_view_press( GtkWidget *widget, 
					       GdkEventButton *event,
					       ScreemWindow *window );

static void screem_window_real_show_message( ScreemWindow *window,
					     GtkWidget *view, 
					     const gchar *message );
static void screem_window_real_clear_messages( GtkWidget *view );
static gboolean screem_window_clear_status( ScreemWindow *window );

static void screem_window_mdi_change( ScreemMDI *mdi, ScreemPage *page,
					ScreemWindow *window );
static void screem_window_highlight_spelling( ScreemSpell *spell,
					      gint start, gint end,
					      ScreemWindow *window );

static void screem_window_select_context_callback( ScreemTreeView *view, 
						   guint start,
						   guint end, gpointer data );

static void screem_window_view_change( RBSidebarButton *button, 
				       ScreemWindow *window );
static void
screem_window_switch_site_destroy( gpointer data, GClosure *closure );

static GtkWidget *screem_window_get_site_button( ScreemWindow *window,
						  ScreemSite *site );
static void screem_window_site_action( EggAction *action,
					SwitchSiteData *sd );
static void screem_window_site_switch( RBSidebarButton *button,
				       SwitchSiteData *sd );

static void screem_window_drop( GtkWidget *widget, GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *selectionData,
				guint info, guint time, ScreemWindow *window );

static void screem_window_layout_changed( EggDock *dock, ScreemWindow *window );

static void screem_window_item_parent_set( GtkWidget *widget, GtkWidget *prev_parent,
					 GtkWidget *label );


static void screem_window_search_found( ScreemSearch *search,
					ScreemPage *page,
					guint pos,
					guint length,
					gpointer data );
static void screem_window_search_notfound( ScreemSearch *search,
					   ScreemPage *page,
					   gpointer data );

static void screem_window_ctags_jumpto( ScreemCtagsView *view,
					const gchar *pathname,
					const gchar *pattern,
					guint line,
					gpointer data );

void
screem_window_close( ScreemWindow *window )
{
	ScreemSite *site;
	ScreemPage *page;
	ScreemWindowDetails *details;

	g_return_if_fail( SCREEM_IS_WINDOW( window ) );

	details = window->details;
	
	site = screem_window_get_current( window );

	details->current = NULL;
	page = screem_mdi_get_current( details->mdi );
	if( page ) {
		g_signal_handlers_disconnect_matched(G_OBJECT( page ),
						     G_SIGNAL_MATCH_DATA,
						     0, 0, NULL, 
						     NULL,
						     details->tagtree );
		g_signal_handlers_disconnect_matched( G_OBJECT( page ),
						G_SIGNAL_MATCH_DATA,
						0, 0, NULL, NULL,
						details->tree_view );
		g_signal_handlers_disconnect_matched( G_OBJECT( page ),
						G_SIGNAL_MATCH_DATA,
						0, 0, NULL, NULL,
						details->editor );
		g_signal_handlers_disconnect_matched( G_OBJECT( page ),
						G_SIGNAL_MATCH_DATA,
						0, 0, NULL, NULL,
						details->preview );
		g_signal_handlers_disconnect_matched( G_OBJECT( page ),
						G_SIGNAL_MATCH_DATA,
						0, 0, NULL, NULL,
						details->link_view );
	}
	screem_mdi_set_current( details->mdi, NULL );
	
	gtk_widget_destroy( GTK_WIDGET( window ) );
}

static void
screem_window_constructed( ScreemWindow *window )
{
	GtkWidget *box;
	GtkWidget *sbox;
	ScreemWindowDetails *details;
	GtkWidget *button;
       	gboolean offline;
	GtkWidget *widget;
	gchar *file;
	EggAction *action;
	GtkWidget *sw;
	gchar *tmp;
	GtkTooltips *tip;

	GtkPositionType postype;
	
	details = window->details;
	
	gtk_window_set_default_size( GTK_WINDOW( window ), 600, 400 );

	gtk_window_set_wmclass( GTK_WINDOW( window ), "Screem", 
				"mainwindow"  );

	/* content_hbox is the main display area of the window */
	window->content_hbox = gtk_vbox_new( FALSE, 0 );

	screem_window_initialise_menus( window );

	/* we do this here as we need the EggMenuMerge object
	 * to be in existance */
	details->mdi = screem_mdi_new( window );
	g_signal_connect( G_OBJECT( details->mdi ),
			  "changed",
			  G_CALLBACK( screem_window_mdi_change ),
			  window );
	screem_window_mdi_change( details->mdi, NULL, window );
	
	sbox = gtk_vbox_new( FALSE, 0 );
	gtk_widget_show( sbox );
						
	button = gtk_button_new_with_label( _( "Views" ) );
	gtk_widget_show_all( button );
	gtk_box_pack_start( GTK_BOX( sbox ), button, FALSE, FALSE, 0 );
       
	details->bar = rb_sidebar_new();
	gtk_widget_show( details->bar );
	gtk_box_pack_start( GTK_BOX( sbox ), details->bar, TRUE, TRUE, 0 );

	details->sitebar = rb_sidebar_new();

	g_signal_connect_data( G_OBJECT( button ), "clicked",
			       G_CALLBACK( gtk_widget_show_all ),
			       details->bar, NULL, G_CONNECT_SWAPPED );
	g_signal_connect_data( G_OBJECT( button ), "clicked",
			       G_CALLBACK( gtk_widget_hide ),
			       details->sitebar, NULL, G_CONNECT_SWAPPED );

	button = gtk_button_new_with_label( _( "Sites" ) );
	gtk_widget_show_all( button );
	gtk_box_pack_start( GTK_BOX( sbox ), button, FALSE, FALSE, 0 );

	g_signal_connect_data( G_OBJECT( button ), "clicked",
			       G_CALLBACK( gtk_widget_show_all ),
			       details->sitebar, NULL, G_CONNECT_SWAPPED );
	g_signal_connect_data( G_OBJECT( button ), "clicked",
			       G_CALLBACK( gtk_widget_hide ),
			       details->bar, NULL, G_CONNECT_SWAPPED );

	gtk_box_pack_start( GTK_BOX( sbox ), details->sitebar, TRUE, TRUE, 0 );

	/* create the dock */
	window->dock = egg_dock_new();
	window->layout = G_OBJECT(egg_dock_layout_new(EGG_DOCK(window->dock)));
	g_signal_connect( G_OBJECT( window->dock ), "layout_changed",
			  G_CALLBACK( screem_window_layout_changed ),
			  window );
	gtk_box_pack_start( GTK_BOX( window->content_hbox ), window->dock,
				TRUE, TRUE, 0 );
		
	/* create the status bar */
	window->status = gtk_hbox_new( FALSE, 0 );
	gtk_widget_show( window->status );
	gtk_box_pack_start( GTK_BOX( window->content_hbox ), window->status,
				FALSE, TRUE, 0 );


	/* add the editor / preview / link view etc */
	details->editor = GTK_WIDGET( screem_editor_new( window ) );
	g_object_ref( G_OBJECT( details->editor ) );
	gtk_object_sink( GTK_OBJECT( details->editor ) );
	gtk_widget_show_all( details->editor );
	details->current = details->editor;
	
	button = GTK_WIDGET(rb_sidebar_button_new( "SwitchEditor", _( "Editor" ) ) );
	g_object_set_data( G_OBJECT( details->editor ), "button", button );
	g_signal_connect_data( G_OBJECT( button ), "clicked",
			       G_CALLBACK( screem_window_view_change ),
			       window, NULL, 0 );
	rb_sidebar_button_set( RB_SIDEBAR_BUTTON( button ), "Screem_Site",
			       _( "Editor" ), TRUE );
       	rb_sidebar_append( RB_SIDEBAR( details->bar ), 
			   RB_SIDEBAR_BUTTON( button ) );

	details->preview = 
		GTK_WIDGET( screem_preview_new_with_window( window ) );
	g_object_ref( G_OBJECT( details->preview ) );
	gtk_object_sink( GTK_OBJECT( details->preview) );
	gtk_widget_show_all( details->preview );

	button = GTK_WIDGET( rb_sidebar_button_new( "SwitchPreview",
						    _( "Preview" ) ) );
	g_object_set_data( G_OBJECT( details->preview), "button", button );
	g_signal_connect_data( G_OBJECT( button ), "clicked",
			       G_CALLBACK( screem_window_view_change ),
			       window, NULL, 0 );
	rb_sidebar_button_set( RB_SIDEBAR_BUTTON( button ), "Screem_Site",
			       _( "Preview" ), TRUE );
       	rb_sidebar_append( RB_SIDEBAR( details->bar ), 
			   RB_SIDEBAR_BUTTON( button ) );

	details->link_view = GTK_WIDGET( screem_link_view_new( window ) );
	g_object_ref( G_OBJECT( details->link_view ) );
	gtk_object_sink( GTK_OBJECT( details->link_view) );

	gtk_widget_show_all( details->link_view );
	button = GTK_WIDGET( rb_sidebar_button_new( "SwitchLinkView", 
						    _( "Link View" ) ) );
	g_object_set_data( G_OBJECT( details->link_view ), "button", button );
	g_signal_connect_data( G_OBJECT( button ), "clicked",
			       G_CALLBACK( screem_window_view_change ),
			       window, NULL, 0 );
	rb_sidebar_button_set( RB_SIDEBAR_BUTTON( button ), "Screem_LinkView",
			       _( "LinkView" ), TRUE );
       	rb_sidebar_append( RB_SIDEBAR( details->bar ), 
			   RB_SIDEBAR_BUTTON( button ) );	
	
	details->tree_view = GTK_WIDGET( screem_tree_view_new( window ) );
	g_object_ref( G_OBJECT( details->tree_view ) );
	gtk_object_sink( GTK_OBJECT( details->tree_view) );
	gtk_widget_show_all( details->tree_view );
	
	g_signal_connect( G_OBJECT( details->tree_view ),
			  "select_context",
			  G_CALLBACK( screem_window_select_context_callback ),
			  window );
	button = GTK_WIDGET( rb_sidebar_button_new( "SwitchTreeView", 
						    _( "Tree View" ) ) );
	g_object_set_data( G_OBJECT( details->tree_view ), "button", button );
	g_signal_connect_data( G_OBJECT( button ), "clicked",
			       G_CALLBACK( screem_window_view_change ),
			       window, NULL, 0 );
	rb_sidebar_button_set( RB_SIDEBAR_BUTTON( button ), "Screem_Site",
			       _( "Tree View" ), TRUE );
       	rb_sidebar_append( RB_SIDEBAR( details->bar ), 
			   RB_SIDEBAR_BUTTON( button ) );	
	
	gtk_widget_show_all( details->bar );

	/* add all views into a box */
	postype = screem_get_tab_position();
	if( postype == GTK_POS_TOP || postype == GTK_POS_BOTTOM ) {
		box = gtk_vbox_new( FALSE, 0 );
	} else {
		box = gtk_hbox_new( FALSE, 0 );
	}
	
	details->box = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( details->box ),
			    details->editor, TRUE, TRUE, 0 );
	gtk_box_pack_start( GTK_BOX( details->box ),
			    details->preview, TRUE, TRUE, 0 );
	gtk_box_pack_start( GTK_BOX( details->box ),
			    details->link_view, TRUE, TRUE, 0 );
	gtk_box_pack_start( GTK_BOX( details->box ),
			    details->tree_view, TRUE, TRUE, 0 );

	gtk_widget_hide( details->editor );
	gtk_widget_hide( details->preview );
	gtk_widget_hide( details->link_view );
	gtk_widget_hide( details->tree_view );

	if( postype == GTK_POS_TOP || postype == GTK_POS_LEFT ) {
		gtk_box_pack_end( GTK_BOX( box ), details->box, 
				TRUE, TRUE, 0 );
	} else {
		gtk_box_pack_start( GTK_BOX( box ), details->box, 
				TRUE, TRUE, 0 );
	}
	
	/* now create the notebook for the page tabs */
	details->notebook = GTK_WIDGET( screem_mdi_get_notebook( details->mdi ) );
	gtk_box_pack_start( GTK_BOX( box ), details->notebook, 
				FALSE, FALSE, 0 );
	gtk_widget_show( details->notebook );

	gtk_widget_show( box );
	
	/* dock stuff */
	widget = egg_dock_item_new_with_stock( "Main", _( "Main" ), NULL,
					EGG_DOCK_ITEM_BEH_LOCKED |
					EGG_DOCK_ITEM_BEH_NEVER_FLOATING);
	gtk_container_add( GTK_CONTAINER( widget ), box );
	egg_dock_add_item( EGG_DOCK( window->dock ), EGG_DOCK_ITEM( widget ),
				EGG_DOCK_CENTER );
	gtk_widget_show( widget );
	widget = egg_dock_item_new_with_stock( "Shortcut", _( "Shortcuts" ), NULL,
					EGG_DOCK_ITEM_BEH_NORMAL |
					EGG_DOCK_ITEM_BEH_CANT_DOCK_TOP |
					EGG_DOCK_ITEM_BEH_CANT_DOCK_BOTTOM |
					EGG_DOCK_ITEM_BEH_CANT_DOCK_CENTER  |
					EGG_DOCK_ITEM_BEH_NEVER_VERTICAL );
	gtk_container_add( GTK_CONTAINER( widget ), sbox );
	egg_dock_add_item( EGG_DOCK( window->dock ), EGG_DOCK_ITEM( widget ),
				EGG_DOCK_LEFT );
	gtk_widget_show( widget );
	
	/* create site tab */
	widget = gtk_tree_view_new();
	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_AUTOMATIC, 
					GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	gtk_container_add( GTK_CONTAINER( sw ), widget );
	screem_site_init_tree_view( widget );
	g_object_set_data( G_OBJECT( window ), "siteview", widget );
	g_signal_connect( G_OBJECT( widget ), "row_activated",
			G_CALLBACK( screem_window_file_view_click ),
			window );
	g_signal_connect( G_OBJECT( widget ), "button_press_event",
			G_CALLBACK( screem_window_file_view_press ),
			window );
	gtk_widget_show_all( sw );

	screem_window_add_dock_item( window, _( "Files" ), GTK_STOCK_HOME,
					"Files", sw, GTK_POS_RIGHT );

	/* create ctags model/tree */
	details->ctagsmodel = screem_ctags_model_new();
	details->ctagsview = screem_ctags_view_new();
	g_signal_connect( G_OBJECT( details->ctagsview ), "jumpto",
			G_CALLBACK( screem_window_ctags_jumpto ),
			window );
	gtk_tree_view_set_model( GTK_TREE_VIEW( details->ctagsview ),
				GTK_TREE_MODEL( details->ctagsmodel ) );
	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC );
	gtk_container_add( GTK_CONTAINER( sw ), 
			GTK_WIDGET( details->ctagsview ) );
	gtk_widget_show_all( sw );
	screem_window_add_dock_item( window, _( "Symbols" ), 
				     GTK_STOCK_PROPERTIES, 
				     "Symbols",
				     sw, GTK_POS_RIGHT );
	
	/* create resources tree */
	details->tagtree = GTK_WIDGET( screem_tag_tree_new() );
	g_object_ref( G_OBJECT( details->tagtree ) );
	gtk_object_sink( GTK_OBJECT( details->tagtree ) );
	g_object_set( G_OBJECT( details->tagtree ), "window",
		      window, NULL );
	gtk_widget_show_all( details->tagtree );

	screem_window_add_dock_item( window, _( "Resources" ), 
				     "Screem_Resources", 
				     "Resources",
				     details->tagtree, GTK_POS_RIGHT );
	
	gtk_widget_show( window->dock );
		
	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC );
	details->messages = gtk_text_view_new();
	gtk_text_view_set_editable( GTK_TEXT_VIEW( details->messages ), FALSE );
	gtk_text_view_set_cursor_visible( GTK_TEXT_VIEW( details->messages ),
					  FALSE );
	gtk_container_add( GTK_CONTAINER( sw ), details->messages );
	gtk_widget_show_all( sw );

	screem_window_add_dock_item( window, _( "Messages" ), GTK_STOCK_DIALOG_INFO,
					"Messages", sw, GTK_POS_BOTTOM );
	

	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC );
	details->errors = gtk_text_view_new();
	gtk_text_view_set_editable( GTK_TEXT_VIEW( details->errors ), FALSE );
	gtk_text_view_set_cursor_visible( GTK_TEXT_VIEW( details->errors ), 
					  FALSE );
	gtk_container_add( GTK_CONTAINER( sw ), details->errors );
	gtk_widget_show_all( sw );

	screem_window_add_dock_item( window, _( "Errors" ), GTK_STOCK_DIALOG_ERROR,
					"Errors", sw, GTK_POS_BOTTOM );
	
	gtk_widget_show( details->box );
	gtk_widget_show( details->current );
	gtk_widget_show( window->content_hbox );

	gtk_container_add( GTK_CONTAINER( window ), window->content_hbox );

	/* add recent pages and sites menu */
	details->recent_pages = 
		gedit_recent_new_with_merge( "screem-pages", 4,
					     EGG_MENU_MERGE(window->merge),
					     "/menu/File Menu/RecentPages" );
	details->recent_sites =
		gedit_recent_new_with_merge( "screem-sites", 4,
					     EGG_MENU_MERGE(window->merge),
					     "/menu/Site/RecentSites" );
	g_signal_connect( G_OBJECT( details->recent_pages ),
			  "activate",
			  G_CALLBACK( screem_window_recent_page_cb ),
			  (gpointer)window );
	g_signal_connect( G_OBJECT( details->recent_sites ),
			  "activate",
			  G_CALLBACK( screem_window_recent_site_cb ),
			  (gpointer)window );

	/* add online status control to status bar */
	offline = window->details->offline = window->application->offline;

	button = gtk_button_new();
	tip = gtk_tooltips_new();
	gtk_tooltips_set_tip( tip, button, "", "" );
	gtk_button_set_relief( GTK_BUTTON( button ),
			       GTK_RELIEF_NONE );
	details->online_status = gtk_image_new();
	gtk_container_add( GTK_CONTAINER( button ), details->online_status );
	gtk_widget_show_all( button );
	g_signal_connect_data( G_OBJECT( button ), "clicked",
			       G_CALLBACK( screem_window_toggle_online_status),
			       window, NULL, G_CONNECT_SWAPPED );

	gtk_box_pack_start( GTK_BOX( window->status ), button,
				FALSE, FALSE, 0 );
	gtk_widget_set_size_request( button, -1, 16 );
	
	screem_window_toggle_online_status( window, NULL );

	window->status_text = gtk_statusbar_new();
	gtk_box_pack_start( GTK_BOX( window->status ), window->status_text,
				TRUE, TRUE, 0 );
	gtk_statusbar_set_has_resize_grip( GTK_STATUSBAR( window->status_text ),
						FALSE );
	gtk_widget_show( window->status_text );

	action = egg_action_group_get_action( window->action_group,
						"SwitchEditor" );
	egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ), TRUE );

	tmp = screem_get_dot_dir();
	file = g_strconcat( tmp, G_DIR_SEPARATOR_S,
		    	    "layout", NULL );
	g_free( tmp );
	if( ! uri_exists( file, NULL ) ) {
		g_free( file );
		file = g_strconcat( UIDATADIR, "layout", NULL );
	}
	egg_dock_layout_load_from_file( EGG_DOCK_LAYOUT(window->layout),
					file );
	egg_dock_layout_load_layout( EGG_DOCK_LAYOUT( window->layout ),
					"__default__" );
	g_free( file );
	
	/* get visible status of dockitems so we can set the toggle items
		in the menu correctly */
	{
		EggDockItem *item;
		EggActionGroup *group;
		const gchar *items[] = { 
			"ViewMessages", "Messages",
			"ViewErrors", "Errors",
			"ViewShortcut", "Shortcut",
			"ViewFiles", "Files",
			"ViewResources", "Resources",
			"ViewTree", "Structure",
			"ViewAttributes", "Attributes",
			"ViewSymbols", "Symbols",
			NULL
		};
		gint i;
		gboolean show;
		
		group = window->action_group;
		for( i = 0; items[ i ]; i += 2 ) {
			action = egg_action_group_get_action( group,
							items[ i ] );
			item = egg_dock_get_item_by_name(EGG_DOCK(window->dock),
							items[ i + 1 ] );
			show = ( GTK_WIDGET( item )->parent != NULL );
			egg_toggle_action_set_active( EGG_TOGGLE_ACTION(action),
							show );
			g_object_set_data( G_OBJECT( action ), "item", item );
		}
		action = egg_action_group_get_action( group,
						"ViewMainToolbar");
		show = gconf_client_get_bool( window->application->client,
						"/apps/screem/ui/show_main_toolbar",
						NULL );
		egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ), ! show );
		egg_action_activate( action );
		
		action = egg_action_group_get_action( group,
						"ViewWizardToolbar");
		show = gconf_client_get_bool( window->application->client,
						"/apps/screem/ui/show_wizard_toolbar",
						NULL );
		egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ), ! show );
		egg_action_activate( action );
	}

	window->search = screem_search_new();
	g_object_set( G_OBJECT( window->search ),
			"window", window,
			"editor", window->details->editor,
			NULL );
	g_signal_connect( G_OBJECT( window->search ),
			"found",
			G_CALLBACK( screem_window_search_found ),
			window );
	g_signal_connect( G_OBJECT( window->search ),
			"notfound",
			G_CALLBACK( screem_window_search_notfound ),
			window );
	
	gtk_widget_show_all( window->status );
}

void screem_window_change_view( ScreemWindow *window,
				GtkWidget *view )
{
	gboolean activate = FALSE;
	ScreemWindowDetails *details;
	RBSidebarButton *button;

	details = window->details;

	if( view == details->editor &&
	    details->current != details->editor ) {
		/* switch to it */
		gtk_widget_hide( details->current );
		details->current = details->editor;
		activate = TRUE;
	} else if( view == details->preview &&
		   details->current != details->preview ) {
		/* switch to it */
		gtk_widget_hide( details->current );
		details->current = details->preview;
		activate = TRUE;
	} else if( view == details->link_view &&
		   details->current != details->link_view ) {
		/* switch to it */
		gtk_widget_hide( details->current );
		details->current = details->link_view;
		activate = TRUE;
	} else if( view == details->tree_view &&
		   details->current != details->tree_view ) {
		/* switch to it */
		gtk_widget_hide( details->current );
		details->current = details->tree_view;
		activate = TRUE;
	} else {
		activate = FALSE;
	}
		
	if( activate ) {
		gtk_widget_show( details->current );
		screem_window_display_view( window );
		
		/* set shortcutbar button */
		button = g_object_get_data( G_OBJECT( details->current ),
					    "button" );
		if( ! gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( button ) ) ) {
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( button ), 
						      TRUE );
		}
	} 
}

void screem_window_add_site( ScreemWindow *window, ScreemSite *site )
{
	const gchar *name;
	EggMenuMerge* merge;
	const gchar *pathname;
	SwitchSiteData *sd;
	gchar *verb_name;
	RBSidebarButton *button;
	
	GString *xml;
	gchar **paths;
	gint i;
	guint mergeid;
	EggActionGroupEntry *entries;
	EggAction *action;
	
	static EggActionGroupEntry site_switch_action_entries[] = {
		{ "Action Name", "Label",
	  	"Screem_Site", NULL, "Tip",
	  	G_CALLBACK( NULL ), NULL },
	};

	name = screem_site_get_name( site );
	
	/* sidebar stuff */
	
	button = rb_sidebar_button_new( name, name );
	g_object_set_data( G_OBJECT( button ), "site", site );

	rb_sidebar_button_set( button, "Screem_Site", name, TRUE );
	rb_sidebar_append( RB_SIDEBAR( window->details->sitebar ),
			   button );
	gtk_widget_show( GTK_WIDGET( button ) );

	sd = g_new0( SwitchSiteData, 1 );
	sd->site = site;
	sd->window = window;
	g_signal_connect( G_OBJECT( button ), "clicked",
			  G_CALLBACK( screem_window_site_switch ), sd );

	window->details->sitebargroup = 
		g_slist_prepend( window->details->sitebargroup, button );

	/* EggMenuMerge stuff */

	merge = EGG_MENU_MERGE( window->merge );
	verb_name = g_strconcat( "SwitchSite_", name, NULL );
	entries = site_switch_action_entries;
	entries->name = verb_name;
	entries->label = (gchar*)name;
	pathname = screem_site_get_pathname( site );
	if( ! pathname ) {
		pathname = _( "Individual Files" );
	}
	entries->tooltip = (gchar*)pathname;
	entries->callback = G_CALLBACK( screem_window_site_action );
	entries->user_data = sd;
	
	egg_action_group_add_actions( EGG_ACTION_GROUP( window->action_group ),
					entries, 1 );

	paths = g_strsplit ("menu/Site/SwitchTo", "/", 0);
	xml = g_string_new ("<Root>");
	for (i=0; paths[i]; ++i) {
		if (!strcmp("menu",paths[i])) {
			g_string_append (xml, "<menu>");
		} else {
			g_string_append_printf (xml, "<submenu name=\"%s\" verb=\"%s\">", paths[i], paths[i]);
		}
	}
	g_string_append_printf (xml, "<menuitem name=\"%s\" verb=\"%s\" />", verb_name, verb_name);
	for ( ; i > 0; -- i ) {
		if (!strcmp("menu",paths[i-1])) {
			g_string_append (xml, "</menu>");
		} else {
			g_string_append (xml, "</submenu>");
		}
	}
	g_string_append (xml, "</Root>");
		
	mergeid = egg_menu_merge_add_ui_from_string (merge, xml->str, xml->len, NULL);
	
	action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
						verb_name );
	g_object_set_data( G_OBJECT( action ), "mergeid",
			   GUINT_TO_POINTER( mergeid ) );
	g_object_set_data( G_OBJECT( action ), "sitedata", sd );
	
	g_string_free( xml, TRUE );
	g_free( verb_name );
}


void screem_window_set_current( ScreemWindow *window, ScreemSite *site )
{
	ScreemWindowDetails *details;
	GList *docs;
	GList *tmp;
	GtkTreeViewColumn *column;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkWidget *tree;
	EggActionGroup *group;
	
	gboolean enable = ( ! screem_site_get_fake_flag( site ) );
	static const gchar* elements[] = {
		"Close Site",

		"SiteSettings",
		"UpdateSite",
		"ImportSite",
		"CommitSite",
		"TaskList",
		"FixLinks",
		NULL
	};
	gint i;

	GtkWidget *button;
	const gchar *pathname;
	
	g_return_if_fail( SCREEM_IS_WINDOW( window ) );
        g_return_if_fail( SCREEM_IS_SITE( site ) );

	details = window->details;
	
	if( site == details->current_site ) {
		return;
	}
	
	/* bit of a hack here to ensure the correct sidebar
	 * button is selected*/
	button = screem_window_get_site_button( window, site );
	if( ! gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( button ) ) ) {
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( button ), TRUE );
		return;
	}
	
	model = screem_site_get_model( site );
	tree = GTK_WIDGET( g_object_get_data( G_OBJECT( window ), "siteview" ) );
	g_object_ref( G_OBJECT( model ) );
	gtk_tree_view_set_model( GTK_TREE_VIEW( tree ), model );
	
        if( details->current_site ) {
		/* store documents list */

		docs = screem_mdi_get_document_list( details->mdi );
		if( docs ) {
			docs = g_list_copy( docs );
		}
		screem_site_set_documents( details->current_site, 
					   details->name,
					   docs );
		screem_mdi_remove_all( details->mdi );
       }

        details->current_site = site;

	column = gtk_tree_view_get_column( GTK_TREE_VIEW( tree ), 0 );
	gtk_tree_view_column_set_title( column, screem_site_get_name( site ) );

	g_object_set_data( G_OBJECT( tree ), "site", site );

	column = gtk_tree_view_get_column( GTK_TREE_VIEW( tree ), 1 );

	gtk_tree_view_column_set_visible( column, 
					  ! screem_site_get_fake_flag(site) );

	/* hide header for individual files as per HIG guidelines */
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( tree ), 
					   ! screem_site_get_fake_flag( site ) ); 
	
	/* expand root node */
	path = gtk_tree_path_new_first();
	gtk_tree_view_expand_row( GTK_TREE_VIEW( tree ),
				  path, FALSE );
	gtk_tree_path_free( path );

	/* now loop through the documents and insert each page */
	docs = screem_site_get_documents( site, details->name );

	for( tmp = docs; docs; docs = docs->next ) {
		screem_mdi_add_document( details->mdi,
					SCREEM_PAGE( docs->data ) );
	}
	if( tmp ) {
		screem_mdi_set_current( details->mdi,
					SCREEM_PAGE( tmp->data ) );
	}
	
	pathname = screem_site_get_pathname( site );
	screem_ctags_model_set_pathname( details->ctagsmodel,
					pathname );
	
	/* enable || disable ui elements */
	group = window->action_group;
	for( i = 0; elements[ i ]; ++ i ) {
		EggAction *action;
			
		action = egg_action_group_get_action( group,
						elements[ i ] );
		g_object_set( G_OBJECT( action ),
				"sensitive", enable, NULL );
	}
}

ScreemSite *screem_window_get_current( ScreemWindow *window )
{
	return window->details->current_site;
}

void screem_window_set_document( ScreemWindow *window, ScreemPage *page )
{
        const gchar *pathname;
        ScreemWindowDetails *details;
	ScreemSite *site;
	gboolean loaded;

        g_return_if_fail( SCREEM_IS_WINDOW( window ) );

        details = window->details;

	if( ! details->current ) {
		/* no current view, don't display, probably
		   destroying the window */
		return;
	}

	loaded = screem_mdi_set_current( details->mdi, page );

	site = screem_window_get_current( window );
	if( loaded && screem_site_get_fake_flag( site ) ) {
		pathname = screem_page_get_pathname( page );
		if( pathname ) {
			gedit_recent_add( details->recent_pages, pathname );
		}
	}
}

ScreemPage* screem_window_get_document( ScreemWindow *window )
{
	return screem_mdi_get_current( window->details->mdi );
}

void screem_window_remove_document( ScreemWindow *window, ScreemPage *page )
{
	screem_mdi_remove_document( window->details->mdi, page );
}

void screem_window_show_message( ScreemWindow *window, const gchar *message )
{
	screem_window_real_show_message( window, window->details->messages,
					 message );
}

void screem_window_show_error( ScreemWindow *window, const gchar *error )
{
	screem_window_real_show_message( window, window->details->errors,
					 error );
}

void screem_window_clear_messages( ScreemWindow *window )
{
	screem_window_real_clear_messages( window->details->messages );
}

void screem_window_clear_errors( ScreemWindow *window )
{
	screem_window_real_clear_messages( window->details->errors );
}

void screem_window_print( ScreemWindow *window, gboolean preview )
{
	ScreemWindowDetails *details;

	details = window->details;

	screem_view_print( SCREEM_VIEW( details->current ),
			   preview );
}

void screem_window_display_view( ScreemWindow *window )
{
	ScreemSite *site;
	ScreemPage *page;
	ScreemWindowDetails *details;

	details = window->details;
	site = screem_window_get_current( window );
	page = screem_mdi_get_current( details->mdi );

	if( site && details->current ) {
		g_object_set( G_OBJECT( details->current ), "online",
			      ! window->details->offline, NULL );
		
		screem_view_display( SCREEM_VIEW( details->current ) );

		if( details->current != details->tree_view ) {
			screem_view_display( SCREEM_VIEW( details->tree_view ) );
		}
	}
}


void screem_window_toggle_online_status( ScreemWindow *window, gpointer data )
{
	ScreemApplication *application;
	GConfClient *client;
	gboolean offline;
	EggAction *action;
	GtkWidget *widget;
	GtkTooltips *tip;
	GtkTooltipsData *tip_data;
	application = window->application;
	client = application->client;

	if( data ) {
		/* toggle actual setting */
		application->offline = ! application->offline;
	}

	offline = application->offline;

	if( window->details->offline != offline ) {
		gconf_client_set_bool( client,
				       "/apps/screem/general/work_offline",
				       offline,
				       NULL );
		window->details->offline = offline;
	}

	widget = gtk_widget_get_parent( window->details->online_status );
	tip_data = gtk_tooltips_data_get( widget );
	tip = tip_data->tooltips;
	
	action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
						"Offline" );
	g_object_set( G_OBJECT( action ), "visible", ! offline, NULL );
	if( ! offline ) {
		gtk_tooltips_set_tip( tip, widget,
				action->tooltip, "" );
	}
	
	action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
						"Online" );
	g_object_set( G_OBJECT( action ), "visible", offline, NULL );
	if( offline ) {
		gtk_tooltips_set_tip( tip, widget,
				action->tooltip, "" );

	}
	
	gtk_image_set_from_stock( GTK_IMAGE( window->details->online_status ),
				  offline ? "Screem_Offline" : "Screem_Online",
				  GTK_ICON_SIZE_BUTTON );
}

GList *screem_window_get_documents( ScreemWindow *window )
{
	g_return_val_if_fail( SCREEM_IS_WINDOW( window ), NULL );
	
	return screem_mdi_get_document_list( window->details->mdi );
}


gchar *screem_window_get_selection( ScreemWindow *window )
{
	ScreemEditor *editor;
	ScreemPage *page;
	guint start;
	guint end;
	gchar *ret;
	
	editor = SCREEM_EDITOR( window->details->editor );
	page = screem_window_get_document( window );

	if( screem_editor_has_selection( editor, &start, &end ) ) {
		ret = screem_editor_get_text( editor, start, end - start ); 
	} else if( page ) {
		ret = screem_page_get_data( page );
	} else {
		ret = NULL;
	}

	return ret;
}

void screem_window_replace_selection( ScreemWindow *window, const gchar *data )
{
	ScreemEditor *editor;
	ScreemPage *page;
	guint start;
	guint end;
	
	editor = SCREEM_EDITOR( window->details->editor );
	page = screem_window_get_document( window );

	if( page && ! screem_editor_has_selection( editor, &start, &end ) ) {
		start = 0;
		end = gtk_text_buffer_get_char_count( GTK_TEXT_BUFFER( page ) );
	}
	screem_editor_delete_forward( editor, start, end - start );
	screem_editor_insert( editor, start, data );
}

void screem_window_add_dock_item( ScreemWindow *window, const gchar *text,
				  const gchar *stock_id, 
				  const gchar *name,
				  GtkWidget *item,
				  GtkPositionType pos )
{
	GtkWidget *box;
	GtkWidget *label;
	GtkWidget *image;
	GtkWidget *lbox;
	GtkWidget *widget;

	EggDockPlacement place;
	
	switch( pos ) {
		case GTK_POS_LEFT:
			place = EGG_DOCK_LEFT;
			break;
		case GTK_POS_RIGHT:
			place = EGG_DOCK_RIGHT;
			break;
		case GTK_POS_TOP:
			place = EGG_DOCK_TOP;
			break;
		case GTK_POS_BOTTOM:
			place = EGG_DOCK_BOTTOM;
			break;
		default:
			g_warning( "Unsupported pos for dock item" );
			return;
			break;

	}
	
	label = gtk_label_new( text );
	gtk_misc_set_alignment( GTK_MISC( label ), 0.0, 0.5 );
	image = gtk_image_new_from_stock( stock_id, GTK_ICON_SIZE_MENU );

	lbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( lbox ), image, FALSE, FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( lbox ), label, TRUE, TRUE, 0 );
	gtk_widget_show_all( lbox );
	
	box = gtk_vbox_new( FALSE, 0 );
	gtk_container_set_border_width( GTK_CONTAINER( box ), 6 );
	gtk_widget_show( box );
	gtk_box_pack_start( GTK_BOX( box ), lbox, FALSE, FALSE, 6 );
	gtk_box_pack_start( GTK_BOX( box ), item, TRUE, TRUE, 0 );
	
	widget = egg_dock_item_new_with_stock( name, text, stock_id,
					EGG_DOCK_ITEM_BEH_NORMAL );
	gtk_container_add( GTK_CONTAINER( widget ), box );
	g_signal_connect( G_OBJECT( widget ), "parent_set",
			  G_CALLBACK( screem_window_item_parent_set ),
			  lbox );
	egg_dock_add_item( EGG_DOCK( window->dock ), EGG_DOCK_ITEM( widget ),
			   place );
	gtk_widget_show( widget );
}


static void screem_window_real_show_message( ScreemWindow *window,
					     GtkWidget *view, 
					     const gchar *message )
{
	ScreemApplication *application;
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;
	gint buffersize;
	
	g_return_if_fail( GTK_IS_TEXT_VIEW( view ) );

	application = SCREEM_APPLICATION( window->application );
	buffersize = gconf_client_get_int( application->client,
					   "/apps/screem/messagelines",
					   NULL );

	/* put in status bar, or message box */
	if( ! strchr( message, '\n' ) ) {
		if( window->details->status_timeout ) {
			g_source_remove( window->details->status_timeout );
			gtk_statusbar_pop( GTK_STATUSBAR( window->status_text ), 0 );
		}
		gtk_statusbar_push( GTK_STATUSBAR( window->status_text ),
					0, message );
	} else {
		GtkWidget *dockitem;
		EggDockObject *parent;
	
		/* first parent is the scrolled window */
		dockitem = gtk_widget_get_parent( view );
		/* second is the GtkVBox with the window / label in */
		dockitem = gtk_widget_get_parent( dockitem );
		/* third is the EggDockItem */
		dockitem = gtk_widget_get_parent( dockitem );
			
		parent = egg_dock_object_get_parent_object( EGG_DOCK_OBJECT( dockitem ) );
		
		if( EGG_IS_DOCK_NOTEBOOK( parent ) ) {
			GtkWidget *notebook;
			guint pnum;
			
			notebook = EGG_DOCK_ITEM( parent )->child;
			pnum=gtk_notebook_page_num( GTK_NOTEBOOK( notebook ),
							dockitem );
			gtk_notebook_set_current_page( GTK_NOTEBOOK(notebook),
							pnum );
		}
		
		buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(view));
		
		/* first trim the buffer if needed*/
		gtk_text_buffer_get_end_iter( buffer, &eit );
		if( gtk_text_iter_get_line( &eit ) > buffersize ) {
			gtk_text_buffer_get_iter_at_line( buffer, &it, 0 );
			gtk_text_buffer_get_iter_at_line( buffer, &eit, 1 );	
			gtk_text_buffer_delete( buffer, &it, &eit );
		}
		
		/* insert the message */
		gtk_text_buffer_get_end_iter( buffer, &eit );
		gtk_text_buffer_insert( buffer, &eit, message, 
					strlen( message ) );
		gtk_text_buffer_get_end_iter( buffer, &eit );
		gtk_text_buffer_insert( buffer, &eit, "\n", strlen( "\n" ) );
		
	}

	/* set timeout to clear status bar */
	window->details->status_timeout = g_timeout_add( 5000, 
			(GSourceFunc)screem_window_clear_status, window );
}

static void screem_window_real_clear_messages( GtkWidget *view )
{
	GtkTextBuffer *buffer;
	GtkTextIter it;
	GtkTextIter eit;

	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(view));

	gtk_text_buffer_get_start_iter( buffer, &it );
	gtk_text_buffer_get_end_iter( buffer, &eit );
	gtk_text_buffer_delete( buffer, &it, &eit );

	buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( view ) );
}

static gboolean screem_window_clear_status( ScreemWindow *window )
{
gdk_threads_enter();
	
	gtk_statusbar_pop( GTK_STATUSBAR( window->status_text ), 0 );
	window->details->status_timeout = 0;

	gdk_threads_leave();
	
	return FALSE;
}

static gboolean screem_window_file_view_click( GtkTreeView *tree_view,
					       GtkTreePath *path,
					       GtkTreeViewColumn *column,
					       gpointer data )
{
	ScreemWindow *window;
	ScreemSite *site;

	GtkTreeSelection *sel;

	window = SCREEM_WINDOW( data );
	site = screem_window_get_current( window );

	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree_view ) );

	if( sel ) {
		ScreemPage *page;

		const gchar *current;

		current = g_object_get_data( G_OBJECT( tree_view ), 
					     "file_activated" );
		if( current && screem_page_is_file_page( current ) ) {
			page = screem_site_locate_page( site, current );
			if( page ) {
				screem_window_set_document( window, page );
			} else if( screem_site_get_fake_flag( site ) ) {
				screem_page_open_with_filename( site, window,
								current );
			} else {
				g_warning( "%s not in the current site\n",
					current );
			}
		} else if( current && ! screem_uri_is_dir( current ) ) {
			/* non page, open with default gnome app */
			if( ! screem_execute_default_app( current ) ) {
				/* failed */

			}
		}
		g_object_set_data( G_OBJECT( tree_view ),
				   "file_activated", NULL );
	} else {
		/* hmm, no iterator */
		
	}

	return (sel != NULL);
}

static gboolean screem_window_file_view_press( GtkWidget *widget,
					       GdkEventButton *event,
					       ScreemWindow *window )
{
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	gint cx;
	gint cy;

	GtkTreeModel *model;
	GtkTreeIter it;

	ScreemSiteViewNodeInfo *info;
	GString *pathname;
	ScreemSite *site;
	ScreemPage *page;
	GtkWidget *menu;
	EggAction *action;
	gboolean status;

	if( event && event->button != 3 ) {
		return FALSE;
	}

	site = screem_window_get_current( window );

	/* right mouse clicked */
	gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW( widget ), 
				       event->x, event->y, 
				       &path, &column, &cx, &cy );

	if( path ) {
		GtkTreeSelection *selection;

		if( event ) {
			g_signal_stop_emission_by_name( G_OBJECT( widget ),
							"button_press_event" );
		}
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
		gtk_tree_model_get_iter( GTK_TREE_MODEL( model ), &it, path );
		
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
		gtk_tree_selection_select_iter( GTK_TREE_SELECTION( selection ),
						&it );
		
		if( ! screem_site_get_fake_flag( site ) ) {
			gtk_tree_model_get( model, &it, 
					FILE_BROWSER_DATA_COL, &info,
					-1 );
			pathname = g_string_new( info->fullname );
		} else {
			gchar *tmp;
			
			gtk_tree_model_get( model, &it, 
					FILE_BROWSER_URI_COL, &tmp,
					-1 );
			pathname = g_string_new( tmp );
			info = NULL;
		}
		page = screem_site_locate_page( site, pathname->str );

		/* set upload status toggle buttons */
		menu = egg_menu_merge_get_widget( EGG_MENU_MERGE( window->merge ), "/popups/siteviewmenu" );
							
		status = screem_site_is_excluded( site, pathname->str );
		action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ), "ExcludeFlag" );
		egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ),
						status );
		g_object_set( G_OBJECT( action ), "sensitive",
				( info != NULL ), NULL );

		status = screem_site_is_ignored( site, pathname->str );
		action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ), "IgnoreFlag" );
		egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ),
						status );
		g_object_set( G_OBJECT( action ), "sensitive",
				( info != NULL ), NULL );
	
		status = screem_site_is_ascii( site, pathname->str );
		action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ), "ASCIIFlag" );
		egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ),
						status );
		g_object_set( G_OBJECT( action ), "sensitive",
				( info != NULL ), NULL );
						
		gnome_popup_menu_do_popup_modal( menu, 0, 0, event,
						 0, widget );

		if( info ) {
			action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ), "ExcludeFlag" );
			if( egg_toggle_action_get_active( EGG_TOGGLE_ACTION( action ) ) ) {
				screem_site_add_exclude( site, pathname->str );
			} else {
				screem_site_remove_exclude( site, 
							    pathname->str );
			}
			action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ), "IgnoreFlag" );
			if( egg_toggle_action_get_active( EGG_TOGGLE_ACTION( action ) ) ) {
				screem_site_add_ignore( site, pathname->str );
			} else {
				screem_site_remove_ignore( site, 
							   pathname->str );
			}
			action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ), "ASCIIFlag" );
			if( egg_toggle_action_get_active( EGG_TOGGLE_ACTION( action ) ) ) {
				screem_site_add_ascii( site, pathname->str );
			} else {
				screem_site_remove_ascii( site, 
							  pathname->str );
			}
		}
		g_string_free( pathname, TRUE );

		gtk_tree_path_free( path );
	}
	
	return ( path != NULL );
}

static void screem_window_mdi_change( ScreemMDI *mdi, ScreemPage *page,
					ScreemWindow *window )
{
	gboolean enable;
	static const gchar* elements[] = {
		"Save Document",
		"Save Document As",
		"Save Template",
		"Close Document",
		"CloseAll",
		
		"Print",
		"PrintPreview",

		"Undo", "Redo",
		"Cut", "Copy",
		"Paste", "PasteEncoded",
		"Clear", "SelectAll",
		"SelectContext",
		"Find","FindAgain", "Replace", 
		"GotoLine",
		"Bookmarks",

		"Auto Indent", "Indent", "UnIndent",
		"Encode Entities", "URL Encode",
		"TagLower", "TagUpper",
		
		"MarkEditable",
			
		"SpellCheck",
		"Browser0", "Browser1",
		"Browser2", "Browser3",

		"Bold", "Italic",
		"Underline", "Para",
		"Bullet", "Numbered",
		"AlignLeft", "AlignCentre",
		"AlignRight", "Justify",
		"Font", "Pre",
		"Subscript", "Superscript",
		"Table", "TBody",
		"Tr", "Th", "Td",
		"Caption", "Summary",
		"Textarea", "Input",
		"Fileinput", "Button",
		"Checkbutton", "Radiobutton",
		"Optionmenu",
	
		"Basic Structure",
		"Generator", "Author",
		"Content-Type", "MetaRefresh",
		
		"InsertDoctype", "InsertFile",
			
		"CommitPage", "UpdatePage",

		"TemplateUpdate",
		NULL
	};
	gint i;
	EggActionGroup *group;
	
	enable = ( page != NULL );
	group = window->action_group;
	for( i = 0; elements[ i ]; ++ i ) {
		EggAction *action;
			
		action = egg_action_group_get_action( group,
						elements[ i ] );
		g_object_set( G_OBJECT( action ), 
				"sensitive", enable, NULL );
	}
	screem_window_display_view( window );
	if( enable ) {
		const gchar *pathname;
		gchar *title;
		gchar *temp;

		pathname = screem_page_get_pathname( page );
		if( ! pathname ) {
			pathname = _( "Untitled" );
		}

		temp = gnome_vfs_unescape_string_for_display( pathname );
		if( screem_page_get_changed( page ) ) {
			title = g_strdup_printf( "%s * - Screem", temp );
		} else {
			title = g_strdup_printf( "%s - Screem", temp );
		}
		gtk_window_set_title( GTK_WINDOW( window ),
				      title );
		g_free( temp );
		g_free( title );
	} else {
		gtk_window_set_title( GTK_WINDOW( window ),
				      "Screem" );
	}
	if( window->details->tagtree ) {
		g_object_set( G_OBJECT( window->details->tagtree ), 
				"page", page, NULL );
	}
}

static void screem_window_highlight_spelling( ScreemSpell *spell,
					      gint start, gint end,
					      ScreemWindow *window )
{
	ScreemEditor *editor;

	editor = SCREEM_EDITOR( window->details->editor );

	screem_editor_set_pos( editor, start );
	screem_editor_select_region( editor, start, end - start );
}

static void screem_window_select_context_callback( ScreemTreeView *view, 
						   guint start,
						   guint end, gpointer data )
{
	ScreemWindow *window;
	ScreemEditor *editor;

	window = SCREEM_WINDOW( data );
	editor = SCREEM_EDITOR( window->details->editor );

	if( GTK_WIDGET( editor ) == window->details->current ) {
		screem_editor_select_region( editor, start,
					     end - start );
		gtk_widget_grab_focus( GTK_WIDGET( editor ) );
	}
}

static void screem_window_view_change( RBSidebarButton *button, 
				       ScreemWindow *window )
{
	EggAction *action;
	EggActionGroup *group;

	group = window->action_group;
	
	action = egg_action_group_get_action( group, button->unique_id );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( button ) ) ) {
		egg_action_activate( action );
	}
}

static void
screem_window_switch_site_destroy( gpointer data, GClosure *closure )
{
	SwitchSiteData *sd = ( SwitchSiteData * ) data;

	g_return_if_fail( sd );

	g_free( sd );
}

static GtkWidget *screem_window_get_site_button( ScreemWindow *window,
						  ScreemSite *site )
{
	RBSidebarButton *button;
	GSList *sites;
	const gchar *sitename;
	
	sitename = screem_site_get_name( site );

	button = NULL;
     	for( sites = window->details->sitebargroup; sites;
	     sites = sites->next ) {
		button = RB_SIDEBAR_BUTTON( sites->data );
		if( ! strcmp( sitename, button->unique_id ) ) { 
			break;
		}
	}

	g_assert( button );

	return GTK_WIDGET( button );
}

static void screem_window_site_action( EggAction *action,
					SwitchSiteData *sd )
{
	GtkWidget *button;
	
	button = screem_window_get_site_button( sd->window, sd->site );
	
	if( ! gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( button ) ) ) {
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( button ),
					      TRUE );
	}
}

static void screem_window_site_switch( RBSidebarButton *button,
				       SwitchSiteData *sd )
{
	GtkToggleButton *tb;

	tb = GTK_TOGGLE_BUTTON( button );
	if( gtk_toggle_button_get_active( tb ) ) {
		screem_window_set_current( sd->window, sd->site );
	}
}

static void screem_window_drop( GtkWidget *widget, GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *selectionData,
				guint info, guint time, ScreemWindow *window )
{
	ScreemApplication *app;
	ScreemSite *dsite;
	ScreemSite *site;
	gboolean handled;
	GList *uris;
	GList *tmp;
	gchar *text;

	handled = FALSE;

	app = window->application;

	if( info == TARGET_MOZ ) {
		glong read;
		glong written;

		text = g_utf16_to_utf8( (const gunichar2*)selectionData->data,
					selectionData->length,
					&read, &written,
					NULL );
	} else {
		text = g_strdup( selectionData->data );
	}

	dsite = screem_application_get_default_site( app );
	
	switch( info ) {
	case TARGET_MOZ:
	case TARGET_URL:
	case TARGET_URI_LIST:
		uris = gnome_vfs_uri_list_parse( text );
		for( tmp = uris; tmp; tmp = tmp->next ) {
			GnomeVFSURI *uri;
			gchar *url;

			uri = (GnomeVFSURI*)tmp->data;
			url = gnome_vfs_uri_to_string( uri, 0 );

			if( ( ! window->details->offline ) ||
			    gnome_vfs_uri_is_local( uri ) ) {
				handled = TRUE;
				site = screem_window_get_current( window );
				
				if( ! screem_site_locate_page( site, url ) ) {
					site = dsite;
				}
				screem_page_open_with_filename( site, window,
								url );
			}
			
			gnome_vfs_uri_unref( uri );
		}
		g_list_free( uris );
		break;
	default:
		/* we don't know about this type, and as such
		   this case should never execute */
		screem_window_show_error( SCREEM_WINDOW( window ),
					  _("Error: invalid drop type for window occured\n") );
		break;
	}

	g_free( text );

	gtk_drag_finish( context, handled, FALSE, time );
}

static void screem_window_layout_changed( EggDock *dock, ScreemWindow *window )
{
	gchar *file;
	gchar *tmp;

	tmp = screem_get_dot_dir();
	file = g_strconcat( tmp, G_DIR_SEPARATOR_S,
			    "layout", NULL );
	g_free( tmp );
	egg_dock_layout_save_layout( EGG_DOCK_LAYOUT( window->layout ),
				     "__default__" );
	egg_dock_layout_save_to_file( EGG_DOCK_LAYOUT( window->layout ), file );
	g_free( file );
}

static void screem_window_item_parent_set( GtkWidget *widget, GtkWidget *prev_parent,
					 GtkWidget *label )
{
	GtkWidget *parent;

	parent = gtk_widget_get_parent( widget );
	
	if( GTK_IS_NOTEBOOK( parent ) ) {
		gtk_widget_hide( label );
	} else {
		gtk_widget_show( label );
	}
}

static void screem_window_search_found( ScreemSearch *search,
					ScreemPage *page,
					guint pos,
					guint length,
					gpointer data )
{
	ScreemWindow *window;
	ScreemEditor *editor;
	
	window = SCREEM_WINDOW( data );
	
	editor = SCREEM_EDITOR( window->details->editor );

	screem_editor_set_pos( editor, pos );
	screem_editor_select_region( editor, pos, length );
}

static void screem_window_search_notfound( ScreemSearch *search,
					   ScreemPage *page,
					   gpointer data )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( data );
}

static gboolean screem_window_ctags_jumpto_idle( ScreemWindow *window )
{
	ScreemEditor *editor;
	ScreemPage *page;
	gchar *pattern;
	gchar *text;
	gchar *found;	
	guint len;
	guint pos;
	
	editor = SCREEM_EDITOR( window->details->editor );

gdk_threads_enter();
	
	page = screem_window_get_document( window );
	pattern = g_object_get_data( G_OBJECT( window ),
			"ctagspattern" );
	
	text = screem_page_get_data( page );
	found = find_text( text, pattern, NULL, &len );
	if( found ) {
		pos = found - text;
		screem_editor_set_pos( editor, pos );
		screem_editor_select_region( editor, pos, len );
	} else {
		g_print( "FAILED TO FIND: %s\n", pattern );
	}
	g_free( pattern );
	g_free( text );

	gdk_threads_leave();
	
	return FALSE;
}
						
static void screem_window_ctags_jumpto( ScreemCtagsView *view,
					const gchar *pathname,
					const gchar *pattern,
					guint line,
					gpointer data )
{
	ScreemWindow *window;
	ScreemSite *site;
	ScreemPage *page;
	ScreemEditor *editor;
	gchar *tmp;

	window = SCREEM_WINDOW( data );
	site = screem_window_get_current( window );
	editor = SCREEM_EDITOR( window->details->editor );

	if( screem_page_open_with_filename( site, window, pathname ) ) {
		page = screem_site_locate_page( site, pathname );

		if( ! pattern ) {
			/* jump to line */
			screem_editor_goto_line( editor, line );
		} else {
			tmp = screem_support_escape_regexp( pattern );
			g_object_set_data( G_OBJECT( window ),
					"ctagspattern", tmp );
			g_idle_add( (GSourceFunc)screem_window_ctags_jumpto_idle, window );

		}
	}
}

/* GObject Stuff */

#define PARENT_TYPE GTK_TYPE_WINDOW

static gpointer parent_class;

static void
screem_window_initialise_class( ScreemWindowClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = G_OBJECT_CLASS( klass );
	widget_class = (GtkWidgetClass *)klass;
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_window_finalize;
	object_class->get_property = screem_window_get_prop;
	object_class->set_property = screem_window_set_prop;

	widget_class->show = screem_window_show;

	g_object_class_install_property(object_class,
					ARG_APP_ID,
					g_param_spec_string("app_id",
							    "Application ID",
							    "The ID",
							    "",
							    G_PARAM_READWRITE)
					);

	g_object_class_install_property(object_class,
					ARG_APP,
					g_param_spec_object("app",
							    "Application",
							    "The Application",
							    G_TYPE_OBJECT,
							    G_PARAM_READWRITE |
							    G_PARAM_CONSTRUCT)
					);

	widget_class->realize = screem_window_realize;
	widget_class->size_request = screem_window_size_request;
}

static void
screem_window_initialise( ScreemWindow *window )
{
	ScreemWindowDetails *details;
	
	details = window->details = g_new0( ScreemWindowDetails, 1 );

	/* as we only ever open 1 window at the moment we
	   don't need to gen a unique one yet */
	details->name = g_strdup( "fixme" );
	
	details->spell = screem_spell_new();
	/* hookup the highlight signal so we can scroll the editor to
	   show it */
	g_signal_connect( G_OBJECT( details->spell ),
			  "highlight",
			  G_CALLBACK( screem_window_highlight_spelling ),
			  window );

	details->cvs = screem_cvs_new( window );

	gtk_drag_dest_set( GTK_WIDGET( window ),
			   GTK_DEST_DEFAULT_MOTION | 
			   GTK_DEST_DEFAULT_HIGHLIGHT |
			   GTK_DEST_DEFAULT_DROP,
			   drop_types, num_drop_types,
			   GDK_ACTION_COPY );
	
	g_signal_connect( G_OBJECT( window ), "drag_data_received",
			  G_CALLBACK( screem_window_drop ), window );

}

static void
screem_window_set_prop( GObject *object, guint prop_id, 
			const GValue *value, GParamSpec *spec )
{
	const gchar *name;
	gchar *old_name;
	GObject *obj;
	ScreemWindow *window;

	window = SCREEM_WINDOW( object );

	switch( prop_id ) {
	case ARG_APP_ID:
		if( ! G_VALUE_HOLDS_STRING( value ) )
			return;
		/* FIXME: set the name */
		screem_window_constructed( SCREEM_WINDOW( object ) );
		break;
	case ARG_APP:
		obj = g_value_get_object( value );
		window->application = SCREEM_APPLICATION( obj );
		break;
	}
}

static void
screem_window_get_prop( GObject *object, guint prop_id, 
			GValue *value, GParamSpec *spec )
{
	GObject *obj;
	gchar *name;

	switch( prop_id ) {
	case ARG_APP_ID:
		/* FIXME set the name */
		g_value_set_string( value, "" );
		break;
	case ARG_APP:
		obj = G_OBJECT( SCREEM_WINDOW( object )->application );
		g_value_set_object( value, obj );
		break;
	}
}

static void
screem_window_finalize( GObject *object )
{
	ScreemWindow *window;
	ScreemWindowDetails *details;

	window = SCREEM_WINDOW( object );
	details = window->details;

	g_object_unref( window->search );

	if( details->status_timeout ) {
		g_source_remove( details->status_timeout );
	}
	
	if( details->name ) {
		g_free( details->name );
	}

	g_object_unref( window->layout );

	g_object_unref( details->mdi );

	/* remove initial references */
	g_object_unref( details->tagtree );

	g_object_unref( details->editor );
	g_object_unref( details->preview );
	g_object_unref( details->link_view );
	g_object_unref( details->tree_view );
	
	g_object_unref( details->cvs );
	g_object_unref( details->spell );

	g_object_unref( details->recent_pages );
	g_object_unref( details->recent_sites );

	g_object_unref( details->ctagsmodel );
	
	g_free( details );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}


static void
screem_window_realize( GtkWidget *widget )
{
	/* Create GdkWindow */
	GTK_WIDGET_CLASS( parent_class )->realize( widget );
}

static void
screem_window_size_request( GtkWidget *widget, GtkRequisition *req )
{
	g_return_if_fail( SCREEM_IS_WINDOW( widget ) );
	g_return_if_fail( req != NULL );

	GTK_WIDGET_CLASS( parent_class )->size_request( widget, req );
}

static void
screem_window_show( GtkWidget *widget )
{
	ScreemWindow *window;

	window = SCREEM_WINDOW( widget );
	
	GTK_WIDGET_CLASS( parent_class )->show( widget );
}

GType screem_window_get_type()
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemWindowClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_window_initialise_class,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemWindow ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_window_initialise
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemWindow",
					       &info, 0 );
	}

	return type;
}
