/*  Screem:  screem-view.c
 *
 *  The base view widget
 *
 *  Copyright (C) 2002  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include "screem-window.h"
#include "screem-site.h"
#include "screem-page.h"

#include "screem-view.h"

enum {
	PROP_0,
	PROP_WINDOW,
	PROP_SITE,
	PROP_PAGE,
	PROP_ONLINE
};

static void screem_view_class_init( ScreemViewClass *klass );
static void screem_view_init( ScreemView *view );
static void screem_view_finalize( GObject *view );
static void screem_view_set_prop( GObject *object, guint property_id,
				  const GValue *value, 
				  GParamSpec *pspec );
static void screem_view_get_prop( GObject *object, guint property_id,
				  GValue *value, GParamSpec *pspec );
static void screem_view_size_request( GtkWidget *widget,
                           		GtkRequisition *requisition );
static void screem_view_size_allocate( GtkWidget *widget,
                              		GtkAllocation *allocation );

struct ScreemViewPrivate {
	ScreemWindow *window;

	ScreemPage *page;

	gboolean online;
};

void screem_view_display( ScreemView *view )
{
	if( view->display ) {
		view->display( view );
	}
}

void screem_view_print( ScreemView *view, gboolean preview )
{
	if( view->print ) {
		view->print( view, preview );
	}
}

void screem_view_undo( ScreemView *view )
{
	if( view->undo ) {
		view->undo( view );
	}
}

void screem_view_redo( ScreemView *view )
{
	if( view->redo ) {
		view->redo( view );
	}
}

/* G Object stuff */
#define PARENT_TYPE GTK_TYPE_BIN

static gpointer parent_class;

static void screem_view_class_init( ScreemViewClass *klass )
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	GParamSpec *pspec;

	object_class = G_OBJECT_CLASS( klass );
	widget_class = (GtkWidgetClass *)klass;
	parent_class = g_type_class_peek_parent( klass );


	object_class->finalize = screem_view_finalize;
	object_class->get_property = screem_view_get_prop;
	object_class->set_property = screem_view_set_prop;

	widget_class->size_request = screem_view_size_request;
	widget_class->size_allocate = screem_view_size_allocate;
	
	pspec = g_param_spec_pointer( "window", "window",
				      "window",
				      G_PARAM_READABLE |
				      G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_WINDOW,
					 pspec );
	pspec = g_param_spec_object( "site", "site",
				     "site",
				     SCREEM_TYPE_SITE,
				     G_PARAM_READABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_SITE,
					 pspec );
	pspec = g_param_spec_object( "page", "page",
				     "page",
				     SCREEM_TYPE_PAGE,
				     G_PARAM_READABLE |
				     G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_PAGE,
					 pspec );
	pspec = g_param_spec_boolean( "online", "online",
				      "online",
				      FALSE,
				      G_PARAM_READABLE |
				      G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_ONLINE,
					 pspec );
}

static void screem_view_init( ScreemView *view )
{
	view->private = g_new0( ScreemViewPrivate, 1 );
}

static void screem_view_finalize( GObject *screem_view )
{
	ScreemView *view;

	view = SCREEM_VIEW( screem_view );

	g_free( view->private );

	G_OBJECT_CLASS( parent_class )->finalize( screem_view );
}

static void screem_view_set_prop( GObject *object, guint property_id,
				  const GValue *value, GParamSpec *pspec )
{
	ScreemView *view;
	
	view = SCREEM_VIEW( object );
	
	switch( property_id ) {
	case PROP_WINDOW:
		view->private->window = 
			SCREEM_WINDOW( g_value_get_pointer( value ) );
		if( view->window_set ) {
			view->window_set( view );
		}
		break;
	case PROP_PAGE:
		view->private->page = SCREEM_PAGE( g_value_get_object(value) );
		break;
	case PROP_ONLINE:
		view->private->online = g_value_get_boolean( value );
		break;
	}
}

static void screem_view_get_prop( GObject *object, guint property_id,
				  GValue *value, GParamSpec *pspec)
{
	ScreemView *view;
	ScreemWindow *window;
	ScreemSite *site;
	ScreemPage *page;

	view = SCREEM_VIEW( object );

	window = view->private->window;
	if( window ) {
		site = screem_window_get_current( window );
		page = screem_window_get_document( window );
	} else {
		site = NULL;
		page = view->private->page;
	}

	switch( property_id ) {
	case PROP_WINDOW:
		g_value_set_pointer( value, view->private->window );
		break;
	case PROP_SITE:
		g_value_set_object( value, site );
		break;
	case PROP_PAGE:
		g_value_set_object( value, page );
		break;
	case PROP_ONLINE:
		g_value_set_boolean( value, view->private->online );
		break;
	}
}

static void screem_view_size_request( GtkWidget *widget,
                           		GtkRequisition *requisition )
{
	GtkBin *bin;

	bin = GTK_BIN (widget);

	requisition->width = GTK_CONTAINER( widget )->border_width * 2;
	requisition->height = GTK_CONTAINER( widget )->border_width * 2;

	if( bin->child && GTK_WIDGET_VISIBLE( bin->child ) ) {
		GtkRequisition child_requisition;

		gtk_widget_size_request( bin->child, &child_requisition );
		requisition->width += child_requisition.width;
		requisition->height += child_requisition.height;
	}
}

static void screem_view_size_allocate( GtkWidget *widget,
                            		GtkAllocation *allocation )
{
	GtkBin *bin;
	GtkAllocation child_allocation;

	bin = GTK_BIN( widget );
	widget->allocation = *allocation;
	
	if( bin->child ) {
	child_allocation.x = allocation->x + 
				GTK_CONTAINER( widget )->border_width; 
	child_allocation.y = allocation->y + 
				GTK_CONTAINER (widget)->border_width;
	child_allocation.width = MAX( allocation->width - 
				      GTK_CONTAINER( widget )->border_width * 2,
					0);
	child_allocation.height = MAX( allocation->height - 
				       GTK_CONTAINER (widget)->border_width * 2,
					0);
	gtk_widget_size_allocate( bin->child, &child_allocation );
	}
}

GType screem_view_get_type()
{
	static guint type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemViewClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_view_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemView ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_view_init
		};
		
		type = g_type_register_static( PARENT_TYPE,
					       "ScreemView",
					       &info, 0 );
	}

	return type;
}
