/*  Screem:  screem-site-druid.c,
 *  A site druid for guiding users through the process of creating a site
 *
 *  Copyright (C) 2002  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <pwd.h>
#include <unistd.h>
#include <sys/types.h>

#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>

#include <libgnome/gnome-i18n.h>

#include <libgnomeui/gnome-file-entry.h>
#include <libgnomeui/gnome-entry.h>

#include <glade/glade.h>


#include "screem-application.h"
#include "screem-window.h"
#include "screem-site.h"
#include "screem-page.h"

#include "screem-site-druid.h"

void screem_site_druid_finish( GtkWidget *widget );
void screem_site_druid_cancel( GtkWidget *widget );

void screem_site_druid_switch_page( GtkNotebook *book,
				    GtkNotebookPage *page,
				    guint page_num );

void screem_site_druid_init_publish_1( GtkWidget *widget );

static void screem_site_druid_create_site( ScreemSite *site, GladeXML *xml );

static gboolean screem_site_druid_create_initial_page( ScreemWindow *window );

/* MUST have an entry in the array for each page in the druid,
   use NULL if there is to be no init function */
typedef void (*InitPage)( GtkWidget *widget );
static const InitPage init_funcs[] = {
	NULL,
	NULL,
	screem_site_druid_init_publish_1,
	NULL
};


void screem_site_druid_begin( ScreemApplication *application, 
			      ScreemWindow *window )
{
	GladeXML *xml;
	gchar *glade_path;
	GtkWidget *druid;
	GtkWidget *book;
	GtkWidget *image;
	ScreemSite *site;

	glade_path = gconf_client_get_string( application->client,
					      "/apps/screem/general/glade_path",
					      NULL );
	xml = glade_xml_new( glade_path, "sitedruid", NULL );

	druid = glade_xml_get_widget( xml, "sitedruid" );
	site = screem_site_new( G_OBJECT( application ) );

	g_object_set_data( G_OBJECT( druid ), "site", site );
	g_object_set_data( G_OBJECT( druid ), "application", application );
	g_object_set_data( G_OBJECT( druid ), "window", window );

	book = glade_xml_get_widget( xml, "druid-notebook" );
	gtk_notebook_set_show_tabs( GTK_NOTEBOOK( book ), FALSE );

	image = glade_xml_get_widget( xml, "druid-logo" );
	gtk_image_set_from_file( GTK_IMAGE( image ),
				 DATADIR"/pixmaps/screem.png" );

	glade_xml_signal_autoconnect( xml );

	gtk_window_set_transient_for( GTK_WINDOW( druid ),
				      GTK_WINDOW( window ) );

	screem_site_druid_switch_page( GTK_NOTEBOOK( book ),
				       NULL, 0);
}


void screem_site_druid_finish( GtkWidget *widget )
{
	ScreemSite *site;
	ScreemWindow *window;
	ScreemApplication *application;
	GladeXML *xml;
	GtkWidget *book;

	xml = glade_get_widget_tree( GTK_WIDGET( widget ) );
	widget = glade_xml_get_widget( xml, "sitedruid" );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site" ) );
	application = SCREEM_APPLICATION( g_object_get_data( G_OBJECT( widget),
							     "application" ) );
	window = SCREEM_WINDOW( g_object_get_data( G_OBJECT( widget ),
						   "window" ) );

	screem_site_druid_create_site( site, xml );

	if( screem_site_create( site ) ) {
		screem_application_add_site( application, site );
		screem_window_set_current( window, site );
		g_idle_add_full( G_PRIORITY_LOW,
				 (GSourceFunc)screem_site_druid_create_initial_page,
			   	 window, NULL );
	} else {
		GtkWidget *alert;

		alert = gtk_message_dialog_new( GTK_WINDOW( window ),
						GTK_DIALOG_MODAL |
						GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						_( "Failed to create %s" ),
						screem_site_get_name( site ) );
		gtk_dialog_run( GTK_DIALOG( alert ) );
		gtk_widget_destroy( alert );
		g_object_unref( site );
	}

	book = glade_xml_get_widget( xml, "druid-notebook" );
	g_signal_handlers_disconnect_by_func( G_OBJECT( book ),
					      screem_site_druid_switch_page,
					      NULL );

	gtk_widget_destroy( widget );
}

void screem_site_druid_cancel( GtkWidget *widget )
{
	ScreemSite *site;
	GladeXML *xml;
	GtkWidget *druid;
	GtkWidget *book;

	xml = glade_get_widget_tree( widget );
	druid = glade_xml_get_widget( xml, "sitedruid" );
	
	site = SCREEM_SITE( g_object_get_data( G_OBJECT( druid ), "site" ) );
	g_object_unref( G_OBJECT( site ) );

	book = glade_xml_get_widget( xml, "druid-notebook" );
	g_signal_handlers_disconnect_by_func( G_OBJECT( book ),
					      screem_site_druid_switch_page,
					      NULL );

	gtk_widget_destroy( druid );
}

void screem_site_druid_switch_page( GtkNotebook *book,
				    GtkNotebookPage *page,
				    guint page_num )
{
	GladeXML *xml;

	GtkWidget *back;
	GtkWidget *forward;
	GtkWidget *apply;
	GtkWidget *current;

	xml = glade_get_widget_tree( GTK_WIDGET( book ) );

	back = glade_xml_get_widget( xml, "back" );
	forward = glade_xml_get_widget( xml, "forward" );
	apply = glade_xml_get_widget( xml, "apply" );

	if( page_num == 0 ) {
		gtk_widget_set_sensitive( apply, FALSE );
		gtk_widget_set_sensitive( back, FALSE );
		gtk_widget_set_sensitive( forward, TRUE );
	} else if( page_num == g_list_length( book->children ) - 1 ) {
		gtk_widget_set_sensitive( apply, TRUE );
		gtk_widget_set_sensitive( back, TRUE );
		gtk_widget_set_sensitive( forward, FALSE );
	} else {
		gtk_widget_set_sensitive( apply, FALSE );
		gtk_widget_set_sensitive( back, TRUE );
		gtk_widget_set_sensitive( forward, TRUE );
	}

	/* call init function for page if needed */
	current = gtk_notebook_get_nth_page( book, page_num );
	if( ! g_object_get_data( G_OBJECT( current ), "inited" ) ) {

		if( init_funcs[ page_num ] ) {
			init_funcs[ page_num ]( current );
		}

		g_object_set_data( G_OBJECT( current ), "inited",
				   GINT_TO_POINTER( 1 ) );
	}
}

void screem_site_druid_init_publish_1( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( GTK_WIDGET( widget ) );

	widget = glade_xml_get_widget( xml, "username" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	gtk_entry_set_text( GTK_ENTRY( widget ), g_get_real_name() );
}

static void screem_site_druid_create_site( ScreemSite *site, GladeXML *xml )
{
	GtkWidget *widget;
	const gchar *value;
	gboolean flag;
	UploadMethods method;
	gint port;
	ScreemSitePerms sp;
	ScreemSiteSymlinks sl;
	
	/* process details page */
	widget = glade_xml_get_widget( xml, "sitename" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_name( site, value );

	widget = glade_xml_get_widget( xml, "sitepath" );
	widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_pathname( site, value );

	widget = glade_xml_get_widget( xml, "usesitetemplate" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "sitetemplate" );
		widget = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(widget));
		value = gtk_entry_get_text( GTK_ENTRY( widget ) );
		screem_site_set_site_template( site, value );
	}

	widget = glade_xml_get_widget( xml, "usesitecvs" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "sitecvs" );
		widget = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(widget));
		value = gtk_entry_get_text( GTK_ENTRY( widget ) );
		screem_site_set_cvs_root( site, value );
	}
	widget = glade_xml_get_widget( xml, "usepagetemplate" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "sitepagetemplate" );
		widget = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(widget));
		value = gtk_entry_get_text( GTK_ENTRY( widget ) );
		screem_site_set_template_path( site, value );
	}

	/* process publish settings 1 page */
	method = LOCAL;
	widget = glade_xml_get_widget( xml, "ftp" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		method = FTP;
	widget = glade_xml_get_widget( xml, "webdav" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		method = WEBDAV;
	widget = glade_xml_get_widget( xml, "rcp" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		method = RSH;
	widget = glade_xml_get_widget( xml, "scp" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		method = SSH;
	screem_site_set_remote_method( site, method );

	widget = glade_xml_get_widget( xml, "hostname" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "port" );
	port = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
	{
		gchar *fullhost;

		fullhost = g_strdup_printf( "%s:%i", value, port );
		screem_site_set_remote_url( site, fullhost );
		g_free( fullhost );
	}

	widget = glade_xml_get_widget( xml, "remotepath" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_remote_path( site, value );

	widget = glade_xml_get_widget( xml, "baseuri" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_http_url( site, value );

	widget = glade_xml_get_widget( xml, "username" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_remote_user( site, value );

	widget = glade_xml_get_widget( xml, "password" );
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_remote_pass( site, value );

	/* process publish settings 2 page */
	widget = glade_xml_get_widget( xml, "passiveftp" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_passive_ftp( site, flag );

	widget = glade_xml_get_widget( xml, "nodelete" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_no_delete( site, flag );

	widget = glade_xml_get_widget( xml, "checkmoved" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_check_moved( site, flag );

	widget = glade_xml_get_widget( xml, "nooverwrite" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_no_overwrite( site, flag );

	sp = PERMS_IGNORE;
	widget = glade_xml_get_widget( xml, "p_exec" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		sp = PERMS_EXEC;
	widget = glade_xml_get_widget( xml, "p_all" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		sp = PERMS_ALL;
	screem_site_set_permissions( site, sp );

	sl = SYM_IGNORE;
	widget = glade_xml_get_widget( xml, "s_follow" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		sp = SYM_FOLLOW;
	widget = glade_xml_get_widget( xml, "s_maintain" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		sp = SYM_MAINTAIN;
	screem_site_set_symlinks( site, sl );
}

static gboolean screem_site_druid_create_initial_page( ScreemWindow *window )
{
	GConfClient *client;
	ScreemApplication *app;
	ScreemSite *site;
	ScreemPage *page;
	const gchar *template;
	gchar *default_file;
	gchar *tmp;

	client = gconf_client_get_default();
	
	site = screem_window_get_current( window );
	g_object_get( G_OBJECT( site ), "app", &app, NULL );
	
	tmp = gconf_client_get_string( client,
				       "/apps/screem/default_filename",
				       NULL );
	if( ! tmp ) {
		tmp = g_strdup( "index.html" );
	}

	default_file = g_strconcat( screem_site_get_pathname( site ),
				    "/", tmp, NULL );
	g_free( tmp );

	page = screem_page_new( G_OBJECT( app ) );

	template = screem_site_get_template_path( site );

	if( template ) {
		screem_page_set_pathname( page, template );
		screem_page_load( page );
	}
	       
	screem_page_set_pathname( page, default_file );
	screem_page_set_changed( page, TRUE );
gdk_threads_enter();
	if( screem_page_save( page ) ) {

		screem_site_add_page( site, page );

		screem_window_set_document( window, page );
	} else {
		/* failed to save the page */
		GtkWidget *alert;

		alert = gtk_message_dialog_new( GTK_WINDOW( window ),
						GTK_DIALOG_MODAL |
						GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_MESSAGE_WARNING,
						GTK_BUTTONS_OK,
						_( "Failed to create initial page %s" ),
						default_file );
		gtk_dialog_run( GTK_DIALOG( alert ) );
		gtk_widget_destroy( alert );
	}
	gdk_threads_leave();
	g_free( default_file );
	g_object_unref( client );
	g_object_unref( app );

	return FALSE;
}
