/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * This file incorporates work covered by the following copyright and
 * permission notice:
 *
 *  Copyright 2006 Maurizio Monge <maurizio.monge@gmail.com>
 *
 *  BSD License
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Attention: note that even being named "BSD License" by the
 * copyright holder, the license above is a variation of the license
 * known as BSD License.  This variation is compatible with GNU GPL.
 */

#include <QDebug>
#include <QRegion>
#include <QPainter>
#include <QPaintEvent>

#include "qziongroup.h"
#include "qzionobject_p.h"
#include "qzionrectangle.h"
#include "qzionabstractcanvas_p.h"

QZionGroup::QZionGroup(QZionAbstractCanvas *canvas)
    : QZionObject(canvas), QZionAbstractCanvas(), _childRectChanged(true)
{
}

QZionGroup::~QZionGroup()
{
    QZDP(QZionAbstractCanvas, da);
    foreach (QZionObject *obj, da->items) {
        delete obj;
    }
}

void QZionGroup::ensurePendingUpdate()
{
    if(!_changed || !_childRectChanged) {
        _childRectChanged = true;
        QZionObject::changed();
    }
}

void QZionGroup::updateChanges()
{
    if(!_changed)
        return;

    QZDP(QZionAbstractCanvas, da);
    foreach (QZionObject *obj, da->items) {
        if (obj->_changed)
            obj->updateChanges();
    }

    _changed = false;
}

void QZionGroup::changed()
{
    QZionObject::changed();
    QZDP(QZionAbstractCanvas, da);
    foreach (QZionObject *obj, da->items)
        obj->changed();
}

void QZionGroup::invalidate(const QRect &r, bool translate)
{
    QZD(QZionObject);

    if(d->canvas)
        d->canvas->invalidate(translate ? r.translated(pos()) : r, translate);
    if(!_changed)
        ensurePendingUpdate();
}

void QZionGroup::invalidate(const QRegion &r, bool translate)
{
    QZD(QZionObject);

    if(d->canvas)
        d->canvas->invalidate(translate ? r.translated(pos()) : r, translate);
    if(!_changed)
        ensurePendingUpdate();
}

void QZionGroup::paintInternal(QPainter *p, const QRect &prect,
                               const QRegion &preg, const QPoint &delta,
                               double cumulativeOpacity)
{
    QZD(QZionObject);

    cumulativeOpacity *= (d->opacity / 255.0);

    p->save();

    QPoint adelta = delta + pos();
    QRegion clipRegion = p->clipRegion();

    if (d->clipper != NULL) {
        QRect effectiveClipper = d->clipper->effectiveRect();
        int effectiveOpacity = d->clipper->effectiveColor().alpha();

        if (p->clipRegion().boundingRect().isNull())
            p->setClipRect(effectiveClipper);
        else
            p->setClipRegion(p->clipRegion() & effectiveClipper);

        cumulativeOpacity *= (effectiveOpacity / 255.0);
    }

    p->translate(pos());

    QZDP(QZionAbstractCanvas, da);
    foreach (QZionObject *obj, da->items) {
        QRect r = obj->rect().translated(adelta);

        if (obj->visible() && prect.intersects(r) && preg.contains(r)) {
            obj->paintInternal(p, prect, preg, adelta, cumulativeOpacity);
        }
    }

    p->restore();
}

void QZionGroup::paint(QPainter *p)
{
    paintInternal(p, rect(), QRegion(rect()), QPoint(), 1.0);
}

QRect QZionGroup::rect() const
{
    if(!_childRectChanged)
        return _lastChildRect.translated(pos());

    _childRectChanged = false;
    _lastChildRect = QRect();

    QZDP(QZionAbstractCanvas, da);
    foreach (QZionObject *obj, da->items) {
        if (obj->visible())
            _lastChildRect |= obj->rect();
    }

    return _lastChildRect.translated(pos());
}

int QZionGroup::canvasWidth() const
{
    QZD(QZionObject);

    return d->canvas ? d->canvas->canvasWidth() : 0;
}

int QZionGroup::canvasHeight() const
{
    QZD(QZionObject);

    return d->canvas ? d->canvas->canvasHeight() : 0;
}

void QZionGroup::setColor(const QColor &color)
{
    _QZionObject_data->opacity = color.alpha();
}

QSize QZionGroup::size() const
{
    return _size;
}

void QZionGroup::setSize(const QSize &size)
{
    QZionObject::setSize(size);
    _size = size;
}

void QZionGroup::setClipper(QZionRectangle *rect)
{
    QZionObject::setClipper(rect);
}

QColor QZionGroup::effectiveColor()
{
    QZionObject::effectiveColor();
}

QPoint QZionGroup::mapToLocal(const QPoint &pt) const
{
    return pt - pos();
}

void QZionGroup::mouseMoveEvent(QMouseEvent *e)
{
    QZionAbstractCanvas::mouseMoveEvent(e);
}

void QZionGroup::mousePressEvent(QMouseEvent *e)
{
    QZionAbstractCanvas::mousePressEvent(e);
}

void QZionGroup::mouseReleaseEvent(QMouseEvent *e)
{
    QZionAbstractCanvas::mouseReleaseEvent(e);
}

void QZionGroup::wheelEvent(QWheelEvent *e)
{
    QZionAbstractCanvas::wheelEvent(e);
}

bool QZionGroup::contains(const QPoint &point) const
{
    // XXX: need to support testAttribute(HandleOutOfBoundsEvents)
    return QZionObject::contains(point);
}
