/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QEdje.
 *
 * QEdje is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QEdje is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with QEdje.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include <QtGui>

#include <qzion.h>
#include <qedje.h>

#include "viewer.h"


Viewer::Viewer(QWidget *parent)
    : QWidget(parent), _canvas(this), _entry_emission(this),
      _entry_source(this), _label_emission("Emission:", this),
      _label_source("Source:", this), _label_groups("Groups:", this),
      _current(NULL)
{
    setWindowTitle("QEdje Viewer");
    QObject::connect(&_entry_emission, SIGNAL(returnPressed()),
                     this, SLOT(emitSignal()));
    QObject::connect(&_entry_source, SIGNAL(returnPressed()),
                     this, SLOT(emitSignal()));
    QObject::connect(&_combo_groups, SIGNAL(currentIndexChanged(int)),
                     this, SLOT(groupChanged(int)));

    QGridLayout *grid = new QGridLayout();

    _label_groups.setBuddy(&_combo_groups);
    grid->addWidget(&_label_groups, 0, 0);
    grid->addWidget(&_combo_groups, 0, 1);

    _label_emission.setBuddy(&_entry_emission);
    grid->addWidget(&_label_emission, 1, 0);
    grid->addWidget(&_entry_emission, 1, 1);

    _label_source.setBuddy(&_entry_source);
    grid->addWidget(&_label_source, 2, 0);
    grid->addWidget(&_entry_source, 2, 1);

    QVBoxLayout *vbox = new QVBoxLayout();
    vbox->addWidget(_canvas.widget(), 1); // canvas will stretch when there's space
    vbox->addLayout(grid, 0);

    setLayout(vbox);
}

bool Viewer::setGroup(const QString &group){
    int groupPos = _combo_groups.findText(group);
    if (groupPos == -1){
        return false;
    }
    _combo_groups.setCurrentIndex(groupPos);
    return true;
}

bool Viewer::load(const QString &filename)
{
    QStringList lst = groupNamesFromFile(filename);

    if (lst.count() <= 0)
        return false;

    _filename = filename;
    _combo_groups.clear();
    _combo_groups.addItems(lst);

    return true;
};

bool Viewer::loadGroup(const QString &group)
{
    if (_current) {
        _current->disconnectEdjeSignal("*", "*",
                                       this, SLOT(debugSignal(const QString &,
                                                              const QString &)));
        delete _current;
    }

    _current = new QEdje(&_canvas, _filename, group);

    if (_current->group().isEmpty())
        return false;

    _current->connectEdjeSignal("*", "*",
                                this, SLOT(debugSignal(const QString &,
                                                       const QString &)));
    _current->show();

    // XXX: enforce minimum/maximum size
    QSize min = _current->propMin();
    if (min == QSize(0, 0))
        min = QSize(200, 200);

    return true;
}

void Viewer::debugSignal(const QString &emission, const QString &source)
{
    std::cout << qPrintable(emission) << " "
              << qPrintable(source) << std::endl;
}

void Viewer::emitSignal()
{
    if (_current) {
        _current->emitEdjeSignal(_entry_emission.text(),
                                 _entry_source.text());
    }
}

void Viewer::groupChanged(int index)
{
    if (index < 0 || _filename.isNull())
        return;

    QString groupName = _combo_groups.itemText(index);
    loadGroup(groupName);
}
