
#include <Python.h>
#include <stdio.h>
#include <math.h>
#include "libnumarray.h"

#ifdef MS_WIN32
#pragma warning(once : 4244)
#endif

#define logical_and(arg1, arg2) (arg1 != 0) & (arg2 != 0)
#define logical_or(arg1, arg2)  (arg1 != 0) | (arg2 != 0)
#define logical_xor(arg1, arg2) ((arg1 != 0) ^ (arg2 != 0)) & 1
#define ufmaximum(arg1, arg2) (((temp1=arg1) > (temp2=arg2)) ? temp1 : temp2)
#define ufminimum(arg1, arg2) (((temp1=arg1) < (temp2=arg2)) ? temp1 : temp2)

static PyObject *_Error;
/*********************  minus  *********************/

static int minus_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = -*tin;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(minus_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  add  *********************/

static int add_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 + *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(add_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int add_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 + tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(add_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int add_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar + *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(add_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _add_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval + *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _add_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int add_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _add_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(add_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _add_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = net + *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _add_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int add_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _add_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(add_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  subtract  *********************/

static int subtract_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 - *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(subtract_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int subtract_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 - tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(subtract_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int subtract_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar - *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(subtract_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _subtract_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval - *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _subtract_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int subtract_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _subtract_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(subtract_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _subtract_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = net - *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _subtract_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int subtract_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _subtract_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(subtract_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/

static int multiply_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 * *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(multiply_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int multiply_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 * tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(multiply_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int multiply_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar * *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(multiply_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _multiply_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval * *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _multiply_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int multiply_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _multiply_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(multiply_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _multiply_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = net * *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _multiply_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int multiply_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _multiply_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(multiply_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  divide  *********************/
/*********************  divide  *********************/

static int divide_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 / *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(divide_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int divide_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 / tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(divide_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int divide_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar / *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(divide_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _divide_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval / *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _divide_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int divide_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _divide_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(divide_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _divide_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = net / *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _divide_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int divide_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _divide_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(divide_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  floor_divide  *********************/
/*********************  floor_divide  *********************/

static int floor_divide_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        NUM_FLOORDIVIDE(*tin1, *tin2, *tout);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(floor_divide_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int floor_divide_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        NUM_FLOORDIVIDE(*tin1, tscalar, *tout);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(floor_divide_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int floor_divide_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        NUM_FLOORDIVIDE(tscalar, *tin2, *tout);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(floor_divide_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _floor_divide_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            NUM_FLOORDIVIDE(lastval ,*tin, *tout);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _floor_divide_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int floor_divide_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _floor_divide_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(floor_divide_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _floor_divide_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            NUM_FLOORDIVIDE(net, *tin, net);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _floor_divide_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int floor_divide_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _floor_divide_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(floor_divide_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  true_divide  *********************/
/*********************  true_divide  *********************/

static int true_divide_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 / *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(true_divide_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int true_divide_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 / tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(true_divide_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int true_divide_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar / *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(true_divide_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _true_divide_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval / *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _true_divide_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int true_divide_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _true_divide_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(true_divide_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _true_divide_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = net / *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _true_divide_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int true_divide_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _true_divide_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(true_divide_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  remainder  *********************/
/*********************  remainder  *********************/

static int remainder_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = fmod(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(remainder_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int remainder_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = fmod(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(remainder_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int remainder_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = fmod(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(remainder_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _remainder_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = fmod(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _remainder_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int remainder_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _remainder_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(remainder_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _remainder_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = fmod(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _remainder_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int remainder_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _remainder_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(remainder_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  power  *********************/

static int power_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = num_pow(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(power_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int power_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = num_pow(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(power_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int power_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = num_pow(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(power_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _power_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = num_pow(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _power_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int power_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _power_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(power_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _power_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = num_pow(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _power_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int power_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _power_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(power_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  abs  *********************/

static int abs_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = fabs(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(abs_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  sin  *********************/

static int sin_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sin(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(sin_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  cos  *********************/

static int cos_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = cos(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(cos_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  tan  *********************/

static int tan_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = tan(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(tan_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  arcsin  *********************/

static int arcsin_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = asin(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arcsin_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  arccos  *********************/

static int arccos_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = acos(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arccos_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  arctan  *********************/

static int arctan_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = atan(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arctan_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  arctan2  *********************/

static int arctan2_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = atan2(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(arctan2_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int arctan2_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = atan2(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(arctan2_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int arctan2_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = atan2(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(arctan2_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _arctan2_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = atan2(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _arctan2_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int arctan2_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _arctan2_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(arctan2_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _arctan2_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = atan2(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _arctan2_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int arctan2_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _arctan2_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(arctan2_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  log  *********************/

static int log_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_log(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(log_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  log10  *********************/

static int log10_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_log10(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(log10_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  exp  *********************/

static int exp_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = exp(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(exp_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  sinh  *********************/

static int sinh_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sinh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(sinh_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  cosh  *********************/

static int cosh_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = cosh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(cosh_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  tanh  *********************/

static int tanh_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = tanh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(tanh_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  arcsinh  *********************/

static int arcsinh_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_asinh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arcsinh_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  arccosh  *********************/

static int arccosh_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_acosh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arccosh_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  arctanh  *********************/

static int arctanh_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_atanh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arctanh_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  ieeemask  *********************/

static int ieeemask_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Int32  *tin2  = (Int32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = NA_IeeeSpecial32(tin1, tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(ieeemask_Float32_vector_vector, sizeof(Float32), sizeof(Int32), sizeof(Bool));

/*********************  ieeemask  *********************/
/*********************  sqrt  *********************/

static int sqrt_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sqrt(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(sqrt_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  equal  *********************/

static int equal_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 == *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(equal_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int equal_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 == tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(equal_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int equal_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar == *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(equal_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));

/*********************  not_equal  *********************/

static int not_equal_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 != *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(not_equal_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int not_equal_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 != tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(not_equal_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int not_equal_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar != *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(not_equal_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));

/*********************  greater  *********************/

static int greater_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 > *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(greater_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int greater_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 > tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(greater_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int greater_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar > *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(greater_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));

/*********************  greater_equal  *********************/

static int greater_equal_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 >= *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(greater_equal_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int greater_equal_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 >= tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(greater_equal_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int greater_equal_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar >= *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(greater_equal_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));

/*********************  less  *********************/

static int less_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 < *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(less_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int less_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 < tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(less_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int less_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar < *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(less_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));

/*********************  less_equal  *********************/

static int less_equal_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 <= *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(less_equal_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int less_equal_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 <= tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(less_equal_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int less_equal_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar <= *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(less_equal_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));

/*********************  logical_and  *********************/

static int logical_and_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_and(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(logical_and_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int logical_and_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_and(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(logical_and_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int logical_and_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_and(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(logical_and_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));

/*********************  logical_or  *********************/

static int logical_or_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_or(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(logical_or_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int logical_or_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_or(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(logical_or_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int logical_or_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_or(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(logical_or_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));

/*********************  logical_xor  *********************/

static int logical_xor_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_xor(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(logical_xor_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int logical_xor_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_xor(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(logical_xor_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Bool));


static int logical_xor_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_xor(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(logical_xor_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Bool));

/*********************  logical_not  *********************/

static int logical_not_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Bool *tout = (Bool *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = !*tin;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(logical_not_Float32_vector, sizeof(Float32), sizeof(Bool));
/*********************  bitwise_and  *********************/
/*********************  bitwise_or  *********************/
/*********************  bitwise_xor  *********************/
/*********************  bitwise_not  *********************/
/*********************  bitwise_not  *********************/
/*********************  rshift  *********************/
/*********************  lshift  *********************/
/*********************  floor  *********************/
/*********************  floor  *********************/

static int floor_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = floor(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(floor_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  ceil  *********************/
/*********************  ceil  *********************/

static int ceil_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = ceil(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(ceil_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  maximum  *********************/

static int maximum_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    Float32 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ufmaximum(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(maximum_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int maximum_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    Float32 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ufmaximum(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(maximum_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int maximum_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    Float32 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ufmaximum(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(maximum_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _maximum_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    Float32 temp1, temp2;
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = ufmaximum(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _maximum_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int maximum_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _maximum_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(maximum_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _maximum_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    Float32 temp1, temp2;
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = ufmaximum(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _maximum_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int maximum_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _maximum_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(maximum_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  minimum  *********************/

static int minimum_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    Float32 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ufminimum(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(minimum_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int minimum_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    Float32 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ufminimum(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(minimum_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int minimum_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    Float32 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ufminimum(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(minimum_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _minimum_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    Float32 temp1, temp2;
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = ufminimum(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _minimum_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int minimum_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _minimum_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(minimum_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _minimum_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    Float32 temp1, temp2;
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = ufminimum(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _minimum_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int minimum_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _minimum_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(minimum_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  fabs  *********************/

static int fabs_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = fabs(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(fabs_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  _round  *********************/

static int _round_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_round(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(_round_Float32_vector, sizeof(Float32), sizeof(Float32));
/*********************  hypot  *********************/

static int hypot_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = hypot(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(hypot_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int hypot_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = hypot(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(hypot_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int hypot_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = hypot(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(hypot_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static void _hypot_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = hypot(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _hypot_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int hypot_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _hypot_Float32_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(hypot_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static void _hypot_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = hypot(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _hypot_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int hypot_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _hypot_Float32_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(hypot_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));
/*********************  minus  *********************/
/*********************  add  *********************/
/*********************  subtract  *********************/
/*********************  multiply  *********************/
/*********************  divide  *********************/
/*********************  true_divide  *********************/
/*********************  remainder  *********************/
/*********************  power  *********************/
/*********************  abs  *********************/
/*********************  sin  *********************/
/*********************  cos  *********************/
/*********************  tan  *********************/
/*********************  arcsin  *********************/
/*********************  arccos  *********************/
/*********************  arctan  *********************/
/*********************  arcsinh  *********************/
/*********************  arccosh  *********************/
/*********************  arctanh  *********************/
/*********************  log  *********************/
/*********************  log10  *********************/
/*********************  exp  *********************/
/*********************  sinh  *********************/
/*********************  cosh  *********************/
/*********************  tanh  *********************/
/*********************  sqrt  *********************/
/*********************  equal  *********************/
/*********************  not_equal  *********************/
/*********************  logical_and  *********************/
/*********************  logical_or  *********************/
/*********************  logical_xor  *********************/
/*********************  logical_not  *********************/
/*********************  floor  *********************/
/*********************  ceil  *********************/
/*********************  maximum  *********************/
/*********************  minimum  *********************/
/*********************  fabs  *********************/
/*********************  _round  *********************/
/*********************  hypot  *********************/

static PyMethodDef _ufuncFloat32Methods[] = {

	{NULL,      NULL}        /* Sentinel */
};

static PyObject *init_funcDict(void) {
    PyObject *dict;
    dict = PyDict_New();
    NA_add_cfunc(dict, "('minus', 'vector', (('Float32',), ('Float32',)))", (void *) &minus_Float32_vector_descr);
    NA_add_cfunc(dict, "('add', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &add_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('add', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &add_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('add', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &add_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('add', 'reduce', (('Float32',), ('Float32',)))", (void *) &add_Float32_reduce_descr);
    NA_add_cfunc(dict, "('add', 'accumulate', (('Float32',), ('Float32',)))", (void *) &add_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('subtract', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &subtract_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('subtract', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &subtract_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('subtract', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &subtract_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('subtract', 'reduce', (('Float32',), ('Float32',)))", (void *) &subtract_Float32_reduce_descr);
    NA_add_cfunc(dict, "('subtract', 'accumulate', (('Float32',), ('Float32',)))", (void *) &subtract_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('multiply', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &multiply_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('multiply', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &multiply_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('multiply', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &multiply_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('multiply', 'reduce', (('Float32',), ('Float32',)))", (void *) &multiply_Float32_reduce_descr);
    NA_add_cfunc(dict, "('multiply', 'accumulate', (('Float32',), ('Float32',)))", (void *) &multiply_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('divide', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &divide_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('divide', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &divide_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('divide', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &divide_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('divide', 'reduce', (('Float32',), ('Float32',)))", (void *) &divide_Float32_reduce_descr);
    NA_add_cfunc(dict, "('divide', 'accumulate', (('Float32',), ('Float32',)))", (void *) &divide_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('floor_divide', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &floor_divide_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('floor_divide', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &floor_divide_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('floor_divide', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &floor_divide_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('floor_divide', 'reduce', (('Float32',), ('Float32',)))", (void *) &floor_divide_Float32_reduce_descr);
    NA_add_cfunc(dict, "('floor_divide', 'accumulate', (('Float32',), ('Float32',)))", (void *) &floor_divide_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('true_divide', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &true_divide_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('true_divide', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &true_divide_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('true_divide', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &true_divide_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('true_divide', 'reduce', (('Float32',), ('Float32',)))", (void *) &true_divide_Float32_reduce_descr);
    NA_add_cfunc(dict, "('true_divide', 'accumulate', (('Float32',), ('Float32',)))", (void *) &true_divide_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('remainder', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &remainder_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('remainder', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &remainder_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('remainder', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &remainder_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('remainder', 'reduce', (('Float32',), ('Float32',)))", (void *) &remainder_Float32_reduce_descr);
    NA_add_cfunc(dict, "('remainder', 'accumulate', (('Float32',), ('Float32',)))", (void *) &remainder_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('power', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &power_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('power', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &power_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('power', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &power_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('power', 'reduce', (('Float32',), ('Float32',)))", (void *) &power_Float32_reduce_descr);
    NA_add_cfunc(dict, "('power', 'accumulate', (('Float32',), ('Float32',)))", (void *) &power_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('abs', 'vector', (('Float32',), ('Float32',)))", (void *) &abs_Float32_vector_descr);
    NA_add_cfunc(dict, "('sin', 'vector', (('Float32',), ('Float32',)))", (void *) &sin_Float32_vector_descr);
    NA_add_cfunc(dict, "('cos', 'vector', (('Float32',), ('Float32',)))", (void *) &cos_Float32_vector_descr);
    NA_add_cfunc(dict, "('tan', 'vector', (('Float32',), ('Float32',)))", (void *) &tan_Float32_vector_descr);
    NA_add_cfunc(dict, "('arcsin', 'vector', (('Float32',), ('Float32',)))", (void *) &arcsin_Float32_vector_descr);
    NA_add_cfunc(dict, "('arccos', 'vector', (('Float32',), ('Float32',)))", (void *) &arccos_Float32_vector_descr);
    NA_add_cfunc(dict, "('arctan', 'vector', (('Float32',), ('Float32',)))", (void *) &arctan_Float32_vector_descr);
    NA_add_cfunc(dict, "('arctan2', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &arctan2_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('arctan2', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &arctan2_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('arctan2', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &arctan2_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('arctan2', 'reduce', (('Float32',), ('Float32',)))", (void *) &arctan2_Float32_reduce_descr);
    NA_add_cfunc(dict, "('arctan2', 'accumulate', (('Float32',), ('Float32',)))", (void *) &arctan2_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('log', 'vector', (('Float32',), ('Float32',)))", (void *) &log_Float32_vector_descr);
    NA_add_cfunc(dict, "('log10', 'vector', (('Float32',), ('Float32',)))", (void *) &log10_Float32_vector_descr);
    NA_add_cfunc(dict, "('exp', 'vector', (('Float32',), ('Float32',)))", (void *) &exp_Float32_vector_descr);
    NA_add_cfunc(dict, "('sinh', 'vector', (('Float32',), ('Float32',)))", (void *) &sinh_Float32_vector_descr);
    NA_add_cfunc(dict, "('cosh', 'vector', (('Float32',), ('Float32',)))", (void *) &cosh_Float32_vector_descr);
    NA_add_cfunc(dict, "('tanh', 'vector', (('Float32',), ('Float32',)))", (void *) &tanh_Float32_vector_descr);
    NA_add_cfunc(dict, "('arcsinh', 'vector', (('Float32',), ('Float32',)))", (void *) &arcsinh_Float32_vector_descr);
    NA_add_cfunc(dict, "('arccosh', 'vector', (('Float32',), ('Float32',)))", (void *) &arccosh_Float32_vector_descr);
    NA_add_cfunc(dict, "('arctanh', 'vector', (('Float32',), ('Float32',)))", (void *) &arctanh_Float32_vector_descr);
    NA_add_cfunc(dict, "('ieeemask', 'vector_vector', (('Float32', 'Int32'), ('Bool',)))", (void *) &ieeemask_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('sqrt', 'vector', (('Float32',), ('Float32',)))", (void *) &sqrt_Float32_vector_descr);
    NA_add_cfunc(dict, "('equal', 'vector_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &equal_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('equal', 'vector_scalar', (('Float32', 'Float32'), ('Bool',)))", (void *) &equal_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('equal', 'scalar_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &equal_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('not_equal', 'vector_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &not_equal_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('not_equal', 'vector_scalar', (('Float32', 'Float32'), ('Bool',)))", (void *) &not_equal_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('not_equal', 'scalar_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &not_equal_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('greater', 'vector_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &greater_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('greater', 'vector_scalar', (('Float32', 'Float32'), ('Bool',)))", (void *) &greater_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('greater', 'scalar_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &greater_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('greater_equal', 'vector_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &greater_equal_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('greater_equal', 'vector_scalar', (('Float32', 'Float32'), ('Bool',)))", (void *) &greater_equal_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('greater_equal', 'scalar_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &greater_equal_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('less', 'vector_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &less_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('less', 'vector_scalar', (('Float32', 'Float32'), ('Bool',)))", (void *) &less_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('less', 'scalar_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &less_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('less_equal', 'vector_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &less_equal_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('less_equal', 'vector_scalar', (('Float32', 'Float32'), ('Bool',)))", (void *) &less_equal_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('less_equal', 'scalar_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &less_equal_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_and', 'vector_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &logical_and_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_and', 'vector_scalar', (('Float32', 'Float32'), ('Bool',)))", (void *) &logical_and_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_and', 'scalar_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &logical_and_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_or', 'vector_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &logical_or_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_or', 'vector_scalar', (('Float32', 'Float32'), ('Bool',)))", (void *) &logical_or_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_or', 'scalar_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &logical_or_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_xor', 'vector_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &logical_xor_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_xor', 'vector_scalar', (('Float32', 'Float32'), ('Bool',)))", (void *) &logical_xor_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_xor', 'scalar_vector', (('Float32', 'Float32'), ('Bool',)))", (void *) &logical_xor_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_not', 'vector', (('Float32',), ('Bool',)))", (void *) &logical_not_Float32_vector_descr);
    NA_add_cfunc(dict, "('floor', 'vector', (('Float32',), ('Float32',)))", (void *) &floor_Float32_vector_descr);
    NA_add_cfunc(dict, "('ceil', 'vector', (('Float32',), ('Float32',)))", (void *) &ceil_Float32_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &maximum_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &maximum_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('maximum', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &maximum_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'reduce', (('Float32',), ('Float32',)))", (void *) &maximum_Float32_reduce_descr);
    NA_add_cfunc(dict, "('maximum', 'accumulate', (('Float32',), ('Float32',)))", (void *) &maximum_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('minimum', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &minimum_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('minimum', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &minimum_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('minimum', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &minimum_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('minimum', 'reduce', (('Float32',), ('Float32',)))", (void *) &minimum_Float32_reduce_descr);
    NA_add_cfunc(dict, "('minimum', 'accumulate', (('Float32',), ('Float32',)))", (void *) &minimum_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('fabs', 'vector', (('Float32',), ('Float32',)))", (void *) &fabs_Float32_vector_descr);
    NA_add_cfunc(dict, "('_round', 'vector', (('Float32',), ('Float32',)))", (void *) &_round_Float32_vector_descr);
    NA_add_cfunc(dict, "('hypot', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &hypot_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('hypot', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &hypot_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('hypot', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &hypot_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('hypot', 'reduce', (('Float32',), ('Float32',)))", (void *) &hypot_Float32_reduce_descr);
    NA_add_cfunc(dict, "('hypot', 'accumulate', (('Float32',), ('Float32',)))", (void *) &hypot_Float32_accumulate_descr);
    return dict;
}

/* platform independent*/
#ifdef MS_WIN32
__declspec(dllexport)
#endif
void init_ufuncFloat32(void) {
    PyObject *m, *d;
    m = Py_InitModule("_ufuncFloat32", _ufuncFloat32Methods);
    d = PyModule_GetDict(m);
    _Error = PyErr_NewException("_ufuncFloat32.error", NULL, NULL);
    PyDict_SetItemString(d, "error", _Error);
    import_libnumarray();
    PyDict_SetItemString(d, "functionDict", init_funcDict());
}
