/*
 * Plasma applet to display indicators from libindicate
 *
 * Copyright 2009 Canonical Ltd.
 *
 * Authors:
 * - Aurélien Gâteau <aurelien.gateau@canonical.com>
 *
 * License: GPL v3
 */
// Self
#include "listenermodel.h"

// Qt
#include <QRegExp>
#include <QTime>

// KDE
#include <KDebug>
#include <KDesktopFile>
#include <KIcon>
#include <KIconLoader>

// libindicate
#include <libindicate/indicator-messages.h>

// libindicate-qt
#include <qindicatedecode.h>

// Local

#define USE_ICONS_FOR_SERVERS

enum
{
    ServerRole = Qt::UserRole+1,
    IndicatorRole,
    OutdatedKeyListRole
};

typedef QPair<QIndicate::Listener::Server*, QIndicate::Listener::Indicator*> ServerIndicatorPair;
typedef QHash<ServerIndicatorPair, QStandardItem*> ItemForIndicatorHash;
typedef QSet<QIndicate::Listener::Indicator*> IndicatorSet;
struct ListenerModelPrivate
{
    ListenerModel* q;

    QIndicate::Listener* mListener;
    QRegExp mAcceptedServerType;
    // Indicators we have received, but for which we haven't received a server
    // yet
    QHash<QIndicate::Listener::Server*, IndicatorSet> mWaitingIndicators;
    QHash<QIndicate::Listener::Server*, QStandardItem*> mItemForServer;
    ItemForIndicatorHash mItemForIndicator;

    void updateIndicatorItem(QStandardItem* item)
    {
        QStringList outdatedKeyList = item->data(OutdatedKeyListRole).toStringList();
        Q_ASSERT(!outdatedKeyList.isEmpty());
        QString key = outdatedKeyList.takeFirst();
        item->setData(QVariant(outdatedKeyList), OutdatedKeyListRole);

        QIndicate::Listener::Server* server = item->parent()->data(ServerRole).value<QIndicate::Listener::Server*>();
        QIndicate::Listener::Indicator* indicator = item->data(IndicatorRole).value<QIndicate::Listener::Indicator*>();

        mListener->getIndicatorProperty(server, indicator, key,
                                        q, SLOT(slotPropertyReceived(
                                                QIndicate::Listener::Server*,
                                                QIndicate::Listener::Indicator*,
                                                const QString&,
                                                const QByteArray&))
                                        );
    }

    void removeIndicatorItem(QStandardItem* item) {
        QIndicate::Listener::Server* server = item->parent()->data(ServerRole).value<QIndicate::Listener::Server*>();
        QIndicate::Listener::Indicator* indicator = item->data(IndicatorRole).value<QIndicate::Listener::Indicator*>();
        mItemForIndicator.remove(ServerIndicatorPair(server, indicator));
        QStandardItem* parent = item->parent();
        parent->removeRow(item->row());
    }
};

ListenerModel::ListenerModel(QIndicate::Listener* listener, const QRegExp& acceptedServerType)
: d(new ListenerModelPrivate)
{
    d->q = this;
    d->mListener = listener;
    d->mAcceptedServerType = acceptedServerType;
    connect(d->mListener,
            SIGNAL(serverAdded(QIndicate::Listener::Server*, const QString&)),
            SLOT(slotServerAdded(QIndicate::Listener::Server*, const QString&))
            );
    connect(d->mListener,
            SIGNAL(serverRemoved(QIndicate::Listener::Server*, const QString&)),
            SLOT(slotServerRemoved(QIndicate::Listener::Server*))
            );
    connect(d->mListener,
            SIGNAL(serverCountChanged(QIndicate::Listener::Server*, int)),
            SLOT(slotServerCountChanged(QIndicate::Listener::Server*, int))
            );

    connect(d->mListener,
            SIGNAL(indicatorAdded(QIndicate::Listener::Server*,
                                  QIndicate::Listener::Indicator*)),
            SLOT(slotIndicatorAdded(QIndicate::Listener::Server*,
                                    QIndicate::Listener::Indicator*))
            );

    connect(d->mListener,
            SIGNAL(indicatorRemoved(QIndicate::Listener::Server*,
                                    QIndicate::Listener::Indicator*)),
            SLOT(slotIndicatorRemoved(QIndicate::Listener::Server*,
                                      QIndicate::Listener::Indicator*))
            );

    connect(d->mListener,
            SIGNAL(indicatorModified(QIndicate::Listener::Server*,
                                     QIndicate::Listener::Indicator*,
                                     const QString&)),
            SLOT(slotIndicatorModified(QIndicate::Listener::Server*,
                                     QIndicate::Listener::Indicator*,
                                     const QString&))
            );
}

ListenerModel::~ListenerModel()
{
    delete d;
}

void ListenerModel::slotServerAdded(QIndicate::Listener::Server* server, const QString& type)
{
    if (d->mAcceptedServerType.indexIn(type) == -1) {
        d->mWaitingIndicators.remove(server);
        return;
    }

    if (d->mItemForServer.contains(server)) {
        kWarning() << "We already know about server" << server;
        return;
    }

    QStandardItem* item = new QStandardItem;
    item->setData(QVariant::fromValue(server), ServerRole);
    item->setData(type, ServerTypeRole);
    d->mItemForServer.insert(server, item);
    appendRow(item);

    // Simulate slotIndicatorAdded for waiting indicators
    IndicatorSet set = d->mWaitingIndicators.take(server);
    Q_FOREACH(QIndicate::Listener::Indicator* indicator, set) {
        slotIndicatorAdded(server, indicator);
    }

    d->mListener->getServerDesktopFile(server,
                                       this,
                                       SLOT(slotDesktopFileReceived(
                                            QIndicate::Listener::Server*,
                                            const QByteArray&)
                                            )
                                       );
}


void ListenerModel::slotDesktopFileReceived(QIndicate::Listener::Server* server, const QByteArray& _fileName)
{
    QString fileName = QIndicate::Decode::stringFromValue(_fileName);
    KDesktopFile desktopFile(fileName);
    QString name = desktopFile.readName();
    if (name.isEmpty()) {
        name = fileName.section('/', -1);
    }

    QStandardItem* item = d->mItemForServer.value(server);
    Q_ASSERT(item);
    item->setText(name);

#ifdef USE_ICONS_FOR_SERVERS
    QPixmap icon = KIconLoader::global()->loadIcon(desktopFile.readIcon(),
                                                   KIconLoader::Small,
                                                   0 /* size */,
                                                   KIconLoader::DefaultState,
                                                   QStringList() /* overlays */,
                                                   0L /* path_store */,
                                                   true /* canReturnNull */);
    if (!icon.isNull()) {
        item->setData(QVariant(icon), Qt::DecorationRole);
    }
#endif
}

void ListenerModel::slotServerRemoved(QIndicate::Listener::Server* server)
{
    if (d->mWaitingIndicators.contains(server)) {
        d->mWaitingIndicators.remove(server);
        return;
    }
    QStandardItem* item = d->mItemForServer.value(server);
    if (!item) {
        kWarning() << "No item found for server" << server;
        return;
    }

    // Remove all indicators for this server
    for (int row = item->rowCount() - 1; row >=0; --row) {
        d->removeIndicatorItem(item->child(row));
    }

    // Delete server item
    d->mItemForServer.remove(server);
    removeRow(item->row());
}

void ListenerModel::slotServerCountChanged(QIndicate::Listener::Server* server, int count)
{
    QStandardItem* item = d->mItemForServer.value(server);
    if (!item) {
        kWarning() << "No item found for server" << server;
        return;
    }

    item->setData(QVariant(count), CountRole);
}

void ListenerModel::slotIndicatorAdded(QIndicate::Listener::Server* server,
                                       QIndicate::Listener::Indicator* indicator)
{
    static QVariant outdatedKeyList = QStringList()
        << INDICATE_INDICATOR_MESSAGES_PROP_NAME
        << INDICATE_INDICATOR_MESSAGES_PROP_ICON
        << INDICATE_INDICATOR_MESSAGES_PROP_TIME
        << INDICATE_INDICATOR_MESSAGES_PROP_ATTENTION
        << INDICATE_INDICATOR_MESSAGES_PROP_COUNT;

    QStandardItem* serverItem = d->mItemForServer.value(server);
    if (!serverItem) {
        kWarning() << "We received indicatorAdded() signal before serverAdded() signal!";
        d->mWaitingIndicators[server] << indicator;
        // We will be called again when the server is ready
        return;
    }

    QStandardItem* indicatorItem = new QStandardItem();
    indicatorItem->setData(QVariant::fromValue(indicator), IndicatorRole);
    indicatorItem->setData(outdatedKeyList, OutdatedKeyListRole);
    d->mItemForIndicator.insert(ServerIndicatorPair(server, indicator), indicatorItem);
    serverItem->appendRow(indicatorItem);

    d->updateIndicatorItem(indicatorItem);
}

void ListenerModel::slotIndicatorModified(QIndicate::Listener::Server* server,
                                          QIndicate::Listener::Indicator* indicator,
                                          const QString& key)
{
    QStandardItem* item = d->mItemForIndicator.value(ServerIndicatorPair(server, indicator));
    if (!item) {
        if (!d->mWaitingIndicators.contains(server)) {
            kError() << "Unknown indicator" << indicator;
        }
        return;
    }

    QStringList outdatedKeyList = item->data(OutdatedKeyListRole).toStringList();
    if (outdatedKeyList.contains(key)) {
        return;
    }
    outdatedKeyList << key;
    item->setData(QVariant(outdatedKeyList), OutdatedKeyListRole);
    d->updateIndicatorItem(item);
}

void ListenerModel::slotIndicatorRemoved(QIndicate::Listener::Server* server,
                                         QIndicate::Listener::Indicator* indicator)
{
    if (d->mWaitingIndicators.contains(server)) {
        d->mWaitingIndicators[server].remove(indicator);
        return;
    }
    QStandardItem* item = d->mItemForIndicator.value(ServerIndicatorPair(server, indicator));
    if (!item) {
        kWarning() << "No item for indicator" << indicator;
        return;
    }

    QStandardItem* serverItem = item->parent();
    if (!serverItem) {
        kError() << "Item for indicator" << indicator << "has no parent!";
        return;
    }
    d->removeIndicatorItem(item);
}

void ListenerModel::getProxiesForIndex(const QModelIndex& index,
                                       QIndicate::Listener::Server** server,
                                       QIndicate::Listener::Indicator** indicator) const
{
    Q_ASSERT(server);
    Q_ASSERT(indicator);
    *server = 0;
    *indicator = 0;
    if (!index.isValid()) {
        return;
    }

    if (index.parent().isValid()) {
        *indicator = index.data(IndicatorRole).value<QIndicate::Listener::Indicator*>();
        *server = index.parent().data(ServerRole).value<QIndicate::Listener::Server*>();
    } else {
        *server = index.data(ServerRole).value<QIndicate::Listener::Server*>();
    }
}

void ListenerModel::slotPropertyReceived(QIndicate::Listener::Server* server,
                                         QIndicate::Listener::Indicator* indicator,
                                         const QString& key,
                                         const QByteArray& value)
{
    QStandardItem* item = d->mItemForIndicator.value(ServerIndicatorPair(server, indicator));
    if (!item) {
        kWarning() << "No item for indicator" << indicator;
        return;
    }

    if (key == INDICATE_INDICATOR_MESSAGES_PROP_NAME) {
        item->setText(QIndicate::Decode::stringFromValue(value));
    } else if (key == INDICATE_INDICATOR_MESSAGES_PROP_ICON) {
        QImage image = QIndicate::Decode::imageFromValue(value);
        QPixmap pix = QPixmap::fromImage(image);
        item->setData(KIcon(pix), Qt::DecorationRole);
    } else if (key == INDICATE_INDICATOR_MESSAGES_PROP_TIME) {
        QDateTime dateTime = QIndicate::Decode::dateTimeFromValue(value);
        item->setData(QVariant(dateTime), IndicatorDateTimeRole);
    } else if (key == INDICATE_INDICATOR_MESSAGES_PROP_ATTENTION) {
        QVariant oldAttention = item->data(IndicatorDrawAttentionRole).toBool();
        bool attention = QIndicate::Decode::boolFromValue(value);
        item->setData(QVariant(attention), IndicatorDrawAttentionRole);
        if (oldAttention != attention) {
            emit drawAttentionChanged(indexFromItem(item));
        }
    } else if (key == INDICATE_INDICATOR_MESSAGES_PROP_COUNT) {
        int count = QIndicate::Decode::intFromValue(value);
        item->setData(QVariant(count), CountRole);
    } else {
        kWarning() << "Unhandled key" << key;
    }

    QStringList outdatedKeyList = item->data(OutdatedKeyListRole).toStringList();
    if (!outdatedKeyList.isEmpty()) {
        d->updateIndicatorItem(item);
    }
}

#include "listenermodel.moc"
