/* --------------------------------------------------------------------------
 * module_mixer.c
 * general code for the mixer module. This module doesn't control the
 * mixer device directly. It delegates all actions to the configured or
 * detected mixer module (OSS or ALSA) and keep only generic actions for
 * itself
 *
 * Copyright 2002-2004 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * -------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif


#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "config_manager.h"
#include "module_mixer.h"
#include "module_ossmixer.h"
#include "module_alsamixer.h"
#include "support.h"

struct moddata_mixer {
	int soundsystem;     /* currently active sound system */
	int soundsystemcfg;  /* soundsystem to write to configfile */
} modbase_mixer;

int
mixer_init ()
{
	struct moddata_mixer *base = &modbase_mixer;
	int sid;

	base->soundsystem    = SOUNDSYSTEM_NONE;
	base->soundsystemcfg = SOUNDSYSTEM_AUTO;

	if ((sid = registerCfgSection ("MODULE MIXER")) == 0) {
		print_msg (PBB_ERR, _("Can't register configuration section %s, out of memory."), "MODULE MIXER");
		return E_NOMEM;
	} else {
		registerCfgOptionList (sid, "SoundSystem", TAG_SOUNDSYSTEMCFG, 0,
				N_("'"SOUNDSYSTEM_NONE_NAME"', '"SOUNDSYSTEM_AUTO_NAME"', '"SOUNDSYSTEM_OSS_NAME"' or '"SOUNDSYSTEM_ALSA_NAME"'"),
				SOUNDSYSTEM_NONE_NAME, SOUNDSYSTEM_AUTO_NAME, SOUNDSYSTEM_OSS_NAME, SOUNDSYSTEM_ALSA_NAME, NULL);
	}	
	register_function (QUERYQUEUE, mixer_query);
	register_function (CONFIGQUEUE, mixer_configure);
	return 0;
}

int
mixer_open (struct tagitem *taglist)
{
	struct moddata_mixer *base = &modbase_mixer;
	char *names[] = {SOUNDSYSTEM_NONE_NAME, SOUNDSYSTEM_AUTO_NAME, SOUNDSYSTEM_OSS_NAME, SOUNDSYSTEM_ALSA_NAME};
	int rc = 0, requested, soundsystem, autodetect = 0;
	
	requested = soundsystem = (int) tagfind (taglist, TAG_SOUNDSYSTEMCFG, SOUNDSYSTEM_NONE);
#if defined(WITH_ALSA) && defined(WITH_OSS)
	int fd;
	char *mixerdev;
	
	if (soundsystem == SOUNDSYSTEM_AUTO) {
		soundsystem = SOUNDSYSTEM_NONE;
		autodetect = 1;
		if ((tagfind (taglist, TAG_MIXERINITDELAY, 0)) == 1) {
			print_msg (PBB_WARN, _("MixerInitDelay is not possible with automatic mixer detection; option disabled.\n"));
			tagskip (taglist, TAG_MIXERINITDELAY);
		}
		
		if ((check_devorfile ("/proc/asound", TYPE_DIR)) == E_OK) {
			soundsystem = SOUNDSYSTEM_ALSA;
		} else {
			mixerdev = (char *) tagfind (taglist, TAG_MIXERDEVICE, (long) DEFAULT_MIXER);
			if ((check_devorfile (mixerdev, TYPE_CHARDEV)) == E_OK)
				if ((fd = open (mixerdev, O_RDWR)) >= 0) {
					soundsystem = SOUNDSYSTEM_OSS;
					close (fd);
				}
		}
	}
#endif
	
	switch (soundsystem) {
		case SOUNDSYSTEM_NONE:
		default:
			rc = 0;
			break;
		case SOUNDSYSTEM_AUTO:
#ifndef WITH_OSS
		case SOUNDSYSTEM_OSS:
		/* if OSS is not compiled in, give ALSA a try.
		 * ALSA would do the same with OSS.
		 */
#endif
		case SOUNDSYSTEM_ALSA: /* ALSA */
#ifdef WITH_ALSA
			rc = alsamixer_open (taglist);
			if (!rc || (rc && !autodetect)) {
				base->soundsystem = rc ? SOUNDSYSTEM_NONE : SOUNDSYSTEM_ALSA;
				break;
			}
#endif
#ifdef WITH_OSS
		case SOUNDSYSTEM_OSS: /* OSS */
			rc = ossmixer_open (taglist);
			base->soundsystem = rc ? SOUNDSYSTEM_NONE : SOUNDSYSTEM_OSS;
#endif
			break;
	}

	if (autodetect)
		print_msg (PBB_INFO, _("Soundsystem requested: %s, detected: %s and at least activated: %s.\n"),
				names[requested], names[soundsystem], names[base->soundsystem]);
	else
		print_msg (PBB_INFO, _("Soundsystem requested: %s and at least activated: %s.\n"),
				names[requested], names[base->soundsystem]);
	return rc;
}

int
mixer_close ()
{
	struct moddata_mixer *base = &modbase_mixer;
	
	switch (base->soundsystem) {
		case SOUNDSYSTEM_ALSA:
#ifdef WITH_ALSA
			alsamixer_close ();
			break;
#endif
		case SOUNDSYSTEM_OSS:
#ifdef WITH_OSS
			ossmixer_close ();
			break;
#endif
		default:
			break;
	}
	return 0;
}

int
mixer_exit ()
{
	return 0;
}

void
mixer_query (struct tagitem *taglist)
{
	mixer_handle_tags (MODE_QUERY, taglist);
}

void
mixer_configure (struct tagitem *taglist)
{
	mixer_handle_tags (MODE_CONFIG, taglist);
}

void
mixer_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct moddata_mixer *base = &modbase_mixer;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_SOUNDSYSTEM:
			if (cfgure)	tagerror (taglist, E_NOWRITE); 
			else		taglist->data = base->soundsystem;
			break;
		case TAG_SOUNDSYSTEMCFG:
			/* This tag configures the soundsystem. This change has to written to
			 * the configfile first and become active with next server start.
			 * Changing the soundsystem on the fly is not possible.
			 */
			if (cfgure)	base->soundsystemcfg = taglist->data;
			else		taglist->data = base->soundsystemcfg;
			break;
		}
		taglist++;
	}
}

